public class QuakePredictor {
    final int fftSample = 1<<14;
    final int mid = (1<<24) / 2;
    final double[] hz = new double[]{ 0, 0.004, 0.005, 0.006, 0.008, 0.01, 0.02, 0.05, 0.1, 0.2, 0.5, 1, 2, 5};
    final int maxHour = 2160;

    int sampleRate;
    double[] sitesData;
    int numOfSites;
    double[] result = null;     //forecast method always return this variable.
    
    int sampleOffset = 0;
    double[][][] prevReal = null;
    double[] average = null;
    double averageAll = -1;

    public int init(int sampleRate, int numOfSites, double[] sitesData){
        this.sampleRate = sampleRate;
        this.numOfSites = numOfSites;
        this.sitesData = sitesData;
        this.prevReal = new double[numOfSites][3][fftSample/2];
        this.average = new double[numOfSites];
        this.result = new double[maxHour * numOfSites];
        return 0;
    }

    public double[] forecast(int hour, int[] data, double K, double[] globalQuakes){
        try {
            double[][][] real = new double[numOfSites][3][fftSample];   //Input for FFT. Indices are [sites][channel][sample]
            int numSamples = data.length / numOfSites / 3;
            for (int j = 0; j < numOfSites; j++) {      //for each site
                //prev is data value for previous sample for 3 channels.
                //prevReal is latter half values of real filled in previous sampling.
                double[] prev = hour == 0 ? new double[]{mid, mid, mid} : new double[]{prevReal[j][0][prevReal.length - 1], prevReal[j][1][prevReal.length - 1], prevReal[j][2][prevReal.length - 1]};
                double pred = 0;        //predicted probability for this site and hour.
                int count = 0;          //not used.
                for (int sample = 0; sample < numSamples; sample++) {
                    try {
                        int i = (sample + sampleOffset) % (fftSample / 2);      //sampleOffset numbers of samples is already filled in previous hour.
                        for (int c = 0; c < 3; c++) {       //for each channel.
                            int v = data[j * 3 * numSamples + c * numSamples + sample];
                            if (v < 0) v = (int) prev[c];       //missing value
                            real[j][c][fftSample / 2 + i] = v;  //To make current and previous samples overlapped, the first half will be filled with the latter half of previous values. Fill only the latter half of real here.
                            prev[c] = v;
                        }
                        if (i == (fftSample / 2) - 1) {     //when real is full.
                            double[][] waveFeatures = new double[3][hz.length];
                            for (int c = 0; c < 3; c++) {
                                double[] imag = new double[fftSample];  //image part of FFT
                                //Fill the first half of real and replace prevReal values.
                                for (int ii = 0; ii < fftSample / 2; ii++) {
                                    real[j][c][ii] = prevReal[j][c][ii];
                                    prevReal[j][c][ii] = real[j][c][(fftSample / 2) + ii];
                                }
                                Fft.transform(real[j][c], imag);
                                double sum = 0;
                                int ihz = 0;
                                for (int ii = 0; ii < fftSample / 2; ii++) {
                                    double a = Math.sqrt(real[j][c][ii] * real[j][c][ii] + imag[ii] * imag[ii]);    //amplitude
                                    sum += a;
                                    if (ihz < hz.length && sampleRate * ii >= fftSample * hz[ihz]) {
                                        waveFeatures[c][ihz] = sum;     //sum of amplitude for samples whose frequency are between hz[ihz-1] and hz[ihz]
                                        sum = 0;
                                        ihz++;
                                    }
                                }
                            }
                            
                            if (hour == 0 && sample < fftSample / 2) continue;  //first sampling result is ignored.
                            double[] features = new double[hz.length * 6 + 1];  //features for prediction.
                            for (int c = 0; c < 3; c++) {
                                for (int ii = 0; ii < hz.length; ii++) {
                                    features[c * hz.length + ii] = waveFeatures[c][ii]; //raw value of waveFeatures
                                    features[(3 + c) * hz.length + ii] = waveFeatures[c][ii] / (waveFeatures[(c + 1) % 3][ii] + 1); //ratio value of waveFeatures.
                                }
                            }
                            features[6 * hz.length] = K;

                            pred += Math.min(Math.max(Predictor.predict(features), 1e-6), 1.0);     //predict probability of quake by features. value is normalize by clipping.
                            count++;
                        }
                    } catch (Exception e) { }
                }
                if(count > 0) pred /= count;
                pred = Math.max(pred, averageAll * 0.5);    //avoid too small value.
                average[j] = (average[j] * hour + pred) / (hour + 1);   //average for each sites
                if (hour == 95){    //fill the result by average predicted value of early days.
                    for(int h = hour; h < maxHour; h++) {
                        result[h * numOfSites + j] = average[j];
                    }
                }
                for(int h = hour; h < maxHour; h++){
                    //pred is quake probablity within 3 weeks.
                    //I use pred as it is for 2 weeks and use the middle of pred and average for 4 weeks.
                    double val = h-hour < 14*24 ? pred : h-hour < 28*24 ? (pred + average[j]) / 2 : average[j];
                    //moving average like calculation. (very simplified and not accurate)
                    result[h * numOfSites + j] += (val - result[h * numOfSites + j]) / (10*24);
                }
            }
            //average of all sites and all hours.
            averageAll = 0;
            for (int j = 0; j < numOfSites; j++) averageAll += average[j];
            averageAll /= numOfSites;

            //calculate sample offset for next hour
            sampleOffset = (sampleOffset + numSamples) % (fftSample / 2);
        }
        catch (Exception e) { }
        return result;
    }
}

//Class for predicting the probability of quake within 21days from features.
//The "predict" method is created by offline training.
class Predictor{
    public static double predict(double[] f){return p0(f)+p1(f)+p2(f)+p3(f)+p4(f)+p5(f)+p6(f)+p7(f)+p8(f)+p9(f)+p10(f)+p11(f)+p12(f)+p13(f)+p14(f)+p15(f)+p16(f)+p17(f)+p18(f)+p19(f)+p20(f)+p21(f)+p22(f)+p23(f);}
    static double p0(double[] f){double r=0;if(f[67]<2.54509){if(f[69]<0.53145){if(f[0]<1.40124e+11){if(f[0]<1.39989e+11){if(f[28]<1.39977e+11){if(f[41]<1.71536e+09){if(f[14]<1.40425e+11){if(f[42]<1.00251){if(f[57]<0.126525){r+=0.25;}else{r+=-0.990074;}}else{r+=0.961538;}}else{if(f[56]<1.00855){r+=0.99831;}else{r+=-0.929825;}}}else{if(f[28]<1.39914e+11){if(f[83]<0.549856){r+=0.636364;}else{if(f[0]<1.39807e+11){if(f[37]<1.88113e+08){if(f[2]<1.87881e+07){r+=-0.5;}else{r+=-0;}}else{r+=-0.996654;}}else{r+=-0;}}}else{if(f[40]<2.40022e+09){if(f[0]<1.39502e+11){r+=0.966942;}else{r+=0.25;}}else{if(f[39]<4.82003e+09){if(f[2]<5.3395e+07){r+=-0.972222;}else{r+=-0.25;}}else{r+=0.333333;}}}}}else{if(f[82]<9.24796){if(f[49]<5.09993){if(f[66]<2.34056){if(f[43]<10.0116){if(f[0]<1.37839e+11){if(f[28]<1.40462e+11){r+=0.866667;}else{r+=-0.75;}}else{if(f[14]<1.41045e+11){if(f[52]<4.27042){r+=-0.973551;}else{r+=0.45098;}}else{if(f[70]<1.00582){r+=0.0392157;}else{r+=-0.980519;}}}}else{if(f[29]<7.80425e+06){if(f[14]<1.40164e+11){if(f[0]<1.23576e+11){r+=0.25;}else{r+=0.942029;}}else{r+=-0.636364;}}else{if(f[0]<1.39445e+11){if(f[0]<1.38593e+11){r+=-0.5;}else{r+=-0;}}else{r+=-0.885714;}}}}else{if(f[0]<1.39985e+11){if(f[14]<1.46307e+11){if(f[70]<1.00178){if(f[28]<1.4018e+11){r+=-0.753846;}else{r+=0.276596;}}else{if(f[28]<1.41547e+11){r+=-0.966102;}else{r+=-0;}}}else{r+=0.866667;}}else{if(f[28]<1.40179e+11){if(f[0]<1.39987e+11){if(f[52]<0.791428){r+=-0.764706;}else{r+=0.25;}}else{r+=0.333333;}}else{if(f[15]<2.42757e+06){if(f[2]<3.3168e+06){r+=-0.5;}else{r+=-0;}}else{if(f[40]<1.72333e+09){r+=0.632353;}else{r+=-0.0526316;}}}}}}else{if(f[28]<1.40185e+11){if(f[27]<3.20142e+07){if(f[0]<1.39983e+11){r+=-0;}else{r+=-0.870968;}}else{if(f[28]<1.40018e+11){if(f[29]<3.86606e+07){r+=-0.818182;}else{r+=0.733333;}}else{if(f[14]<1.40151e+11){if(f[15]<1.50152e+07){r+=0.99241;}else{r+=-0;}}else{r+=-0;}}}}else{r+=-0.972222;}}}else{if(f[0]<1.39984e+11){if(f[0]<1.39981e+11){if(f[72]<0.108982){r+=-0.0909091;}else{if(f[52]<0.364929){r+=-0.111111;}else{if(f[29]<4.09202e+06){if(f[1]<8.65852e+06){r+=-0.84;}else{r+=-0.0666667;}}else{r+=-0.975155;}}}}else{if(f[28]<1.4018e+11){if(f[18]<7.17581e+06){if(f[58]<3.97777){if(f[0]<1.39984e+11){r+=-0.910448;}else{r+=-0.25;}}else{r+=-0;}}else{r+=0.4;}}else{if(f[29]<7.34036e+06){if(f[29]<3.16385e+06){if(f[13]<2.7529e+08){r+=-0.6;}else{r+=-0;}}else{if(f[81]<2.63671){r+=-0.333333;}else{r+=0.577465;}}}else{if(f[5]<1.7402e+06){r+=-0;}else{r+=-0.692308;}}}}}else{if(f[69]<0.149209){if(f[54]<0.477296){if(f[12]<9.83865e+07){r+=-0;}else{r+=-0.897436;}}else{if(f[11]<7.10136e+07){if(f[13]<2.08295e+08){if(f[81]<5.83129){r+=-0.647059;}else{r+=0.555556;}}else{if(f[2]<5.05417e+06){r+=0.857143;}else{r+=-0;}}}else{if(f[14]<1.40097e+11){if(f[57]<0.5243){r+=-0;}else{r+=-0.769231;}}else{if(f[7]<2.1928e+07){r+=-0;}else{r+=0.555556;}}}}}else{if(f[28]<1.40179e+11){if(f[68]<0.182041){if(f[21]<1.81939e+07){if(f[2]<3.55235e+06){r+=-0.777778;}else{r+=0.166667;}}else{if(f[82]<11.8044){r+=-0.142857;}else{r+=0.509434;}}}else{if(f[8]<2.70201e+08){r+=-0.851852;}else{r+=0.25;}}}else{if(f[52]<1.16639){if(f[55]<0.789095){if(f[14]<1.40094e+11){r+=0.47644;}else{r+=0.773678;}}else{r+=-0.454545;}}else{if(f[36]<2.88595e+07){if(f[11]<7.62793e+07){r+=0.7;}else{r+=-0.333333;}}else{if(f[1]<5.50635e+06){r+=-0;}else{r+=-0.714286;}}}}}}}}}else{if(f[55]<0.87416){if(f[14]<1.40123e+11){if(f[28]<1.40182e+11){if(f[0]<1.40068e+11){if(f[70]<1.00103){if(f[7]<2.01721e+08){if(f[68]<0.21811){r+=-0.2;}else{r+=-0.981818;}}else{r+=0.538462;}}else{if(f[14]<1.40095e+11){if(f[57]<1.39773){if(f[45]<4.52311){r+=0.583333;}else{r+=-0.555556;}}else{if(f[28]<1.4018e+11){r+=0.6;}else{r+=-0.5;}}}else{if(f[52]<0.748092){if(f[25]<2.38128e+08){r+=-0.285714;}else{r+=0.714286;}}else{if(f[27]<3.37722e+08){r+=-0;}else{r+=0.854167;}}}}}else{if(f[0]<1.40103e+11){if(f[0]<1.40071e+11){if(f[12]<1.33803e+08){r+=0.777778;}else{r+=-0.5;}}else{if(f[14]<1.40057e+11){r+=0.993871;}else{if(f[0]<1.40079e+11){r+=0.5;}else{r+=-0;}}}}else{r+=-0.538462;}}}else{if(f[70]<1.00373){if(f[32]<3.06391e+08){if(f[0]<1.39992e+11){if(f[69]<0.150475){if(f[9]<3.02509e+07){r+=-0.538462;}else{r+=0.62963;}}else{if(f[14]<1.40093e+11){r+=0.52381;}else{r+=0.917972;}}}else{if(f[0]<1.39992e+11){r+=0;}else{if(f[0]<1.39992e+11){r+=0;}else{r+=0.992767;}}}}else{if(f[0]<1.40076e+11){r+=-0.891892;}else{if(f[0]<1.40097e+11){r+=0.971223;}else{r+=0.25;}}}}else{r+=-0.809524;}}}else{if(f[40]<1.7034e+08){if(f[6]<3.30475e+06){r+=-0.25;}else{r+=0.962264;}}else{if(f[67]<0.514294){if(f[14]<1.40173e+11){if(f[54]<0.75961){r+=0.6;}else{r+=-0.692308;}}else{if(f[83]<13.1901){if(f[56]<1.00276){if(f[0]<1.40123e+11){r+=-0.980328;}else{r+=-0.25;}}else{if(f[1]<2.24931e+08){r+=0.25;}else{r+=-0.636364;}}}else{if(f[1]<5.3221e+06){r+=0.25;}else{r+=-0.555556;}}}}else{if(f[56]<1.00038){r+=-0.891892;}else{if(f[11]<1.77475e+09){if(f[22]<4.58822e+07){r+=-0.25;}else{r+=0.862069;}}else{r+=-0.555556;}}}}}}else{if(f[0]<1.40069e+11){if(f[41]<1.10655e+08){if(f[14]<1.40083e+11){r+=-0.764706;}else{if(f[1]<1.9522e+07){r+=0.993893;}else{r+=0.25;}}}else{if(f[15]<1.94825e+07){if(f[54]<2.72778){if(f[12]<5.98217e+07){if(f[7]<3.83727e+07){r+=-0.555556;}else{r+=0.875;}}else{if(f[55]<0.953547){if(f[70]<1.00123){r+=-0.898876;}else{r+=0.591398;}}else{if(f[13]<1.75969e+09){r+=-0.923851;}else{r+=0.315789;}}}}else{if(f[28]<1.40067e+11){if(f[21]<1.2942e+07){r+=0.982979;}else{if(f[5]<3.84293e+06){r+=0.25;}else{r+=-0.666667;}}}else{if(f[52]<1.10813){if(f[81]<1.48085){r+=-0.777778;}else{r+=0.932203;}}else{if(f[21]<5.49543e+07){r+=-0.990741;}else{r+=-0.157895;}}}}}else{if(f[23]<9.8408e+07){if(f[28]<1.40331e+11){r+=-0.928571;}else{if(f[56]<1.00079){if(f[8]<7.74592e+07){r+=-0.666667;}else{r+=0.25;}}else{r+=0.714286;}}}else{if(f[56]<0.993203){r+=-0.833333;}else{if(f[67]<1.05773){if(f[69]<0.163598){r+=-0.777778;}else{r+=0.832853;}}else{if(f[66]<2.34288){r+=-0.851852;}else{r+=0.5;}}}}}}}else{if(f[27]<2.97335e+08){if(f[14]<1.40059e+11){if(f[27]<7.84668e+07){r+=-0.870968;}else{if(f[0]<1.40118e+11){if(f[78]<0.392787){if(f[0]<1.40099e+11){r+=0.714286;}else{r+=-0.6;}}else{if(f[9]<2.70264e+08){r+=0.961342;}else{r+=-0;}}}else{if(f[7]<6.00299e+07){r+=-0.885714;}else{if(f[14]<1.39004e+11){r+=-0.5;}else{r+=0.793103;}}}}}else{if(f[14]<1.40541e+11){if(f[22]<2.02821e+08){if(f[60]<12.7139){if(f[49]<0.5263){r+=-0.25;}else{r+=-0.98125;}}else{r+=-0.111111;}}else{r+=-0;}}else{if(f[35]<3.23799e+08){if(f[28]<1.40337e+11){r+=-0.25;}else{r+=0.993103;}}else{r+=-0.636364;}}}}else{if(f[0]<1.40091e+11){if(f[41]<4.42606e+09){if(f[52]<2.56738){if(f[4]<2.03289e+06){r+=0.25;}else{r+=0.95;}}else{r+=-0.111111;}}else{r+=-0.684211;}}else{if(f[56]<1.00073){if(f[14]<1.39878e+11){r+=0.454545;}else{if(f[14]<1.40062e+11){if(f[0]<1.40092e+11){r+=-0.25;}else{r+=-0.985455;}}else{if(f[43]<2.47797){r+=-0.692308;}else{r+=0.4;}}}}else{if(f[9]<2.08683e+08){if(f[68]<0.30603){r+=-0.555556;}else{if(f[0]<1.40114e+11){r+=0.818182;}else{r+=-0;}}}else{r+=-0.75;}}}}}}}}else{if(f[70]<1.00947){if(f[83]<2.62022){if(f[23]<1.0974e+08){if(f[37]<1.89525e+07){if(f[56]<1.00117){r+=-0.986885;}else{if(f[27]<6.70658e+07){if(f[14]<1.41047e+11){if(f[7]<1.37141e+07){r+=0.555556;}else{r+=-0.555556;}}else{r+=-0.966387;}}else{if(f[28]<1.40378e+11){if(f[14]<1.40335e+11){r+=0.25;}else{r+=0.990719;}}else{r+=-0;}}}}else{if(f[55]<0.939513){if(f[14]<1.40188e+11){if(f[39]<2.36532e+08){if(f[56]<1.0012){if(f[14]<1.39468e+11){r+=0.636364;}else{r+=-0.943662;}}else{r+=0.979487;}}else{if(f[70]<0.996101){if(f[14]<1.39808e+11){r+=-0.333333;}else{r+=0.666667;}}else{if(f[77]<1.2546){r+=-0.909091;}else{r+=-0;}}}}else{if(f[9]<1.29487e+08){if(f[0]<1.40165e+11){if(f[14]<1.40529e+11){r+=-0.555556;}else{r+=0.636364;}}else{if(f[14]<1.4023e+11){r+=-0.611111;}else{r+=-0.97619;}}}else{if(f[0]<1.41672e+11){r+=-0.826087;}else{r+=0.75;}}}}else{if(f[27]<9.91325e+08){if(f[52]<1.30435){if(f[56]<1.00118){if(f[37]<2.08875e+08){r+=-0.936458;}else{r+=0.473684;}}else{if(f[37]<7.66016e+07){r+=0.361257;}else{r+=-0.820225;}}}else{if(f[37]<2.64855e+10){if(f[35]<8.49e+06){r+=-0.391304;}else{r+=-0.967356;}}else{if(f[7]<1.10121e+10){r+=0.714286;}else{r+=-0;}}}}else{if(f[56]<0.997867){r+=0.923077;}else{if(f[54]<1.52441){r+=-0.888889;}else{if(f[14]<1.40058e+11){r+=-0.692308;}else{r+=0.454545;}}}}}}}else{if(f[56]<1.00779){if(f[41]<7.49418e+08){if(f[25]<9.73084e+07){if(f[56]<1.00124){if(f[0]<1.40179e+11){if(f[14]<1.39816e+11){r+=0.866667;}else{r+=-0.75;}}else{if(f[0]<1.41674e+11){r+=-0.797625;}else{r+=0.207101;}}}else{if(f[56]<1.00461){if(f[28]<1.3944e+11){r+=-0.842105;}else{r+=0.738749;}}else{if(f[14]<1.41124e+11){r+=-0.978022;}else{r+=0.894737;}}}}else{if(f[42]<1.0023){if(f[0]<1.41126e+11){if(f[39]<9.48529e+07){r+=0.5;}else{r+=-0.914132;}}else{if(f[1]<1.3297e+08){r+=0.885714;}else{r+=-0;}}}else{if(f[28]<1.40852e+11){if(f[14]<1.38841e+11){r+=-0.876161;}else{r+=0.359635;}}else{if(f[56]<0.991798){r+=-0.818182;}else{r+=0.998309;}}}}}else{if(f[14]<1.4006e+11){if(f[1]<2.20713e+08){if(f[54]<5.95028){if(f[0]<1.40261e+11){r+=-0.714286;}else{r+=-0.9924;}}else{if(f[70]<1.00046){r+=-0.789474;}else{r+=0.75;}}}else{if(f[28]<1.40034e+11){r+=0.684211;}else{r+=-0.6;}}}else{if(f[42]<1.00451){if(f[14]<1.40183e+11){if(f[67]<0.397258){r+=-0.657143;}else{r+=0.606838;}}else{if(f[0]<1.42639e+11){r+=-0.882993;}else{r+=0.875;}}}else{if(f[28]<1.39888e+11){if(f[0]<1.41543e+11){r+=-0.976879;}else{r+=-0;}}else{if(f[14]<1.40431e+11){r+=0.316327;}else{r+=0.867769;}}}}}}else{if(f[14]<1.40279e+11){if(f[0]<1.42692e+11){r+=0.25;}else{r+=-0.6;}}else{r+=-0.9968;}}}}else{if(f[12]<2.19216e+08){if(f[0]<1.40261e+11){if(f[42]<1.00161){if(f[68]<0.351907){if(f[41]<1.54241e+09){if(f[67]<0.442105){if(f[42]<0.997065){r+=-0.208333;}else{r+=-0.937334;}}else{if(f[56]<1.00032){r+=-0.954023;}else{r+=0.529412;}}}else{if(f[14]<1.40187e+11){if(f[0]<1.40154e+11){r+=-0.793103;}else{r+=0.376206;}}else{if(f[1]<2.471e+06){r+=-0.25;}else{r+=-0.961538;}}}}else{if(f[56]<1.00039){if(f[0]<1.40215e+11){r+=-0.980392;}else{r+=-0.25;}}else{if(f[13]<7.39576e+07){r+=-0.764706;}else{if(f[28]<1.40385e+11){r+=0.943548;}else{r+=-0.4;}}}}}else{if(f[14]<1.4001e+11){if(f[56]<0.998706){if(f[28]<1.40156e+11){if(f[70]<0.999854){r+=-0.9375;}else{r+=-0.111111;}}else{if(f[25]<6.03866e+07){r+=-0.777778;}else{r+=0.869565;}}}else{if(f[28]<1.39462e+11){r+=-0.823529;}else{if(f[29]<9.37545e+07){r+=0.965035;}else{r+=-0.5;}}}}else{if(f[70]<0.999304){if(f[15]<1.61564e+07){r+=-0.960784;}else{r+=0.25;}}else{r+=0.764706;}}}}else{if(f[14]<1.40188e+11){if(f[14]<1.40158e+11){if(f[28]<1.39562e+11){if(f[28]<1.39423e+11){r+=-0.733333;}else{if(f[37]<1.69934e+08){r+=-0;}else{r+=0.896552;}}}else{if(f[9]<2.61861e+08){if(f[25]<1.22836e+09){r+=-0.984932;}else{r+=0.4;}}else{if(f[64]<1.14506){r+=0.636364;}else{r+=-0.938776;}}}}else{if(f[81]<2.58423){if(f[0]<1.51501e+11){if(f[39]<2.71132e+08){r+=0.961207;}else{r+=-0.583333;}}else{r+=-0.924528;}}else{if(f[41]<1.47619e+09){if(f[39]<3.10943e+08){r+=-0.176471;}else{r+=-0.816296;}}else{if(f[28]<1.40002e+11){r+=0.649789;}else{r+=-0.859649;}}}}}else{if(f[82]<2.14383){if(f[42]<1.00508){if(f[55]<0.511255){r+=0.454545;}else{if(f[16]<7.34007e+07){r+=-0.947566;}else{r+=-0.230769;}}}else{if(f[0]<1.42885e+11){if(f[28]<1.42021e+11){r+=0.960265;}else{r+=-0.5;}}else{r+=-0.777778;}}}else{if(f[67]<1.18131){if(f[8]<1.10826e+08){if(f[42]<0.992669){r+=-0.25;}else{r+=-0.995253;}}else{if(f[0]<1.41185e+11){r+=-0.97892;}else{r+=-0.127572;}}}else{if(f[78]<0.889923){if(f[69]<0.369408){r+=-0.5;}else{r+=0.789474;}}else{r+=-0.793103;}}}}}}else{if(f[28]<1.4353e+11){if(f[13]<4.40395e+08){if(f[0]<1.4042e+11){if(f[54]<0.804921){if(f[23]<9.04124e+07){r+=-0.666667;}else{if(f[52]<0.858831){r+=0.927273;}else{r+=0.166667;}}}else{if(f[0]<1.4026e+11){if(f[0]<1.40196e+11){r+=-0.75;}else{r+=0.421053;}}else{if(f[25]<2.46373e+08){r+=-0.961538;}else{r+=-0.111111;}}}}else{if(f[71]<0.18956){r+=0.0909091;}else{if(f[81]<0.753188){if(f[1]<3.22642e+07){r+=0.2;}else{r+=-0.714286;}}else{if(f[81]<16.0438){r+=-0.963214;}else{r+=-0.2;}}}}}else{if(f[49]<0.242971){if(f[0]<1.40895e+11){r+=0.647059;}else{r+=-0.906977;}}else{if(f[36]<7.72946e+07){if(f[23]<6.47598e+07){if(f[70]<0.999285){r+=-0.987055;}else{r+=0.230769;}}else{if(f[0]<1.40255e+11){r+=-0.0333333;}else{r+=-0.841808;}}}else{if(f[14]<1.38992e+11){if(f[0]<1.40804e+11){r+=0.25;}else{r+=-0.5;}}else{if(f[52]<0.233867){r+=-0.272727;}else{r+=-0.99772;}}}}}}else{r+=0.952381;}}}}else{if(f[13]<2.708e+08){if(f[0]<1.4033e+11){if(f[0]<1.40191e+11){if(f[1]<3.82858e+07){r+=-0.891892;}else{r+=-0.25;}}else{if(f[6]<3.64662e+07){r+=0.971223;}else{r+=0.25;}}}else{if(f[28]<1.40176e+11){if(f[25]<1.06182e+08){r+=-0.714286;}else{r+=0.764706;}}else{r+=-0.995238;}}}else{if(f[28]<2.28937e+11){if(f[0]<1.40171e+11){if(f[14]<1.40634e+11){r+=-0.945205;}else{r+=0.862069;}}else{if(f[56]<0.97411){if(f[0]<1.40617e+11){if(f[14]<1.40191e+11){r+=0.986159;}else{r+=-0.5;}}else{r+=-0.927273;}}else{if(f[57]<0.128753){r+=-0.555556;}else{if(f[28]<1.42083e+11){r+=-0.25;}else{if(f[1]<3.31717e+09){r+=0.998794;}else{r+=0.25;}}}}}}else{r+=-0.963303;}}}}}else{if(f[14]<1.39854e+11){if(f[14]<1.39531e+11){if(f[28]<1.40056e+11){if(f[28]<1.39614e+11){if(f[14]<1.39491e+11){if(f[27]<4.18503e+10){r+=-0.995489;}else{if(f[33]<3.53979e+09){r+=0.538462;}else{r+=-0.6;}}}else{if(f[0]<1.39764e+11){r+=0.949367;}else{r+=0.25;}}}else{if(f[14]<1.38845e+11){if(f[14]<1.3883e+11){if(f[37]<4.52007e+10){r+=-0.989343;}else{r+=0.2;}}else{if(f[9]<2.75264e+08){r+=-0.789474;}else{if(f[2]<2.06784e+07){r+=-0;}else{r+=0.733333;}}}}else{if(f[27]<1.16192e+08){if(f[28]<1.39814e+11){r+=0.975904;}else{if(f[28]<1.39962e+11){if(f[40]<1.05336e+08){if(f[0]<1.40856e+11){r+=0.8;}else{r+=-0.25;}}else{r+=-0.885714;}}else{r+=-0.989779;}}}else{if(f[29]<7.37493e+07){if(f[28]<1.39689e+11){if(f[0]<1.40186e+11){r+=0.98939;}else{r+=-0.987461;}}else{if(f[35]<4.98448e+07){if(f[12]<6.76775e+07){r+=-0.25;}else{r+=-0.961165;}}else{if(f[9]<2.05894e+08){r+=0.736655;}else{r+=0.974708;}}}}else{if(f[9]<2.3717e+08){if(f[28]<1.39853e+11){r+=0.955056;}else{if(f[36]<1.3414e+08){r+=0.769231;}else{r+=-0.956656;}}}else{if(f[7]<1.92951e+09){if(f[27]<1.36636e+08){r+=-0.333333;}else{r+=0.909091;}}else{if(f[8]<7.2285e+09){r+=-0;}else{r+=-0.636364;}}}}}}}}else{if(f[42]<1.00783){if(f[14]<1.39248e+11){if(f[0]<1.40066e+11){if(f[43]<131.509){if(f[57]<18.1949){r+=-0.998945;}else{if(f[0]<1.39401e+11){r+=-0.6;}else{r+=0.5;}}}else{r+=0.733333;}}else{r+=0.727273;}}else{if(f[56]<0.992738){r+=-0.996494;}else{if(f[28]<1.40147e+11){if(f[66]<0.692713){if(f[25]<1.26481e+08){if(f[0]<1.40202e+11){r+=0.921569;}else{r+=-0.25;}}else{r+=-0.846154;}}else{if(f[33]<6.37307e+07){if(f[28]<1.40144e+11){r+=-0.97654;}else{r+=-0.473684;}}else{if(f[5]<4.79922e+07){r+=0.5;}else{r+=-0.6;}}}}else{if(f[13]<1.31573e+09){if(f[28]<1.40154e+11){if(f[81]<0.828725){r+=-0.444444;}else{r+=0.585366;}}else{if(f[41]<1.66178e+09){r+=0.983054;}else{r+=-0.2;}}}else{if(f[71]<0.0092999){r+=0.882353;}else{if(f[14]<1.39508e+11){r+=-0.842105;}else{r+=0.25;}}}}}}}else{if(f[28]<1.40103e+11){if(f[41]<2.49757e+08){if(f[55]<0.845559){if(f[47]<1.13985){r+=0.5;}else{r+=-0.692308;}}else{r+=-0.994382;}}else{if(f[27]<1.95781e+08){if(f[55]<0.860413){if(f[0]<1.4055e+11){r+=-0.333333;}else{r+=0.666667;}}else{if(f[30]<9.09082e+06){r+=0.111111;}else{if(f[64]<1.10831){r+=-0.931035;}else{r+=-0.0769231;}}}}else{if(f[0]<1.40241e+11){if(f[13]<2.31849e+08){r+=-0.818182;}else{r+=0.25;}}else{if(f[14]<1.38991e+11){if(f[0]<1.40561e+11){r+=0.52809;}else{r+=0.883171;}}else{r+=-0.692308;}}}}}else{if(f[70]<0.990188){if(f[14]<1.39504e+11){if(f[2]<3.25527e+08){if(f[28]<1.40107e+11){r+=0.5;}else{if(f[15]<2.41539e+08){r+=-0.984568;}else{r+=0.25;}}}else{if(f[29]<2.85284e+09){r+=0.777778;}else{r+=-0.5;}}}else{if(f[0]<1.43572e+11){r+=-0.8;}else{if(f[8]<9.78671e+07){r+=0.25;}else{r+=0.978723;}}}}else{if(f[28]<1.40115e+11){if(f[41]<3.56875e+08){if(f[55]<0.895771){if(f[13]<1.81096e+08){r+=-0.68;}else{r+=0.555556;}}else{r+=-0.990385;}}else{if(f[13]<2.07522e+08){if(f[63]<0.998097){r+=-0.727273;}else{r+=0.285714;}}else{if(f[19]<4.47893e+07){r+=0.794872;}else{r+=-0.4;}}}}else{if(f[57]<84.0656){if(f[78]<2.04666e+07){if(f[7]<3.8446e+10){r+=-0.996658;}else{r+=-0.25;}}else{if(f[28]<1.40176e+11){r+=0.3;}else{r+=-0.916667;}}}else{if(f[14]<1.38954e+11){r+=-0.25;}else{r+=0.555556;}}}}}}}}else{if(f[28]<1.39441e+11){if(f[56]<1.00171){r+=0.934426;}else{if(f[28]<1.39428e+11){if(f[55]<0.218963){if(f[14]<1.39839e+11){r+=-0.666667;}else{r+=0.5;}}else{if(f[14]<1.39854e+11){r+=-0.994577;}else{r+=-0.25;}}}else{if(f[14]<1.39824e+11){if(f[54]<0.289772){r+=-0;}else{if(f[14]<1.39817e+11){r+=-0.958333;}else{if(f[0]<1.40269e+11){r+=-0.636364;}else{r+=0.25;}}}}else{if(f[58]<2.18062){if(f[9]<1.29769e+08){if(f[14]<1.39841e+11){r+=-0.636364;}else{r+=0.25;}}else{if(f[30]<4.70433e+07){if(f[11]<9.16056e+07){r+=-0;}else{r+=0.860465;}}else{r+=-0.25;}}}else{if(f[6]<1.05113e+07){r+=-0;}else{r+=-0.692308;}}}}}}else{if(f[22]<4.17886e+08){if(f[66]<0.738929){if(f[42]<1.0062){if(f[25]<6.89479e+07){if(f[68]<0.789972){if(f[29]<8.68055e+07){r+=-0.990654;}else{r+=-0.25;}}else{if(f[37]<1.50802e+08){r+=-0.6;}else{r+=0.272727;}}}else{if(f[56]<0.993111){if(f[70]<1.01804){r+=-0.976331;}else{if(f[0]<1.39287e+11){r+=0.9;}else{r+=0.25;}}}else{if(f[25]<1.0252e+09){if(f[66]<0.269272){r+=-0.48;}else{r+=0.875965;}}else{if(f[0]<1.40352e+11){r+=-0.25;}else{r+=-0.888889;}}}}}else{if(f[35]<2.54426e+07){r+=0.960396;}else{if(f[42]<1.00632){if(f[12]<1.05808e+08){r+=0.6;}else{r+=-0.714286;}}else{if(f[31]<1.39591e+06){r+=-0.25;}else{r+=-0.994924;}}}}}else{if(f[27]<5.1437e+08){if(f[52]<1.69813){if(f[66]<0.963384){if(f[0]<1.40838e+11){if(f[70]<1.01186){r+=0.777009;}else{r+=0.270929;}}else{if(f[56]<0.983987){r+=0.252632;}else{r+=-0.914168;}}}else{if(f[41]<1.5234e+08){if(f[70]<1.01169){r+=0.940562;}else{r+=0.671293;}}else{if(f[0]<1.40835e+11){r+=0.484472;}else{r+=-0.424;}}}}else{if(f[42]<0.996879){if(f[7]<8.23333e+07){if(f[0]<1.3059e+11){r+=-0.555556;}else{r+=0.6;}}else{if(f[0]<1.39323e+11){r+=0.985294;}else{r+=0.25;}}}else{if(f[28]<1.40176e+11){if(f[25]<9.61666e+07){r+=-0.611111;}else{r+=0.777778;}}else{if(f[0]<1.42568e+11){r+=-0.955556;}else{r+=0.571429;}}}}}else{if(f[56]<0.993531){if(f[0]<1.42631e+11){if(f[28]<1.41853e+11){r+=-0.956989;}else{r+=0.666667;}}else{r+=0.692308;}}else{if(f[70]<1.4018e+11){if(f[43]<0.127706){r+=-0.538462;}else{if(f[41]<1.80465e+09){r+=0.986867;}else{r+=0.454545;}}}else{if(f[25]<1.35264e+08){r+=-0.888889;}else{if(f[14]<1.39658e+11){r+=-0.555556;}else{r+=0.5;}}}}}}}else{if(f[28]<1.39793e+11){if(f[28]<1.39453e+11){if(f[14]<1.39797e+11){r+=-0.538462;}else{r+=0.571429;}}else{if(f[15]<8.52055e+08){if(f[70]<1.00001){if(f[4]<4.20749e+06){r+=0.25;}else{r+=0.968504;}}else{r+=0.2;}}else{if(f[17]<2.8241e+08){r+=0.25;}else{r+=-0.555556;}}}}else{if(f[42]<1.02352){if(f[52]<0.30068){if(f[25]<1.53986e+09){r+=-0.6;}else{if(f[0]<1.39607e+11){r+=0.960784;}else{r+=0.25;}}}else{if(f[70]<1.01836){if(f[67]<0.54089){if(f[23]<4.02364e+08){r+=0.545455;}else{r+=-0.8;}}else{if(f[28]<1.41997e+11){r+=-0.984229;}else{r+=-0.333333;}}}else{if(f[12]<7.02657e+08){if(f[0]<1.39293e+11){r+=0.914894;}else{r+=0.25;}}else{r+=-0.5;}}}}else{if(f[5]<7.73903e+08){r+=0.964602;}else{r+=0.25;}}}}}}}else{if(f[28]<1.39894e+11){if(f[14]<1.39877e+11){if(f[28]<1.3942e+11){r+=-0.980099;}else{if(f[28]<1.39607e+11){if(f[28]<1.39427e+11){r+=-0;}else{if(f[11]<6.32485e+08){if(f[69]<0.542268){r+=0.25;}else{r+=0.988604;}}else{r+=0.25;}}}else{r+=-0.777778;}}}else{if(f[12]<1.77963e+08){if(f[73]<312.342){if(f[14]<1.39907e+11){if(f[9]<1.7457e+08){if(f[1]<1.15268e+08){r+=-0.932203;}else{r+=-0.25;}}else{if(f[11]<1.89808e+08){if(f[16]<5.41237e+07){r+=0.777778;}else{r+=-0;}}else{r+=-0.666667;}}}else{if(f[12]<1.59134e+08){if(f[70]<1.00248){r+=-0.999334;}else{r+=-0.25;}}else{if(f[42]<1.00881){if(f[57]<0.0939641){r+=0.5;}else{if(f[41]<4.50802e+08){r+=-0.995893;}else{r+=-0.487179;}}}else{if(f[42]<1.0111){if(f[0]<1.41638e+11){r+=0.25;}else{r+=0.897436;}}else{r+=-0.972414;}}}}}else{if(f[0]<1.4007e+11){r+=-0.75;}else{r+=0.789474;}}}else{if(f[14]<1.40298e+11){if(f[14]<1.40038e+11){if(f[27]<2.23162e+09){if(f[69]<4.25585){if(f[63]<0.417496){if(f[0]<1.39758e+11){r+=-0.6;}else{r+=-0;}}else{r+=-0.996368;}}else{r+=-0;}}else{if(f[14]<1.4002e+11){if(f[1]<3.64715e+08){r+=-0.764706;}else{r+=-0;}}else{r+=0.666667;}}}else{if(f[42]<1.01083){if(f[66]<0.936559){if(f[81]<2.25079){if(f[42]<1.00021){r+=-0.642857;}else{r+=0.960784;}}else{if(f[15]<2.07898e+07){r+=0.538462;}else{r+=-0.947368;}}}else{if(f[0]<1.41453e+11){if(f[14]<1.40088e+11){r+=0.223881;}else{r+=-0.928571;}}else{if(f[0]<1.41555e+11){r+=0.25;}else{r+=0.940298;}}}}else{if(f[70]<0.985186){r+=-0.990971;}else{r+=0.5;}}}}else{if(f[12]<5.18996e+10){if(f[30]<9.78131e+08){if(f[56]<1.00346){if(f[15]<6.60331e+07){r+=-0.692308;}else{r+=0.333333;}}else{if(f[67]<2.53433){r+=-0.993817;}else{r+=-0.25;}}}else{if(f[41]<1.79639e+10){if(f[6]<1.31055e+09){r+=0.5;}else{r+=-0.25;}}else{r+=-0.692308;}}}else{if(f[12]<6.11036e+10){r+=0.636364;}else{r+=-0.555556;}}}}}}else{if(f[28]<1.40121e+11){if(f[0]<1.42872e+11){if(f[14]<1.39991e+11){if(f[83]<0.383198){if(f[41]<1.4122e+08){if(f[14]<1.39983e+11){r+=-0.764706;}else{r+=0.986532;}}else{if(f[54]<1.01036){r+=0.25;}else{if(f[0]<1.40143e+11){r+=-0.961538;}else{r+=-0;}}}}else{if(f[8]<1.43248e+07){if(f[54]<0.956668){r+=0.943662;}else{r+=-0.5;}}else{if(f[83]<0.424322){if(f[40]<8.34758e+07){if(f[11]<1.10453e+08){r+=-0.25;}else{r+=0.857143;}}else{r+=-0.950617;}}else{if(f[12]<3.49559e+10){if(f[22]<1.31898e+07){r+=0.0526316;}else{r+=-0.996764;}}else{r+=0.555556;}}}}}else{if(f[0]<1.40086e+11){if(f[28]<1.4e+11){if(f[14]<1.40505e+11){if(f[0]<1.40072e+11){if(f[27]<1.94171e+09){r+=-0.990442;}else{r+=-0.176471;}}else{if(f[9]<2.39677e+07){r+=-0.764706;}else{r+=0.959732;}}}else{if(f[0]<1.39128e+11){r+=-0.979592;}else{if(f[81]<4.46771){r+=0.994763;}else{r+=-0.666667;}}}}else{if(f[69]<1.05817){if(f[83]<0.891248){if(f[14]<1.4005e+11){r+=-0.905455;}else{r+=0.626877;}}else{if(f[56]<1.00045){r+=0.386201;}else{r+=-0.84326;}}}else{if(f[0]<1.39843e+11){if(f[14]<1.40094e+11){r+=0.2;}else{r+=-0.99548;}}else{if(f[36]<3.03142e+07){r+=0.94941;}else{r+=0.591729;}}}}}else{if(f[0]<1.40145e+11){if(f[28]<1.39952e+11){if(f[14]<1.40018e+11){r+=-0.945205;}else{if(f[14]<1.40094e+11){r+=0.996979;}else{r+=-0.84;}}}else{if(f[29]<2.54322e+07){if(f[14]<1.40166e+11){r+=-0.980949;}else{r+=-0.0244821;}}else{if(f[14]<1.40079e+11){r+=0.573333;}else{r+=-0.75;}}}}else{if(f[8]<2.43969e+08){if(f[41]<7.94246e+07){if(f[36]<7.34358e+06){r+=0.515982;}else{r+=-0.984724;}}else{if(f[28]<1.39986e+11){r+=0.797076;}else{r+=0.368667;}}}else{if(f[66]<0.301183){if(f[14]<1.40086e+11){r+=0.844926;}else{r+=-0.146747;}}else{if(f[42]<1.00954){r+=-0.69625;}else{r+=0.582697;}}}}}}}else{if(f[71]<0.108231){if(f[0]<1.43825e+11){r+=0.870968;}else{r+=-0.333333;}}else{if(f[70]<0.980641){if(f[28]<1.40118e+11){if(f[70]<0.980562){if(f[39]<6.11251e+07){r+=-0.2;}else{r+=-0.99839;}}else{r+=-0;}}else{if(f[14]<1.41246e+11){r+=0.666667;}else{r+=-0.904762;}}}else{r+=0.111111;}}}}else{if(f[0]<1.40162e+11){if(f[81]<1.25322){if(f[56]<1.03669){if(f[28]<1.41467e+11){if(f[63]<0.354984){if(f[0]<1.40095e+11){if(f[0]<1.4005e+11){r+=-0.688172;}else{r+=0.967277;}}else{if(f[14]<1.40089e+11){r+=0.52183;}else{r+=-0.666179;}}}else{if(f[28]<1.40142e+11){if(f[0]<1.40094e+11){r+=0.0360713;}else{r+=-0.83865;}}else{if(f[56]<1.00127){r+=-0.968008;}else{r+=-0.625959;}}}}else{if(f[28]<1.41729e+11){if(f[14]<1.40108e+11){r+=-0.971014;}else{if(f[14]<1.40973e+11){r+=0.982984;}else{r+=-0.789474;}}}else{if(f[36]<9.34949e+07){r+=-0.996212;}else{if(f[25]<1.16425e+08){r+=0.853659;}else{r+=-0.703704;}}}}}else{if(f[14]<1.51997e+11){if(f[28]<1.41229e+11){r+=-0.777778;}else{r+=0.999208;}}else{r+=-0.997828;}}}else{if(f[28]<1.40223e+11){if(f[63]<0.362378){if(f[14]<1.40238e+11){if(f[68]<0.736081){if(f[14]<1.40093e+11){r+=0.487463;}else{r+=-0.674242;}}else{if(f[14]<1.40168e+11){r+=0.653759;}else{r+=0.992523;}}}else{if(f[11]<4.61758e+09){r+=-0.99322;}else{r+=0.25;}}}else{if(f[28]<1.40141e+11){if(f[14]<1.40092e+11){if(f[14]<1.40072e+11){r+=-0.590778;}else{r+=0.748517;}}else{if(f[0]<1.4009e+11){r+=0.812734;}else{r+=-0.944149;}}}else{if(f[41]<3.38983e+09){if(f[66]<1.96063){r+=-0.939838;}else{r+=0.623529;}}else{if(f[14]<1.40103e+11){r+=0.664865;}else{r+=-0.752212;}}}}}else{if(f[82]<2.82957){if(f[64]<0.77105){if(f[14]<1.40064e+11){if(f[0]<1.40006e+11){r+=-0.976744;}else{r+=0.642325;}}else{if(f[0]<1.39225e+11){r+=0.914286;}else{r+=-0.837735;}}}else{if(f[14]<1.40872e+11){if(f[25]<7.49182e+07){r+=-0.389423;}else{r+=0.746959;}}else{if(f[1]<3.65106e+08){r+=-0.976471;}else{r+=-0;}}}}else{if(f[54]<0.564514){if(f[27]<7.18778e+07){if(f[0]<1.39989e+11){r+=-0.955056;}else{r+=0.882353;}}else{if(f[70]<1.00601){r+=0.981595;}else{r+=-0.607143;}}}else{if(f[0]<1.40001e+11){if(f[0]<1.39169e+11){r+=0.692308;}else{r+=-0.962963;}}else{if(f[0]<1.40075e+11){r+=0.92;}else{r+=-0.6;}}}}}}}else{if(f[0]<1.4028e+11){if(f[25]<5.96164e+07){if(f[25]<5.54742e+07){if(f[14]<1.41332e+11){if(f[43]<8.38561){if(f[83]<2.45123){r+=-0.979071;}else{r+=-0.25;}}else{r+=-0.0909091;}}else{r+=0.5;}}else{if(f[28]<1.40147e+11){if(f[0]<1.4017e+11){r+=0.953488;}else{r+=0.25;}}else{if(f[28]<1.61595e+11){if(f[56]<1.00056){r+=-0.98524;}else{r+=0.538462;}}else{r+=0.862069;}}}}else{if(f[78]<0.556379){if(f[28]<1.42035e+11){if(f[56]<0.999426){if(f[67]<0.546303){r+=-0.25;}else{r+=-0.979781;}}else{if(f[28]<1.40145e+11){r+=0.964497;}else{r+=-0.447099;}}}else{r+=0.993475;}}else{if(f[28]<1.42013e+11){if(f[56]<0.999252){if(f[68]<0.509514){r+=-0.581395;}else{r+=-0.988251;}}else{if(f[27]<1.00326e+08){r+=-0.712737;}else{r+=0.917434;}}}else{if(f[14]<1.39987e+11){r+=0.25;}else{if(f[15]<1.30586e+08){r+=0.999355;}else{r+=0.25;}}}}}}else{if(f[14]<1.40875e+11){if(f[28]<1.40916e+11){if(f[0]<1.41582e+11){if(f[29]<1.69873e+08){if(f[70]<0.989901){r+=-0.125;}else{r+=-0.990149;}}else{if(f[66]<0.515133){r+=0.692308;}else{r+=-0.672727;}}}else{if(f[28]<1.40223e+11){if(f[14]<1.4011e+11){r+=-0.794872;}else{r+=0.638158;}}else{if(f[42]<1.01019){r+=0.166667;}else{r+=-0.893939;}}}}else{if(f[70]<1.00227){if(f[42]<1.00254){if(f[14]<1.40423e+11){r+=0.571429;}else{r+=-0.955056;}}else{if(f[8]<5.56486e+07){r+=-0.692308;}else{r+=0.996086;}}}else{if(f[0]<1.40312e+11){if(f[28]<1.41678e+11){r+=-0.971831;}else{r+=0.989591;}}else{if(f[0]<1.40337e+11){r+=-0.372727;}else{r+=-0.982502;}}}}}else{if(f[70]<1.00003){if(f[14]<1.41457e+11){if(f[0]<1.42929e+11){if(f[27]<1.25167e+09){r+=0.98702;}else{r+=-0.376812;}}else{if(f[56]<1.00791){r+=0.755102;}else{r+=-0.967742;}}}else{if(f[52]<0.53681){if(f[57]<1.99748){r+=-0.555556;}else{r+=0.818182;}}else{if(f[0]<1.40392e+11){r+=0.555556;}else{r+=-0.98696;}}}}else{if(f[37]<4.17513e+08){if(f[0]<1.42312e+11){if(f[68]<0.585547){r+=-0.555556;}else{r+=-0.987519;}}else{r+=0.952941;}}else{if(f[40]<1.6139e+09){if(f[0]<1.42408e+11){r+=-0.927711;}else{r+=0.555556;}}else{if(f[56]<1.00292){r+=0.884101;}else{r+=-0.75;}}}}}}}}}}}}else{if(f[56]<0.768903){if(f[0]<1.40131e+11){r+=-0.936508;}else{if(f[0]<1.4067e+11){r+=0.997175;}else{r+=-0.5;}}}else{if(f[14]<1.41212e+11){if(f[41]<1.61572e+09){if(f[66]<2.18241){if(f[13]<2.31478e+08){if(f[27]<9.31762e+08){if(f[12]<3.74117e+08){if(f[28]<1.41385e+11){if(f[52]<0.161925){r+=0.636364;}else{if(f[14]<1.41084e+11){if(f[12]<2.69902e+08){r+=-0.975745;}else{r+=-0.416667;}}else{if(f[0]<1.40341e+11){r+=-0.6;}else{r+=0.555556;}}}}else{if(f[14]<1.40227e+11){r+=0.833333;}else{r+=-0.789474;}}}else{if(f[9]<5.03774e+07){r+=0.809524;}else{r+=-0.6;}}}else{if(f[14]<1.39949e+11){if(f[28]<1.3943e+11){r+=-0.25;}else{if(f[31]<5.20552e+07){r+=0.977143;}else{r+=-0;}}}else{if(f[7]<2.19074e+07){r+=-0.25;}else{r+=-0.983264;}}}}else{if(f[28]<1.40103e+11){if(f[14]<1.40096e+11){if(f[68]<2.98196){if(f[56]<1.0004){if(f[64]<0.744395){r+=-0.857143;}else{if(f[11]<2.20502e+08){r+=-0.285714;}else{r+=0.853659;}}}else{r+=-0.946667;}}else{if(f[41]<1.04058e+08){if(f[0]<1.40002e+11){r+=0.636364;}else{if(f[1]<6.21175e+07){r+=-0.918367;}else{r+=-0.25;}}}else{if(f[28]<1.39444e+11){if(f[13]<2.4309e+08){r+=-0;}else{r+=-0.875;}}else{if(f[23]<1.28561e+09){r+=0.966535;}else{r+=-0.793103;}}}}}else{if(f[21]<8.70422e+06){if(f[0]<1.4044e+11){r+=-0.25;}else{r+=0.733333;}}else{if(f[66]<1.33928){if(f[14]<1.40136e+11){r+=0.636364;}else{r+=-0.25;}}else{if(f[4]<633292){r+=0.0909091;}else{if(f[69]<1.68038){r+=-0.47619;}else{r+=-0.935484;}}}}}}else{if(f[28]<1.40115e+11){if(f[42]<0.99998){if(f[82]<0.345602){if(f[1]<8.64112e+06){r+=0.934426;}else{r+=0.25;}}else{if(f[54]<1.03441){r+=0.692308;}else{r+=-0.828571;}}}else{if(f[0]<1.40132e+11){r+=-0.956522;}else{r+=-0.25;}}}else{if(f[37]<2.10067e+07){if(f[41]<7.48724e+07){r+=-0.6;}else{r+=0.73913;}}else{if(f[70]<1.20299){if(f[0]<1.41711e+11){if(f[57]<8.20053){r+=-0.968683;}else{r+=-0.142857;}}else{if(f[48]<0.75181){r+=-0.777778;}else{r+=0.4;}}}else{r+=0.666667;}}}}}}else{if(f[22]<5.13908e+08){if(f[71]<0.501646){if(f[69]<2.70378){if(f[28]<1.41419e+11){if(f[14]<1.41004e+11){if(f[40]<6.35922e+08){if(f[37]<9.29505e+07){r+=-0.955801;}else{r+=-0.769424;}}else{if(f[1]<5.80254e+07){r+=0.714286;}else{r+=-0.666667;}}}else{if(f[70]<0.996938){r+=0.8;}else{r+=-0.928571;}}}else{if(f[28]<1.41998e+11){r+=0.888889;}else{r+=-0.882353;}}}else{if(f[41]<8.9676e+08){if(f[66]<2.42729){if(f[41]<9.31158e+07){r+=-0.98961;}else{if(f[23]<5.6075e+07){r+=0.756098;}else{r+=-0.92381;}}}else{if(f[37]<1.19565e+08){if(f[17]<52191.4){r+=-0.25;}else{r+=-0.995091;}}else{if(f[70]<1.01612){r+=-0.847059;}else{r+=0.2;}}}}else{if(f[37]<3.4601e+07){r+=0.8;}else{r+=-0.833333;}}}}else{if(f[13]<2.5403e+08){if(f[28]<1.63331e+11){if(f[29]<1.2387e+08){if(f[14]<1.41146e+11){if(f[28]<1.3974e+11){r+=-0.404255;}else{r+=-0.969305;}}else{if(f[1]<6.01932e+07){r+=-0;}else{r+=0.5;}}}else{r+=0.6;}}else{r+=0.851852;}}else{if(f[22]<3.57656e+07){if(f[41]<1.08152e+08){if(f[0]<1.40224e+11){if(f[67]<2.83272){r+=-0.0909091;}else{r+=-0.898305;}}else{r+=0.538462;}}else{if(f[82]<0.344877){if(f[82]<0.095621){r+=-0.444444;}else{r+=0.793103;}}else{if(f[11]<1.551e+08){r+=0.4;}else{r+=-0.806452;}}}}else{if(f[28]<1.40117e+11){if(f[28]<1.40065e+11){if(f[56]<0.999787){r+=0.551724;}else{r+=-0.841191;}}else{if(f[0]<1.4015e+11){r+=-0.371429;}else{r+=0.917808;}}}else{if(f[69]<0.60508){r+=0.647059;}else{if(f[0]<1.39184e+11){r+=0.444444;}else{r+=-0.961178;}}}}}}}else{if(f[14]<1.4109e+11){if(f[70]<1.01633){if(f[42]<1.00017){if(f[69]<1.59012){if(f[56]<1.00479){if(f[5]<1.06946e+07){r+=0.5;}else{r+=-0.857143;}}else{r+=0.4;}}else{if(f[13]<7.90376e+09){if(f[57]<0.333547){r+=-0.25;}else{r+=-0.993697;}}else{r+=-0.2;}}}else{if(f[42]<1.00454){if(f[28]<1.40017e+11){if(f[14]<1.40029e+11){r+=0.733333;}else{r+=-0.8;}}else{if(f[69]<2.41055){r+=0.00917431;}else{r+=-0.852273;}}}else{if(f[37]<2.09903e+10){if(f[31]<2.79329e+08){r+=-0.875776;}else{r+=0.0666667;}}else{if(f[0]<1.43273e+11){r+=-0;}else{r+=0.764706;}}}}}else{if(f[70]<1.01803){if(f[1]<1.6287e+08){if(f[0]<1.39176e+11){r+=0.942857;}else{r+=0.25;}}else{r+=-0.25;}}else{if(f[14]<1.39752e+11){r+=0.2;}else{r+=-0.931035;}}}}else{if(f[28]<1.40735e+11){if(f[28]<1.40468e+11){r+=-0;}else{r+=0.965517;}}else{r+=-0.891892;}}}}}else{if(f[35]<1.00224e+08){if(f[14]<1.40178e+11){if(f[14]<1.40063e+11){if(f[2]<9.59555e+06){r+=-0.862069;}else{r+=-0;}}else{if(f[0]<1.39991e+11){if(f[57]<1.71412){if(f[6]<1.87003e+06){r+=0.6;}else{if(f[48]<0.741869){r+=-0.8;}else{if(f[36]<7.67809e+07){r+=-0.375;}else{r+=0.538462;}}}}else{if(f[11]<1.14432e+09){if(f[15]<2.03874e+07){r+=0.769231;}else{r+=-0;}}else{r+=-0.25;}}}else{if(f[52]<0.79421){if(f[55]<0.226844){if(f[5]<2.44117e+06){r+=0.555556;}else{r+=-0;}}else{if(f[14]<1.4008e+11){r+=0.25;}else{if(f[15]<1.23871e+06){r+=0.25;}else{r+=0.98494;}}}}else{if(f[2]<6.18497e+06){r+=-0;}else{r+=0.5;}}}}}else{if(f[2]<1.0899e+08){r+=-0.878788;}else{r+=-0.25;}}}else{if(f[28]<1.40222e+11){if(f[28]<1.40181e+11){if(f[28]<1.40101e+11){if(f[42]<1.00593){if(f[82]<0.0900894){r+=-0.73913;}else{if(f[0]<1.39589e+11){if(f[81]<0.184873){r+=0.333333;}else{r+=-0.636364;}}else{if(f[17]<6.93905e+06){r+=0.0588235;}else{r+=0.851852;}}}}else{if(f[47]<3.08292){if(f[14]<1.38947e+11){r+=0.111111;}else{r+=-0.918367;}}else{r+=0.333333;}}}else{if(f[77]<0.411306){if(f[1]<2.82607e+08){r+=0.5;}else{r+=-0.25;}}else{if(f[37]<3.16069e+10){if(f[30]<2.34692e+06){r+=-0.25;}else{r+=-0.947368;}}else{r+=-0.111111;}}}}else{if(f[2]<6.49697e+07){if(f[2]<1.28498e+06){r+=0.25;}else{r+=0.925926;}}else{r+=-0.0909091;}}}else{if(f[30]<5.0316e+08){if(f[0]<1.39069e+11){r+=-0;}else{if(f[12]<2.72501e+08){r+=-0;}else{r+=-0.960265;}}}else{if(f[18]<4.12553e+08){r+=0.5;}else{if(f[48]<0.581844){r+=0.25;}else{r+=-0.692308;}}}}}}}else{if(f[67]<4.42616){if(f[56]<1.00958){if(f[28]<1.40754e+11){if(f[0]<1.40266e+11){if(f[11]<5.13999e+07){r+=-0.75;}else{r+=-0;}}else{if(f[1]<3.17512e+08){r+=0.998915;}else{if(f[5]<4.6932e+07){r+=-0;}else{r+=0.5;}}}}else{if(f[28]<1.43352e+11){if(f[0]<1.42917e+11){r+=-0.980861;}else{r+=0.5;}}else{r+=0.946667;}}}else{if(f[29]<7.08422e+06){if(f[27]<1.68906e+08){if(f[0]<1.40382e+11){r+=-0.916667;}else{r+=0.5;}}else{if(f[55]<0.789945){if(f[14]<1.45977e+11){r+=-0.647059;}else{r+=0.4;}}else{if(f[28]<1.41289e+11){if(f[64]<1.04285){r+=0.25;}else{r+=0.965517;}}else{r+=-0.6;}}}}else{if(f[77]<0.102503){if(f[56]<1.02918){r+=-0.4;}else{r+=0.666667;}}else{if(f[1]<2.33135e+09){if(f[15]<6.73408e+06){if(f[15]<4.34568e+06){r+=-0.5;}else{r+=0.25;}}else{r+=-0.986517;}}else{r+=0.4;}}}}}else{if(f[67]<4.97751){if(f[55]<0.572866){if(f[0]<1.4103e+11){if(f[7]<5.65113e+07){r+=0.555556;}else{r+=-0.75;}}else{if(f[6]<1.5535e+07){r+=0.25;}else{r+=0.888889;}}}else{if(f[35]<8.02919e+06){r+=0.555556;}else{if(f[82]<0.923943){if(f[18]<1.73263e+08){r+=-0.955556;}else{if(f[18]<2.31354e+08){r+=0.25;}else{r+=-0.555556;}}}else{if(f[11]<6.50466e+08){r+=0.555556;}else{r+=-0.25;}}}}}else{if(f[0]<1.42194e+11){if(f[68]<0.986245){if(f[2]<3.2056e+07){r+=0.6;}else{r+=-0.666667;}}else{if(f[8]<1.10526e+07){if(f[9]<9.73833e+06){r+=-0.5;}else{r+=0.5;}}else{if(f[67]<5.6597){if(f[55]<0.533334){if(f[78]<0.410556){r+=0.647059;}else{r+=-0.7;}}else{if(f[9]<5.36152e+07){r+=0.25;}else{r+=-0.970588;}}}else{if(f[9]<1.33213e+10){if(f[32]<1.44527e+08){r+=-0.991597;}else{r+=-0.25;}}else{if(f[0]<1.41725e+11){r+=-0;}else{r+=-0.5;}}}}}}else{if(f[0]<1.43593e+11){r+=0.833333;}else{r+=-0.875;}}}}}}} return r;}
    static double p1(double[] f){double r=0;if(f[40]<6.05128e+07){if(f[28]<1.40254e+11){if(f[41]<8.4369e+07){if(f[41]<8.0205e+07){if(f[14]<1.38722e+11){r+=1.2984;}else{if(f[41]<6.29445e+07){if(f[28]<1.4011e+11){r+=-0.685622;}else{if(f[14]<1.413e+11){if(f[67]<2.54408){if(f[28]<1.40121e+11){if(f[35]<5.85983e+06){r+=-1.34089;}else{r+=-0.259301;}}else{if(f[13]<2.29824e+07){r+=-0.152082;}else{if(f[42]<0.993526){r+=-0.763043;}else{r+=-0.9881;}}}}else{if(f[1]<1.93327e+08){r+=-0.685923;}else{r+=0.0224297;}}}else{r+=0.247461;}}}else{if(f[36]<5.6979e+06){if(f[14]<1.41242e+11){if(f[70]<0.997712){if(f[28]<1.3984e+11){r+=-0.657219;}else{r+=0.735869;}}else{r+=-0.808176;}}else{r+=0.76205;}}else{if(f[43]<11.6647){if(f[51]<6.65234){if(f[35]<2.68925e+06){r+=0.127833;}else{if(f[70]<1.00033){if(f[35]<9.61608e+06){r+=-0.437271;}else{r+=-0.695582;}}else{if(f[42]<0.999184){r+=-0.688488;}else{r+=0.582608;}}}}else{r+=0.266145;}}else{if(f[45]<4.07823){r+=-0.441347;}else{r+=0.68706;}}}}}}else{if(f[42]<0.999957){if(f[84]<0.85){if(f[1]<3.58034e+07){r+=-0.646282;}else{r+=-0.0960669;}}else{if(f[28]<1.40114e+11){if(f[28]<1.39916e+11){r+=-0.527266;}else{if(f[8]<2.9552e+06){r+=0.141912;}else{if(f[74]<16.9404){if(f[9]<7.87275e+07){r+=0.718027;}else{r+=0.234194;}}else{r+=0.178839;}}}}else{if(f[43]<2.98806){r+=-0.684456;}else{r+=0.449834;}}}}else{if(f[42]<1.00301){if(f[51]<0.631065){if(f[14]<1.4014e+11){r+=-0.205364;}else{r+=0.477546;}}else{if(f[36]<2.86085e+06){r+=0.23707;}else{if(f[50]<0.583388){r+=0.056885;}else{if(f[21]<3.61767e+06){r+=-0.0460569;}else{if(f[43]<13.5846){if(f[37]<6.06263e+06){r+=-0.191329;}else{r+=-0.697177;}}else{r+=-0.0496499;}}}}}}else{if(f[15]<2.12834e+07){r+=0.65427;}else{r+=-0.419564;}}}}}else{if(f[14]<1.40334e+11){if(f[42]<1.00317){if(f[38]<5.42965e+07){if(f[51]<0.770065){if(f[84]<3.15){if(f[77]<1.55094){if(f[42]<1.00161){if(f[65]<1.17755){if(f[84]<1.5){r+=-0.50953;}else{r+=0.375828;}}else{r+=-0.673889;}}else{r+=0.521057;}}else{if(f[38]<4.94705e+07){r+=-0.087961;}else{r+=0.579288;}}}else{r+=0.662214;}}else{if(f[52]<6.20189){if(f[69]<0.782136){if(f[28]<1.39943e+11){r+=-0.612109;}else{r+=-1.16136;}}else{if(f[42]<0.999434){if(f[14]<1.40149e+11){if(f[10]<6.67294e+07){r+=-0.307466;}else{r+=0.693523;}}else{r+=-0.578047;}}else{if(f[67]<0.803415){if(f[35]<1.43328e+07){r+=0.542663;}else{r+=-0.626555;}}else{if(f[52]<0.693439){r+=0.021113;}else{r+=-0.679389;}}}}}else{r+=0.568876;}}}else{if(f[42]<0.999951){if(f[28]<1.40039e+11){if(f[53]<0.825574){r+=-0.144536;}else{r+=-0.753615;}}else{if(f[28]<1.40167e+11){if(f[84]<1){r+=-0.109234;}else{r+=0.738655;}}else{r+=-0.583987;}}}else{if(f[70]<0.999852){if(f[52]<1.20828){if(f[25]<5.84532e+07){r+=-0.503097;}else{if(f[26]<8.6596e+07){if(f[14]<1.40085e+11){r+=0.0774255;}else{r+=0.687898;}}else{r+=-0.0608395;}}}else{if(f[49]<3.29649){r+=-0.809254;}else{r+=-0.102419;}}}else{if(f[31]<7.04078e+06){r+=-0.69847;}else{r+=-0.203181;}}}}}else{if(f[14]<1.40135e+11){r+=-0.0482092;}else{r+=0.707923;}}}else{if(f[84]<1.15){if(f[70]<0.997036){if(f[7]<1.01525e+08){r+=0.731908;}else{r+=-0.0485086;}}else{r+=-0.551535;}}else{r+=-0.715612;}}}}else{if(f[84]<0.85){if(f[13]<7.31141e+07){if(f[14]<1.41258e+11){if(f[10]<1.07727e+08){r+=-0.702024;}else{if(f[14]<1.39919e+11){r+=0.294834;}else{r+=-0.472313;}}}else{if(f[10]<3.81209e+07){r+=-0.588471;}else{r+=0.589016;}}}else{if(f[67]<7.68229){if(f[36]<1.14134e+08){if(f[67]<1.39671){if(f[28]<1.4125e+11){if(f[28]<1.40364e+11){r+=0.655375;}else{if(f[14]<1.39844e+11){r+=-1.33475;}else{r+=-0.67929;}}}else{if(f[14]<1.40434e+11){r+=0.700038;}else{if(f[53]<1.04386){r+=-0.572633;}else{if(f[33]<2.09324e+07){if(f[28]<1.42306e+11){r+=0.650664;}else{r+=0.122233;}}else{if(f[9]<1.22307e+08){r+=-0.251651;}else{r+=0.246198;}}}}}}else{if(f[25]<2.61823e+08){if(f[10]<3.03948e+07){r+=-0.525339;}else{if(f[23]<5.22019e+06){r+=-0.34633;}else{if(f[82]<0.895102){if(f[11]<3.32871e+08){r+=0.67807;}else{r+=0.0889375;}}else{if(f[61]<2.79532){r+=-0.400565;}else{r+=0.249993;}}}}}else{if(f[1]<2.96202e+07){r+=0.437377;}else{r+=-0.644055;}}}}else{if(f[42]<1.00557){if(f[14]<1.3985e+11){r+=-1.2747;}else{if(f[53]<1.23152){r+=-0.65259;}else{if(f[28]<1.41304e+11){r+=-0.595723;}else{if(f[28]<1.42318e+11){r+=0.491821;}else{r+=-0.214056;}}}}}else{if(f[14]<1.39633e+11){r+=-0.938818;}else{r+=0.675929;}}}}else{if(f[28]<1.63366e+11){if(f[28]<1.40396e+11){r+=0.0868482;}else{r+=-0.684201;}}else{r+=0.680864;}}}}else{if(f[82]<2.33149){if(f[38]<1.08276e+08){if(f[14]<1.40146e+11){if(f[41]<6.65927e+07){r+=-0.686909;}else{if(f[28]<1.40372e+11){r+=-0.625057;}else{if(f[1]<2.32898e+07){if(f[41]<9.99727e+07){if(f[67]<0.952615){r+=0.11088;}else{r+=0.749202;}}else{r+=1.15957;}}else{if(f[70]<1.00121){r+=0.0753775;}else{r+=-0.527456;}}}}}else{if(f[41]<1.70165e+08){if(f[34]<2.53744e+07){if(f[84]<3.5){if(f[41]<1.45189e+08){if(f[28]<1.40622e+11){if(f[42]<0.996774){r+=-0.485913;}else{r+=-0.690172;}}else{if(f[28]<1.40644e+11){r+=0.546906;}else{r+=-0.673302;}}}else{if(f[28]<1.40377e+11){r+=0.888902;}else{r+=-0.65282;}}}else{r+=-0.76432;}}else{if(f[84]<1.5){if(f[14]<1.40514e+11){if(f[28]<1.40573e+11){r+=-0.449994;}else{r+=1.43765;}}else{if(f[75]<1.93611){r+=-0.609847;}else{r+=-0.10223;}}}else{if(f[38]<1.05832e+08){r+=-0.679894;}else{r+=-0.12519;}}}}else{if(f[28]<1.40376e+11){r+=1.08779;}else{if(f[20]<3.60069e+07){r+=-0.60609;}else{r+=0.286415;}}}}}else{if(f[28]<1.41448e+11){if(f[14]<1.40068e+11){if(f[28]<1.40614e+11){if(f[27]<7.74349e+07){r+=-0.367291;}else{r+=0.775211;}}else{r+=-0.667886;}}else{if(f[41]<1.06858e+08){if(f[20]<8.44846e+07){r+=-0.69446;}else{r+=-0.100828;}}else{if(f[2]<4.38431e+06){r+=0.598462;}else{r+=-0.57916;}}}}else{if(f[28]<1.41582e+11){r+=0.67458;}else{if(f[28]<1.42019e+11){r+=-0.729449;}else{r+=0.527135;}}}}}else{if(f[42]<0.999318){if(f[27]<7.4074e+07){r+=-0.688777;}else{if(f[82]<2.81962){r+=-0.552527;}else{r+=-1.31553;}}}else{if(f[14]<1.39937e+11){r+=-0.595524;}else{if(f[82]<2.81963){r+=0.92783;}else{r+=0.684135;}}}}}}}else{if(f[69]<0.554217){if(f[14]<1.40039e+11){if(f[26]<1.51087e+08){if(f[9]<2.61122e+08){if(f[84]<1.85){if(f[42]<1.00579){if(f[28]<1.40145e+11){if(f[84]<1){if(f[14]<1.39636e+11){r+=0.743714;}else{if(f[42]<1.00067){r+=0.631803;}else{if(f[28]<1.39496e+11){r+=-0.94057;}else{r+=-0.703675;}}}}else{if(f[28]<1.40082e+11){if(f[10]<7.07706e+07){r+=-0.652018;}else{if(f[14]<1.40026e+11){r+=0.780868;}else{r+=0.219284;}}}else{r+=-0.670815;}}}else{if(f[42]<1.00161){if(f[84]<1.35){if(f[31]<5.14825e+08){r+=-0.719424;}else{if(f[31]<8.34635e+08){r+=0.121664;}else{r+=0.523811;}}}else{r+=0.659745;}}else{if(f[84]<0.8){if(f[27]<1.22404e+08){r+=-0.528094;}else{if(f[70]<0.99916){r+=-0.374852;}else{r+=0.714607;}}}else{if(f[37]<2.04864e+08){r+=1.41705;}else{r+=0.398387;}}}}}else{if(f[1]<1.88422e+08){r+=-0.718409;}else{r+=-0.181599;}}}else{if(f[14]<1.39394e+11){if(f[84]<3.5){if(f[10]<1.39821e+08){if(f[36]<9.45526e+07){r+=-0.636437;}else{if(f[38]<1.87184e+08){if(f[14]<1.38901e+11){r+=-0.100042;}else{r+=1.03663;}}else{r+=-0.489881;}}}else{if(f[70]<0.991267){if(f[18]<2.98066e+07){r+=0.834785;}else{r+=-0.172999;}}else{if(f[55]<0.950774){if(f[69]<0.472528){r+=-0.681395;}else{r+=0.579133;}}else{if(f[25]<9.72804e+07){r+=-0.695135;}else{r+=-1.02432;}}}}}else{if(f[38]<1.34375e+08){if(f[28]<1.40179e+11){if(f[28]<1.40159e+11){if(f[52]<0.899827){r+=-0.189951;}else{r+=-0.776673;}}else{r+=-0.0369115;}}else{r+=0.633212;}}else{if(f[7]<2.62672e+08){if(f[84]<4.85){if(f[14]<1.3911e+11){r+=0.993402;}else{r+=0.64202;}}else{r+=-0.409247;}}else{if(f[28]<1.40162e+11){if(f[14]<1.3935e+11){r+=0.262526;}else{r+=-0.805283;}}else{r+=0.650074;}}}}}else{if(f[70]<1.01798){if(f[42]<1.00587){if(f[14]<1.39471e+11){if(f[28]<1.40162e+11){if(f[2]<1.25745e+07){r+=-0.00342408;}else{r+=-0.892463;}}else{r+=0.756423;}}else{if(f[12]<6.2736e+08){if(f[14]<1.39606e+11){r+=-0.378059;}else{r+=-0.698805;}}else{r+=-0.0253063;}}}else{if(f[25]<9.78885e+07){if(f[66]<1.19832){r+=-0.706303;}else{r+=-0.152021;}}else{if(f[23]<1.09744e+08){r+=-0.620054;}else{if(f[40]<3.32467e+08){r+=-1.10374;}else{r+=-0.43305;}}}}}else{if(f[42]<1.00001){r+=1.54459;}else{r+=-0.507013;}}}}}else{if(f[28]<1.40061e+11){if(f[14]<1.39243e+11){if(f[14]<1.38859e+11){if(f[69]<0.494015){if(f[4]<4.07044e+06){r+=-0.176583;}else{r+=-0.680978;}}else{if(f[7]<2.41901e+08){r+=0.0734202;}else{r+=0.482803;}}}else{if(f[25]<8.69743e+07){r+=-0.390184;}else{if(f[69]<0.531496){if(f[51]<0.995244){r+=-0.18122;}else{if(f[84]<1.65){r+=0.0283476;}else{r+=0.841702;}}}else{r+=0.683417;}}}}else{if(f[84]<1){if(f[14]<1.39659e+11){r+=0.748892;}else{if(f[14]<1.39933e+11){r+=-0.989638;}else{r+=-0.226886;}}}else{if(f[84]<1.5){if(f[70]<0.995656){if(f[64]<0.511007){r+=0.211793;}else{r+=0.763231;}}else{r+=-0.0550242;}}else{if(f[42]<1.00799){if(f[10]<1.00426e+10){if(f[28]<1.40019e+11){r+=-0.826399;}else{r+=-0.207568;}}else{r+=0.478434;}}else{if(f[41]<2.88876e+08){r+=-0.701605;}else{r+=0.726977;}}}}}}else{if(f[84]<0.5){if(f[42]<1.0019){if(f[28]<1.40146e+11){r+=0.0538597;}else{r+=0.712625;}}else{if(f[8]<7.22446e+08){r+=-0.821412;}else{r+=-0.106916;}}}else{if(f[84]<2.15){if(f[84]<1.85){if(f[66]<0.317114){if(f[42]<1.00766){r+=0.884653;}else{r+=-0.575118;}}else{if(f[8]<2.05642e+08){if(f[1]<3.51985e+07){r+=0.34339;}else{r+=-0.453118;}}else{r+=-0.810826;}}}else{if(f[42]<1.00862){r+=-0.651265;}else{if(f[8]<2.28765e+08){r+=0.238499;}else{r+=0.875153;}}}}else{if(f[15]<1.10881e+08){if(f[42]<0.998307){r+=-0.146891;}else{r+=-0.7235;}}else{r+=-0.0053638;}}}}}}else{if(f[70]<1.0201){if(f[43]<8.2097){if(f[42]<0.995903){if(f[1]<4.32499e+07){r+=0.798855;}else{if(f[27]<9.01998e+08){r+=0.0445301;}else{r+=-0.541003;}}}else{if(f[14]<1.39004e+11){if(f[14]<1.38861e+11){if(f[69]<0.549069){if(f[73]<3.94107){r+=-0.694591;}else{if(f[9]<3.03566e+08){r+=-0.547408;}else{r+=0.309561;}}}else{r+=-0.0199437;}}else{if(f[9]<2.68205e+08){if(f[41]<6.70798e+08){r+=-1.01632;}else{r+=-0.6295;}}else{if(f[25]<2.97035e+08){if(f[51]<1.08988){r+=-0.1046;}else{r+=0.787455;}}else{r+=-0.582674;}}}}else{if(f[73]<127.555){if(f[84]<0.5){if(f[37]<9.63345e+07){if(f[28]<1.40147e+11){r+=-0.670117;}else{r+=0.973052;}}else{if(f[28]<1.3984e+11){r+=0.54101;}else{r+=-0.674696;}}}else{if(f[42]<0.998539){if(f[10]<2.29019e+08){r+=0.99308;}else{r+=-0.612829;}}else{if(f[51]<6.12288){r+=-0.683718;}else{r+=-0.1138;}}}}else{if(f[42]<1.00075){r+=0.610893;}else{r+=-0.586116;}}}}}else{if(f[14]<1.3985e+11){if(f[28]<1.40228e+11){if(f[9]<2.03573e+10){r+=0.791957;}else{r+=-0.0474133;}}else{r+=-0.585285;}}else{r+=-0.662407;}}}else{if(f[70]<1.40176e+11){if(f[14]<1.39492e+11){r+=-0.547328;}else{if(f[42]<1.00257){if(f[55]<39.0752){if(f[72]<3.70169e+06){r+=0.689902;}else{r+=0.100228;}}else{r+=-0.388675;}}else{r+=-0.495049;}}}else{if(f[29]<2.89269e+06){r+=-0.122826;}else{r+=-0.617685;}}}}}else{if(f[14]<1.40105e+11){if(f[14]<1.40094e+11){if(f[10]<1.22298e+08){if(f[84]<0.5){if(f[42]<0.999412){if(f[81]<3.30437){r+=0.0748928;}else{r+=0.749556;}}else{r+=-0.7669;}}else{if(f[84]<0.85){if(f[42]<0.999888){if(f[25]<1.09936e+08){r+=-0.69649;}else{r+=0.403183;}}else{if(f[42]<1.00059){if(f[69]<0.0609118){r+=-0.154201;}else{if(f[67]<1.6915){r+=0.701413;}else{r+=0.113523;}}}else{r+=-0.554899;}}}else{if(f[84]<1.5){if(f[28]<1.40181e+11){if(f[51]<0.325631){r+=0.980039;}else{if(f[81]<4.03207){r+=-0.795042;}else{r+=-0.527888;}}}else{if(f[53]<0.973756){if(f[69]<0.151207){r+=-0.455969;}else{r+=0.41677;}}else{if(f[76]<0.490399){r+=0.275004;}else{r+=-0.635862;}}}}else{if(f[84]<1.85){if(f[42]<1.00061){if(f[9]<2.97862e+07){r+=0.231898;}else{r+=0.695888;}}else{r+=-0.622688;}}else{if(f[82]<8.01196){if(f[14]<1.40093e+11){r+=-0.713794;}else{r+=-0.105274;}}else{if(f[14]<1.40079e+11){r+=-0.348394;}else{r+=0.729506;}}}}}}}else{if(f[35]<6.17475e+07){if(f[84]<1.5){if(f[64]<2.33651){if(f[66]<2.61994){if(f[42]<0.999178){r+=0.347056;}else{if(f[82]<0.791928){r+=0.173411;}else{r+=-0.741277;}}}else{if(f[10]<5.36203e+08){r+=0.451636;}else{r+=-0.315699;}}}else{if(f[40]<1.44792e+09){if(f[27]<2.96081e+08){r+=-0.358176;}else{if(f[36]<1.0057e+08){r+=0.773589;}else{r+=0.13542;}}}else{if(f[12]<1.43876e+08){if(f[61]<1.4007){r+=-0.0271021;}else{r+=0.565922;}}else{if(f[53]<0.714703){r+=-0.121221;}else{r+=-0.743407;}}}}}else{if(f[42]<1.00058){if(f[33]<8.70178e+06){if(f[51]<2.45702){if(f[4]<722279){r+=0.0441141;}else{r+=0.71337;}}else{if(f[55]<4.41088){r+=-0.487607;}else{r+=0.482369;}}}else{if(f[84]<1.85){r+=0.367529;}else{r+=-0.438402;}}}else{if(f[14]<1.40078e+11){r+=-0.724329;}else{r+=0.229672;}}}}else{if(f[12]<6.35045e+07){if(f[42]<0.999277){r+=-0.374672;}else{r+=0.698516;}}else{if(f[73]<41.4216){if(f[84]<2.85){if(f[70]<1.00116){if(f[11]<3.32994e+10){r+=-0.727443;}else{r+=0.4723;}}else{if(f[7]<1.01779e+08){r+=0.602883;}else{r+=-0.623973;}}}else{if(f[27]<3.65308e+08){r+=-0.682212;}else{if(f[14]<1.40062e+11){r+=-0.180876;}else{r+=0.816451;}}}}else{if(f[84]<1){if(f[81]<6.98956){r+=0.705751;}else{r+=-0.123234;}}else{r+=-0.666817;}}}}}}else{if(f[28]<1.4018e+11){if(f[84]<3.5){if(f[50]<4.24131){if(f[23]<1.12658e+09){if(f[81]<5.02586){if(f[20]<517693){r+=-0.0990585;}else{if(f[61]<0.0598426){r+=-0.157319;}else{r+=-0.733482;}}}else{if(f[69]<0.153381){if(f[67]<0.199149){r+=-0.73517;}else{r+=-0.201412;}}else{if(f[35]<2.57188e+07){r+=0.579103;}else{r+=-0.24001;}}}}else{if(f[12]<1.8434e+08){if(f[2]<3.88484e+06){r+=-0.521674;}else{if(f[62]<1.1306){r+=0.00578585;}else{r+=0.681125;}}}else{if(f[42]<0.999233){r+=-0.631103;}else{r+=0.00553596;}}}}else{if(f[42]<0.999267){r+=0.640667;}else{r+=-0.401281;}}}else{if(f[26]<1.88525e+08){r+=-0.547741;}else{r+=0.82758;}}}else{if(f[28]<1.40239e+11){if(f[28]<1.40182e+11){if(f[84]<3){if(f[40]<1.64885e+09){if(f[81]<4.22682){if(f[63]<2.53501){r+=-0.595459;}else{r+=0.137874;}}else{if(f[76]<0.662385){r+=-0.319718;}else{r+=0.464769;}}}else{if(f[12]<1.12504e+08){if(f[17]<2.23797e+06){r+=-0.0512087;}else{r+=-0.71544;}}else{if(f[11]<7.21215e+07){r+=0.652097;}else{r+=0.0555955;}}}}else{r+=0.778354;}}else{if(f[42]<1.00031){if(f[28]<1.40185e+11){if(f[84]<2.5){if(f[33]<3.52512e+06){r+=0.432306;}else{r+=0.0736067;}}else{r+=0.730829;}}else{if(f[55]<0.949911){if(f[84]<1.65){r+=0.650975;}else{r+=0.690808;}}else{if(f[15]<1.1577e+07){r+=1.39076;}else{r+=0.425449;}}}}else{r+=-0.573954;}}}else{if(f[8]<1.01368e+08){if(f[28]<1.40393e+11){r+=-0.441326;}else{if(f[3]<1.01205e+07){r+=0.757426;}else{r+=0.210681;}}}else{r+=-0.669436;}}}}}else{if(f[70]<1.00972){if(f[81]<1.47459){if(f[28]<1.39896e+11){if(f[14]<1.40309e+11){if(f[65]<0.938784){if(f[9]<2.26762e+08){r+=-0.359162;}else{if(f[17]<3.88134e+07){r+=0.794289;}else{r+=0.214084;}}}else{r+=-0.642776;}}else{if(f[70]<1.00278){if(f[42]<1.00394){r+=-0.693998;}else{if(f[41]<7.49788e+08){r+=-1.00574;}else{r+=-0.657761;}}}else{if(f[1]<1.06957e+08){r+=0.12332;}else{r+=-0.396735;}}}}else{if(f[42]<1.00223){if(f[40]<1.14093e+08){if(f[84]<1.5){if(f[28]<1.39986e+11){if(f[84]<1.15){r+=0.707795;}else{r+=-0.624593;}}else{if(f[14]<1.41144e+11){r+=-0.702482;}else{r+=0.181958;}}}else{if(f[14]<1.4082e+11){if(f[42]<0.99922){r+=0.724635;}else{r+=-0.567856;}}else{if(f[70]<0.99768){r+=0.663809;}else{r+=-0.703571;}}}}else{if(f[28]<1.40001e+11){if(f[52]<1.37433){if(f[14]<1.40156e+11){r+=-0.538005;}else{r+=0.713281;}}else{if(f[70]<0.999294){r+=-0.676383;}else{r+=0.638948;}}}else{if(f[25]<9.78236e+08){if(f[84]<3.5){r+=-0.604384;}else{r+=0.290862;}}else{if(f[84]<2.85){r+=-0.344916;}else{r+=0.798841;}}}}}else{if(f[14]<1.40147e+11){if(f[84]<2.15){if(f[25]<8.57914e+07){if(f[42]<1.00449){r+=-0.190722;}else{r+=-0.661986;}}else{r+=-0.996618;}}else{if(f[55]<1.51115){r+=-0.6284;}else{r+=0.794343;}}}else{if(f[70]<0.980679){if(f[40]<1.14949e+08){r+=-0.0962012;}else{r+=-0.713299;}}else{if(f[12]<9.46696e+07){if(f[84]<1.5){r+=-0.898318;}else{r+=0.0896502;}}else{if(f[35]<6.94058e+08){r+=0.684021;}else{r+=-0.108156;}}}}}}}else{if(f[23]<5.87483e+07){if(f[38]<4.44924e+07){if(f[14]<1.40523e+11){r+=-0.682699;}else{if(f[84]<2.65){if(f[10]<1.96558e+07){r+=0.0756941;}else{r+=0.718523;}}else{r+=-0.635058;}}}else{if(f[14]<1.40189e+11){if(f[42]<0.999316){if(f[84]<1){r+=-0.641818;}else{if(f[29]<1.05158e+07){r+=0.696309;}else{r+=-0.43132;}}}else{if(f[65]<0.80659){if(f[27]<6.65338e+08){r+=-0.681012;}else{r+=0.0257285;}}else{if(f[28]<1.40005e+11){r+=0.254735;}else{r+=-0.652227;}}}}else{if(f[20]<3.83959e+07){if(f[40]<1.57116e+08){if(f[28]<1.39986e+11){r+=0.624103;}else{r+=-0.566013;}}else{if(f[14]<1.41377e+11){r+=-0.686663;}else{r+=-0.560192;}}}else{if(f[6]<2.8571e+07){r+=-0.372305;}else{r+=0.748268;}}}}}else{if(f[28]<1.40027e+11){if(f[28]<1.39902e+11){if(f[42]<1.00253){if(f[28]<1.39881e+11){if(f[10]<2.3621e+09){r+=-0.692329;}else{r+=-0.140024;}}else{if(f[84]<2.5){r+=-0.650601;}else{r+=0.547971;}}}else{if(f[41]<7.49491e+08){if(f[69]<0.530929){r+=-1.15362;}else{r+=-0.679929;}}else{if(f[29]<3.02053e+08){r+=-0.692898;}else{r+=0.376396;}}}}else{if(f[84]<0.85){if(f[14]<1.40187e+11){if(f[41]<1.66191e+09){r+=-0.746515;}else{r+=-1.05162;}}else{r+=-0.698557;}}else{if(f[84]<1.15){if(f[81]<2.49483){r+=0.775001;}else{r+=0.994939;}}else{if(f[14]<1.40417e+11){r+=-0.687705;}else{r+=0.364819;}}}}}else{if(f[27]<1.04327e+08){if(f[70]<1.00238){if(f[28]<1.40588e+11){if(f[62]<0.391074){r+=0.464912;}else{r+=-0.58339;}}else{if(f[41]<2.83841e+08){r+=0.94425;}else{r+=1.58909;}}}else{if(f[29]<1.91329e+08){if(f[13]<2.76709e+07){r+=0.0201211;}else{r+=-0.727434;}}else{r+=0.828682;}}}else{if(f[70]<1.00147){if(f[84]<1.85){if(f[70]<0.99144){r+=0.499716;}else{r+=-0.678716;}}else{if(f[14]<1.4138e+11){r+=0.606564;}else{r+=-0.655519;}}}else{if(f[84]<12){if(f[28]<1.43241e+11){r+=-0.682768;}else{r+=0.598498;}}else{if(f[42]<0.996029){r+=0.890543;}else{r+=-0.686969;}}}}}}}}else{if(f[84]<1.5){if(f[28]<1.41235e+11){r+=-0.673749;}else{if(f[14]<1.40752e+11){r+=-0.613601;}else{if(f[14]<1.45146e+11){if(f[28]<1.42025e+11){r+=-0.559419;}else{if(f[26]<8.41013e+09){r+=0.686613;}else{r+=0.176399;}}}else{if(f[66]<2.33885){if(f[69]<0.541429){r+=1.59311;}else{r+=0.396266;}}else{r+=0.642353;}}}}}else{if(f[13]<1.1301e+08){if(f[35]<9.56865e+07){r+=-0.576533;}else{r+=1.06821;}}else{if(f[28]<1.40109e+11){r+=-0.159491;}else{r+=-0.69114;}}}}}}}else{if(f[14]<1.40238e+11){if(f[38]<5.40573e+07){if(f[14]<1.40128e+11){if(f[70]<1.0008){if(f[84]<1.85){if(f[42]<0.999943){if(f[84]<1.15){if(f[28]<1.40053e+11){r+=-0.521823;}else{r+=-1.09845;}}else{if(f[28]<1.40048e+11){if(f[8]<1.49905e+07){r+=-1.19703;}else{r+=-0.493214;}}else{if(f[28]<1.40121e+11){r+=0.694085;}else{r+=0.948403;}}}}else{if(f[14]<1.4002e+11){if(f[82]<0.77137){if(f[42]<1.00058){if(f[28]<1.39997e+11){r+=-0.485034;}else{r+=0.683599;}}else{r+=-0.60618;}}else{if(f[54]<0.987634){r+=-0.739113;}else{if(f[14]<1.39999e+11){r+=-0.433216;}else{r+=0.405507;}}}}else{if(f[55]<1.13463){if(f[35]<6.00316e+06){if(f[25]<6.61014e+07){r+=-0.531852;}else{r+=0.611164;}}else{if(f[69]<0.673569){r+=0.450693;}else{r+=-0.675111;}}}else{if(f[14]<1.40057e+11){if(f[70]<1.00018){r+=-0.665612;}else{r+=0.711493;}}else{if(f[42]<1.00037){r+=-0.672088;}else{r+=0.708452;}}}}}}else{if(f[14]<1.40028e+11){if(f[40]<6.84603e+07){if(f[12]<1.61938e+08){if(f[42]<1.00047){if(f[84]<2.5){r+=0.255609;}else{r+=-0.482553;}}else{r+=-0.673729;}}else{r+=0.329855;}}else{if(f[28]<1.39993e+11){r+=-0.625109;}else{if(f[70]<0.999793){if(f[51]<0.772254){r+=0.224715;}else{r+=0.712977;}}else{r+=-0.642589;}}}}else{if(f[14]<1.40101e+11){if(f[70]<0.998929){if(f[3]<1.34376e+06){r+=0.236434;}else{r+=0.754066;}}else{if(f[42]<0.999968){if(f[69]<1.06133){r+=-0.857674;}else{r+=-1.33939;}}else{r+=-0.703738;}}}else{if(f[13]<1.21302e+08){if(f[69]<1.06998){r+=-0.724098;}else{if(f[27]<1.00211e+08){r+=-0.68934;}else{r+=-1.35334;}}}else{if(f[69]<1.23263){if(f[41]<1.23655e+08){r+=-1.2885;}else{r+=-0.873634;}}else{if(f[67]<2.54063){r+=-1.65549;}else{r+=-0.641191;}}}}}}}else{if(f[1]<1.33541e+07){if(f[42]<0.998794){if(f[26]<1.00378e+08){r+=-0.652415;}else{r+=-0.154989;}}else{if(f[69]<0.871142){if(f[29]<9.45788e+06){r+=-0.574877;}else{r+=0.469471;}}else{if(f[23]<2.1711e+07){r+=-0.33382;}else{r+=0.680648;}}}}else{if(f[26]<3.18426e+08){r+=-0.681901;}else{r+=0.217108;}}}}else{if(f[84]<1.85){if(f[14]<1.40221e+11){if(f[27]<8.271e+07){if(f[49]<2.15385){r+=-0.586778;}else{if(f[81]<0.656512){r+=-0.0300522;}else{r+=0.734329;}}}else{if(f[28]<1.39984e+11){if(f[42]<1.00118){r+=-0.687394;}else{r+=0.637734;}}else{if(f[28]<1.40144e+11){if(f[41]<9.19911e+07){if(f[9]<4.04431e+07){r+=-0.641547;}else{r+=-0.183783;}}else{if(f[69]<0.980212){r+=-0.899717;}else{r+=-1.22393;}}}else{if(f[84]<2.60001){r+=-0.69783;}else{r+=0.411547;}}}}}else{if(f[84]<3.40001){if(f[42]<1.00092){r+=-0.849558;}else{if(f[2]<7.66695e+06){r+=-0.056475;}else{r+=0.457989;}}}else{if(f[9]<2.24049e+07){r+=0.0111779;}else{r+=0.699507;}}}}else{if(f[28]<1.40145e+11){if(f[14]<1.40196e+11){if(f[70]<0.998718){if(f[8]<2.48112e+08){if(f[8]<1.18429e+07){r+=-0.191482;}else{r+=0.726131;}}else{if(f[42]<1.01014){r+=1.3124;}else{r+=0.561637;}}}else{if(f[28]<1.39993e+11){if(f[54]<1.25725){r+=1.54925;}else{r+=0.563743;}}else{if(f[14]<1.40132e+11){r+=0.230718;}else{r+=0.848192;}}}}else{if(f[37]<3.17368e+07){r+=-0.822863;}else{r+=0.00689565;}}}else{if(f[28]<1.41194e+11){if(f[27]<5.93945e+08){r+=-0.688283;}else{r+=-0.188452;}}else{if(f[67]<2.79741){r+=0.663347;}else{r+=1.23914;}}}}}}else{if(f[40]<7.54917e+07){if(f[84]<0.5){if(f[28]<1.39794e+11){if(f[38]<1.13402e+08){r+=-0.517071;}else{r+=0.680368;}}else{if(f[28]<1.60776e+11){if(f[13]<1.1034e+08){if(f[14]<1.3953e+11){r+=-0.635653;}else{if(f[7]<5.50788e+07){r+=-0.454947;}else{r+=-1.61231;}}}else{if(f[42]<0.999648){if(f[69]<0.967525){r+=-0.76123;}else{if(f[28]<1.40121e+11){r+=-1.39173;}else{r+=-0.915228;}}}else{if(f[28]<1.40364e+11){r+=-0.703656;}else{if(f[14]<1.395e+11){r+=-0.665752;}else{r+=-1.47919;}}}}}else{r+=0.653556;}}}else{if(f[14]<1.39509e+11){if(f[28]<1.40252e+11){if(f[28]<1.40161e+11){if(f[42]<1.00541){if(f[9]<1.98853e+08){r+=0.698071;}else{r+=-0.0958913;}}else{if(f[28]<1.40151e+11){if(f[37]<8.81688e+07){r+=-0.10175;}else{r+=-0.687077;}}else{if(f[14]<1.3936e+11){r+=-0.722157;}else{r+=0.530325;}}}}else{if(f[42]<1.00783){if(f[9]<1.80933e+08){if(f[13]<1.71819e+08){r+=0.678324;}else{r+=0.144758;}}else{if(f[1]<7.2761e+07){r+=-0.083669;}else{r+=0.399242;}}}else{r+=-0.665463;}}}else{if(f[14]<1.395e+11){r+=-0.683785;}else{if(f[4]<3.44015e+07){r+=-0.599927;}else{r+=0.438618;}}}}else{if(f[14]<1.40026e+11){if(f[28]<1.39435e+11){if(f[14]<1.3964e+11){r+=0.456155;}else{if(f[64]<0.606991){r+=0.118488;}else{if(f[41]<1.11056e+08){r+=-0.678186;}else{r+=-0.325513;}}}}else{if(f[70]<1.01125){if(f[41]<6.4086e+07){r+=-0.672171;}else{if(f[69]<0.839476){r+=-0.412324;}else{r+=0.664935;}}}else{if(f[70]<1.01801){if(f[52]<1.69834){r+=-1.23206;}else{r+=-0.681306;}}else{if(f[84]<2.5){r+=-0.746584;}else{r+=0.753648;}}}}}else{if(f[14]<1.40053e+11){if(f[82]<1.14333){if(f[14]<1.40032e+11){if(f[42]<1.00044){r+=0.628426;}else{r+=-0.406731;}}else{if(f[64]<0.44901){r+=0.803577;}else{r+=-0.718269;}}}else{if(f[84]<1.2){r+=-0.564258;}else{if(f[25]<8.93395e+07){r+=0.717737;}else{r+=0.186183;}}}}else{if(f[69]<1.87732){if(f[28]<1.4017e+11){if(f[28]<1.40106e+11){r+=0.364395;}else{r+=0.689897;}}else{if(f[54]<0.804355){r+=0.533522;}else{r+=-0.663426;}}}else{if(f[42]<0.999937){if(f[14]<1.40101e+11){r+=0.698141;}else{r+=-0.322034;}}else{if(f[42]<1.00046){r+=-0.699139;}else{r+=0.137491;}}}}}}}}else{if(f[28]<1.40487e+11){if(f[84]<1.5){if(f[28]<1.3984e+11){if(f[28]<1.39432e+11){if(f[28]<1.39417e+11){if(f[57]<22.7414){r+=-0.68666;}else{r+=-0.0233716;}}else{if(f[84]<1){r+=-0.739003;}else{if(f[42]<1.00157){r+=-0.401308;}else{r+=0.819361;}}}}else{if(f[14]<1.39912e+11){if(f[28]<1.39459e+11){if(f[84]<1){r+=-1.15476;}else{r+=0.698058;}}else{if(f[42]<1.00597){r+=0.695786;}else{r+=-0.590875;}}}else{if(f[31]<1.15144e+09){if(f[45]<0.277522){r+=-0.0983101;}else{r+=-0.696088;}}else{r+=0.71165;}}}}else{if(f[37]<8.70276e+07){if(f[26]<1.36919e+08){if(f[23]<5.99886e+07){if(f[14]<1.401e+11){r+=-0.313773;}else{r+=-0.872694;}}else{if(f[14]<1.4016e+11){r+=-0.647822;}else{r+=0.617894;}}}else{if(f[14]<1.39848e+11){if(f[41]<5.69164e+08){r+=-0.670857;}else{r+=0.668;}}else{if(f[23]<5.95495e+07){r+=-0.331453;}else{r+=0.484954;}}}}else{if(f[31]<3.85969e+08){if(f[65]<1.02106){if(f[41]<2.31249e+09){r+=-0.755255;}else{r+=-1.27484;}}else{if(f[14]<1.4009e+11){r+=-0.618727;}else{r+=0.0252361;}}}else{if(f[35]<2.81864e+09){r+=0.797779;}else{r+=-0.664048;}}}}}else{if(f[12]<2.93827e+08){if(f[14]<1.39068e+11){if(f[14]<1.3883e+11){if(f[84]<2.15){if(f[2]<4.58372e+06){r+=-0.192814;}else{r+=0.810773;}}else{r+=-0.699549;}}else{if(f[41]<1.55233e+08){if(f[84]<2.85){r+=-0.917398;}else{r+=0.695451;}}else{if(f[66]<0.804025){r+=0.330412;}else{r+=0.664224;}}}}else{if(f[14]<1.40021e+11){if(f[28]<1.40154e+11){if(f[14]<1.39245e+11){r+=-0.245701;}else{r+=-0.663112;}}else{if(f[70]<0.999515){r+=-0.575555;}else{r+=0.458052;}}}else{if(f[27]<1.76031e+08){if(f[41]<1.62169e+08){r+=0.489358;}else{r+=0.0530435;}}else{if(f[42]<1.00134){r+=0.0285044;}else{r+=-0.571403;}}}}}else{if(f[84]<2.15){if(f[65]<0.516742){if(f[70]<1.00136){if(f[63]<0.357881){r+=-0.971393;}else{r+=-0.743237;}}else{if(f[31]<2.50484e+08){r+=0.700046;}else{r+=-0.67875;}}}else{if(f[84]<4.00001){if(f[70]<1.00136){r+=0.249617;}else{r+=0.697649;}}else{if(f[28]<1.40153e+11){r+=-0.693657;}else{r+=0.330286;}}}}else{if(f[28]<1.40039e+11){if(f[27]<2.72812e+08){if(f[42]<1.00137){r+=-0.129679;}else{r+=0.665001;}}else{if(f[84]<2.5){r+=-0.672037;}else{r+=-0.131983;}}}else{if(f[14]<1.39528e+11){if(f[84]<2.85){r+=0.61797;}else{r+=-0.686895;}}else{if(f[40]<2.54573e+08){r+=0.205038;}else{r+=0.766885;}}}}}}}else{if(f[37]<1.44854e+08){if(f[14]<1.39797e+11){if(f[14]<1.39527e+11){if(f[14]<1.39522e+11){r+=-0.681179;}else{if(f[11]<1.26888e+08){r+=0.242233;}else{r+=-0.418742;}}}else{if(f[84]<3.35){if(f[84]<0.5){if(f[25]<7.17672e+08){r+=-1.55181;}else{r+=-0.620531;}}else{if(f[28]<1.42196e+11){r+=0.585521;}else{r+=-1.03086;}}}else{if(f[52]<1.69719){if(f[66]<0.742285){r+=-0.631668;}else{r+=-1.24069;}}else{r+=-0.67703;}}}}else{if(f[28]<1.74187e+11){if(f[42]<0.996448){if(f[14]<1.39907e+11){r+=0.750428;}else{r+=-0.687657;}}else{if(f[13]<2.39811e+09){if(f[14]<1.39849e+11){r+=-0.945416;}else{r+=-0.739312;}}else{if(f[14]<1.40135e+11){r+=0.784564;}else{r+=-0.0574491;}}}}else{r+=0.673617;}}}else{if(f[28]<1.71492e+11){if(f[70]<0.983209){if(f[42]<1.02938){if(f[66]<0.952045){if(f[9]<2.09664e+08){r+=1.42743;}else{r+=0.502753;}}else{if(f[28]<1.40686e+11){r+=-0.117313;}else{r+=0.714242;}}}else{if(f[25]<1.42973e+10){if(f[64]<1.62046){r+=-0.690349;}else{r+=-0.124189;}}else{if(f[28]<1.40793e+11){r+=0.525551;}else{r+=-0.440896;}}}}else{if(f[13]<6.90684e+07){if(f[1]<4.04212e+06){r+=0.15998;}else{r+=0.658463;}}else{if(f[37]<1.51666e+08){if(f[84]<12){r+=-0.557774;}else{r+=0.615002;}}else{if(f[57]<0.0413992){r+=0.471571;}else{r+=-0.678895;}}}}}else{r+=0.682809;}}}}}}else{if(f[69]<1.42647){if(f[28]<1.39904e+11){if(f[14]<1.40276e+11){if(f[42]<1.00896){if(f[31]<1.60007e+07){r+=-0.685685;}else{r+=0.135993;}}else{if(f[42]<1.01257){r+=0.693845;}else{r+=-0.617013;}}}else{if(f[70]<1.00278){if(f[14]<1.40309e+11){if(f[65]<0.934731){if(f[81]<0.780621){if(f[49]<2.17064){r+=0.812041;}else{r+=-0.295604;}}else{r+=-0.682862;}}else{if(f[77]<1.06969){r+=-0.671224;}else{r+=-0.191329;}}}else{if(f[28]<1.39894e+11){if(f[37]<4.71044e+10){r+=-0.683195;}else{r+=-0.0289459;}}else{if(f[15]<1.0297e+08){if(f[82]<0.520487){r+=-0.218498;}else{if(f[41]<7.93814e+07){r+=-0.504697;}else{r+=-1.11961;}}}else{r+=0.140793;}}}}else{if(f[42]<0.987406){r+=-0.507858;}else{if(f[62]<2.30146){r+=0.689391;}else{r+=-0.000433121;}}}}}else{if(f[82]<0.872225){if(f[84]<0.85){if(f[14]<1.41161e+11){if(f[70]<0.984281){if(f[14]<1.40992e+11){r+=-0.16626;}else{r+=0.652268;}}else{r+=-0.693556;}}else{if(f[28]<1.40131e+11){if(f[14]<1.41233e+11){if(f[66]<1.05169){r+=0.718137;}else{r+=-0.0489582;}}else{if(f[36]<1.66846e+08){if(f[28]<1.40121e+11){r+=-0.681422;}else{r+=-0.894081;}}else{r+=0.00121295;}}}else{if(f[27]<1.12836e+09){if(f[70]<0.980384){r+=-0.561949;}else{if(f[14]<1.41181e+11){r+=0.339518;}else{r+=0.69369;}}}else{if(f[28]<1.4274e+11){if(f[55]<1.86994){r+=-0.655626;}else{r+=0.204881;}}else{r+=0.667298;}}}}}else{if(f[28]<1.41451e+11){if(f[70]<1.00538){if(f[84]<1.5){if(f[36]<7.89742e+06){if(f[28]<1.40084e+11){r+=0.723617;}else{r+=-0.698201;}}else{if(f[10]<4.06793e+07){r+=-1.12356;}else{r+=-0.719169;}}}else{if(f[42]<0.997141){if(f[14]<1.40534e+11){r+=-0.702189;}else{r+=0.83747;}}else{if(f[42]<1.0028){r+=-0.605598;}else{r+=0.0552288;}}}}else{if(f[14]<1.41791e+11){if(f[28]<1.41356e+11){r+=-0.691282;}else{r+=-0.110996;}}else{r+=-0.772827;}}}else{if(f[14]<1.40999e+11){if(f[70]<1.01246){if(f[14]<1.40873e+11){r+=0.206965;}else{r+=-0.67503;}}else{if(f[28]<1.42852e+11){r+=0.691738;}else{r+=-0.579759;}}}else{if(f[84]<6.00001){r+=-0.684681;}else{if(f[9]<2.11115e+08){r+=0.582133;}else{r+=-0.587026;}}}}}}else{if(f[84]<1.5){if(f[25]<5.04174e+07){if(f[70]<0.997452){if(f[21]<5.73804e+07){r+=0.730804;}else{r+=-0.192818;}}else{if(f[41]<1.22077e+08){if(f[14]<1.41172e+11){r+=-0.688104;}else{r+=0.372476;}}else{if(f[28]<1.40116e+11){r+=-1.19008;}else{r+=-0.485745;}}}}else{if(f[70]<1.01152){if(f[66]<0.310933){if(f[28]<1.39986e+11){if(f[8]<2.5143e+08){r+=-1.58068;}else{r+=-0.664516;}}else{if(f[70]<0.998528){r+=-1.20767;}else{r+=-0.43242;}}}else{if(f[42]<1.01059){if(f[28]<1.39987e+11){r+=-0.0147427;}else{r+=-0.710872;}}else{if(f[70]<0.980679){r+=-0.679569;}else{r+=0.671342;}}}}else{if(f[13]<2.09536e+08){if(f[14]<1.45148e+11){if(f[70]<1.01462){r+=-0.232625;}else{r+=-0.691266;}}else{r+=0.669034;}}else{if(f[28]<1.41224e+11){r+=-0.706303;}else{if(f[41]<2.14748e+08){r+=-0.249309;}else{r+=0.715478;}}}}}}else{if(f[9]<2.45411e+07){if(f[42]<0.997142){if(f[70]<1.00202){if(f[84]<2.8){if(f[69]<1.04181){r+=1.36706;}else{r+=0.748163;}}else{r+=-0.696142;}}else{if(f[70]<1.00222){if(f[14]<1.40539e+11){r+=-0.475095;}else{r+=0.536357;}}else{r+=-0.698413;}}}else{if(f[25]<3.76918e+07){if(f[55]<1.27453){r+=-0.640857;}else{r+=-1.13383;}}else{if(f[1]<6.28089e+07){if(f[26]<1.62986e+09){r+=-0.687954;}else{r+=-0.0521732;}}else{if(f[1]<7.09913e+07){r+=0.289295;}else{r+=-0.372891;}}}}}else{if(f[9]<2.74662e+08){if(f[13]<1.95413e+09){if(f[37]<2.29612e+08){if(f[41]<4.77965e+08){r+=0.424735;}else{r+=-0.299166;}}else{if(f[7]<5.99875e+07){r+=-0.61632;}else{r+=0.731544;}}}else{if(f[11]<1.4054e+10){if(f[66]<0.377928){r+=-0.07893;}else{r+=-0.687629;}}else{r+=0.99621;}}}else{if(f[31]<8.17809e+06){if(f[28]<1.40382e+11){if(f[28]<1.40337e+11){r+=-0.212174;}else{r+=1.3243;}}else{if(f[52]<0.673025){r+=0.549763;}else{r+=-0.572231;}}}else{if(f[42]<1.00526){if(f[52]<0.643371){r+=0.212002;}else{r+=-0.603215;}}else{if(f[14]<1.41052e+11){r+=0.771106;}else{r+=-0.692961;}}}}}}}}}else{if(f[28]<1.42037e+11){if(f[14]<1.45774e+11){if(f[84]<4.85){if(f[70]<1.01662){if(f[35]<6.1059e+06){if(f[84]<1.15){if(f[70]<1.00269){if(f[28]<1.39944e+11){r+=-0.240597;}else{r+=0.753385;}}else{r+=-0.633497;}}else{r+=-0.707941;}}else{if(f[70]<0.982364){if(f[28]<1.40132e+11){if(f[84]<2.5){r+=-0.681612;}else{r+=-1.05126;}}else{if(f[70]<0.980609){r+=-0.106505;}else{r+=0.686311;}}}else{if(f[42]<1.01078){if(f[42]<1.00965){r+=-0.681431;}else{r+=-0.0212181;}}else{if(f[41]<7.93997e+07){r+=-0.677356;}else{r+=-1.32927;}}}}}else{if(f[42]<0.987709){if(f[9]<1.02003e+09){r+=-0.687651;}else{r+=-0.0562077;}}else{if(f[43]<3.63189){r+=0.720749;}else{r+=0.212285;}}}}else{if(f[42]<0.995159){r+=-0.700164;}else{r+=0.717667;}}}else{if(f[49]<0.252099){r+=-0.490204;}else{if(f[54]<0.725989){r+=1.09245;}else{if(f[15]<8.60206e+07){r+=0.692246;}else{r+=-0.050108;}}}}}else{if(f[70]<1.01589){if(f[14]<1.41001e+11){if(f[38]<1.75528e+08){r+=0.68891;}else{r+=-0.451477;}}else{if(f[28]<1.43654e+11){if(f[41]<8.90606e+07){r+=0.418236;}else{r+=-0.669545;}}else{r+=0.644843;}}}else{if(f[9]<9.42791e+08){r+=-0.689711;}else{if(f[69]<2.34403){r+=0.589006;}else{r+=0.0748463;}}}}}}}} return r;}
    static double p2(double[] f){double r=0;if(f[40]<6.08173e+07){if(f[0]<1.40246e+11){if(f[83]<3.133){if(f[41]<6.78895e+07){if(f[83]<2.18706){if(f[14]<1.39657e+11){if(f[28]<1.39501e+11){r+=0.873205;}else{r+=-0.584278;}}else{if(f[0]<1.40242e+11){r+=-0.596577;}else{if(f[28]<1.69745e+11){r+=-0.515981;}else{r+=0.385988;}}}}else{if(f[21]<1.94371e+07){if(f[35]<1.51564e+07){r+=0.00573653;}else{r+=0.504801;}}else{r+=-0.525444;}}}else{if(f[42]<0.951564){if(f[80]<0.417312){r+=0.599233;}else{r+=0.148951;}}else{if(f[28]<1.41394e+11){if(f[52]<5.60373){if(f[54]<0.384616){if(f[42]<0.999244){if(f[54]<0.249173){r+=-0.14505;}else{r+=-0.555673;}}else{if(f[0]<1.40064e+11){r+=0.611422;}else{r+=-0.17694;}}}else{if(f[28]<1.40142e+11){if(f[80]<0.720196){if(f[0]<1.4001e+11){if(f[28]<1.40028e+11){r+=-0.581307;}else{r+=0.593596;}}else{if(f[38]<5.58253e+07){r+=-0.601072;}else{r+=-0.2703;}}}else{if(f[41]<8.90065e+07){if(f[53]<0.708128){r+=0.0238719;}else{r+=-0.560961;}}else{if(f[28]<1.40086e+11){r+=-0.573449;}else{r+=0.330219;}}}}else{if(f[41]<1.14559e+08){if(f[55]<0.374422){if(f[56]<0.997773){r+=0.559364;}else{r+=-0.434266;}}else{if(f[56]<0.997401){r+=-0.481051;}else{r+=-0.605083;}}}else{if(f[28]<1.40375e+11){if(f[84]<2.15){r+=-0.444006;}else{r+=0.819346;}}else{if(f[14]<1.40096e+11){r+=0.783683;}else{r+=-0.617645;}}}}}}else{if(f[10]<1.94591e+08){r+=0.615634;}else{r+=-0.42054;}}}else{if(f[56]<0.987818){if(f[42]<1.0012){r+=-0.595767;}else{r+=0.363461;}}else{if(f[84]<1.8){r+=-0.439664;}else{r+=0.627346;}}}}}}else{if(f[0]<1.39992e+11){r+=-0.611126;}else{if(f[0]<1.40065e+11){r+=0.608355;}else{r+=-0.388141;}}}}else{if(f[84]<1.15){if(f[28]<1.40148e+11){if(f[84]<0.85){r+=-0.601942;}else{if(f[0]<1.40359e+11){r+=-0.527436;}else{if(f[39]<4.927e+07){r+=0.633855;}else{r+=0.209065;}}}}else{if(f[13]<6.8599e+07){if(f[28]<1.4031e+11){r+=0.0901718;}else{r+=-0.596124;}}else{if(f[39]<4.32733e+07){if(f[84]<0.85){if(f[36]<743159){r+=-0.430701;}else{if(f[11]<3.44888e+08){r+=0.596239;}else{r+=0.1513;}}}else{r+=-0.470069;}}else{if(f[14]<1.40124e+11){if(f[28]<1.41408e+11){r+=-0.750245;}else{if(f[0]<1.41427e+11){r+=0.599947;}else{r+=-0.430885;}}}else{if(f[28]<1.40387e+11){r+=0.567735;}else{if(f[0]<1.40252e+11){r+=-0.143655;}else{r+=-0.607602;}}}}}}}else{if(f[40]<5.86457e+07){if(f[56]<0.999111){if(f[56]<0.998603){if(f[65]<1.01973){if(f[42]<1.00623){if(f[3]<2.7222e+07){r+=0.518328;}else{r+=0.0811207;}}else{r+=-0.378157;}}else{if(f[67]<1.05048){r+=0.114862;}else{r+=-0.587132;}}}else{if(f[11]<8.86098e+07){if(f[28]<1.40628e+11){r+=0.0716894;}else{r+=1.12307;}}else{if(f[55]<1.04406){r+=-0.531217;}else{r+=-0.150248;}}}}else{if(f[40]<5.78198e+07){if(f[84]<4.65){r+=-0.59561;}else{if(f[0]<1.4028e+11){r+=-0.58132;}else{r+=-1.00937;}}}else{if(f[14]<1.40159e+11){if(f[34]<4.38191e+06){r+=0.500564;}else{r+=0.1222;}}else{r+=-0.580249;}}}}else{if(f[14]<1.40155e+11){if(f[28]<1.40596e+11){if(f[7]<1.71354e+08){r+=0.616886;}else{if(f[60]<0.952322){r+=0.425285;}else{r+=-0.161585;}}}else{r+=-0.570591;}}else{if(f[28]<1.42035e+11){if(f[41]<1.88879e+08){r+=-0.60083;}else{r+=0.0649543;}}else{r+=0.478479;}}}}}}else{if(f[67]<0.198601){if(f[0]<1.40098e+11){if(f[14]<1.40176e+11){if(f[84]<1.85){if(f[84]<0.5){if(f[82]<4.58031){r+=0.511822;}else{r+=-0.57672;}}else{if(f[26]<1.96306e+08){if(f[28]<1.40557e+11){if(f[84]<1.5){if(f[53]<1.84111){if(f[54]<0.532163){if(f[65]<0.705725){r+=-0.0221478;}else{r+=0.41041;}}else{r+=0.596531;}}else{r+=-0.388266;}}else{if(f[34]<5.89581e+06){r+=0.636783;}else{r+=0.803005;}}}else{r+=-0.16809;}}else{if(f[53]<0.87416){if(f[17]<1.03444e+06){r+=-0.10672;}else{if(f[18]<1.28257e+06){r+=-0.0117133;}else{if(f[12]<2.60182e+08){r+=0.614466;}else{r+=0.158011;}}}}else{if(f[80]<2.12067){if(f[61]<1.53004){r+=-0.0528801;}else{r+=0.35217;}}else{if(f[75]<20.8605){if(f[44]<0.448648){r+=-0.200053;}else{r+=-0.772971;}}else{r+=0.181264;}}}}}}else{if(f[55]<0.708458){if(f[41]<1.55204e+09){r+=-0.562282;}else{r+=0.588593;}}else{if(f[35]<2.7455e+07){if(f[14]<1.40088e+11){r+=-0.518298;}else{r+=0.591294;}}else{if(f[18]<6.50516e+07){if(f[0]<1.40069e+11){if(f[64]<3.28072){r+=-0.602308;}else{r+=-0.151151;}}else{r+=-0.926702;}}else{r+=0.104486;}}}}}else{if(f[28]<1.39985e+11){if(f[84]<2.5){r+=-0.601629;}else{r+=0.662124;}}else{if(f[14]<1.40181e+11){r+=-0.139566;}else{if(f[25]<2.03288e+09){r+=-0.601648;}else{r+=-0.0825646;}}}}}else{if(f[65]<0.284661){if(f[28]<1.68894e+11){if(f[80]<0.946046){if(f[0]<1.40613e+11){if(f[84]<1){r+=-0.492289;}else{if(f[14]<1.40145e+11){r+=0.63945;}else{r+=-0.170308;}}}else{if(f[0]<1.41645e+11){r+=-0.55277;}else{r+=0.0811291;}}}else{if(f[40]<7.58397e+08){if(f[53]<0.975577){if(f[20]<2.91778e+06){r+=-0.444189;}else{if(f[22]<1.16233e+08){r+=0.815422;}else{r+=0.216623;}}}else{if(f[39]<1.08141e+09){if(f[55]<0.92953){if(f[4]<1.25619e+07){r+=-0.406126;}else{r+=0.315607;}}else{if(f[58]<10.8427){if(f[52]<0.977303){r+=-0.084623;}else{r+=-0.595562;}}else{r+=-0.0612596;}}}else{if(f[28]<1.39925e+11){r+=-0.488523;}else{if(f[8]<1.64542e+08){r+=0.78972;}else{r+=0.160978;}}}}}else{if(f[35]<4.62485e+07){if(f[14]<1.40187e+11){if(f[41]<4.44649e+09){if(f[19]<4.81600e+06){if(f[14]<1.40049e+11){r+=0.0115326;}else{r+=0.759973;}}else{r+=-0.321014;}}else{if(f[12]<3.49806e+09){if(f[30]<5.05034e+06){r+=-0.550316;}else{r+=-0.0309105;}}else{r+=0.328744;}}}else{r+=-0.586387;}}else{if(f[29]<2.01641e+06){r+=-0.0922916;}else{if(f[71]<0.0756207){r+=-0.150187;}else{if(f[0]<1.40105e+11){r+=-0.191069;}else{r+=-0.593771;}}}}}}}else{if(f[18]<8.99337e+06){r+=0.549854;}else{r+=-0.00352034;}}}else{if(f[84]<1.5){if(f[84]<0.85){if(f[28]<1.4299e+11){if(f[0]<1.40261e+11){if(f[0]<1.40245e+11){if(f[0]<1.40236e+11){r+=-0.603697;}else{r+=-0.209269;}}else{if(f[28]<1.40076e+11){r+=-0.369512;}else{r+=0.57555;}}}else{r+=-0.6412;}}else{r+=0.555279;}}else{if(f[84]<1.15){if(f[37]<3.40209e+08){if(f[7]<1.29801e+08){r+=0.719634;}else{r+=0.222858;}}else{r+=0.182553;}}else{if(f[28]<1.39732e+11){r+=0.60738;}else{if(f[54]<3.97869){r+=-0.639132;}else{r+=0.0470977;}}}}}else{if(f[11]<6.34148e+08){if(f[37]<6.52621e+08){if(f[0]<1.40119e+11){r+=-0.11526;}else{if(f[12]<6.75863e+07){r+=-0.0715359;}else{r+=-0.603752;}}}else{if(f[42]<1.00665){if(f[14]<1.4053e+11){r+=0.557249;}else{r+=-0.40056;}}else{r+=-0.580292;}}}else{if(f[8]<3.14465e+08){if(f[28]<1.39948e+11){r+=-0.168961;}else{if(f[7]<1.23298e+08){r+=0.0900401;}else{r+=0.705834;}}}else{if(f[0]<1.41627e+11){r+=-0.573866;}else{r+=0.0255846;}}}}}}}else{if(f[14]<1.40239e+11){if(f[0]<1.40095e+11){if(f[84]<3.15){if(f[14]<1.3916e+11){if(f[14]<1.3914e+11){if(f[42]<0.829102){r+=0.384904;}else{if(f[38]<5.0342e+10){if(f[14]<1.39125e+11){r+=-0.599502;}else{if(f[84]<0.5){if(f[6]<4.69536e+07){r+=0.367745;}else{r+=0.000210421;}}else{r+=-0.569521;}}}else{if(f[20]<2.84231e+09){r+=0.469412;}else{r+=-0.432757;}}}}else{if(f[84]<0.5){if(f[0]<1.39219e+11){r+=-0.323496;}else{r+=0.555475;}}else{r+=-0.567883;}}}else{if(f[14]<1.40154e+11){if(f[22]<1.83927e+08){if(f[28]<1.39998e+11){if(f[14]<1.39648e+11){if(f[28]<1.39888e+11){if(f[28]<1.39841e+11){r+=0.605287;}else{r+=0.905526;}}else{r+=-0.639999;}}else{if(f[29]<1.1137e+08){if(f[14]<1.39658e+11){r+=-0.107323;}else{r+=-0.632036;}}else{if(f[14]<1.40037e+11){r+=0.023061;}else{r+=0.618695;}}}}else{if(f[38]<1.25953e+08){if(f[41]<1.8507e+09){if(f[0]<1.40089e+11){r+=0.589038;}else{r+=-0.0147801;}}else{if(f[0]<1.39994e+11){r+=-0.262724;}else{r+=0.530068;}}}else{if(f[84]<1.5){if(f[0]<1.39991e+11){r+=-0.586265;}else{r+=0.314662;}}else{if(f[67]<1.40812){r+=0.491455;}else{r+=-0.493621;}}}}}else{if(f[77]<1.30659){if(f[28]<1.40428e+11){if(f[14]<1.39666e+11){if(f[0]<1.39238e+11){r+=-0.505674;}else{r+=0.603459;}}else{if(f[28]<1.40185e+11){r+=-0.149811;}else{r+=0.570906;}}}else{if(f[0]<1.39343e+11){if(f[84]<2.65){r+=-0.547944;}else{r+=0.665078;}}else{if(f[14]<1.40125e+11){r+=-0.616124;}else{r+=-0.117022;}}}}else{if(f[40]<1.04586e+09){if(f[28]<1.3981e+11){if(f[14]<1.39669e+11){r+=0.58542;}else{r+=-0.325465;}}else{if(f[0]<1.40058e+11){r+=-0.617092;}else{r+=0.562855;}}}else{if(f[64]<0.639341){if(f[28]<1.40478e+11){r+=0.55333;}else{r+=-0.518162;}}else{if(f[0]<1.3999e+11){r+=-0.41243;}else{r+=0.41856;}}}}}}else{if(f[84]<2.85){if(f[21]<9.38123e+06){if(f[9]<2.74798e+07){r+=-0.519417;}else{if(f[13]<1.54933e+08){r+=0.629533;}else{r+=-0.0251992;}}}else{if(f[0]<1.40062e+11){if(f[84]<0.5){if(f[26]<1.33865e+08){r+=-0.729957;}else{r+=-0.94114;}}else{if(f[21]<1.89455e+09){r+=-0.647303;}else{r+=0.308506;}}}else{if(f[41]<1.28377e+08){if(f[84]<1.5){r+=-0.805706;}else{r+=-1.37814;}}else{if(f[83]<0.891364){r+=-1.00021;}else{r+=-0.786449;}}}}}else{if(f[28]<1.40034e+11){if(f[55]<0.618546){if(f[83]<1.73687){r+=-1.30335;}else{r+=-0.717928;}}else{if(f[14]<1.40213e+11){r+=-0.634979;}else{r+=0.0479955;}}}else{if(f[9]<6.11449e+08){if(f[28]<1.40142e+11){r+=0.797756;}else{if(f[39]<8.29753e+08){r+=1.28877;}else{r+=0.681911;}}}else{if(f[40]<1.49862e+09){if(f[28]<1.40121e+11){r+=0.703953;}else{r+=0.606201;}}else{if(f[28]<1.40142e+11){r+=0.686697;}else{r+=0.865049;}}}}}}}}else{if(f[41]<1.37087e+09){if(f[0]<1.40072e+11){if(f[39]<8.51212e+07){if(f[38]<5.40672e+07){if(f[13]<1.17627e+08){r+=-0.6106;}else{r+=-0.717719;}}else{if(f[13]<2.66576e+08){if(f[14]<1.40019e+11){if(f[0]<1.39389e+11){r+=-1.3032;}else{r+=-0.676199;}}else{if(f[13]<1.20708e+08){r+=-0.84355;}else{r+=-1.03163;}}}else{if(f[67]<2.54729){if(f[25]<8.93268e+07){r+=-0.817049;}else{r+=-1.59232;}}else{r+=-0.672195;}}}}else{if(f[82]<7.34912){if(f[42]<1.00471){if(f[82]<1.95379){if(f[54]<0.925004){r+=-0.815552;}else{r+=-0.671898;}}else{if(f[80]<0.529373){r+=0.538559;}else{r+=-0.609811;}}}else{if(f[28]<1.4063e+11){r+=0.61302;}else{r+=-0.594343;}}}else{if(f[0]<1.39986e+11){r+=-0.437556;}else{r+=0.562359;}}}}else{if(f[14]<1.39538e+11){if(f[51]<1.53441){r+=0.616522;}else{r+=-0.0960898;}}else{r+=-0.804317;}}}else{if(f[14]<1.4007e+11){if(f[17]<5.68359e+07){if(f[62]<9.3515){if(f[40]<1.9725e+10){r+=-0.619652;}else{r+=-0.163037;}}else{r+=-0.130512;}}else{if(f[38]<2.33337e+09){if(f[25]<9.69328e+09){r+=0.689452;}else{r+=-0.355725;}}else{if(f[46]<0.176845){r+=0.0572258;}else{r+=-0.680983;}}}}else{if(f[14]<1.40133e+11){if(f[28]<1.40171e+11){r+=-0.338717;}else{if(f[13]<6.04715e+08){if(f[0]<1.39989e+11){if(f[82]<8.94447){r+=0.83322;}else{r+=0.636609;}}else{r+=0.596958;}}else{r+=0.716679;}}}else{if(f[65]<3.00263){r+=-0.6148;}else{r+=0.34291;}}}}}}else{if(f[28]<1.40002e+11){if(f[67]<0.731444){if(f[9]<2.41154e+08){if(f[84]<1.5){if(f[84]<0.85){if(f[14]<1.39637e+11){r+=0.593183;}else{if(f[31]<5.86994e+08){if(f[83]<1.4087){r+=-0.833599;}else{r+=-0.646675;}}else{r+=0.335339;}}}else{if(f[84]<1.15){if(f[80]<1.73491){r+=0.702832;}else{r+=0.864851;}}else{if(f[28]<1.39673e+11){if(f[14]<1.39964e+11){r+=0.651283;}else{r+=-0.472183;}}else{if(f[83]<2.23677){r+=-0.771975;}else{r+=-0.621096;}}}}}else{if(f[39]<8.40746e+07){if(f[84]<5.40001){r+=-0.792545;}else{if(f[0]<1.40192e+11){r+=0.698229;}else{r+=-0.476065;}}}else{if(f[84]<3.15){if(f[83]<1.77901){if(f[84]<2.5){r+=-0.447496;}else{r+=-1.04751;}}else{if(f[84]<2.85){r+=-0.605207;}else{r+=-0.238682;}}}else{if(f[0]<1.7738e+11){r+=0.735629;}else{r+=-0.589648;}}}}}else{if(f[84]<0.85){if(f[14]<1.39653e+11){r+=0.597745;}else{if(f[28]<1.39441e+11){r+=-0.618209;}else{if(f[28]<1.39585e+11){if(f[28]<1.39459e+11){r+=-0.743291;}else{r+=-1.52568;}}else{if(f[18]<4.50126e+07){r+=-0.72424;}else{r+=0.0989832;}}}}}else{if(f[41]<5.53949e+08){if(f[38]<1.68013e+08){if(f[14]<1.40162e+11){if(f[18]<3.20181e+07){r+=-0.764883;}else{r+=-0.202432;}}else{r+=0.47942;}}else{if(f[42]<1.00242){if(f[4]<2.20933e+07){r+=-0.736138;}else{r+=0.165431;}}else{if(f[14]<1.38861e+11){r+=-0.447585;}else{r+=0.608539;}}}}else{if(f[28]<1.39756e+11){if(f[42]<1.00531){r+=0.614692;}else{if(f[82]<0.955842){r+=0.34543;}else{r+=-0.70274;}}}else{if(f[78]<0.704484){if(f[84]<1.15){r+=0.69902;}else{r+=-0.492105;}}else{if(f[0]<1.40155e+11){r+=-0.2728;}else{r+=0.612432;}}}}}}}else{if(f[84]<2.85){if(f[39]<6.43e+07){if(f[67]<0.94551){if(f[84]<1.15){if(f[84]<0.85){r+=-0.426424;}else{if(f[41]<1.43508e+08){r+=0.707865;}else{r+=0.148397;}}}else{if(f[42]<1.00083){r+=-0.580709;}else{r+=-0.925847;}}}else{if(f[0]<1.4018e+11){if(f[56]<1.00026){if(f[84]<4.60001){r+=-0.802174;}else{r+=0.670064;}}else{if(f[0]<1.40145e+11){r+=-0.609741;}else{r+=-1.11371;}}}else{if(f[84]<5.40001){if(f[35]<6.36569e+06){r+=0.327179;}else{r+=-0.780703;}}else{if(f[38]<5.40571e+07){r+=-0.700059;}else{r+=-1.33372;}}}}}else{if(f[84]<2.5){if(f[84]<0.85){if(f[14]<1.39677e+11){if(f[42]<1.01851){r+=0.599984;}else{r+=-0.543675;}}else{if(f[14]<1.39815e+11){r+=-1.23469;}else{r+=-0.716585;}}}else{if(f[84]<1.15){if(f[1]<2.81242e+07){r+=0.727056;}else{r+=-0.462131;}}else{if(f[10]<9.9126e+07){r+=-0.908196;}else{r+=0.379726;}}}}else{if(f[41]<1.79131e+08){if(f[28]<1.39981e+11){if(f[14]<1.38984e+11){r+=-0.784847;}else{r+=-1.87246;}}else{if(f[42]<1.01691){r+=-1.03205;}else{r+=-0.564322;}}}else{if(f[41]<4.6039e+08){if(f[12]<1.09949e+08){r+=-0.679624;}else{r+=-1.02383;}}else{if(f[64]<4.39844){r+=-0.621943;}else{r+=-0.129432;}}}}}}else{if(f[80]<1.25616){if(f[14]<1.38851e+11){r+=-0.574713;}else{if(f[29]<8.50047e+07){if(f[30]<6.93126e+07){if(f[0]<1.40145e+11){r+=1.01914;}else{r+=0.637252;}}else{r+=-0.266195;}}else{if(f[80]<1.04664){if(f[8]<2.73086e+08){r+=-0.00830417;}else{r+=0.575364;}}else{r+=-0.556041;}}}}else{if(f[38]<1.87991e+08){if(f[0]<1.7719e+11){r+=0.619574;}else{r+=-0.556688;}}else{if(f[9]<3.06812e+08){if(f[9]<1.26354e+08){if(f[1]<1.09437e+07){r+=-0.464338;}else{r+=0.0349832;}}else{if(f[29]<7.36053e+07){r+=-1.19454;}else{r+=-0.575274;}}}else{r+=0.34212;}}}}}}else{if(f[14]<1.40181e+11){if(f[42]<1.02349){if(f[84]<2.85){if(f[0]<1.40159e+11){if(f[14]<1.40176e+11){if(f[14]<1.39995e+11){if(f[21]<6.61192e+07){r+=0.591969;}else{r+=-0.462764;}}else{if(f[40]<7.6024e+07){r+=-0.613564;}else{r+=-0.696153;}}}else{if(f[84]<1.85){if(f[83]<0.70135){r+=-0.814182;}else{r+=-0.560358;}}else{if(f[28]<1.40149e+11){r+=0.72203;}else{r+=-0.481172;}}}}else{if(f[0]<1.40175e+11){if(f[14]<1.40068e+11){if(f[82]<3.1193e+08){r+=-0.652513;}else{r+=0.516966;}}else{if(f[36]<1.50384e+08){r+=0.67739;}else{r+=-0.663507;}}}else{if(f[28]<1.72086e+11){if(f[14]<1.40011e+11){r+=-0.307832;}else{r+=-0.616593;}}else{r+=0.59241;}}}}else{if(f[12]<7.3535e+08){if(f[28]<1.42028e+11){if(f[14]<1.39441e+11){if(f[0]<1.40201e+11){r+=0.443328;}else{r+=-0.320614;}}else{if(f[28]<1.40009e+11){r+=0.549987;}else{r+=-0.573647;}}}else{if(f[14]<1.39298e+11){r+=-0.571984;}else{if(f[37]<1.51686e+08){r+=0.62696;}else{r+=1.0362;}}}}else{if(f[14]<1.40018e+11){if(f[26]<3.07994e+08){if(f[14]<1.39373e+11){r+=0.72136;}else{r+=-0.450093;}}else{if(f[57]<12.8092){r+=-0.602935;}else{r+=0.204119;}}}else{if(f[84]<3.5){if(f[21]<5.69145e+08){r+=0.675684;}else{r+=0.863819;}}else{if(f[56]<1.00035){r+=-0.637554;}else{r+=0.827122;}}}}}}else{if(f[14]<1.39991e+11){if(f[84]<2.65){if(f[14]<1.39515e+11){if(f[84]<0.5){if(f[0]<1.70321e+11){r+=-0.582258;}else{r+=0.320789;}}else{if(f[57]<8.55762){r+=0.573107;}else{r+=-0.412527;}}}else{if(f[83]<1.15625){if(f[9]<3.45205e+10){r+=0.609111;}else{r+=-0.176237;}}else{if(f[84]<0.5){r+=0.587267;}else{r+=0.817207;}}}}else{if(f[28]<1.40057e+11){if(f[22]<9.50211e+07){r+=-0.406483;}else{if(f[50]<1.14983){r+=0.15227;}else{r+=0.533854;}}}else{r+=-0.607195;}}}else{if(f[55]<0.496751){r+=-0.614375;}else{r+=-0.710316;}}}}else{if(f[84]<1.85){if(f[28]<1.40009e+11){if(f[38]<1.3138e+08){if(f[41]<2.03528e+08){r+=-0.699157;}else{if(f[80]<1.64765){if(f[37]<5.06157e+07){r+=1.16198;}else{r+=0.733641;}}else{r+=-0.439084;}}}else{if(f[37]<7.465e+07){if(f[39]<3.20737e+08){r+=0.625894;}else{if(f[12]<1.10902e+08){r+=-0.119553;}else{r+=-0.536719;}}}else{if(f[39]<9.61156e+07){r+=0.371074;}else{if(f[39]<1.21047e+08){r+=-0.192998;}else{r+=-0.614445;}}}}}else{if(f[83]<0.701935){if(f[40]<7.54782e+07){if(f[38]<5.40618e+07){r+=-0.670136;}else{if(f[28]<1.40174e+11){r+=-1.57383;}else{r+=-0.565534;}}}else{if(f[28]<1.40154e+11){r+=-0.824515;}else{r+=-0.586382;}}}else{if(f[36]<7.82455e+09){if(f[47]<11.8518){if(f[28]<1.40009e+11){r+=-0.336569;}else{r+=-0.621829;}}else{r+=-0.127492;}}else{r+=-0.11294;}}}}else{if(f[28]<1.40178e+11){if(f[0]<1.41755e+11){if(f[67]<0.516567){if(f[28]<1.40033e+11){if(f[29]<2.47076e+07){r+=-0.644675;}else{r+=0.420773;}}else{if(f[10]<7.79772e+08){r+=1.01059;}else{r+=0.233023;}}}else{if(f[84]<3.15){if(f[25]<5.27931e+07){r+=-0.789693;}else{r+=0.693065;}}else{if(f[35]<1.58479e+07){r+=-0.107223;}else{r+=-0.853015;}}}}else{if(f[28]<1.40044e+11){r+=-0.631339;}else{r+=0.330719;}}}else{if(f[42]<1.00953){if(f[10]<8.16126e+09){if(f[28]<1.40843e+11){if(f[11]<1.64681e+09){r+=-0.620197;}else{r+=-1.4393;}}else{r+=0.380469;}}else{r+=0.895309;}}else{if(f[6]<1.08343e+08){if(f[28]<1.4022e+11){r+=0.622085;}else{if(f[55]<2.50521){r+=1.19031;}else{r+=0.445033;}}}else{r+=0.211831;}}}}}}}}else{if(f[28]<1.41476e+11){if(f[14]<1.45836e+11){if(f[0]<1.41151e+11){if(f[38]<4.6513e+07){if(f[0]<1.40021e+11){r+=-0.626976;}else{if(f[14]<1.40531e+11){if(f[0]<1.40358e+11){if(f[8]<3.19051e+08){r+=-0.642559;}else{r+=-0.0734795;}}else{if(f[28]<1.39998e+11){r+=0.613719;}else{r+=-0.200053;}}}else{if(f[84]<2.8){if(f[0]<1.40205e+11){if(f[9]<2.44921e+07){r+=0.712327;}else{r+=0.940161;}}else{if(f[84]<1.15){r+=0.631215;}else{r+=-0.399058;}}}else{r+=-0.64389;}}}}else{if(f[80]<6.82293){if(f[14]<1.40808e+11){if(f[84]<1.85){if(f[35]<5.46136e+06){if(f[0]<1.40391e+11){r+=-0.597518;}else{r+=0.654372;}}else{if(f[10]<7.01267e+10){r+=-0.608619;}else{r+=1.04187;}}}else{if(f[26]<1.51082e+08){if(f[28]<1.39813e+11){r+=-0.593346;}else{r+=0.287587;}}else{if(f[28]<1.40387e+11){r+=-0.0385392;}else{r+=-0.571775;}}}}else{if(f[80]<1.57042){if(f[21]<8.84564e+06){if(f[0]<1.40328e+11){r+=-0.620145;}else{r+=0.719963;}}else{if(f[42]<0.95861){r+=-0.19571;}else{r+=-0.610044;}}}else{if(f[84]<12){if(f[28]<1.39996e+11){r+=0.256315;}else{r+=-0.6184;}}else{if(f[0]<1.39805e+11){r+=-0.700045;}else{r+=0.659101;}}}}}else{if(f[84]<12){if(f[13]<1.30336e+08){r+=0.717439;}else{if(f[12]<9.84917e+07){r+=-0.174211;}else{r+=-0.609684;}}}else{if(f[14]<1.40849e+11){r+=-0.518389;}else{if(f[34]<4.32687e+08){r+=0.594264;}else{r+=0.915126;}}}}}}else{if(f[28]<1.39956e+11){if(f[14]<1.40288e+11){if(f[0]<1.41895e+11){if(f[84]<2.65){r+=0.646012;}else{r+=-0.614214;}}else{r+=-0.584199;}}else{if(f[14]<1.40312e+11){if(f[0]<1.41728e+11){if(f[84]<2.5){if(f[3]<1.02997e+07){r+=0.17104;}else{r+=0.764986;}}else{r+=-0.663653;}}else{r+=-0.583767;}}else{if(f[41]<2.65233e+08){if(f[28]<1.39895e+11){if(f[14]<1.4033e+11){r+=0.0424673;}else{r+=-0.594817;}}else{if(f[41]<7.95282e+07){r+=-0.61803;}else{r+=-1.10354;}}}else{if(f[0]<1.4134e+11){if(f[56]<1.00468){r+=0.429522;}else{r+=-0.687867;}}else{if(f[38]<5.27536e+10){r+=-0.601814;}else{r+=0.088132;}}}}}}else{if(f[84]<1.5){if(f[42]<1.01039){if(f[28]<1.40219e+11){if(f[55]<0.967534){r+=-0.733904;}else{if(f[14]<1.40418e+11){r+=-0.904341;}else{r+=-1.40334;}}}else{if(f[14]<1.42937e+11){r+=-0.60842;}else{r+=0.323898;}}}else{if(f[28]<1.40138e+11){if(f[0]<1.42899e+11){if(f[56]<1.00804){r+=0.332052;}else{r+=-0.704657;}}else{if(f[56]<1.00774){r+=-0.386601;}else{r+=-0.615005;}}}else{if(f[56]<1.00813){if(f[14]<1.40974e+11){r+=-0.538969;}else{r+=0.598787;}}else{if(f[0]<1.42927e+11){r+=-1.47068;}else{r+=-0.625194;}}}}}else{if(f[83]<0.738491){if(f[41]<2.89984e+08){if(f[14]<1.40289e+11){if(f[42]<1.00908){r+=-0.661331;}else{r+=0.795654;}}else{if(f[14]<1.40852e+11){r+=-0.765563;}else{r+=0.506957;}}}else{if(f[84]<2.15){if(f[13]<1.66715e+09){r+=0.741413;}else{r+=1.12941;}}else{if(f[84]<4.85){r+=-0.61841;}else{r+=0.54304;}}}}else{if(f[35]<3.99194e+08){if(f[38]<1.06135e+08){if(f[84]<4.85){r+=-1.00525;}else{r+=0.346518;}}else{if(f[14]<1.41531e+11){r+=0.635837;}else{r+=-0.527363;}}}else{if(f[84]<2.15){if(f[42]<1.00657){r+=0.852881;}else{r+=-0.184865;}}else{if(f[84]<4.65){r+=-0.621622;}else{r+=0.492526;}}}}}}}}else{if(f[84]<2.85){if(f[14]<1.54903e+11){if(f[56]<1.0366){r+=0.94585;}else{if(f[43]<0.127167){if(f[1]<1.4544e+07){r+=0.367009;}else{r+=-0.465937;}}else{if(f[82]<0.872477){r+=0.59395;}else{r+=0.638162;}}}}else{if(f[0]<1.40958e+11){r+=-0.638846;}else{r+=0.271697;}}}else{r+=-0.637011;}}}else{if(f[56]<0.99677){if(f[42]<1.00577){if(f[56]<0.981678){if(f[0]<1.41157e+11){r+=-0.608538;}else{r+=0.438309;}}else{if(f[14]<1.40945e+11){if(f[84]<1.15){r+=-0.670618;}else{if(f[42]<0.998794){if(f[0]<1.40313e+11){if(f[84]<2.15){r+=0.575204;}else{r+=0.618357;}}else{if(f[83]<1.30865){r+=0.944758;}else{r+=0.568109;}}}else{r+=-0.304775;}}}else{if(f[84]<0.8){if(f[28]<1.4197e+11){r+=-0.393558;}else{if(f[38]<8.54934e+07){r+=-0.321904;}else{if(f[32]<7.06726e+08){r+=0.601221;}else{r+=-0.155437;}}}}else{r+=-0.607384;}}}}else{if(f[84]<0.5){if(f[0]<1.42735e+11){r+=0.0968494;}else{r+=0.600804;}}else{if(f[84]<1.65){if(f[62]<0.0993379){r+=-0.126417;}else{r+=-0.616192;}}else{if(f[0]<1.42114e+11){r+=-0.142222;}else{r+=-0.769692;}}}}}else{if(f[0]<1.41341e+11){if(f[21]<2.23768e+10){if(f[65]<0.299766){if(f[82]<0.872539){r+=-0.583569;}else{r+=-0.744483;}}else{r+=-0.607451;}}else{r+=0.121253;}}else{if(f[28]<1.43121e+11){r+=-0.410253;}else{if(f[41]<1.88179e+10){r+=0.604551;}else{r+=-0.00348166;}}}}}}}} return r;}
    static double p3(double[] f){double r=0;if(f[28]<1.39435e+11){if(f[56]<1.00186){if(f[28]<1.39131e+11){if(f[7]<3.89891e+10){if(f[56]<1.00107){r+=-0.942857;}else{r+=-0.25;}}else{r+=-0.25;}}else{if(f[0]<1.40016e+11){if(f[42]<1.0015){r+=-0.111111;}else{r+=0.925926;}}else{r+=-0.714286;}}}else{if(f[13]<4.25751e+10){if(f[28]<1.39428e+11){if(f[9]<2.06493e+08){r+=-0.999191;}else{if(f[41]<2.96005e+08){if(f[42]<1.0095){if(f[41]<2.93053e+08){if(f[55]<0.22476){if(f[19]<1.28502e+07){r+=0.25;}else{r+=-0.684211;}}else{if(f[28]<1.39427e+11){r+=-0.985185;}else{if(f[1]<6.1629e+07){r+=-0;}else{r+=-0.5;}}}}else{r+=-0;}}else{if(f[14]<1.4002e+11){r+=-0.666667;}else{r+=0.954023;}}}else{if(f[29]<2.99429e+09){if(f[17]<1.80369e+09){r+=-0.990055;}else{r+=-0.25;}}else{if(f[13]<1.68921e+10){if(f[8]<4.08121e+09){r+=-0.2;}else{r+=0.6;}}else{r+=-0.8;}}}}}else{if(f[14]<1.39879e+11){if(f[14]<1.39823e+11){if(f[19]<4.99148e+06){r+=-0;}else{if(f[5]<6.7775e+07){r+=-0.948718;}else{r+=-0.25;}}}else{if(f[57]<1.75292){if(f[2]<1.81781e+07){if(f[1]<3.99396e+07){r+=0.5;}else{r+=-0.272727;}}else{r+=0.8125;}}else{if(f[14]<1.39838e+11){r+=-0.555556;}else{r+=-0;}}}}else{if(f[42]<1.00692){r+=-0.980198;}else{r+=-0;}}}}else{if(f[76]<1.0059){if(f[13]<6.41186e+10){r+=0.777778;}else{r+=-0.555556;}}else{r+=-0.666667;}}}}else{if(f[69]<0.476461){if(f[0]<1.40101e+11){if(f[0]<1.39989e+11){if(f[28]<1.39984e+11){if(f[41]<2.20026e+09){if(f[70]<1.00282){if(f[14]<1.39651e+11){r+=0.941176;}else{if(f[49]<1.65858){r+=-0.971223;}else{if(f[8]<2.88924e+08){r+=0.636364;}else{r+=-0.5;}}}}else{if(f[0]<1.39395e+11){r+=-0.851852;}else{r+=0.998023;}}}else{if(f[83]<4.10434){if(f[67]<0.193915){if(f[28]<1.39638e+11){r+=-0.5;}else{if(f[0]<1.39428e+11){r+=0.25;}else{r+=0.942029;}}}else{if(f[83]<2.31929){if(f[8]<3.13643e+08){r+=0.25;}else{r+=-0.5;}}else{r+=-0.963964;}}}else{if(f[66]<0.117962){r+=-0.25;}else{r+=-0.996835;}}}}else{if(f[82]<9.24796){if(f[49]<5.09993){if(f[66]<1.33276){if(f[43]<10.2352){if(f[0]<1.39779e+11){if(f[0]<1.33029e+11){if(f[1]<5.58506e+06){r+=-0;}else{r+=0.714286;}}else{if(f[14]<1.39673e+11){r+=0.0588235;}else{r+=-0.995895;}}}else{if(f[54]<0.758909){if(f[42]<0.995603){r+=0.810526;}else{r+=-0.867238;}}else{if(f[6]<1.58022e+07){r+=-0.946865;}else{r+=-0.72876;}}}}else{if(f[14]<1.40163e+11){if(f[21]<4.20141e+07){r+=0.918367;}else{r+=-0.555556;}}else{r+=-0.862069;}}}else{if(f[0]<1.39779e+11){if(f[14]<1.46847e+11){if(f[77]<0.998814){if(f[54]<3.5838){r+=-0.982456;}else{r+=-0.25;}}else{if(f[39]<1.60194e+08){r+=0.777778;}else{r+=-0.789474;}}}else{r+=0.733333;}}else{if(f[28]<1.40548e+11){if(f[55]<0.787163){if(f[28]<1.40179e+11){r+=-0.717391;}else{r+=0.390135;}}else{if(f[27]<5.13244e+08){r+=-0.894231;}else{r+=-0.254902;}}}else{if(f[28]<1.40641e+11){r+=0.982609;}else{r+=-0.909091;}}}}}else{if(f[28]<1.40101e+11){if(f[27]<8.63493e+07){if(f[55]<2.51712){r+=0.0909091;}else{r+=0.991886;}}else{r+=-0.555556;}}else{if(f[13]<1.52978e+09){if(f[27]<3.96107e+08){r+=-0.976879;}else{r+=-0.111111;}}else{if(f[23]<4.822e+07){r+=0.806452;}else{r+=-0.6;}}}}}else{if(f[14]<1.40092e+11){if(f[49]<0.824333){if(f[0]<1.39984e+11){r+=-0.764706;}else{if(f[2]<1.72264e+06){r+=-0.25;}else{r+=0.75;}}}else{r+=-0.9799;}}else{if(f[70]<1.00566){if(f[28]<1.40181e+11){if(f[56]<1.00144){if(f[0]<1.39984e+11){if(f[28]<1.4018e+11){r+=-0.861635;}else{r+=-0.0196078;}}else{if(f[28]<1.40177e+11){r+=-0.62963;}else{r+=0.279743;}}}else{if(f[28]<1.40069e+11){r+=-0.454545;}else{r+=0.973856;}}}else{if(f[69]<0.15009){if(f[54]<0.475521){if(f[2]<5.07182e+06){r+=-0.84;}else{r+=0.2;}}else{if(f[54]<0.548273){r+=0.586207;}else{r+=-0.210526;}}}else{if(f[42]<0.994827){r+=-0.733333;}else{if(f[11]<7.02889e+07){r+=0.88412;}else{r+=0.627178;}}}}}else{r+=-0.946667;}}}}}else{if(f[14]<1.40037e+11){if(f[29]<1.79491e+07){if(f[1]<4.54468e+07){if(f[27]<1.43025e+08){if(f[0]<1.40025e+11){r+=-0.666667;}else{r+=0.454545;}}else{if(f[13]<1.5531e+09){r+=-0.996937;}else{r+=0.25;}}}else{if(f[11]<1.0715e+08){r+=-0;}else{r+=0.636364;}}}else{if(f[25]<2.81112e+08){if(f[27]<7.43741e+07){r+=-0.833333;}else{if(f[56]<0.992852){r+=-0.6;}else{if(f[40]<6.73076e+08){if(f[4]<2.08658e+06){r+=0.25;}else{r+=0.975;}}else{if(f[0]<1.40074e+11){r+=-0.25;}else{r+=0.5;}}}}}else{if(f[28]<1.40045e+11){r+=-0;}else{r+=-0.902439;}}}}else{if(f[14]<1.40184e+11){if(f[30]<6.69586e+08){if(f[55]<5.39433){if(f[28]<1.40184e+11){if(f[0]<1.40071e+11){if(f[42]<0.999452){if(f[40]<1.22114e+09){r+=0.988194;}else{r+=0.705545;}}else{if(f[0]<1.40067e+11){r+=-0.937349;}else{r+=0.423077;}}}else{if(f[8]<2.86008e+08){if(f[0]<1.40071e+11){r+=0.25;}else{r+=0.991518;}}else{r+=-0.666667;}}}else{if(f[14]<1.40114e+11){if(f[82]<1.09187){if(f[63]<0.406376){r+=0.75;}else{r+=-0.777778;}}else{if(f[83]<3.07667){r+=0.833002;}else{r+=0.985776;}}}else{if(f[0]<1.40024e+11){r+=0.25;}else{r+=-0.866667;}}}}else{if(f[28]<1.40067e+11){if(f[0]<1.40022e+11){r+=0.692308;}else{r+=-0.25;}}else{r+=-0.955056;}}}else{if(f[0]<1.40075e+11){r+=-0.992366;}else{if(f[1]<5.68095e+07){r+=0.993266;}else{r+=-0.692308;}}}}else{if(f[14]<1.40481e+11){if(f[11]<1.01844e+09){if(f[6]<1.41136e+07){r+=-0.984981;}else{if(f[30]<1.53784e+06){r+=0.4;}else{if(f[62]<0.342096){r+=0.25;}else{r+=-0.785714;}}}}else{if(f[39]<1.69932e+10){if(f[66]<0.688024){if(f[0]<1.40088e+11){r+=0.932203;}else{r+=0.25;}}else{if(f[2]<4.20962e+07){r+=-0.5;}else{r+=0.25;}}}else{r+=-0.636364;}}}else{if(f[55]<1.41769){if(f[28]<1.40698e+11){if(f[39]<1.46891e+08){if(f[28]<1.40399e+11){if(f[28]<1.40378e+11){r+=-0.636364;}else{r+=0.674419;}}else{r+=-0.851852;}}else{if(f[1]<3.30228e+08){if(f[28]<1.40374e+11){r+=-0;}else{r+=0.959677;}}else{r+=-0.555556;}}}else{if(f[39]<1.98995e+08){r+=0.25;}else{r+=-0.846154;}}}else{if(f[14]<1.40516e+11){r+=0.571429;}else{if(f[63]<1.58276){if(f[22]<1.58269e+08){r+=0.272727;}else{r+=-0.538462;}}else{r+=-0.948718;}}}}}}}}else{if(f[83]<2.35815){if(f[25]<8.0668e+07){if(f[70]<1.26218){if(f[56]<1.00126){if(f[42]<0.996571){r+=0.870968;}else{if(f[0]<1.40208e+11){if(f[22]<1.01408e+08){if(f[52]<1.17304){if(f[67]<0.848793){r+=-0.607843;}else{r+=0.428571;}}else{if(f[77]<1.16968){r+=-0.9819;}else{r+=-0.285714;}}}else{if(f[32]<6.7942e+06){if(f[35]<9.62311e+07){r+=-0.6;}else{r+=-0;}}else{r+=0.906977;}}}else{if(f[37]<2.64855e+10){if(f[16]<9.11293e+07){if(f[22]<5.16844e+08){r+=-0.990598;}else{r+=-0.25;}}else{r+=-0.25;}}else{r+=0.5;}}}}else{if(f[39]<1.74663e+08){if(f[37]<1.78756e+07){if(f[27]<6.87228e+07){if(f[14]<1.40589e+11){r+=-0;}else{r+=-0.949367;}}else{if(f[0]<1.40189e+11){if(f[14]<1.4061e+11){r+=0.142857;}else{r+=-0.684211;}}else{if(f[9]<6.00325e+06){r+=-0.111111;}else{r+=0.944828;}}}}else{if(f[56]<1.00461){if(f[27]<6.43888e+07){r+=-0.977901;}else{if(f[14]<1.40178e+11){r+=-0.783784;}else{r+=0.114428;}}}else{if(f[0]<1.41986e+11){if(f[36]<6.55924e+06){r+=-0.2;}else{r+=-0.991957;}}else{if(f[0]<1.42856e+11){r+=0.666667;}else{r+=-0.25;}}}}}else{if(f[56]<1.00367){if(f[28]<1.39875e+11){if(f[14]<1.39778e+11){if(f[28]<1.39491e+11){r+=-0.714286;}else{r+=0.571429;}}else{if(f[14]<1.40267e+11){r+=0.92029;}else{r+=-0.5;}}}else{if(f[14]<1.40305e+11){if(f[0]<1.40233e+11){r+=-0.111111;}else{r+=-0.870968;}}else{if(f[0]<1.41351e+11){r+=0.897436;}else{r+=0.25;}}}}else{if(f[15]<6.8499e+06){r+=-0;}else{r+=-0.909091;}}}}}else{r+=0.973684;}}else{if(f[56]<1.00455){if(f[14]<1.40199e+11){if(f[9]<3.0384e+08){if(f[28]<1.39871e+11){if(f[67]<0.557351){if(f[56]<1.00313){if(f[28]<1.39458e+11){r+=0.222222;}else{r+=0.910112;}}else{r+=-0.25;}}else{if(f[18]<2.48598e+07){r+=-0.714286;}else{r+=0.555556;}}}else{if(f[70]<0.999345){if(f[37]<8.23215e+07){if(f[28]<1.39995e+11){r+=0.483516;}else{r+=-0.543307;}}else{if(f[66]<1.0426){r+=-0.944411;}else{r+=-0.22449;}}}else{if(f[14]<1.39738e+11){if(f[0]<1.40478e+11){r+=0.917808;}else{r+=-0.666667;}}else{if(f[41]<7.37828e+08){r+=-0.823009;}else{r+=0.535714;}}}}}else{if(f[14]<1.39208e+11){if(f[28]<1.4006e+11){if(f[0]<1.40262e+11){r+=-0.666667;}else{if(f[63]<2.19405){r+=0.986207;}else{r+=-0;}}}else{if(f[28]<1.40094e+11){if(f[40]<2.26483e+08){r+=-0.6;}else{r+=0.5;}}else{r+=-0.916667;}}}else{if(f[29]<2.46949e+07){if(f[56]<1.00193){if(f[11]<1.23587e+08){r+=0.411765;}else{r+=-0.838188;}}else{if(f[5]<1.28935e+07){r+=-0.538462;}else{r+=0.846154;}}}else{if(f[42]<1.006){if(f[67]<0.125713){r+=-0.8;}else{r+=0.575092;}}else{if(f[14]<1.40121e+11){r+=-0.799043;}else{r+=0.630769;}}}}}}else{if(f[14]<1.41058e+11){if(f[7]<8.86533e+07){if(f[14]<1.40752e+11){if(f[0]<1.402e+11){if(f[8]<4.99377e+07){r+=-0.692308;}else{r+=0.692308;}}else{if(f[42]<1.0025){r+=-0.964664;}else{r+=-0.0769231;}}}else{r+=0.684211;}}else{if(f[8]<4.48124e+08){if(f[83]<1.69541){if(f[67]<1.00121){r+=0.88173;}else{r+=-0.416667;}}else{if(f[42]<1.00178){r+=-0.887218;}else{r+=0.801394;}}}else{if(f[14]<1.40232e+11){r+=0.769231;}else{if(f[58]<3.08607){r+=-0.710843;}else{r+=0.307692;}}}}}else{if(f[28]<1.4073e+11){if(f[0]<1.41178e+11){r+=-0.25;}else{r+=0.75;}}else{if(f[1]<1.99933e+09){r+=-0.987013;}else{r+=-0.25;}}}}}else{if(f[42]<1.00869){if(f[8]<2.42937e+07){r+=0.666667;}else{if(f[64]<5.24006){if(f[56]<1.00487){if(f[0]<1.41087e+11){r+=-0.764706;}else{r+=0.4;}}else{r+=-0.995825;}}else{r+=0.25;}}}else{if(f[70]<0.980709){r+=-0.924528;}else{r+=0.970149;}}}}}else{if(f[12]<2.17096e+08){if(f[0]<1.40264e+11){if(f[42]<1.00161){if(f[14]<1.40465e+11){if(f[67]<0.438321){if(f[40]<1.0825e+09){if(f[68]<0.340352){if(f[27]<4.40375e+08){r+=-0.938165;}else{r+=-0.457143;}}else{if(f[52]<1.03137){r+=0.34375;}else{r+=-0.913043;}}}else{if(f[37]<9.23146e+07){if(f[14]<1.40187e+11){r+=0.825243;}else{r+=-0.733333;}}else{if(f[70]<0.998796){r+=-0.160976;}else{r+=-0.890411;}}}}else{if(f[23]<3.74303e+07){if(f[48]<0.436979){if(f[22]<2.34106e+07){r+=-0.692308;}else{r+=0.5;}}else{if(f[18]<196674){r+=-0;}else{r+=-0.971429;}}}else{if(f[70]<0.999527){if(f[39]<2.53584e+08){r+=0.938776;}else{r+=-0.571429;}}else{if(f[46]<0.344034){r+=-0.2;}else{r+=-0.857143;}}}}}else{if(f[28]<1.40403e+11){if(f[14]<1.40609e+11){if(f[15]<1.06341e+07){if(f[0]<1.40171e+11){r+=-0.714286;}else{r+=0.364706;}}else{if(f[33]<3.45941e+06){r+=-0.876923;}else{r+=-0;}}}else{if(f[2]<502311){r+=-0;}else{if(f[49]<0.847059){r+=-0.444444;}else{r+=-0.904348;}}}}else{if(f[28]<1.40933e+11){if(f[14]<1.40617e+11){if(f[0]<1.40177e+11){r+=-0.804878;}else{r+=0.84573;}}else{if(f[29]<7.01516e+06){r+=-0.515152;}else{r+=0.673469;}}}else{r+=-0.957447;}}}}else{if(f[42]<1.00627){if(f[25]<5.32117e+07){r+=-0.933333;}else{if(f[14]<1.40011e+11){if(f[28]<1.39462e+11){if(f[14]<1.39801e+11){r+=-0.75;}else{r+=-0;}}else{if(f[14]<1.40009e+11){r+=0.972923;}else{r+=0.36;}}}else{if(f[0]<1.40256e+11){r+=0.6;}else{if(f[2]<8.00648e+06){r+=-0.914894;}else{r+=-0.25;}}}}}else{if(f[2]<4.5428e+07){r+=-0.983539;}else{r+=-0.25;}}}}else{if(f[23]<5.31966e+07){if(f[82]<4.91207e+08){if(f[28]<1.42849e+11){if(f[82]<1.59714){if(f[22]<2.56713e+07){r+=-0.692308;}else{r+=0.5;}}else{if(f[23]<3.81531e+07){if(f[11]<1.70721e+08){r+=-0.996573;}else{r+=-0.25;}}else{if(f[14]<1.40189e+11){r+=-0.754839;}else{r+=-0.99278;}}}}else{r+=0.25;}}else{r+=0.769231;}}else{if(f[70]<0.998585){if(f[42]<1.01371){if(f[37]<7.74089e+07){if(f[14]<1.40189e+11){if(f[28]<1.40011e+11){r+=0.881326;}else{r+=-0.692308;}}else{if(f[35]<2.12722e+07){r+=0.4;}else{r+=-0.809524;}}}else{if(f[28]<1.40578e+11){if(f[14]<1.39108e+11){r+=0.774194;}else{r+=-0.618064;}}else{if(f[35]<3.86288e+08){r+=0.965937;}else{r+=-0.801653;}}}}else{if(f[35]<2.16782e+08){if(f[83]<2.60743){if(f[70]<0.992089){r+=0.153846;}else{r+=-0.929825;}}else{r+=-0.998532;}}else{if(f[81]<1.69891){if(f[22]<3.16532e+08){r+=0.901639;}else{r+=-0.25;}}else{if(f[69]<0.449087){r+=-0.932584;}else{r+=0.272727;}}}}}else{if(f[27]<9.84715e+07){if(f[25]<8.60281e+07){r+=-0.963303;}else{r+=0.932203;}}else{if(f[41]<2.73557e+08){if(f[11]<7.71484e+07){r+=-0.75;}else{r+=0.25;}}else{if(f[43]<3.60877e+08){r+=-0.997395;}else{r+=-0.25;}}}}}}}else{if(f[56]<0.711343){if(f[14]<1.40601e+11){if(f[0]<1.40262e+11){r+=0.947368;}else{r+=0.25;}}else{r+=-0.857143;}}else{if(f[36]<8.65504e+07){if(f[0]<1.4026e+11){if(f[64]<2.93966){if(f[52]<1.0884){if(f[28]<1.3999e+11){if(f[29]<6.94055e+06){r+=0.764706;}else{r+=-0.454545;}}else{if(f[55]<0.60192){r+=0.384615;}else{r+=-0.636364;}}}else{if(f[64]<2.16004){if(f[32]<5.73673e+06){r+=-0.938462;}else{r+=-0.333333;}}else{r+=0.111111;}}}else{if(f[15]<2.57776e+07){r+=0.888889;}else{r+=-0.2;}}}else{if(f[22]<1.29386e+07){if(f[11]<1.532e+08){r+=-0.5;}else{r+=0.714286;}}else{if(f[82]<12.1484){if(f[71]<0.0942381){r+=-0.111111;}else{if(f[81]<0.750402){r+=-0.384615;}else{r+=-0.963576;}}}else{if(f[28]<1.39999e+11){if(f[2]<1.22811e+06){r+=-0;}else{r+=0.555556;}}else{r+=-0.692308;}}}}}else{if(f[14]<1.38992e+11){if(f[0]<1.41029e+11){r+=0.666667;}else{r+=-0.555556;}}else{if(f[49]<0.217935){if(f[0]<1.40239e+11){r+=0.5;}else{r+=-0.714286;}}else{if(f[41]<1.66174e+09){if(f[52]<0.557659){if(f[0]<1.40449e+11){r+=0.906977;}else{r+=-0.733333;}}else{if(f[37]<1.34999e+08){r+=-0.446809;}else{r+=-0.96778;}}}else{if(f[29]<1.54472e+06){r+=-0.0909091;}else{if(f[12]<2.23877e+10){r+=-0.997095;}else{r+=-0;}}}}}}}}}}}else{if(f[14]<1.39855e+11){if(f[28]<1.40231e+11){if(f[25]<1.08564e+08){if(f[0]<1.40325e+11){if(f[56]<0.992857){if(f[0]<1.40288e+11){if(f[0]<1.39836e+11){r+=0.25;}else{r+=-0.99308;}}else{r+=0.878788;}}else{if(f[25]<6.61158e+07){if(f[28]<1.39638e+11){r+=0.571429;}else{if(f[14]<1.39436e+11){r+=0.5;}else{r+=-0.984252;}}}else{if(f[28]<1.39459e+11){if(f[14]<1.39819e+11){if(f[14]<1.39669e+11){r+=0.948052;}else{if(f[14]<1.39811e+11){r+=-0.874016;}else{r+=-0.16129;}}}else{if(f[28]<1.39441e+11){if(f[8]<1.69366e+08){r+=-0.375;}else{r+=0.714286;}}else{if(f[52]<0.996793){r+=0.142857;}else{r+=0.896774;}}}}else{if(f[66]<0.247426){if(f[0]<1.40277e+11){r+=0.25;}else{r+=-0.833333;}}else{if(f[11]<7.18858e+07){if(f[13]<9.45961e+07){r+=-0.846154;}else{r+=0.478261;}}else{if(f[27]<7.33679e+07){r+=-0.789474;}else{r+=0.972431;}}}}}}}else{if(f[14]<1.39073e+11){if(f[28]<1.40087e+11){if(f[27]<1.09466e+08){r+=-0.924528;}else{if(f[39]<1.53882e+08){if(f[41]<1.38469e+08){r+=-0.636364;}else{if(f[14]<1.38849e+11){r+=-0;}else{r+=0.985321;}}}else{if(f[41]<2.71195e+08){r+=-0.647059;}else{if(f[7]<1.55519e+08){r+=0.666667;}else{r+=-0;}}}}}else{if(f[28]<1.40096e+11){if(f[27]<1.97275e+08){r+=-0.555556;}else{r+=0.5;}}else{if(f[5]<3.22959e+06){r+=-0.25;}else{r+=-0.969231;}}}}else{if(f[56]<1.00017){if(f[82]<3.10346e+08){if(f[55]<0.88009){if(f[14]<1.39126e+11){if(f[41]<2.72466e+08){r+=-0.538462;}else{r+=0.914286;}}else{if(f[12]<3.54302e+07){r+=0.25;}else{r+=-0.813953;}}}else{if(f[13]<6.97532e+08){if(f[27]<5.2566e+08){r+=-0.995079;}else{r+=-0.0909091;}}else{if(f[35]<2.16969e+07){r+=0.809524;}else{r+=-0.75;}}}}else{if(f[70]<1.40178e+11){if(f[35]<1.13675e+07){r+=0.25;}else{r+=0.940298;}}else{r+=-0.692308;}}}else{if(f[28]<1.39445e+11){if(f[54]<1.07734){r+=0.5;}else{r+=-0.625;}}else{if(f[29]<9.15432e+07){if(f[63]<0.543079){r+=0.25;}else{if(f[69]<0.484866){r+=0.25;}else{r+=0.971014;}}}else{r+=0.25;}}}}}}else{if(f[37]<8.46364e+08){if(f[14]<1.39459e+11){if(f[28]<1.40086e+11){if(f[28]<1.39706e+11){if(f[11]<5.22359e+08){if(f[0]<1.39966e+11){r+=0.666667;}else{r+=-0.995316;}}else{r+=0.932203;}}else{if(f[9]<8.80467e+07){if(f[43]<1.0986){r+=-0.983471;}else{r+=-0.272727;}}else{if(f[9]<2.04343e+08){if(f[36]<1.97736e+08){r+=0.731876;}else{r+=-0.696477;}}else{if(f[55]<0.132356){r+=-0.866667;}else{r+=0.963127;}}}}}else{if(f[81]<1.22147){if(f[42]<1.01208){if(f[54]<0.809211){if(f[42]<1.00612){r+=0.933333;}else{r+=-0.625;}}else{if(f[42]<1.00573){r+=0.662651;}else{r+=-0.935484;}}}else{if(f[28]<1.40124e+11){if(f[27]<2.07526e+08){r+=-0.290323;}else{r+=0.878378;}}else{if(f[41]<5.78701e+08){r+=-0.909091;}else{r+=0.897436;}}}}else{if(f[14]<1.39277e+11){if(f[28]<1.40114e+11){if(f[13]<2.07386e+08){r+=-0.703704;}else{r+=0.563636;}}else{if(f[13]<5.87069e+08){r+=-0.738397;}else{r+=0.647059;}}}else{if(f[0]<1.40231e+11){if(f[18]<3.19931e+06){r+=0.25;}else{r+=0.994092;}}else{r+=-0;}}}}}else{if(f[40]<1.32472e+09){if(f[66]<0.303285){if(f[0]<1.40277e+11){r+=0.75;}else{if(f[4]<2.54114e+07){r+=-0.947368;}else{r+=-0;}}}else{if(f[43]<0.0561925){r+=-0.823529;}else{if(f[27]<1.01748e+08){if(f[28]<1.39663e+11){r+=0.571429;}else{r+=-0.862069;}}else{if(f[28]<1.39446e+11){r+=0.101449;}else{r+=0.982002;}}}}}else{if(f[0]<1.40322e+11){r+=-0.25;}else{r+=-0.936508;}}}}else{if(f[66]<0.98282){if(f[39]<9.09839e+09){if(f[54]<2.33825){if(f[6]<1.37888e+07){r+=-0.333333;}else{if(f[54]<0.551474){r+=-0;}else{if(f[70]<1.00062){r+=0.952381;}else{r+=0.25;}}}}else{r+=-0.555556;}}else{if(f[16]<4.86246e+07){r+=0.272727;}else{if(f[27]<4.96708e+10){r+=-0.789474;}else{r+=0.25;}}}}else{if(f[37]<1.35495e+10){if(f[28]<1.39874e+11){if(f[15]<5.11694e+08){r+=0.862069;}else{r+=-0.789474;}}else{if(f[0]<1.39765e+11){r+=0.538462;}else{if(f[0]<1.41765e+11){if(f[41]<1.25131e+10){r+=-0.991984;}else{r+=-0.25;}}else{r+=-0.25;}}}}else{if(f[23]<2.00295e+10){if(f[81]<0.844344){r+=-0;}else{r+=0.846154;}}else{if(f[8]<1.20323e+10){if(f[0]<1.40461e+11){r+=0.636364;}else{r+=-0;}}else{if(f[16]<5.50823e+08){r+=0.111111;}else{if(f[37]<4.66899e+10){r+=-0.846154;}else{r+=-0.111111;}}}}}}}}}else{if(f[14]<1.39532e+11){if(f[28]<1.40244e+11){if(f[42]<1.00772){if(f[0]<1.40041e+11){r+=0.25;}else{r+=0.978723;}}else{if(f[0]<1.41539e+11){r+=-0.979695;}else{r+=-0.25;}}}else{if(f[42]<1.00156){if(f[0]<1.39271e+11){r+=-0.571429;}else{r+=0.954023;}}else{if(f[0]<1.4283e+11){if(f[28]<1.40256e+11){if(f[42]<1.00757){r+=0.928571;}else{r+=-0.982759;}}else{if(f[22]<5.72628e+10){if(f[28]<1.40274e+11){if(f[42]<1.00777){r+=0.882353;}else{r+=-0.991416;}}else{if(f[8]<1.25694e+10){r+=-0.999277;}else{r+=-0.533333;}}}else{r+=0.0909091;}}}else{if(f[14]<1.39499e+11){if(f[14]<1.38545e+11){if(f[23]<2.25283e+10){r+=0.826087;}else{r+=-0.454545;}}else{if(f[15]<2.93733e+08){if(f[31]<2.84088e+08){r+=-0.992169;}else{r+=0.25;}}else{r+=0.6;}}}else{if(f[28]<1.40684e+11){r+=-0.4;}else{r+=0.980769;}}}}}}else{if(f[52]<1.19337){if(f[70]<1.01171){if(f[9]<2.89507e+08){if(f[69]<0.790242){if(f[70]<1.0003){if(f[0]<1.43069e+11){if(f[0]<1.40488e+11){r+=-0.166667;}else{r+=-0.993388;}}else{r+=0.733333;}}else{if(f[27]<6.68048e+08){if(f[25]<7.32672e+07){r+=0.0588235;}else{r+=0.842975;}}else{r+=-0.941176;}}}else{if(f[23]<7.95673e+07){if(f[27]<9.27114e+07){r+=-0.25;}else{r+=-0.985075;}}else{if(f[66]<0.760579){if(f[42]<1.00633){r+=0.88697;}else{r+=-0.979136;}}else{if(f[67]<0.99751){r+=0.704836;}else{r+=0.970137;}}}}}else{if(f[81]<2.54543){if(f[70]<0.982695){if(f[0]<1.43685e+11){r+=0.9375;}else{r+=0.25;}}else{if(f[0]<1.40821e+11){if(f[11]<2.42696e+08){r+=0.789474;}else{r+=-0.632653;}}else{if(f[12]<9.9517e+09){r+=-0.994638;}else{r+=0.0909091;}}}}else{if(f[70]<1.00055){r+=-0.454545;}else{if(f[7]<1.32176e+09){r+=0.955056;}else{r+=0.25;}}}}}else{r+=-0.995092;}}else{if(f[14]<1.3974e+11){if(f[70]<1.00028){if(f[0]<1.43396e+11){if(f[0]<1.40525e+11){if(f[9]<1.18063e+08){r+=0.6;}else{r+=-0.5;}}else{r+=-0.997017;}}else{if(f[0]<1.43641e+11){r+=0.963636;}else{r+=0.25;}}}else{if(f[70]<1.01171){if(f[25]<2.42903e+08){if(f[28]<1.40248e+11){r+=-0.636364;}else{if(f[70]<1.00039){r+=0.0344828;}else{r+=0.879925;}}}else{if(f[56]<0.9936){r+=-0.891892;}else{r+=-0;}}}else{r+=-0.973154;}}}else{if(f[40]<6.53751e+07){if(f[0]<1.40301e+11){r+=-0.976608;}else{r+=0.945205;}}else{if(f[8]<1.18656e+08){if(f[70]<1.00067){r+=-0.971014;}else{if(f[9]<1.45675e+08){if(f[0]<1.40263e+11){r+=-0.555556;}else{r+=0.785714;}}else{if(f[1]<2.84949e+07){r+=-0.25;}else{r+=-0.891892;}}}}else{if(f[9]<1.05359e+08){r+=-0.2;}else{if(f[29]<1.89594e+08){r+=-0.991034;}else{if(f[2]<9.96319e+07){r+=-0.6;}else{r+=-0;}}}}}}}}}}else{if(f[28]<1.39818e+11){if(f[14]<1.39888e+11){if(f[28]<1.39606e+11){if(f[0]<1.4014e+11){r+=-0;}else{if(f[11]<6.32485e+08){r+=0.98324;}else{r+=0.25;}}}else{r+=-0.826087;}}else{if(f[41]<1.83269e+08){if(f[9]<2.24483e+08){if(f[6]<8.00868e+07){if(f[31]<1.12518e+08){if(f[28]<1.39809e+11){if(f[14]<1.39906e+11){if(f[0]<1.39912e+11){r+=-0.6;}else{r+=-0;}}else{if(f[75]<49.5999){r+=-0.995597;}else{r+=-0.25;}}}else{if(f[41]<1.47614e+08){if(f[69]<4.51308){r+=-0.985481;}else{r+=-0.179487;}}else{if(f[14]<1.4032e+11){r+=0.703704;}else{r+=-0.791667;}}}}else{r+=0.4;}}else{if(f[56]<1.00402){if(f[45]<1.64732){if(f[28]<1.3979e+11){r+=-0.2;}else{r+=0.777778;}}else{if(f[9]<9.70564e+07){r+=-0.636364;}else{r+=-0;}}}else{r+=-0.969697;}}}else{if(f[42]<1.009){if(f[14]<1.39921e+11){r+=-0.25;}else{r+=-0.986425;}}else{if(f[0]<1.4239e+11){r+=0.97037;}else{r+=-0.902439;}}}}else{if(f[14]<1.40318e+11){if(f[0]<1.41443e+11){if(f[28]<1.39794e+11){if(f[14]<1.40111e+11){if(f[0]<1.40077e+11){r+=-0.870968;}else{if(f[39]<3.68068e+09){r+=0.939394;}else{r+=-0.111111;}}}else{if(f[56]<1.00281){r+=0.111111;}else{if(f[0]<1.41004e+11){r+=-0.991361;}else{r+=-0;}}}}else{if(f[83]<1.47825){if(f[14]<1.40034e+11){r+=-0.714286;}else{if(f[14]<1.40306e+11){r+=0.954545;}else{r+=0.395349;}}}else{if(f[29]<7.66871e+08){r+=-0.846154;}else{r+=0.25;}}}}else{if(f[42]<1.01104){r+=0.988506;}else{r+=-0.809524;}}}else{if(f[13]<3.46845e+10){if(f[15]<1.30987e+08){if(f[37]<7.47714e+06){r+=-0;}else{if(f[46]<24.9151){r+=-0.991753;}else{r+=-0.25;}}}else{if(f[28]<1.39772e+11){if(f[74]<4.00932){r+=-0.929825;}else{r+=-0.2;}}else{if(f[14]<1.40591e+11){r+=0.571429;}else{if(f[14]<1.40914e+11){r+=-0.5;}else{r+=-0;}}}}}else{r+=0.5;}}}}}else{if(f[28]<1.40118e+11){if(f[70]<0.986314){if(f[14]<1.40156e+11){if(f[40]<8.38009e+07){if(f[28]<1.39986e+11){if(f[0]<1.4377e+11){r+=0.981818;}else{r+=-0;}}else{if(f[2]<1.3158e+07){r+=0.25;}else{r+=-0.555556;}}}else{if(f[56]<1.00117){r+=-0.955556;}else{if(f[0]<1.77526e+11){r+=0.7;}else{r+=-0.733333;}}}}else{if(f[14]<1.43152e+11){if(f[28]<1.40118e+11){if(f[2]<2.72777e+09){r+=-0.997787;}else{r+=-0.25;}}else{r+=-0.0909091;}}else{r+=0.5;}}}else{if(f[14]<1.40104e+11){if(f[28]<1.40002e+11){if(f[40]<8.80838e+08){if(f[69]<0.617258){if(f[28]<1.3994e+11){if(f[7]<1.0249e+08){r+=0.897181;}else{r+=-0.937984;}}else{if(f[31]<5.6645e+08){r+=-0.978208;}else{r+=0.753846;}}}else{if(f[31]<5.54422e+08){if(f[64]<0.608216){r+=-0.441026;}else{r+=-0.971604;}}else{if(f[0]<1.40073e+11){r+=-0.73913;}else{r+=0.964912;}}}}else{if(f[70]<0.999389){if(f[14]<1.40086e+11){if(f[66]<1.09913){r+=0.990097;}else{r+=-0.653846;}}else{if(f[28]<1.39918e+11){r+=-0;}else{r+=-0.932203;}}}else{if(f[4]<2.23295e+08){r+=-0.972414;}else{r+=-0.25;}}}}else{if(f[0]<1.40088e+11){if(f[14]<1.39962e+11){if(f[7]<1.21969e+07){r+=0.25;}else{if(f[7]<2.73084e+09){r+=-0.99634;}else{r+=-0;}}}else{if(f[27]<1.38798e+08){if(f[0]<1.40005e+11){r+=0.975365;}else{r+=-0.392054;}}else{if(f[28]<1.40113e+11){r+=0.955252;}else{r+=-0.721519;}}}}else{if(f[13]<1.27219e+09){if(f[77]<4.12995){if(f[14]<1.40102e+11){r+=-0.990148;}else{r+=-0.308411;}}else{if(f[67]<1.23206){r+=0.913043;}else{r+=-0.789474;}}}else{if(f[14]<1.40087e+11){if(f[67]<0.607671){r+=0.572131;}else{r+=-0.755352;}}else{if(f[11]<1.47618e+09){r+=0.913043;}else{r+=-0.81978;}}}}}}else{if(f[9]<1.03657e+08){if(f[83]<0.0618212){if(f[56]<1.00285){r+=-0.996075;}else{r+=0.5;}}else{if(f[67]<0.60337){if(f[56]<1.00238){if(f[28]<1.40048e+11){r+=-0.906137;}else{r+=0.840319;}}else{if(f[13]<8.01654e+07){r+=-0.84;}else{r+=0.954997;}}}else{if(f[0]<1.39822e+11){if(f[28]<1.40075e+11){r+=-0.501023;}else{r+=0.61313;}}else{if(f[14]<1.4048e+11){r+=0.558045;}else{r+=0.923745;}}}}}else{if(f[14]<1.40861e+11){if(f[67]<1.01817){if(f[14]<1.40161e+11){if(f[0]<1.41571e+11){r+=-0.717476;}else{r+=0.843373;}}else{if(f[14]<1.40253e+11){r+=0.875349;}else{r+=0.450977;}}}else{if(f[64]<1.43889){if(f[42]<1.00969){r+=-0.670782;}else{r+=0.822222;}}else{if(f[81]<0.092501){r+=-0.835294;}else{r+=0.538604;}}}}else{if(f[14]<1.45288e+11){if(f[13]<8.55148e+08){if(f[7]<6.22395e+07){r+=0.25;}else{r+=-0.99767;}}else{if(f[67]<0.960689){r+=0.785714;}else{r+=-0.555556;}}}else{if(f[0]<1.40176e+11){r+=0.943662;}else{r+=-0.733333;}}}}}}}else{if(f[69]<1.61988){if(f[54]<0.486497){if(f[28]<1.40221e+11){if(f[28]<1.40141e+11){if(f[14]<1.40039e+11){if(f[27]<4.7187e+08){if(f[0]<1.40089e+11){r+=-0.25;}else{r+=0.555556;}}else{r+=-0.982063;}}else{if(f[67]<2.27901){if(f[52]<0.517054){r+=0.961772;}else{r+=0.455108;}}else{if(f[42]<0.999959){r+=-0.927273;}else{r+=0.727273;}}}}else{if(f[66]<3.41354){if(f[30]<4.37771e+07){if(f[69]<0.597058){r+=0.391304;}else{r+=-0.982009;}}else{if(f[0]<1.40102e+11){r+=0.704348;}else{r+=-0.647059;}}}else{if(f[28]<1.40185e+11){if(f[83]<2.39125){r+=-0.733333;}else{r+=0.207547;}}else{if(f[0]<1.39987e+11){r+=0.166667;}else{r+=0.983193;}}}}}else{if(f[0]<1.40463e+11){if(f[70]<1.00584){if(f[27]<6.76211e+07){if(f[42]<0.999499){r+=-0.947368;}else{r+=0.809524;}}else{if(f[67]<1.54519){r+=0.952813;}else{r+=0.458143;}}}else{if(f[35]<2.30696e+09){if(f[72]<0.140276){r+=0.0769231;}else{r+=-0.907514;}}else{r+=0.272727;}}}else{if(f[11]<1.07526e+08){if(f[14]<1.41003e+11){if(f[81]<0.633109){r+=0.428571;}else{r+=-0.98995;}}else{if(f[28]<1.41165e+11){r+=0.733333;}else{r+=-0.555556;}}}else{if(f[27]<4.36436e+08){if(f[70]<1.00222){r+=0.94702;}else{r+=-0.764706;}}else{if(f[64]<4.41826){r+=-0.789474;}else{r+=0.571429;}}}}}}else{if(f[0]<1.41648e+11){if(f[28]<1.4146e+11){if(f[42]<0.952319){if(f[0]<1.39638e+11){r+=0.99922;}else{r+=-0;}}else{if(f[28]<1.41005e+11){if(f[42]<0.99385){r+=-0.993882;}else{r+=-0.13999;}}else{if(f[0]<1.41334e+11){r+=-0.960527;}else{r+=0.939623;}}}}else{if(f[70]<1.02008){if(f[70]<1.01245){if(f[39]<4.85433e+07){r+=0.894737;}else{r+=-0.983683;}}else{if(f[22]<4.76021e+07){r+=-0.938462;}else{r+=0.997319;}}}else{if(f[27]<2.31238e+08){if(f[81]<1.84269){r+=-0.987778;}else{r+=0.130435;}}else{if(f[56]<0.78974){r+=0.98983;}else{r+=-0.798561;}}}}}else{if(f[28]<1.41444e+11){if(f[0]<1.42929e+11){if(f[14]<1.40142e+11){if(f[77]<0.729415){r+=0.217391;}else{r+=-0.950249;}}else{if(f[14]<1.41311e+11){r+=0.992087;}else{r+=-0.852632;}}}else{if(f[70]<0.980632){if(f[0]<1.42947e+11){r+=-0.545455;}else{r+=-0.980488;}}else{if(f[28]<1.40304e+11){r+=0.866667;}else{r+=-0.538462;}}}}else{r+=-0.995395;}}}}else{if(f[56]<0.992566){if(f[0]<1.40243e+11){if(f[14]<1.40121e+11){if(f[28]<1.75364e+11){if(f[14]<1.40115e+11){r+=-0.996877;}else{if(f[0]<1.40049e+11){r+=-0;}else{r+=-0.5;}}}else{r+=0.8;}}else{if(f[28]<1.42418e+11){if(f[71]<9.35574){if(f[43]<0.191345){r+=-0.25;}else{r+=0.992167;}}else{r+=-0.5;}}else{r+=-0.992172;}}}else{if(f[0]<1.40312e+11){if(f[35]<1.98769e+06){if(f[57]<2.13979){r+=0.2;}else{if(f[30]<436809){r+=-0.963636;}else{r+=-0.25;}}}else{if(f[7]<1.26245e+07){r+=-0;}else{if(f[21]<1.04053e+07){r+=0.25;}else{r+=0.998813;}}}}else{if(f[28]<1.4254e+11){if(f[14]<1.40192e+11){r+=-0.777778;}else{if(f[0]<1.4107e+11){r+=0.75;}else{r+=-0;}}}else{if(f[28]<1.88333e+11){r+=-0.980488;}else{r+=-0.25;}}}}}else{if(f[40]<5.20621e+07){if(f[28]<1.40559e+11){if(f[28]<1.40132e+11){if(f[36]<7.53538e+06){if(f[0]<1.39857e+11){r+=0.2;}else{r+=-0.979036;}}else{if(f[11]<1.02704e+08){r+=0.115108;}else{r+=0.888268;}}}else{if(f[28]<1.40147e+11){if(f[36]<1.02847e+07){r+=-0.970675;}else{r+=0.5;}}else{if(f[63]<0.395048){r+=-0.0967742;}else{r+=-0.987639;}}}}else{if(f[14]<1.40509e+11){if(f[0]<1.4023e+11){if(f[28]<1.40623e+11){r+=0.987008;}else{r+=-0.571429;}}else{if(f[0]<1.4071e+11){r+=-0.983264;}else{r+=0.987229;}}}else{if(f[0]<1.39914e+11){if(f[28]<1.40666e+11){r+=0.986348;}else{r+=-0.9;}}else{if(f[42]<1.0031){r+=-0.971631;}else{r+=0.352941;}}}}}else{if(f[0]<1.41143e+11){if(f[36]<8.182e+08){if(f[69]<1.92293){if(f[28]<1.40144e+11){r+=-0.275645;}else{r+=-0.884061;}}else{if(f[9]<4.68065e+08){r+=-0.945791;}else{r+=-0.670791;}}}else{if(f[0]<1.40088e+11){if(f[0]<1.39802e+11){r+=-0.772727;}else{r+=0.830588;}}else{if(f[41]<2.98893e+09){r+=-0.835052;}else{r+=0.0149254;}}}}else{if(f[14]<1.40835e+11){if(f[0]<1.41602e+11){if(f[28]<1.40893e+11){r+=-0.986014;}else{r+=-0.25;}}else{if(f[28]<1.4022e+11){r+=0.533333;}else{r+=-0.638095;}}}else{if(f[36]<7.09454e+07){r+=-0.935484;}else{if(f[41]<9.04274e+08){r+=0.970181;}else{r+=-0.491525;}}}}}}}}}}}} return r;}
    static double p4(double[] f){double r=0;if(f[28]<1.39432e+11){if(f[28]<1.3942e+11){if(f[10]<7.00404e+10){if(f[19]<5.45623e+09){if(f[14]<1.38438e+11){if(f[28]<1.39216e+11){r+=-0.624893;}else{if(f[1]<7.37284e+08){r+=0.0848082;}else{r+=0.495083;}}}else{if(f[20]<4.61237e+09){if(f[35]<1.87345e+08){r+=-0.683313;}else{if(f[84]<2.15){r+=-0.686281;}else{if(f[14]<1.40319e+11){if(f[1]<9.42231e+07){r+=0.79853;}else{r+=-0.235965;}}else{if(f[65]<0.379298){r+=-0.129425;}else{r+=-0.677017;}}}}}else{r+=0.0310992;}}}else{if(f[62]<1.04028){r+=-0.469883;}else{r+=0.533307;}}}else{if(f[69]<1.31225){r+=-0.401109;}else{if(f[65]<1.17483){r+=0.599079;}else{r+=-0.0114733;}}}}else{if(f[14]<1.39876e+11){if(f[84]<1){r+=-0.714683;}else{if(f[42]<1.00164){r+=-0.464344;}else{if(f[49]<0.551874){r+=0.0680293;}else{if(f[42]<1.00296){r+=0.54286;}else{r+=1.12213;}}}}}else{if(f[82]<0.709141){r+=-0.134099;}else{r+=-0.678661;}}}}else{if(f[69]<0.476229){if(f[15]<8.54812e+06){if(f[14]<1.40185e+11){if(f[55]<0.859611){if(f[28]<1.40182e+11){if(f[84]<1.15){if(f[84]<0.85){if(f[12]<1.12924e+08){if(f[42]<1.00189){if(f[14]<1.40055e+11){if(f[69]<0.459788){r+=0.690484;}else{r+=0.165871;}}else{if(f[9]<4.30596e+07){r+=0.234937;}else{r+=-0.539412;}}}else{r+=-0.662921;}}else{if(f[70]<0.999158){if(f[29]<2.13701e+08){r+=-0.705346;}else{r+=0.46903;}}else{if(f[42]<0.999915){if(f[2]<4.96144e+06){r+=-0.585122;}else{r+=-0.100758;}}else{if(f[25]<2.15738e+08){r+=0.674962;}else{r+=-0.0342889;}}}}}else{if(f[37]<7.74536e+07){if(f[51]<1.20613){if(f[23]<5.56747e+07){r+=1.15531;}else{if(f[70]<0.9984){r+=0.704318;}else{r+=0.929838;}}}else{r+=-0.210836;}}else{if(f[28]<1.39812e+11){r+=-0.420486;}else{if(f[70]<0.998085){r+=1.41144;}else{if(f[40]<3.41149e+08){r+=0.204862;}else{r+=1.02414;}}}}}}else{if(f[28]<1.40178e+11){if(f[10]<5.90989e+08){if(f[84]<3.5){r+=-0.720714;}else{if(f[8]<4.14036e+07){r+=-0.569956;}else{r+=0.27081;}}}else{if(f[84]<2){if(f[66]<3.88683){r+=0.436661;}else{if(f[77]<0.857255){r+=-0.676457;}else{r+=-0.218616;}}}else{r+=0.807948;}}}else{if(f[84]<2.5){if(f[53]<0.953692){if(f[82]<11.6761){if(f[66]<10.0586){r+=-0.423186;}else{r+=0.476267;}}else{if(f[82]<15.9292){r+=0.189029;}else{r+=-0.631076;}}}else{if(f[66]<5.05032){if(f[37]<3.96035e+07){r+=-0.0195819;}else{r+=-0.893037;}}else{r+=0.31294;}}}else{if(f[13]<1.76374e+08){r+=-0.222556;}else{r+=0.815292;}}}}}else{if(f[70]<1.00041){r+=-0.608342;}else{if(f[42]<0.999126){r+=-0.620866;}else{if(f[28]<1.40186e+11){if(f[84]<2.5){if(f[69]<0.150709){if(f[12]<1.1578e+08){r+=-0.755528;}else{r+=0.217466;}}else{if(f[12]<1.79202e+08){r+=0.37632;}else{r+=-0.111288;}}}else{if(f[65]<0.423573){r+=0.206244;}else{r+=0.721745;}}}else{if(f[14]<1.4013e+11){if(f[27]<5.23981e+07){r+=-0.230419;}else{if(f[30]<1.22179e+09){r+=0.676259;}else{r+=0.262946;}}}else{r+=-0.392823;}}}}}}else{if(f[13]<2.19791e+08){if(f[84]<0.5){if(f[8]<3.24625e+08){if(f[28]<1.90765e+11){if(f[65]<1.65124){if(f[55]<0.873087){if(f[26]<9.67365e+07){r+=0.375894;}else{r+=-0.558022;}}else{if(f[27]<2.33615e+08){r+=-0.731363;}else{r+=-0.225152;}}}else{r+=0.0908952;}}else{r+=0.463614;}}else{r+=0.582457;}}else{if(f[63]<0.422799){if(f[70]<0.997859){if(f[29]<5.17337e+08){r+=-0.646695;}else{r+=0.598917;}}else{if(f[84]<1.85){if(f[14]<1.39968e+11){r+=-0.582929;}else{if(f[42]<0.9987){r+=-0.449566;}else{r+=0.711134;}}}else{r+=-0.690464;}}}else{if(f[66]<0.760967){if(f[25]<6.5366e+07){if(f[54]<2.78653){if(f[77]<1.4682){r+=-0.674118;}else{r+=0.00342239;}}else{r+=0.3277;}}else{if(f[42]<1.0004){if(f[14]<1.40076e+11){r+=0.751166;}else{r+=-0.245597;}}else{if(f[70]<0.996556){r+=0.461724;}else{r+=-0.658209;}}}}else{if(f[84]<1.85){if(f[9]<2.24854e+07){r+=-0.613247;}else{if(f[84]<1.15){r+=0.696674;}else{r+=0.0951613;}}}else{if(f[28]<1.39983e+11){r+=0.140104;}else{if(f[42]<0.999352){r+=-0.228256;}else{r+=-0.718901;}}}}}}}else{if(f[84]<1.15){if(f[84]<0.85){if(f[73]<24.465){if(f[70]<1.05778){if(f[37]<7.34743e+07){if(f[70]<0.99917){r+=-0.568575;}else{r+=0.455394;}}else{if(f[77]<0.204212){r+=0.28255;}else{r+=-0.685527;}}}else{r+=0.722241;}}else{if(f[52]<2.38952){if(f[14]<1.39997e+11){r+=-0.119827;}else{if(f[59]<0.0332459){r+=0.709756;}else{r+=0.239122;}}}else{r+=-0.556086;}}}else{if(f[51]<1.20367){if(f[14]<1.40108e+11){r+=-0.417522;}else{if(f[37]<7.74534e+07){r+=0.845911;}else{if(f[53]<1.39213){r+=1.2787;}else{r+=0.316957;}}}}else{r+=-0.650468;}}}else{if(f[42]<1.00017){if(f[70]<1.00128){if(f[14]<1.40019e+11){if(f[15]<1.74896e+06){r+=0.905759;}else{r+=-0.126627;}}else{if(f[13]<1.00969e+10){if(f[42]<0.999467){r+=-0.388369;}else{r+=-0.716523;}}else{r+=0.552498;}}}else{if(f[77]<3.57972){if(f[84]<1.5){if(f[28]<1.40184e+11){r+=-0.656221;}else{r+=0.55727;}}else{if(f[40]<1.90219e+08){r+=-0.390107;}else{r+=0.714751;}}}else{r+=-0.635076;}}}else{if(f[82]<1.37129){if(f[27]<6.71104e+08){if(f[28]<1.39938e+11){if(f[84]<1.5){r+=-0.344485;}else{r+=0.668771;}}else{if(f[82]<0.662483){r+=-0.0246675;}else{r+=-0.707049;}}}else{if(f[52]<1.11224){r+=-0.0323811;}else{r+=1.24205;}}}else{if(f[28]<1.39937e+11){if(f[70]<0.998403){if(f[17]<1.48913e+07){r+=-0.669784;}else{r+=-0.190288;}}else{r+=0.719528;}}else{if(f[70]<0.988946){if(f[63]<1.13096){r+=-0.412285;}else{r+=0.228444;}}else{if(f[27]<1.09421e+10){r+=-0.688134;}else{r+=-0.111289;}}}}}}}}}else{if(f[84]<1.15){if(f[42]<0.997331){if(f[27]<7.76708e+07){if(f[28]<1.40403e+11){if(f[28]<1.40017e+11){r+=0.244147;}else{if(f[28]<1.40395e+11){r+=-0.692295;}else{r+=-0.155725;}}}else{if(f[14]<1.40611e+11){r+=0.464738;}else{r+=0.0269537;}}}else{if(f[14]<1.4046e+11){r+=-0.643885;}else{if(f[84]<0.85){if(f[28]<1.4049e+11){r+=-0.951493;}else{if(f[54]<0.7594){r+=0.67788;}else{if(f[66]<1.32515){r+=1.39909;}else{r+=0.505117;}}}}else{if(f[42]<0.990663){if(f[14]<1.40973e+11){r+=-0.493062;}else{r+=0.155349;}}else{if(f[17]<1.28016e+07){r+=0.806883;}else{r+=-0.0299111;}}}}}}else{if(f[84]<0.85){if(f[28]<1.3988e+11){if(f[70]<0.996924){r+=-0.601719;}else{r+=0.708509;}}else{if(f[20]<1.92212e+07){if(f[23]<5.41991e+07){if(f[20]<1.07125e+07){r+=-0.687238;}else{r+=-0.0603066;}}else{r+=-0.744872;}}else{if(f[19]<8.91307e+06){r+=0.383839;}else{r+=-0.383448;}}}}else{if(f[27]<7.77128e+07){if(f[14]<1.40439e+11){r+=-0.681796;}else{r+=0.260764;}}else{if(f[28]<1.39843e+11){r+=-0.593028;}else{if(f[28]<1.40276e+11){if(f[9]<9.06939e+06){r+=0.208729;}else{r+=1.01157;}}else{r+=-0.356478;}}}}}}else{if(f[40]<4.26559e+07){if(f[13]<4.28338e+07){r+=-0.499704;}else{r+=0.959731;}}else{if(f[17]<1.08693e+07){if(f[25]<1.16784e+09){if(f[6]<1.42182e+07){if(f[59]<51.4124){if(f[28]<1.39819e+11){r+=-0.291324;}else{r+=-0.691958;}}else{r+=-0.172531;}}else{if(f[13]<1.54578e+08){if(f[28]<1.40117e+11){if(f[23]<2.98012e+07){r+=1.06874;}else{r+=0.00283304;}}else{if(f[14]<1.40502e+11){r+=-0.0774355;}else{r+=-0.648952;}}}else{if(f[30]<274263){r+=0.129125;}else{if(f[42]<1.00044){r+=-0.690135;}else{r+=0.033002;}}}}}else{if(f[26]<1.40037e+09){r+=0.307853;}else{r+=-0.534045;}}}else{if(f[84]<2.85){if(f[40]<3.74547e+08){if(f[70]<1.00223){if(f[70]<1.00059){if(f[30]<1.07664e+07){r+=-0.416687;}else{r+=0.222602;}}else{r+=0.542914;}}else{r+=-0.493648;}}else{r+=-0.685261;}}else{if(f[28]<1.40186e+11){if(f[38]<1.65301e+08){r+=0.0988813;}else{r+=0.559035;}}else{r+=-0.481838;}}}}}}}else{if(f[27]<6.66458e+08){if(f[14]<1.40991e+11){if(f[35]<1.48143e+08){if(f[28]<1.42047e+11){if(f[82]<8.66928){if(f[82]<2.11177){if(f[14]<1.40173e+11){if(f[28]<1.39792e+11){if(f[40]<4.48525e+08){r+=0.519784;}else{r+=-0.479956;}}else{if(f[70]<0.999215){r+=-0.681061;}else{r+=-0.314753;}}}else{if(f[84]<1.85){if(f[42]<1.0017){r+=-0.577172;}else{r+=0.287018;}}else{if(f[70]<1.00373){r+=0.459518;}else{r+=-0.520654;}}}}else{if(f[41]<3.47165e+08){if(f[37]<1.31313e+08){if(f[42]<0.997315){r+=-0.121325;}else{r+=-0.63029;}}else{if(f[84]<1.15){r+=-0.342755;}else{r+=0.380695;}}}else{if(f[66]<1.55718){if(f[42]<0.992744){r+=-0.332494;}else{r+=-0.628652;}}else{if(f[40]<9.14992e+08){r+=-0.321285;}else{r+=0.521393;}}}}}else{if(f[42]<0.999448){if(f[28]<1.40632e+11){if(f[28]<1.40182e+11){if(f[84]<1.85){r+=-0.333428;}else{r+=0.646348;}}else{if(f[54]<0.754962){r+=0.652345;}else{r+=-0.229222;}}}else{r+=-0.564405;}}else{if(f[84]<1.15){if(f[84]<0.85){if(f[14]<1.40097e+11){r+=0.00236972;}else{r+=-0.662199;}}else{if(f[27]<2.74205e+08){r+=0.0372912;}else{r+=1.15525;}}}else{if(f[54]<0.31958){r+=-0.0897114;}else{r+=-0.734831;}}}}}else{if(f[14]<1.40596e+11){if(f[84]<0.8){r+=0.471989;}else{r+=-0.678897;}}else{r+=0.70615;}}}else{if(f[28]<1.40092e+11){if(f[84]<2.85){if(f[28]<1.40089e+11){if(f[41]<8.30066e+08){if(f[67]<0.435056){if(f[14]<1.40253e+11){r+=0.492981;}else{r+=-0.0143646;}}else{if(f[70]<0.990811){r+=0.204811;}else{r+=-0.483381;}}}else{if(f[36]<2.85134e+08){if(f[73]<32.7713){r+=-0.646413;}else{r+=0.526992;}}else{if(f[42]<1.00471){r+=-0.449225;}else{r+=0.344046;}}}}else{if(f[70]<0.999639){r+=1.41615;}else{if(f[72]<82.2417){r+=-0.0814569;}else{r+=0.666678;}}}}else{if(f[52]<0.928204){if(f[14]<1.40453e+11){if(f[66]<1.39283){if(f[29]<2.335e+07){r+=-0.189204;}else{r+=-0.725074;}}else{r+=-0.00518009;}}else{r+=0.621174;}}else{if(f[14]<1.38869e+11){if(f[40]<3.62314e+08){if(f[63]<1.30326){r+=-0.789328;}else{r+=-0.178207;}}else{r+=-0.0452856;}}else{if(f[38]<1.03711e+09){if(f[52]<0.995837){r+=0.327725;}else{r+=0.681255;}}else{if(f[64]<0.655305){r+=0.386696;}else{r+=-0.663627;}}}}}}else{if(f[14]<1.40036e+11){if(f[84]<1.5){if(f[29]<4.75162e+07){if(f[63]<2.63767){if(f[82]<0.943237){r+=-0.136848;}else{r+=-0.659475;}}else{r+=0.458817;}}else{if(f[42]<1.00615){if(f[42]<1.00423){r+=-0.473505;}else{r+=0.505622;}}else{if(f[17]<7.96666e+07){r+=-0.672323;}else{r+=0.213447;}}}}else{if(f[70]<0.981885){r+=0.144143;}else{if(f[54]<0.53946){r+=0.125695;}else{if(f[28]<1.40096e+11){r+=-0.236982;}else{r+=-0.71248;}}}}}else{if(f[23]<2.17179e+08){if(f[14]<1.40341e+11){if(f[31]<5.02798e+08){if(f[84]<1.5){r+=-0.697797;}else{r+=-0.127081;}}else{if(f[84]<0.5){r+=-0.774214;}else{r+=0.675628;}}}else{if(f[25]<8.91187e+07){if(f[70]<1.01095){r+=-0.653839;}else{r+=0.395362;}}else{if(f[42]<0.99415){r+=-0.630386;}else{r+=0.500789;}}}}else{if(f[14]<1.40539e+11){if(f[42]<0.99565){if(f[14]<1.40442e+11){r+=-0.398903;}else{r+=0.909628;}}else{if(f[23]<2.19505e+09){r+=-0.500025;}else{r+=0.473882;}}}else{if(f[28]<1.42125e+11){if(f[84]<12){r+=-0.705686;}else{r+=-0.124906;}}else{r+=0.504149;}}}}}}}else{if(f[84]<1.15){if(f[42]<0.995329){if(f[66]<0.267273){r+=-0.619255;}else{if(f[28]<1.40645e+11){if(f[70]<1.00452){r+=0.64579;}else{if(f[54]<0.788613){r+=0.661438;}else{if(f[28]<1.40559e+11){r+=-0.100739;}else{r+=1.49249;}}}}else{if(f[6]<6.2208e+07){if(f[42]<0.992477){if(f[67]<0.407773){r+=1.16946;}else{r+=0.681904;}}else{r+=0.705273;}}else{r+=0.0950442;}}}}else{if(f[82]<1.42704){if(f[84]<0.5){if(f[21]<1.12408e+08){r+=-0.0564207;}else{r+=-0.626439;}}else{if(f[70]<0.980777){r+=-0.402349;}else{r+=0.724829;}}}else{if(f[1]<2.80415e+08){if(f[25]<5.60611e+07){r+=-0.201014;}else{r+=-0.678564;}}else{r+=-0.106228;}}}}else{if(f[55]<0.362368){if(f[40]<6.30228e+08){r+=0.552077;}else{r+=-0.488813;}}else{if(f[84]<4.85){r+=-0.690191;}else{if(f[70]<0.997771){if(f[29]<1.99308e+07){r+=-0.0531258;}else{r+=0.696052;}}else{r+=-0.682788;}}}}}}else{if(f[66]<3.79965){if(f[14]<1.33157e+11){if(f[3]<6.96165e+07){r+=0.800324;}else{r+=0.246084;}}else{if(f[55]<8.18315){if(f[38]<2.01653e+08){if(f[82]<1.78478){if(f[51]<1.00324){r+=-0.115263;}else{r+=0.633241;}}else{if(f[82]<5.94028){if(f[26]<6.15112e+08){if(f[9]<4.25896e+08){r+=-0.675211;}else{r+=-0.0981056;}}else{r+=0.227542;}}else{if(f[28]<1.4018e+11){if(f[9]<3.85651e+07){r+=-0.106252;}else{r+=-0.491733;}}else{if(f[10]<7.31682e+07){r+=0.0733893;}else{r+=0.628228;}}}}}else{if(f[70]<0.991105){if(f[42]<1.01193){if(f[29]<6.58945e+07){if(f[28]<1.39916e+11){r+=-0.418809;}else{r+=0.647048;}}else{if(f[12]<1.76612e+09){r+=-0.540796;}else{r+=-0.0220461;}}}else{if(f[2]<2.827e+08){r+=-0.682479;}else{r+=-0.0971071;}}}else{if(f[28]<1.60067e+11){if(f[29]<8.50422e+06){if(f[14]<1.40071e+11){r+=-0.627949;}else{r+=0.0440203;}}else{if(f[11]<3.73317e+10){r+=-0.676878;}else{r+=0.313136;}}}else{if(f[13]<9.31311e+08){r+=-0.420369;}else{r+=0.594615;}}}}}else{if(f[21]<3.07402e+09){if(f[51]<3.5632){r+=0.94058;}else{r+=-0.145278;}}else{r+=-0.489665;}}}}else{if(f[28]<1.40226e+11){if(f[28]<1.40182e+11){if(f[50]<0.743687){if(f[48]<0.504482){r+=-0.744177;}else{r+=-0.0500365;}}else{if(f[34]<1.08512e+07){if(f[17]<4.45509e+06){r+=0.165517;}else{r+=0.717482;}}else{r+=-0.330978;}}}else{if(f[69]<0.188746){r+=0.11492;}else{if(f[38]<1.47686e+08){r+=0.120957;}else{if(f[34]<3.78324e+07){r+=0.713877;}else{r+=0.159594;}}}}}else{r+=-0.659375;}}}}}else{if(f[14]<1.39856e+11){if(f[28]<1.40227e+11){if(f[25]<1.11532e+08){if(f[42]<1.00692){if(f[66]<0.388267){if(f[28]<1.39854e+11){r+=0.663572;}else{if(f[14]<1.39495e+11){r+=0.375094;}else{r+=-0.707277;}}}else{if(f[14]<1.39138e+11){if(f[1]<1.48049e+08){r+=-0.690358;}else{r+=0.280613;}}else{if(f[26]<5.94258e+07){if(f[14]<1.39513e+11){r+=0.168015;}else{if(f[70]<0.996584){r+=-0.117534;}else{r+=-0.70526;}}}else{if(f[14]<1.39661e+11){if(f[38]<6.55645e+07){r+=-0.888984;}else{if(f[42]<1.00659){r+=0.689728;}else{r+=0.0602814;}}}else{if(f[84]<1){if(f[41]<5.43023e+08){r+=-1.1441;}else{r+=0.654999;}}else{if(f[51]<0.882449){r+=-0.244641;}else{r+=0.67094;}}}}}}}else{if(f[14]<1.39079e+11){if(f[84]<1.85){if(f[77]<1.87124e+07){r+=-0.694629;}else{r+=-0.105014;}}else{if(f[27]<1.13729e+08){if(f[42]<1.01353){r+=0.151222;}else{r+=-0.697173;}}else{if(f[14]<1.38842e+11){if(f[75]<1.77334){r+=-0.657806;}else{r+=-0.0598443;}}else{if(f[38]<2.1071e+08){if(f[26]<1.34752e+08){r+=0.672744;}else{r+=-0.306549;}}else{if(f[14]<1.39031e+11){r+=0.473387;}else{r+=-0.693369;}}}}}}else{if(f[41]<2.60781e+08){if(f[25]<1.08569e+08){if(f[10]<3.99381e+08){if(f[41]<2.4629e+08){r+=-0.684864;}else{if(f[55]<0.896873){r+=0.359958;}else{r+=-0.602248;}}}else{r+=-0.0998095;}}else{if(f[70]<0.995802){if(f[23]<1.09963e+08){r+=-0.384103;}else{if(f[36]<1.99186e+08){r+=-1.32597;}else{r+=-0.688934;}}}else{r+=-0.696104;}}}else{if(f[70]<1.40176e+11){if(f[13]<1.5963e+08){if(f[38]<2.0551e+08){if(f[38]<6.88656e+07){r+=-0.22264;}else{r+=0.790505;}}else{if(f[9]<1.77411e+08){r+=-0.534559;}else{r+=0.281639;}}}else{if(f[8]<2.35124e+08){if(f[26]<2.5145e+08){r+=-0.616397;}else{r+=0.504089;}}else{r+=0.661477;}}}else{if(f[40]<3.60654e+08){r+=-0.703827;}else{r+=0.191118;}}}}}}else{if(f[21]<6.6024e+07){if(f[41]<5.69164e+08){if(f[42]<1.00383){r+=0.428656;}else{if(f[35]<1.03846e+08){if(f[57]<0.420647){r+=-0.323155;}else{if(f[14]<1.39507e+11){r+=-0.724587;}else{r+=-1.31572;}}}else{r+=-0.0593167;}}}else{if(f[12]<2.43341e+08){if(f[15]<7.88907e+06){r+=0.44514;}else{r+=-0.695876;}}else{if(f[70]<1.40182e+11){if(f[25]<4.98401e+09){if(f[51]<0.545224){r+=0.108253;}else{r+=0.688373;}}else{r+=-0.403284;}}else{r+=-0.817631;}}}}else{if(f[28]<1.40061e+11){if(f[38]<1.31985e+09){if(f[29]<8.35469e+06){if(f[35]<1.56348e+08){if(f[28]<1.39929e+11){r+=0.501233;}else{if(f[8]<1.52733e+08){r+=-0.669576;}else{r+=-1.22843;}}}else{if(f[28]<1.39723e+11){r+=-0.030706;}else{if(f[8]<3.85835e+08){r+=0.632154;}else{r+=0.148721;}}}}else{if(f[81]<1.29444){if(f[28]<1.39708e+11){if(f[42]<1.00486){r+=0.657059;}else{r+=-0.638193;}}else{if(f[27]<1.10143e+08){r+=-0.963206;}else{r+=0.658431;}}}else{if(f[55]<1.00517){if(f[41]<6.79005e+08){r+=0.545209;}else{r+=0.0525982;}}else{if(f[28]<1.39892e+11){r+=0.60405;}else{r+=-0.587687;}}}}}else{if(f[70]<0.972695){if(f[53]<0.650894){r+=0.100816;}else{r+=0.70053;}}else{if(f[28]<1.39851e+11){if(f[51]<1.00039){if(f[57]<0.973598){r+=0.380433;}else{r+=-0.360773;}}else{if(f[75]<0.50447){r+=0.102643;}else{r+=0.623256;}}}else{if(f[13]<8.10396e+09){if(f[53]<1.82096){r+=-0.687249;}else{r+=-1.13356;}}else{if(f[84]<1.5){r+=-0.561846;}else{r+=0.580522;}}}}}}else{if(f[84]<1.5){if(f[35]<4.88589e+07){if(f[40]<3.2036e+08){r+=-0.103532;}else{r+=0.682981;}}else{if(f[28]<1.40086e+11){if(f[51]<0.708444){r+=-0.53106;}else{r+=-1.5937;}}else{if(f[14]<1.39459e+11){r+=-0.733564;}else{if(f[1]<7.4479e+09){r+=-1.49549;}else{r+=0.346214;}}}}}else{if(f[84]<2.85){if(f[82]<0.715241){if(f[14]<1.39007e+11){r+=0.664788;}else{if(f[70]<0.999001){r+=-0.678787;}else{r+=0.0355351;}}}else{if(f[14]<1.39727e+11){if(f[40]<1.26951e+09){r+=0.731302;}else{r+=0.114532;}}else{if(f[13]<8.57656e+08){r+=-0.692391;}else{r+=0.442347;}}}}else{if(f[84]<3.35){if(f[28]<1.40086e+11){if(f[14]<1.38873e+11){r+=-1.61291;}else{r+=-0.453407;}}else{r+=-0.764794;}}else{if(f[14]<1.39012e+11){if(f[2]<4.39646e+07){r+=0.60018;}else{r+=0.1285;}}else{r+=-0.706642;}}}}}}}}else{if(f[14]<1.39538e+11){if(f[28]<1.40245e+11){if(f[42]<1.0086){if(f[50]<1.56773){r+=0.683314;}else{r+=0.130254;}}else{if(f[13]<1.2855e+09){r+=-0.676883;}else{r+=-0.129685;}}}else{if(f[42]<1.00156){if(f[54]<0.752916){r+=0.676442;}else{r+=-0.610784;}}else{if(f[70]<0.987059){if(f[14]<1.39499e+11){if(f[15]<1.40075e+08){if(f[14]<1.30201e+11){r+=0.45205;}else{r+=-0.683492;}}else{if(f[84]<2.5){if(f[8]<4.17909e+10){r+=0.776736;}else{r+=0.0324255;}}else{r+=-0.634004;}}}else{if(f[28]<1.407e+11){r+=-0.228629;}else{r+=0.678667;}}}else{if(f[28]<1.40254e+11){if(f[42]<1.00757){r+=0.636117;}else{r+=-0.666369;}}else{if(f[14]<1.39532e+11){if(f[20]<5.15365e+09){if(f[28]<1.40289e+11){r+=-0.619553;}else{r+=-0.683564;}}else{r+=0.0470424;}}else{if(f[69]<0.796414){r+=-0.663953;}else{if(f[10]<1.43764e+08){r+=-1.34177;}else{r+=-0.74249;}}}}}}}}else{if(f[52]<1.20113){if(f[84]<0.5){if(f[13]<2.85238e+08){r+=-1.57056;}else{r+=-0.708687;}}else{if(f[84]<2.85){if(f[70]<1.01164){if(f[9]<2.87518e+08){if(f[23]<8.12291e+07){if(f[28]<1.40472e+11){r+=-0.679569;}else{r+=0.652671;}}else{if(f[69]<0.812454){r+=0.163274;}else{r+=0.665745;}}}else{if(f[42]<1.00794){if(f[67]<1.77638){r+=0.642138;}else{r+=-0.401669;}}else{if(f[70]<0.983191){r+=0.652936;}else{r+=-0.672094;}}}}else{r+=-0.681954;}}else{if(f[38]<1.58592e+08){if(f[69]<0.790765){r+=-0.666586;}else{if(f[28]<1.4113e+11){if(f[66]<0.758625){r+=-0.373712;}else{r+=-1.44513;}}else{r+=-0.454302;}}}else{r+=-0.717289;}}}}else{if(f[14]<1.39739e+11){if(f[70]<1.00018){if(f[42]<1.02761){if(f[81]<2.17087){r+=-0.680685;}else{r+=0.13457;}}else{if(f[1]<2.25302e+08){r+=0.664754;}else{r+=0.132577;}}}else{if(f[84]<1){if(f[28]<1.40868e+11){if(f[70]<1.00039){r+=-0.754586;}else{r+=-1.48342;}}else{r+=-0.423179;}}else{if(f[70]<1.01171){if(f[26]<1.74833e+08){if(f[26]<5.98863e+07){r+=-0.648228;}else{r+=0.699818;}}else{if(f[28]<1.40724e+11){r+=0.628785;}else{r+=-0.710064;}}}else{r+=-0.659617;}}}}else{if(f[84]<12){if(f[15]<5.6878e+08){if(f[9]<1.24291e+08){if(f[84]<1.5){if(f[25]<8.82535e+07){r+=0.710348;}else{r+=-0.0688344;}}else{r+=-0.685309;}}else{r+=-0.68302;}}else{r+=0.165942;}}else{if(f[28]<1.41239e+11){r+=-0.549544;}else{if(f[40]<6.53189e+07){r+=0.640673;}else{r+=1.12341;}}}}}}}}else{if(f[28]<1.39819e+11){if(f[14]<1.39889e+11){if(f[42]<1.00264){if(f[26]<2.2715e+08){r+=-0.688298;}else{r+=0.32922;}}else{if(f[8]<9.70111e+07){r+=-0.0537537;}else{if(f[28]<1.39436e+11){r+=0.12238;}else{if(f[14]<1.39856e+11){r+=0;}else{r+=0.678539;}}}}}else{if(f[10]<3.19191e+08){if(f[41]<2.00836e+08){if(f[28]<1.39808e+11){if(f[10]<3.12138e+08){if(f[51]<3.17548){if(f[35]<5.88769e+08){if(f[10]<3.05984e+08){r+=-0.681331;}else{r+=-0.33755;}}else{r+=-0.168186;}}else{if(f[10]<1.10185e+08){r+=0.0873373;}else{r+=-0.454279;}}}else{r+=-0.0644865;}}else{if(f[20]<2.96993e+07){if(f[51]<2.15344){if(f[41]<1.5889e+08){if(f[84]<0.85){r+=0.159991;}else{r+=-0.67837;}}else{if(f[14]<1.40318e+11){r+=0.515638;}else{r+=-0.656576;}}}else{if(f[69]<2.92057){r+=-0.517495;}else{r+=0.877848;}}}else{if(f[14]<1.40372e+11){if(f[37]<1.06558e+07){r+=-0.43037;}else{if(f[6]<2.2606e+07){r+=-0.37245;}else{r+=0.676321;}}}else{r+=-0.646642;}}}}else{if(f[14]<1.40316e+11){if(f[28]<1.39794e+11){if(f[14]<1.40049e+11){if(f[84]<1.85){r+=0.637944;}else{r+=-0.450823;}}else{if(f[69]<0.506525){r+=-0.274911;}else{r+=-0.671843;}}}else{if(f[70]<0.998159){if(f[14]<1.40302e+11){if(f[12]<2.5457e+08){r+=0.688038;}else{r+=0.0558792;}}else{if(f[76]<0.294265){r+=-0.625575;}else{r+=0.5287;}}}else{r+=-0.581026;}}}else{if(f[3]<662525){r+=-0.167205;}else{if(f[4]<760351){r+=-0.167453;}else{r+=-0.696256;}}}}}else{if(f[42]<1.00903){if(f[70]<0.996777){if(f[12]<3.50061e+10){if(f[28]<1.39806e+11){if(f[14]<1.40094e+11){if(f[12]<8.86611e+08){r+=0.562394;}else{r+=-0.360929;}}else{if(f[9]<1.12291e+08){r+=-0.18691;}else{r+=-0.678959;}}}else{if(f[14]<1.40352e+11){if(f[18]<8.02312e+06){r+=0.0972934;}else{r+=0.594132;}}else{r+=-0.653907;}}}else{if(f[37]<3.49509e+10){r+=0.0538203;}else{r+=0.663938;}}}else{if(f[14]<1.40029e+11){r+=-0.682878;}else{if(f[14]<1.40314e+11){if(f[82]<2.0435){if(f[62]<0.172087){r+=0.0195687;}else{r+=0.709804;}}else{if(f[4]<1.45604e+07){r+=-0.504521;}else{r+=0.0921016;}}}else{if(f[29]<1.0676e+08){if(f[9]<1.57862e+08){r+=0.294767;}else{r+=-0.618952;}}else{r+=0.947592;}}}}}else{if(f[42]<1.01116){if(f[70]<0.987599){if(f[9]<2.24467e+08){r+=1.12059;}else{r+=0.681733;}}else{r+=0.0637145;}}else{r+=-0.676387;}}}}}else{if(f[28]<1.40144e+11){if(f[84]<0.5){if(f[14]<1.40117e+11){if(f[69]<1.91864){if(f[28]<1.40022e+11){r+=-0.699853;}else{if(f[66]<0.499943){r+=-0.704566;}else{r+=-0.907823;}}}else{r+=-0.701315;}}else{if(f[28]<1.40118e+11){if(f[42]<1.00133){if(f[70]<1.00148){r+=-1.36567;}else{if(f[28]<1.40074e+11){r+=-0.978079;}else{r+=-1.31528;}}}else{if(f[14]<1.4016e+11){if(f[67]<0.578212){r+=-0.725079;}else{r+=-0.968286;}}else{if(f[9]<1.01824e+08){r+=-0.690674;}else{r+=-1.46332;}}}}else{r+=-0.916497;}}}else{if(f[70]<0.984734){if(f[28]<1.40129e+11){if(f[14]<1.40156e+11){if(f[26]<9.93219e+07){if(f[84]<2.3){r+=-0.515515;}else{r+=0.674869;}}else{if(f[1]<1.76969e+08){r+=-0.668772;}else{r+=-0.19249;}}}else{if(f[28]<1.40112e+11){if(f[59]<107.62){r+=-0.683845;}else{r+=0.321431;}}else{if(f[14]<1.41239e+11){if(f[66]<1.01147){r+=0.845197;}else{r+=-0.0388719;}}else{if(f[65]<0.886816){r+=-0.143642;}else{r+=-0.842224;}}}}}else{if(f[14]<1.4125e+11){if(f[37]<1.2231e+08){r+=-0.0704684;}else{r+=0.679125;}}else{if(f[43]<0.395822){r+=-0.231174;}else{r+=-0.860741;}}}}else{if(f[14]<1.40067e+11){if(f[69]<1.46114){if(f[28]<1.39943e+11){if(f[13]<2.96587e+08){if(f[84]<1.35){r+=0.272739;}else{r+=-0.705454;}}else{if(f[42]<1.00022){r+=-0.702082;}else{r+=0.89077;}}}else{if(f[65]<0.243624){if(f[84]<2.5){r+=-0.109948;}else{r+=0.885148;}}else{if(f[84]<1.5){r+=-0.319191;}else{r+=-0.628743;}}}}else{if(f[41]<9.99874e+07){if(f[41]<9.77012e+07){if(f[66]<0.952843){r+=-1.02985;}else{r+=-0.68921;}}else{if(f[14]<1.40018e+11){r+=0.419103;}else{r+=-0.681249;}}}else{if(f[28]<1.40017e+11){if(f[42]<1.00003){r+=-0.689116;}else{r+=0.624595;}}else{if(f[84]<0.85){r+=0.208544;}else{r+=-0.710745;}}}}}else{if(f[84]<1.5){if(f[84]<1.15){if(f[51]<2.18882){if(f[70]<1.00117){r+=0.350875;}else{r+=-1.17555;}}else{if(f[66]<0.942376){r+=0.359455;}else{r+=-0.868594;}}}else{if(f[9]<2.54989e+08){if(f[14]<1.40102e+11){r+=-0.723999;}else{r+=-1.22218;}}else{if(f[14]<1.4086e+11){r+=-0.794488;}else{r+=-0.685218;}}}}else{if(f[14]<1.40346e+11){if(f[67]<0.616952){if(f[84]<2.85){r+=-0.704286;}else{r+=0.792278;}}else{if(f[14]<1.40105e+11){r+=0.215054;}else{r+=0.646694;}}}else{if(f[65]<1.02624){if(f[9]<2.42882e+08){r+=0.578493;}else{r+=-0.367461;}}else{if(f[14]<1.45039e+11){r+=-0.683697;}else{r+=0.692805;}}}}}}}}else{if(f[84]<4.85){if(f[67]<0.742289){if(f[84]<1.85){if(f[28]<1.40375e+11){if(f[31]<4.71601e+08){if(f[70]<0.98539){if(f[70]<0.980756){r+=-0.435255;}else{r+=0.688537;}}else{if(f[41]<2.90988e+09){r+=-0.803047;}else{r+=0.311931;}}}else{if(f[1]<2.64347e+07){if(f[54]<0.486743){r+=0.69527;}else{r+=1.00191;}}else{r+=-0.534041;}}}else{if(f[14]<1.40543e+11){if(f[42]<1.00199){if(f[70]<1.00193){r+=-0.888179;}else{r+=0.652401;}}else{if(f[14]<1.40466e+11){r+=-0.734024;}else{r+=0.386489;}}}else{if(f[70]<1.00231){if(f[14]<1.4117e+11){r+=-0.935715;}else{r+=-0.742509;}}else{if(f[14]<1.40853e+11){r+=-0.323263;}else{r+=-0.685682;}}}}}else{if(f[70]<1.00234){if(f[42]<0.998699){if(f[14]<1.42072e+11){r+=-0.929743;}else{r+=-0.674862;}}else{if(f[28]<1.40762e+11){if(f[14]<1.40234e+11){r+=0.728433;}else{r+=-0.84105;}}else{if(f[14]<1.40604e+11){r+=1.12153;}else{r+=0.668037;}}}}else{if(f[70]<1.01471){if(f[13]<5.52655e+07){if(f[14]<1.40772e+11){r+=-0.964087;}else{r+=-0.646426;}}else{if(f[29]<2.60439e+08){r+=-0.695765;}else{r+=0.127132;}}}else{if(f[70]<1.01819){r+=0.682146;}else{r+=-0.683889;}}}}}else{if(f[28]<1.41462e+11){if(f[14]<1.45832e+11){if(f[70]<0.982571){if(f[70]<0.980448){if(f[28]<1.40711e+11){r+=-0.68577;}else{r+=0.936172;}}else{if(f[14]<1.41325e+11){r+=0.685974;}else{r+=-0.182126;}}}else{if(f[12]<7.07354e+07){if(f[14]<1.40531e+11){r+=0.187041;}else{r+=-0.696269;}}else{if(f[82]<0.996937){r+=-0.631105;}else{r+=-0.327529;}}}}else{if(f[84]<0.8){if(f[42]<0.952364){if(f[28]<1.40713e+11){r+=-0.622027;}else{r+=0.693264;}}else{r+=1.76272;}}else{r+=-0.694023;}}}else{if(f[14]<1.40862e+11){if(f[35]<2.74284e+06){if(f[84]<0.8){r+=0.716521;}else{r+=-0.686446;}}else{if(f[42]<1.00233){if(f[84]<3){r+=0.636005;}else{r+=-0.671351;}}else{if(f[37]<2.79079e+10){r+=-0.682685;}else{r+=0.044964;}}}}else{if(f[14]<1.40874e+11){if(f[28]<1.43138e+11){if(f[70]<1.01254){r+=-0.247509;}else{r+=0.648232;}}else{r+=-0.655595;}}else{if(f[14]<1.40881e+11){if(f[28]<1.4247e+11){r+=0.548746;}else{r+=-0.656601;}}else{if(f[28]<1.41498e+11){r+=-0.126171;}else{r+=-0.692004;}}}}}}}else{if(f[28]<1.40381e+11){if(f[41]<1.75471e+09){if(f[42]<0.999299){if(f[42]<0.996534){r+=-0.689982;}else{if(f[57]<5.99018){if(f[19]<1.66246e+07){r+=0.913928;}else{r+=0.115981;}}else{r+=-0.511946;}}}else{if(f[69]<1.61995){if(f[82]<3.63543){r+=-0.935202;}else{r+=-0.664748;}}else{r+=-0.69844;}}}else{if(f[51]<0.982638){if(f[42]<1.00071){if(f[70]<1.0022){if(f[13]<8.97672e+08){r+=0.698949;}else{r+=1.03543;}}else{r+=-0.303862;}}else{r+=-0.583117;}}else{if(f[20]<1.10587e+08){if(f[33]<2.97344e+06){r+=0.0620956;}else{r+=-0.721582;}}else{if(f[21]<3.39804e+09){r+=0.72466;}else{r+=-0.325624;}}}}}else{if(f[28]<1.40732e+11){if(f[14]<1.40387e+11){if(f[21]<1.05198e+09){if(f[40]<6.00194e+07){r+=0.223339;}else{if(f[54]<0.490529){r+=-1.33342;}else{r+=-0.921548;}}}else{if(f[9]<8.47579e+09){r+=0.921889;}else{r+=-0.140075;}}}else{if(f[70]<1.00628){if(f[41]<7.7272e+07){if(f[28]<1.40605e+11){r+=0.774504;}else{r+=-0.547324;}}else{if(f[41]<8.95841e+08){r+=1.01524;}else{r+=0.685474;}}}else{if(f[12]<9.6588e+07){r+=-0.096587;}else{r+=-0.675518;}}}}else{if(f[28]<1.41928e+11){if(f[29]<1.61484e+09){if(f[28]<1.40743e+11){if(f[10]<2.96525e+08){r+=0.743199;}else{r+=-0.65585;}}else{if(f[77]<11.7008){r+=-0.697119;}else{r+=0.313451;}}}else{if(f[54]<0.875433){r+=-0.0224397;}else{r+=0.862137;}}}else{if(f[1]<7.48415e+08){r+=0.68475;}else{r+=0.122011;}}}}}}}}}} return r;}
    static double p5(double[] f){double r=0;if(f[28]<1.39436e+11){if(f[56]<1.00179){if(f[28]<1.39216e+11){if(f[56]<1.00126){r+=-0.585457;}else{r+=0.133155;}}else{if(f[30]<3.73846e+07){if(f[20]<9.54887e+06){r+=0.00186208;}else{r+=0.665849;}}else{r+=-0.346077;}}}else{if(f[10]<3.57921e+10){if(f[55]<0.270983){if(f[84]<1.15){r+=-0.576575;}else{if(f[14]<1.40157e+11){r+=0.671755;}else{r+=-0.450901;}}}else{if(f[9]<1.61041e+08){r+=-0.592907;}else{if(f[41]<2.58564e+08){if(f[42]<1.00946){if(f[28]<1.39429e+11){if(f[28]<1.3942e+11){r+=-0.58634;}else{if(f[84]<1){r+=-0.530915;}else{r+=0.264767;}}}else{if(f[84]<1){r+=-0.572536;}else{r+=0.469283;}}}else{if(f[11]<2.05122e+08){r+=-0.397012;}else{r+=0.634564;}}}else{if(f[56]<1.00314){if(f[84]<1){r+=-0.60129;}else{if(f[42]<1.00348){r+=0.648159;}else{r+=-0.102012;}}}else{if(f[53]<3.47414){if(f[55]<3.63183){r+=-0.588375;}else{r+=-0.218374;}}else{if(f[84]<2.15){r+=-0.530663;}else{r+=0.315627;}}}}}}}else{if(f[26]<5.51929e+10){if(f[84]<0.85){r+=-0.408858;}else{if(f[50]<0.769546){r+=-0.268161;}else{if(f[3]<5.37351e+09){r+=1.02983;}else{r+=-0.157324;}}}}else{r+=-0.519374;}}}}else{if(f[41]<5.56433e+09){if(f[67]<6.73184){if(f[0]<1.40094e+11){if(f[28]<1.40665e+11){if(f[14]<1.40791e+11){if(f[28]<1.4e+11){if(f[28]<1.39956e+11){if(f[84]<3.5){if(f[37]<3.46794e+07){if(f[84]<0.85){if(f[14]<1.40111e+11){r+=-0.554759;}else{r+=0.721518;}}else{if(f[84]<2.15){r+=-0.38083;}else{r+=-0.821126;}}}else{if(f[38]<5.51974e+08){if(f[84]<1.85){r+=0.45964;}else{r+=0.697686;}}else{if(f[0]<1.39927e+11){r+=-0.512146;}else{r+=0.424005;}}}}else{if(f[0]<1.39982e+11){r+=-0.606577;}else{if(f[14]<1.40321e+11){if(f[52]<1.92168){r+=-0.695209;}else{r+=0.105362;}}else{if(f[51]<2.17143){r+=-1.69869;}else{r+=-0.64366;}}}}}else{if(f[67]<0.763463){if(f[29]<1.38809e+07){if(f[30]<1.04132e+07){r+=-0.652873;}else{if(f[1]<1.17672e+07){r+=-0.355703;}else{r+=0.311536;}}}else{if(f[84]<3.35){if(f[14]<1.40031e+11){r+=-0.512876;}else{r+=0.664303;}}else{r+=-0.6906;}}}else{if(f[36]<1.45658e+07){if(f[51]<2.19932){if(f[84]<1.35){r+=-1.68649;}else{r+=-1.15033;}}else{r+=-0.748551;}}else{if(f[0]<1.40077e+11){if(f[14]<1.40502e+11){r+=-0.644505;}else{r+=0.647105;}}else{if(f[14]<1.4032e+11){r+=0.629457;}else{r+=-1.03866;}}}}}}else{if(f[28]<1.40552e+11){if(f[84]<0.5){if(f[41]<6.48415e+08){if(f[14]<1.39649e+11){if(f[14]<1.39169e+11){r+=-0.617732;}else{r+=0.596738;}}else{if(f[0]<1.39423e+11){r+=0.177425;}else{r+=-0.706895;}}}else{if(f[39]<5.99457e+08){if(f[14]<1.40141e+11){r+=0.562267;}else{r+=-0.64445;}}else{if(f[65]<2.55414){r+=-0.610919;}else{r+=0.449649;}}}}else{if(f[0]<1.40034e+11){if(f[28]<1.40049e+11){if(f[84]<3.15){r+=-0.624721;}else{r+=-0.915003;}}else{if(f[14]<1.40409e+11){r+=0.378786;}else{r+=-0.589633;}}}else{if(f[38]<1.44737e+07){if(f[56]<1.00051){r+=-0.869143;}else{r+=0.534382;}}else{if(f[28]<1.40118e+11){r+=0.477696;}else{r+=0.715786;}}}}}else{if(f[14]<1.40321e+11){if(f[1]<1.1173e+07){if(f[84]<0.5){r+=-0.438776;}else{r+=0.594999;}}else{if(f[11]<6.04496e+07){r+=0.10033;}else{r+=-0.631706;}}}else{if(f[84]<0.5){if(f[12]<7.07176e+07){if(f[41]<5.30382e+07){r+=0.689499;}else{r+=0.923839;}}else{if(f[41]<6.89985e+07){r+=0.908842;}else{r+=1.3908;}}}else{if(f[84]<6.60001){r+=-0.564813;}else{if(f[40]<5.21963e+07){r+=0.637197;}else{r+=0.824858;}}}}}}}else{if(f[0]<1.39814e+11){if(f[84]<2.85){if(f[80]<7.49296){if(f[42]<0.99207){if(f[19]<8.09449e+08){if(f[0]<1.39803e+11){r+=-0.597784;}else{r+=-0.0911945;}}else{r+=0.403394;}}else{if(f[28]<1.39947e+11){r+=-1.42125;}else{r+=-0.434923;}}}else{if(f[0]<1.39728e+11){r+=-0.419055;}else{r+=0.556045;}}}else{if(f[28]<1.40058e+11){r+=0.700172;}else{r+=-0.601465;}}}else{if(f[55]<1.44615){if(f[84]<1.5){if(f[28]<1.39814e+11){r+=-0.549459;}else{if(f[28]<1.40118e+11){r+=-1.01945;}else{r+=-0.553044;}}}else{if(f[84]<1.85){if(f[36]<1.16121e+07){r+=-0.43485;}else{if(f[14]<1.40925e+11){r+=-0.322468;}else{r+=0.607367;}}}else{r+=-0.661635;}}}else{if(f[0]<1.40018e+11){if(f[56]<1.00387){r+=0.275201;}else{r+=-0.604322;}}else{if(f[14]<1.4478e+11){r+=0.00983151;}else{r+=0.522491;}}}}}}else{if(f[0]<1.39582e+11){if(f[28]<1.41235e+11){if(f[30]<2.05338e+08){if(f[14]<1.46549e+11){if(f[12]<7.06527e+07){if(f[28]<1.40988e+11){r+=-0.969281;}else{r+=-0.59518;}}else{r+=-0.611627;}}else{r+=0.0878415;}}else{r+=0.146603;}}else{if(f[84]<2.8){if(f[28]<1.41629e+11){if(f[28]<1.41395e+11){if(f[14]<1.44077e+11){if(f[30]<6.60739e+07){r+=-0.600195;}else{r+=-0.0913613;}}else{if(f[14]<1.45829e+11){r+=2.58592;}else{r+=0.604372;}}}else{if(f[26]<1.0831e+08){r+=0.600769;}else{if(f[67]<1.05967){r+=0.613966;}else{r+=0.931875;}}}}else{r+=-0.595768;}}else{r+=-0.596568;}}}else{if(f[38]<5.12628e+07){if(f[28]<1.42094e+11){if(f[28]<1.41219e+11){r+=-0.410173;}else{r+=0.598526;}}else{r+=-0.72779;}}else{if(f[80]<8.20589){if(f[28]<1.41005e+11){if(f[83]<2.20147){if(f[67]<2.0937){if(f[12]<7.19111e+07){r+=-0.971272;}else{r+=-0.729718;}}else{r+=-0.610723;}}else{if(f[0]<1.39981e+11){if(f[20]<2.36408e+08){r+=-0.616771;}else{r+=0.159348;}}else{if(f[13]<5.02737e+08){r+=0.551969;}else{r+=-0.451056;}}}}else{if(f[77]<0.105031){if(f[14]<1.46412e+11){r+=-0.465675;}else{r+=0.496216;}}else{if(f[37]<2.31781e+07){r+=0.232474;}else{if(f[0]<1.39588e+11){r+=-0.283625;}else{r+=-0.600748;}}}}}else{if(f[7]<5.8676e+07){r+=-0.43486;}else{r+=0.554738;}}}}}}else{if(f[28]<1.40609e+11){if(f[28]<1.40144e+11){if(f[84]<0.85){if(f[14]<1.39643e+11){if(f[0]<1.40275e+11){r+=0.61194;}else{r+=-0.610772;}}else{if(f[0]<1.40098e+11){if(f[14]<1.40048e+11){if(f[11]<6.20506e+07){r+=-0.168523;}else{if(f[82]<0.958505){r+=1.1615;}else{r+=0.66322;}}}else{if(f[34]<3.9685e+07){if(f[57]<3.07515){r+=-0.670489;}else{r+=-0.170463;}}else{r+=0.239894;}}}else{if(f[42]<1.01118){if(f[28]<1.39483e+11){if(f[0]<1.40408e+11){r+=-1.22581;}else{r+=-0.576348;}}else{if(f[20]<2.10181e+08){r+=-0.68385;}else{r+=0.55347;}}}else{if(f[0]<1.42905e+11){if(f[80]<0.78161){r+=-0.533689;}else{r+=0.625575;}}else{if(f[0]<1.42916e+11){r+=-0.119893;}else{r+=-0.594887;}}}}}}else{if(f[0]<1.4619e+11){if(f[0]<1.40124e+11){if(f[28]<1.39965e+11){if(f[37]<2.92477e+07){if(f[83]<0.347871){r+=-0.960697;}else{r+=0.371191;}}else{if(f[84]<2.15){r+=0.631813;}else{r+=-0.165626;}}}else{if(f[56]<1.00122){if(f[14]<1.40146e+11){r+=-0.720505;}else{r+=-1.25434;}}else{if(f[84]<1.15){r+=0.687679;}else{r+=-0.792958;}}}}else{if(f[14]<1.4062e+11){if(f[84]<2.85){if(f[41]<6.20886e+08){r+=0.148653;}else{r+=-0.250244;}}else{if(f[67]<0.267369){r+=-1.24969;}else{r+=0.496263;}}}else{if(f[84]<1.15){if(f[28]<1.39936e+11){r+=-0.0079857;}else{r+=0.646313;}}else{if(f[14]<1.40761e+11){r+=-0.956666;}else{r+=-0.648177;}}}}}else{r+=-0.595124;}}}else{if(f[14]<1.40578e+11){if(f[14]<1.40011e+11){if(f[42]<1.00738){if(f[84]<1.15){if(f[56]<0.995442){r+=-0.724066;}else{if(f[28]<1.40147e+11){r+=-0.202952;}else{r+=0.68027;}}}else{if(f[56]<0.995288){if(f[84]<2.15){r+=0.625702;}else{r+=-0.531073;}}else{if(f[84]<2.65){r+=0.533528;}else{r+=-0.653426;}}}}else{if(f[35]<5.50487e+07){if(f[41]<1.48596e+09){r+=0.594124;}else{r+=0.811921;}}else{if(f[13]<1.05661e+08){if(f[65]<0.759948){r+=-0.592945;}else{r+=-1.16099;}}else{if(f[0]<1.43638e+11){r+=-0.604114;}else{r+=0.306943;}}}}}else{if(f[65]<0.246592){if(f[84]<2.85){if(f[84]<1.85){if(f[29]<2.31224e+08){r+=-0.659442;}else{r+=0.502691;}}else{if(f[0]<1.40512e+11){r+=-1.35543;}else{r+=0.664448;}}}else{if(f[7]<1.60319e+08){r+=-0.0741294;}else{r+=0.777251;}}}else{if(f[28]<1.40578e+11){if(f[0]<1.41606e+11){if(f[10]<4.08122e+09){r+=-0.688476;}else{r+=-0.303697;}}else{if(f[84]<2.15){r+=-0.654117;}else{r+=1.09986;}}}else{if(f[0]<1.41013e+11){if(f[77]<12.4398){r+=-0.783501;}else{r+=0.651177;}}else{if(f[42]<1.00468){r+=1.10959;}else{r+=-0.327504;}}}}}}else{if(f[84]<1.15){if(f[84]<0.5){r+=-0.741435;}else{if(f[0]<1.42962e+11){if(f[14]<1.41325e+11){if(f[56]<1.00221){r+=0.723314;}else{r+=0.586876;}}else{if(f[44]<0.658424){r+=-0.102777;}else{r+=-0.746356;}}}else{r+=-0.592163;}}}else{if(f[10]<4.96337e+10){if(f[0]<1.40241e+11){if(f[54]<0.486607){r+=-1.07178;}else{r+=-0.703762;}}else{if(f[25]<7.02897e+09){r+=-0.635738;}else{r+=-0.135422;}}}else{r+=0.783617;}}}}}else{if(f[36]<1.47186e+08){if(f[82]<1.98409){if(f[14]<1.39489e+11){r+=-0.591966;}else{if(f[26]<3.96505e+08){if(f[0]<1.40704e+11){if(f[28]<1.42008e+11){if(f[0]<1.40692e+11){r+=-0.653174;}else{r+=0.14414;}}else{if(f[28]<1.43147e+11){r+=0.602058;}else{r+=0.0966948;}}}else{if(f[28]<1.42196e+11){if(f[36]<6.88097e+07){r+=1.19086;}else{r+=0.532992;}}else{if(f[14]<1.39742e+11){r+=-0.0238942;}else{r+=-0.59707;}}}}else{if(f[42]<1.00382){if(f[14]<1.40024e+11){if(f[52]<0.719748){r+=0.11666;}else{r+=0.5675;}}else{if(f[84]<4.85){r+=-0.639597;}else{r+=0.563332;}}}else{if(f[28]<1.41053e+11){if(f[3]<5.58894e+06){r+=0.838156;}else{r+=-0.578657;}}else{if(f[84]<1.65){r+=-0.416387;}else{r+=0.845563;}}}}}}else{if(f[42]<1.00135){if(f[56]<1.00222){if(f[28]<1.41001e+11){r+=-0.842216;}else{r+=-0.60491;}}else{if(f[11]<2.76946e+08){r+=0.611425;}else{r+=-0.616719;}}}else{if(f[56]<0.994443){if(f[28]<1.76887e+11){if(f[56]<0.993267){if(f[84]<0.5){r+=-0.17505;}else{r+=-0.586379;}}else{r+=-0.15095;}}else{r+=0.587088;}}else{if(f[10]<2.89682e+08){if(f[73]<0.0940047){r+=0.149249;}else{r+=0.744201;}}else{r+=-0.000732422;}}}}}else{if(f[56]<1.00192){if(f[56]<0.76259){if(f[14]<1.40585e+11){r+=0.600706;}else{r+=-0.556539;}}else{if(f[0]<1.4073e+11){if(f[28]<1.42055e+11){if(f[56]<0.998936){if(f[77]<0.199067){r+=-0.0461728;}else{r+=-0.647539;}}else{if(f[41]<8.02626e+07){r+=-0.694475;}else{r+=-1.08081;}}}else{if(f[28]<1.42768e+11){if(f[29]<3.80809e+08){r+=0.618771;}else{r+=-0.515169;}}else{if(f[14]<1.40156e+11){r+=0.174413;}else{r+=-0.716114;}}}}else{if(f[39]<6.50111e+07){if(f[12]<7.13451e+07){r+=0.909497;}else{if(f[28]<1.40659e+11){r+=1.44137;}else{r+=0.155205;}}}else{if(f[0]<1.40827e+11){if(f[84]<2.15){r+=0.561762;}else{r+=-0.632775;}}else{if(f[42]<1.02729){r+=-0.500554;}else{r+=0.488516;}}}}}}else{if(f[28]<1.40729e+11){if(f[13]<2.22454e+09){if(f[54]<0.486219){if(f[0]<1.41252e+11){r+=0.598692;}else{r+=0.0853967;}}else{r+=0.698052;}}else{if(f[14]<1.41182e+11){r+=0.442558;}else{r+=-0.924741;}}}else{if(f[33]<1.39762e+08){if(f[14]<1.41334e+11){r+=-0.0573906;}else{r+=-0.645362;}}else{if(f[84]<2.15){if(f[78]<1.03007){if(f[0]<1.41072e+11){r+=0.226348;}else{r+=1.07773;}}else{r+=-0.193863;}}else{r+=-0.183512;}}}}}}}}else{if(f[28]<1.79185e+11){if(f[65]<1.39762){if(f[56]<1.00372){if(f[42]<0.99981){r+=-0.63471;}else{if(f[12]<3.97276e+08){r+=-0.166511;}else{if(f[55]<0.789213){r+=0.537886;}else{r+=0.0925983;}}}}else{if(f[84]<2.5){if(f[65]<1.02761){r+=0.595474;}else{r+=0.951975;}}else{r+=-0.538497;}}}else{if(f[41]<1.65044e+09){if(f[65]<3.7437){if(f[28]<1.39879e+11){if(f[84]<2.15){if(f[9]<1.29779e+08){r+=0.759912;}else{r+=-0.302945;}}else{if(f[22]<5.24508e+07){r+=-0.888807;}else{r+=-0.550047;}}}else{if(f[64]<1.53758){if(f[36]<2.92243e+07){if(f[1]<1.62459e+07){if(f[8]<2.60998e+07){r+=-0.088861;}else{r+=0.568598;}}else{if(f[84]<1.5){r+=-0.559552;}else{r+=-0.0956456;}}}else{if(f[3]<852897){r+=0.515553;}else{if(f[0]<1.42603e+11){r+=-0.53232;}else{r+=0.237402;}}}}else{if(f[25]<8.98579e+07){if(f[28]<1.40119e+11){r+=-1.0717;}else{r+=-0.684574;}}else{if(f[28]<1.39919e+11){if(f[0]<1.40015e+11){r+=0.525023;}else{r+=-0.494156;}}else{if(f[5]<289174){r+=0.326833;}else{r+=-0.596667;}}}}}}else{if(f[10]<5.22577e+07){if(f[39]<1.35284e+07){if(f[38]<4.50008e+06){if(f[84]<1.15){r+=-0.758899;}else{r+=-0.59336;}}else{if(f[28]<1.40116e+11){if(f[26]<7.4103e+07){r+=-0.945613;}else{r+=-1.42044;}}else{r+=-0.742338;}}}else{if(f[58]<3.98721){r+=0.38214;}else{r+=-0.405009;}}}else{if(f[64]<1.58389){if(f[42]<0.997252){r+=0.499817;}else{r+=-0.347648;}}else{if(f[13]<5.99976e+09){if(f[8]<6.32208e+06){if(f[58]<6.3911){r+=0.495477;}else{r+=-0.241734;}}else{if(f[0]<1.40222e+11){r+=-0.583318;}else{r+=-0.652205;}}}else{if(f[30]<1.6671e+07){if(f[17]<2.49602e+06){r+=-0.0967454;}else{r+=-0.508556;}}else{if(f[33]<3.09976e+07){r+=0.153001;}else{r+=0.894452;}}}}}}}else{if(f[53]<0.228049){if(f[0]<1.39975e+11){r+=0.2616;}else{if(f[43]<0.575217){r+=-0.597405;}else{r+=-0.056751;}}}else{if(f[22]<9.04327e+09){if(f[84]<1.15){if(f[28]<1.40593e+11){r+=0.699955;}else{r+=0.0128189;}}else{if(f[28]<1.40317e+11){if(f[83]<2.46857){if(f[58]<0.754088){r+=0.493724;}else{r+=1.7831;}}else{r+=0.541877;}}else{r+=0.259687;}}}else{if(f[26]<2.42215e+10){r+=-0.547719;}else{r+=0.122764;}}}}}}else{if(f[14]<1.40114e+11){r+=0.595098;}else{r+=-0.574647;}}}}else{if(f[83]<1.25814){if(f[14]<1.40609e+11){if(f[28]<1.40053e+11){if(f[12]<3.26835e+10){if(f[84]<2.5){if(f[67]<0.995166){if(f[1]<5.70444e+07){if(f[28]<1.39768e+11){r+=-0.339442;}else{if(f[53]<2.51766){if(f[12]<7.68155e+09){r+=-1.42686;}else{r+=-0.459348;}}else{r+=-0.352884;}}}else{if(f[61]<0.492938){if(f[20]<3.19813e+07){r+=-0.164896;}else{r+=-0.766333;}}else{if(f[39]<2.71643e+10){r+=0.460427;}else{r+=-0.565323;}}}}else{if(f[28]<1.39923e+11){if(f[4]<7.7393e+07){if(f[6]<1.63169e+07){r+=0.0932522;}else{r+=0.641367;}}else{r+=-0.466396;}}else{if(f[73]<0.42727){if(f[72]<0.691014){r+=0.241534;}else{r+=-0.245348;}}else{if(f[12]<3.15218e+09){r+=-0.0415783;}else{if(f[32]<1.97389e+08){r+=-0.850365;}else{r+=-0.23012;}}}}}}else{if(f[42]<1.00097){if(f[56]<1.00359){if(f[7]<8.04397e+07){r+=-0.211027;}else{if(f[56]<1.00329){r+=0.668945;}else{r+=0.164154;}}}else{r+=-0.408403;}}else{if(f[39]<2.61613e+10){r+=0.108419;}else{r+=-0.724572;}}}}else{if(f[33]<6.58755e+08){if(f[71]<0.110012){r+=-0.168801;}else{if(f[80]<0.5252){r+=1.25182;}else{r+=0.642218;}}}else{if(f[80]<0.705644){if(f[43]<2.49161){r+=-0.222475;}else{r+=0.62871;}}else{r+=-0.508082;}}}}else{if(f[22]<8.17555e+09){if(f[40]<2.14889e+09){if(f[35]<5.61013e+08){if(f[55]<1.33534){if(f[53]<0.862218){r+=0.66966;}else{r+=1.93214;}}else{r+=0.468786;}}else{r+=-0.356501;}}else{if(f[84]<2.5){if(f[84]<1.15){if(f[28]<1.40378e+11){if(f[29]<9.98263e+07){if(f[54]<0.992567){r+=0.219305;}else{r+=-0.520696;}}else{if(f[47]<2.30592){r+=-0.0689355;}else{r+=0.836168;}}}else{if(f[84]<0.85){if(f[75]<0.456302){r+=0.120258;}else{r+=0.75035;}}else{if(f[0]<1.40098e+11){r+=1.2125;}else{r+=0.0574007;}}}}else{if(f[28]<1.40315e+11){if(f[73]<1.69061){if(f[84]<1.5){r+=-0.380527;}else{r+=0.784857;}}else{if(f[3]<1.0747e+07){r+=0.419265;}else{r+=-0.629998;}}}else{if(f[82]<1.01221){r+=-0.722413;}else{r+=0.59332;}}}}else{if(f[56]<0.995769){r+=-0.146283;}else{if(f[56]<1.00071){if(f[8]<5.78385e+09){r+=0.793564;}else{r+=1.18463;}}else{r+=-0.00954075;}}}}}else{if(f[64]<0.912352){if(f[31]<7.73838e+08){if(f[7]<8.35706e+09){r+=-0.461309;}else{if(f[51]<1.01467){r+=-0.116563;}else{if(f[77]<0.843837){r+=0.10269;}else{r+=0.975998;}}}}else{r+=-0.605181;}}else{if(f[4]<2.63159e+09){if(f[34]<1.74538e+08){r+=-0.112672;}else{if(f[62]<0.502083){r+=-0.182007;}else{r+=-0.763396;}}}else{r+=0.0100292;}}}}}else{if(f[54]<1.09421){if(f[30]<1.77663e+09){if(f[49]<2.08188){if(f[14]<1.40611e+11){r+=-0.154317;}else{if(f[29]<2.30375e+09){r+=-0.644277;}else{r+=-0.179373;}}}else{r+=0.0396924;}}else{r+=0.493203;}}else{if(f[28]<1.40561e+11){if(f[21]<1.2637e+10){if(f[18]<4.05033e+08){if(f[84]<1.15){r+=0.500568;}else{r+=-0.183932;}}else{if(f[39]<3.85038e+10){r+=1.22394;}else{r+=0.350818;}}}else{r+=-0.170368;}}else{if(f[52]<0.57572){r+=0.419137;}else{if(f[80]<1.12847){r+=-0.706603;}else{r+=-0.173692;}}}}}}else{if(f[29]<8.91341e+06){if(f[14]<1.40087e+11){if(f[21]<1.36741e+07){r+=0.539393;}else{if(f[28]<1.39922e+11){r+=0.0702963;}else{if(f[25]<9.65196e+07){r+=0.119478;}else{if(f[13]<5.88372e+09){r+=-0.608599;}else{r+=0.0981816;}}}}}else{if(f[14]<1.40517e+11){if(f[84]<1.5){if(f[56]<1.00181){if(f[41]<9.2664e+09){if(f[22]<4.55818e+07){if(f[77]<1.1107){r+=-0.080976;}else{r+=-0.53022;}}else{if(f[0]<1.39985e+11){r+=-0.463802;}else{if(f[31]<4.91948e+06){r+=0.525651;}else{r+=-0.219467;}}}}else{if(f[84]<0.5){r+=-0.426388;}else{if(f[28]<1.40106e+11){if(f[25]<6.73254e+08){r+=0.698982;}else{r+=-0.348098;}}else{if(f[84]<0.85){r+=-0.0133602;}else{r+=0.57577;}}}}}else{if(f[1]<1.6221e+07){r+=-0.569815;}else{r+=-0.062919;}}}else{if(f[82]<4.30009){if(f[0]<1.40013e+11){r+=0.699092;}else{if(f[54]<0.589572){r+=-0.492989;}else{r+=0.499062;}}}else{if(f[84]<3.5){r+=-0.681575;}else{if(f[11]<2.27963e+08){r+=0.344908;}else{r+=-0.077709;}}}}}else{if(f[31]<629260){r+=-0.146222;}else{r+=-0.616289;}}}}else{if(f[56]<0.922247){if(f[54]<0.573944){r+=-0.492961;}else{if(f[10]<7.04799e+07){r+=-0.0350384;}else{if(f[67]<0.784541){r+=0.616364;}else{r+=0.059711;}}}}else{if(f[83]<1.3673){if(f[0]<1.40168e+11){if(f[78]<1.03916){r+=-0.289103;}else{r+=0.408445;}}else{if(f[31]<1.66909e+08){r+=-0.811323;}else{r+=0.00950452;}}}else{if(f[0]<1.40082e+11){if(f[54]<0.825817){if(f[14]<1.40309e+11){if(f[39]<3.40831e+10){if(f[53]<0.58002){if(f[84]<1.5){r+=-0.19591;}else{r+=0.600461;}}else{r+=0.891369;}}else{r+=-0.422043;}}else{if(f[71]<7.76313){if(f[84]<12){r+=-0.683133;}else{if(f[53]<0.694156){r+=0.348257;}else{r+=-0.431392;}}}else{r+=0.727693;}}}else{if(f[40]<6.49871e+08){r+=0.518025;}else{if(f[84]<2.85){r+=-0.616759;}else{r+=0.194023;}}}}else{if(f[84]<3.15){if(f[48]<0.000393234){if(f[5]<22345.6){r+=0.444112;}else{r+=-0.0662641;}}else{if(f[5]<2.15251e+09){if(f[28]<1.40009e+11){if(f[0]<1.412e+11){r+=-0.576376;}else{r+=0.200553;}}else{if(f[18]<1.12937e+09){r+=-0.59837;}else{r+=-0.203533;}}}else{r+=0.0879806;}}}else{if(f[0]<1.40947e+11){if(f[53]<0.615441){r+=1.11079;}else{r+=0.192705;}}else{r+=-0.562961;}}}}}}}}} return r;}
    static double p6(double[] f){double r=0;if(f[28]<1.39435e+11){if(f[56]<1.00196){if(f[13]<1.8703e+08){if(f[14]<1.29306e+11){r+=-0.777778;}else{if(f[14]<1.39647e+11){r+=0.921569;}else{r+=-0;}}}else{if(f[27]<7.63779e+10){if(f[56]<1.00182){if(f[13]<5.35538e+10){r+=-0.960396;}else{r+=-0.25;}}else{r+=-0.2;}}else{r+=0.333333;}}}else{if(f[13]<4.25751e+10){if(f[28]<1.39428e+11){if(f[71]<1.69413){r+=-0.996777;}else{if(f[42]<1.0095){if(f[7]<2.18487e+09){r+=-0.994421;}else{if(f[29]<2.50156e+09){r+=-0.636364;}else{if(f[5]<3.6538e+08){r+=0.555556;}else{r+=-0;}}}}else{if(f[42]<1.01088){if(f[14]<1.40178e+11){r+=-0.733333;}else{r+=0.9375;}}else{r+=-0.981043;}}}}else{if(f[14]<1.39879e+11){if(f[14]<1.39823e+11){if(f[19]<4.99148e+06){r+=-0;}else{if(f[5]<5.43587e+07){r+=-0.944444;}else{r+=-0.25;}}}else{if(f[57]<1.75292){if(f[25]<7.96388e+07){r+=-0.25;}else{if(f[47]<2.18359){if(f[1]<7.01258e+07){r+=0.878788;}else{r+=0.25;}}else{r+=-0.111111;}}}else{if(f[14]<1.39838e+11){r+=-0.555556;}else{r+=-0;}}}}else{if(f[29]<1.01396e+08){r+=-0.9799;}else{r+=-0;}}}}else{if(f[27]<3.91123e+10){if(f[23]<6.26835e+10){r+=0.75;}else{r+=-0.5;}}else{if(f[48]<0.56775){r+=-0;}else{r+=-0.777778;}}}}}else{if(f[69]<0.537222){if(f[0]<1.4012e+11){if(f[0]<1.39989e+11){if(f[28]<1.39977e+11){if(f[41]<2.20026e+09){if(f[14]<1.40084e+11){if(f[14]<1.39555e+11){r+=0.963964;}else{r+=-0.989873;}}else{if(f[28]<1.39867e+11){if(f[56]<1.00819){r+=-0;}else{r+=-0.936508;}}else{r+=0.998392;}}}else{if(f[83]<4.10434){if(f[67]<0.193915){if(f[28]<1.39638e+11){r+=-0.5;}else{if(f[0]<1.39428e+11){r+=0.25;}else{r+=0.939394;}}}else{if(f[55]<3.67959){if(f[2]<6.87996e+06){r+=-0.25;}else{r+=-0.965217;}}else{r+=0.25;}}}else{r+=-0.995185;}}}else{if(f[82]<9.24796){if(f[49]<5.28222){if(f[66]<1.33276){if(f[52]<0.873694){if(f[14]<1.40446e+11){if(f[0]<1.39988e+11){if(f[56]<1.00141){r+=-0.976471;}else{r+=-0.111111;}}else{if(f[28]<1.40181e+11){r+=-0.764706;}else{r+=0.5;}}}else{if(f[0]<1.39779e+11){if(f[56]<0.999479){r+=0.666667;}else{r+=-0.982609;}}else{if(f[28]<1.4054e+11){r+=-0.952381;}else{r+=0.855072;}}}}else{if(f[49]<4.13349){if(f[43]<10.1431){if(f[0]<1.37964e+11){r+=0.04;}else{r+=-0.944538;}}else{if(f[29]<7.80425e+06){r+=0.515152;}else{r+=-0.833333;}}}else{if(f[70]<1.00081){if(f[7]<3.67052e+07){r+=0.9375;}else{r+=-0.25;}}else{if(f[66]<0.999909){r+=-0.96;}else{r+=0.142857;}}}}}else{if(f[12]<9.35709e+07){if(f[56]<1.00184){if(f[42]<0.99677){if(f[14]<1.40888e+11){r+=0.946844;}else{r+=-0.6;}}else{r+=-0.818182;}}else{if(f[14]<1.46792e+11){r+=-0.940298;}else{r+=0.5;}}}else{if(f[42]<0.958617){r+=0.935484;}else{if(f[0]<1.39786e+11){if(f[77]<1.14036){r+=-0.968153;}else{r+=-0.157895;}}else{if(f[28]<1.40179e+11){r+=-0.793478;}else{r+=0.0392157;}}}}}}else{if(f[70]<1.00082){if(f[9]<3.22511e+07){r+=0.25;}else{r+=0.991361;}}else{if(f[43]<54.5835){if(f[68]<0.58492){if(f[63]<1.89794){r+=-0.977143;}else{r+=-0.25;}}else{if(f[0]<1.39816e+11){r+=0.25;}else{r+=-0.5;}}}else{if(f[23]<3.03052e+07){r+=0.833333;}else{if(f[1]<8.97351e+08){r+=0.333333;}else{r+=-0.692308;}}}}}}else{if(f[70]<1.00566){if(f[28]<1.4018e+11){if(f[56]<1.00145){if(f[0]<1.39984e+11){if(f[28]<1.4018e+11){if(f[74]<0.303625){r+=-0;}else{r+=-0.917051;}}else{if(f[57]<0.93611){r+=-0.454545;}else{r+=0.454545;}}}else{if(f[55]<0.401797){if(f[19]<5.55034e+06){r+=-0.793103;}else{r+=0.25;}}else{if(f[55]<0.585819){r+=0.341935;}else{r+=-0.419355;}}}}else{if(f[28]<1.40069e+11){r+=-0.5;}else{r+=0.973856;}}}else{if(f[14]<1.40092e+11){if(f[29]<2.8801e+06){r+=0.4;}else{r+=-0.897436;}}else{if(f[69]<0.14826){if(f[54]<0.475521){if(f[1]<7.17615e+06){r+=-0.84;}else{r+=-0;}}else{if(f[19]<1.68171e+06){r+=-0.578947;}else{r+=0.469388;}}}else{if(f[42]<0.994956){r+=-0.733333;}else{if(f[11]<6.97181e+07){r+=0.840909;}else{r+=0.570946;}}}}}}else{r+=-0.992218;}}}}else{if(f[55]<0.873866){if(f[14]<1.40123e+11){if(f[83]<3.09771){if(f[28]<1.40155e+11){if(f[30]<2.24042e+07){if(f[0]<1.40074e+11){r+=-0.956044;}else{r+=-0.0909091;}}else{if(f[0]<1.40072e+11){r+=0.25;}else{r+=0.882353;}}}else{if(f[28]<1.4077e+11){if(f[7]<5.46528e+08){r+=0.962264;}else{r+=0.25;}}else{r+=-0.666667;}}}else{if(f[28]<1.40184e+11){if(f[0]<1.40052e+11){if(f[28]<1.40164e+11){if(f[1]<2.0736e+07){r+=-0.943662;}else{r+=0.636364;}}else{if(f[28]<1.40182e+11){if(f[14]<1.40095e+11){r+=0.139535;}else{r+=0.637584;}}else{if(f[69]<0.146694){r+=-0.0769231;}else{r+=0.84058;}}}}else{if(f[0]<1.40103e+11){if(f[14]<1.40057e+11){r+=0.995906;}else{if(f[0]<1.40079e+11){r+=0.5;}else{r+=-0;}}}else{r+=-0.5;}}}else{if(f[28]<1.40805e+11){if(f[0]<1.39992e+11){if(f[69]<0.149912){if(f[25]<7.41834e+07){r+=-0.263158;}else{r+=0.771429;}}else{if(f[14]<1.40093e+11){r+=0.428571;}else{r+=0.93115;}}}else{if(f[1]<7.34457e+07){r+=0.994523;}else{r+=-0;}}}else{r+=-0.555556;}}}}else{if(f[14]<1.4021e+11){if(f[0]<1.4012e+11){if(f[17]<1.58239e+07){if(f[14]<1.40173e+11){if(f[0]<1.40118e+11){r+=-0.5;}else{r+=-0;}}else{r+=-0.984064;}}else{r+=-0.111111;}}else{r+=-0.0909091;}}else{if(f[28]<1.40404e+11){if(f[81]<5.61348){if(f[29]<5.63e+07){if(f[28]<1.40399e+11){if(f[0]<1.40062e+11){r+=0.407407;}else{r+=0.915094;}}else{if(f[36]<3.69175e+07){r+=-0.636364;}else{r+=0.764706;}}}else{if(f[46]<1.83058){r+=0.333333;}else{r+=-0.5;}}}else{r+=-0.555556;}}else{if(f[15]<3.04193e+07){if(f[29]<2.75539e+07){r+=-0.647059;}else{r+=0.636364;}}else{if(f[55]<0.8322){r+=-0.924528;}else{r+=-0.25;}}}}}}else{if(f[0]<1.40059e+11){if(f[41]<1.10222e+08){if(f[27]<3.1712e+07){r+=-0.789474;}else{if(f[0]<1.40014e+11){r+=0.994143;}else{r+=0.25;}}}else{if(f[29]<1.79048e+07){if(f[78]<0.789945){if(f[27]<8.53328e+07){if(f[4]<9.42405e+06){r+=0.97351;}else{r+=0.25;}}else{if(f[54]<0.990316){if(f[28]<1.40178e+11){r+=-0.454545;}else{r+=0.807692;}}else{if(f[22]<1.25361e+08){r+=-0.866667;}else{r+=0.142857;}}}}else{if(f[68]<0.179042){if(f[28]<1.40182e+11){if(f[78]<1.13853){r+=0.452632;}else{r+=-0.948424;}}else{if(f[12]<5.42578e+08){r+=0.935484;}else{r+=-0.555556;}}}else{if(f[54]<0.6048){if(f[43]<0.676338){r+=-0.25;}else{r+=0.733333;}}else{if(f[64]<0.461076){r+=-0.675497;}else{r+=-0.972506;}}}}}else{if(f[81]<1.36055){if(f[28]<1.3983e+11){r+=0.84;}else{if(f[13]<1.90583e+10){if(f[62]<0.219283){r+=-0.5;}else{r+=-0.875;}}else{r+=0.555556;}}}else{if(f[66]<0.332912){if(f[29]<6.07796e+07){if(f[41]<1.09017e+09){r+=0.384615;}else{r+=-0.6;}}else{r+=-0.9;}}else{if(f[69]<0.12824){r+=-0.75;}else{if(f[56]<0.993279){r+=-0.666667;}else{r+=0.934641;}}}}}}}else{if(f[52]<1.67898){if(f[31]<5.71197e+07){if(f[28]<1.40402e+11){if(f[39]<1.37753e+08){if(f[41]<1.67693e+08){r+=-0.714286;}else{if(f[27]<3.04176e+08){r+=0.906897;}else{r+=-0.684211;}}}else{if(f[28]<1.4012e+11){if(f[28]<1.39941e+11){r+=0.836364;}else{r+=-0.882353;}}else{if(f[54]<1.25584){r+=0.879599;}else{r+=0.204082;}}}}else{if(f[56]<1.00214){if(f[0]<1.4007e+11){if(f[37]<2.0394e+07){r+=-0.5;}else{r+=0.6;}}else{if(f[8]<7.10167e+08){r+=-0.944828;}else{r+=-0;}}}else{if(f[1]<7.47625e+07){r+=0.942857;}else{r+=-0;}}}}else{if(f[14]<1.40235e+11){if(f[35]<1.6818e+09){if(f[11]<5.38235e+07){r+=0.25;}else{r+=0.998507;}}else{r+=-0.25;}}else{if(f[56]<1.0012){r+=-0.636364;}else{r+=0.6;}}}}else{if(f[42]<0.997026){if(f[21]<1.96975e+08){if(f[0]<1.40119e+11){if(f[28]<1.40398e+11){r+=0.955556;}else{r+=0.25;}}else{r+=0.25;}}else{if(f[14]<1.40516e+11){r+=0.6;}else{r+=-0.885714;}}}else{if(f[83]<1.79338){if(f[21]<1.11028e+08){if(f[70]<0.998789){r+=0.625;}else{if(f[75]<8.55138){r+=-0.723077;}else{r+=0.6;}}}else{if(f[82]<0.560859){r+=-0.5;}else{if(f[2]<5.34663e+07){r+=0.828571;}else{r+=-0;}}}}else{if(f[73]<147.054){if(f[69]<0.415942){if(f[14]<1.4047e+11){r+=-0.985148;}else{r+=-0.25;}}else{if(f[6]<1.08994e+07){r+=-0.833333;}else{r+=0.4;}}}else{if(f[2]<5.98474e+06){r+=0.6;}else{r+=-0.555556;}}}}}}}}}else{if(f[70]<1.00935){if(f[83]<2.42455){if(f[23]<1.15224e+08){if(f[42]<0.998026){if(f[14]<1.41146e+11){if(f[28]<1.39948e+11){r+=-0.870968;}else{if(f[28]<1.40375e+11){if(f[15]<1.36641e+08){if(f[27]<7.25077e+07){r+=0.25;}else{r+=0.989796;}}else{r+=-0.727273;}}else{if(f[0]<1.40186e+11){if(f[52]<2.45818){r+=-0.291339;}else{r+=0.789474;}}else{if(f[28]<1.40399e+11){r+=-0.357143;}else{r+=0.895105;}}}}}else{if(f[35]<1.17533e+07){if(f[0]<1.40351e+11){r+=-0.971831;}else{r+=0.96748;}}else{if(f[29]<7.08831e+06){if(f[12]<1.02186e+08){r+=-0.75;}else{r+=0.25;}}else{r+=-0.992857;}}}}else{if(f[14]<1.40147e+11){if(f[28]<1.39575e+11){if(f[28]<1.39457e+11){r+=-0.2;}else{if(f[14]<1.39773e+11){r+=0.2;}else{r+=0.891892;}}}else{if(f[27]<6.33964e+08){if(f[55]<0.904108){if(f[36]<9.69643e+07){r+=-0.846154;}else{r+=0.769231;}}else{if(f[12]<1.63921e+10){r+=-0.967076;}else{r+=0.5;}}}else{if(f[40]<6.25416e+08){r+=0.833333;}else{if(f[56]<1.00102){r+=-0.74359;}else{r+=0.5;}}}}}else{if(f[42]<1.00127){if(f[14]<1.40235e+11){if(f[56]<1.00121){if(f[52]<0.944405){r+=0.2;}else{r+=-0.938776;}}else{if(f[40]<4.95966e+08){r+=0.774295;}else{r+=-0.893805;}}}else{if(f[0]<1.40163e+11){if(f[6]<1.7927e+07){r+=-0.538462;}else{r+=0.5625;}}else{if(f[35]<4.13463e+06){r+=-0.223301;}else{r+=-0.959214;}}}}else{if(f[56]<1.00344){if(f[14]<1.40517e+11){if(f[40]<4.96882e+08){r+=0.844113;}else{r+=-0.636364;}}else{if(f[1]<9.46429e+07){r+=-0.949367;}else{r+=-0.25;}}}else{if(f[23]<1.00837e+08){r+=-0.989583;}else{r+=0.538462;}}}}}}else{if(f[14]<1.40618e+11){if(f[14]<1.40183e+11){if(f[14]<1.39122e+11){if(f[9]<2.34253e+08){if(f[55]<0.966696){if(f[14]<1.38858e+11){r+=-0.789474;}else{r+=0.858586;}}else{if(f[0]<1.42294e+11){r+=-0.967347;}else{r+=0.806452;}}}else{if(f[28]<1.40073e+11){if(f[81]<2.27109){r+=0.943147;}else{r+=-0.73913;}}else{if(f[27]<2.57413e+08){r+=-0.747826;}else{r+=0.568627;}}}}else{if(f[28]<1.39576e+11){if(f[14]<1.39741e+11){if(f[4]<7.98135e+07){r+=-0.957447;}else{r+=-0;}}else{if(f[28]<1.39458e+11){r+=0.222222;}else{r+=0.904535;}}}else{if(f[70]<0.999607){if(f[9]<2.99642e+08){r+=-0.940843;}else{r+=-0.500888;}}else{if(f[14]<1.39738e+11){r+=0.77931;}else{r+=-0.333333;}}}}}else{if(f[28]<1.39822e+11){if(f[0]<1.41536e+11){if(f[56]<1.0028){r+=0.6;}else{if(f[6]<2.04926e+08){r+=-0.986755;}else{r+=-0.25;}}}else{r+=0.851852;}}else{if(f[70]<1.00232){if(f[41]<5.07357e+08){if(f[39]<1.64689e+08){r+=0.361607;}else{r+=0.901663;}}else{if(f[0]<1.40609e+11){r+=-0.09699;}else{r+=0.583774;}}}else{if(f[42]<1.00108){r+=-0.976331;}else{r+=0.4;}}}}}else{if(f[0]<1.42568e+11){if(f[52]<0.692069){if(f[27]<1.70781e+08){r+=0.8;}else{if(f[12]<5.32884e+08){r+=-0.789474;}else{r+=0.111111;}}}else{if(f[12]<1.02688e+08){if(f[0]<1.41132e+11){if(f[0]<1.40205e+11){r+=-0.111111;}else{r+=-0.982379;}}else{if(f[42]<1.0008){r+=0.928571;}else{r+=-0.942857;}}}else{if(f[0]<1.40208e+11){if(f[57]<5.52819){r+=-0.538462;}else{r+=0.555556;}}else{if(f[69]<0.158265){r+=-0;}else{r+=-0.989242;}}}}}else{if(f[70]<0.980709){if(f[76]<0.41858){r+=-0;}else{r+=-0.906977;}}else{if(f[12]<6.51467e+09){if(f[55]<1.21258){r+=0.25;}else{r+=0.977012;}}else{if(f[4]<9.54201e+08){r+=-0;}else{r+=0.5;}}}}}}}else{if(f[12]<1.99794e+08){if(f[0]<1.40262e+11){if(f[42]<1.00161){if(f[14]<1.40446e+11){if(f[52]<1.06124){if(f[39]<2.62068e+08){if(f[56]<1.00037){r+=-0.884615;}else{r+=0.702128;}}else{if(f[41]<1.46935e+09){r+=-0.921005;}else{r+=-0.0473684;}}}else{if(f[49]<0.6725){if(f[67]<0.457441){r+=-0.705882;}else{r+=0.272727;}}else{if(f[41]<2.57127e+09){r+=-0.970194;}else{r+=-0.578947;}}}}else{if(f[14]<1.40611e+11){if(f[28]<1.40376e+11){r+=0.974684;}else{if(f[0]<1.40182e+11){r+=-0.742574;}else{r+=0.540373;}}}else{if(f[28]<1.40403e+11){if(f[0]<1.40143e+11){r+=0.4;}else{r+=-0.873874;}}else{if(f[14]<1.40623e+11){r+=0.448819;}else{r+=-0.408163;}}}}}else{if(f[56]<0.998706){if(f[28]<1.40159e+11){if(f[30]<5.2332e+07){if(f[9]<3.62826e+08){r+=-0.979239;}else{r+=-0.25;}}else{r+=-0;}}else{if(f[25]<6.13181e+07){r+=-0.809524;}else{if(f[2]<5.81692e+07){r+=0.935484;}else{r+=0.25;}}}}else{if(f[14]<1.4001e+11){if(f[28]<1.39462e+11){r+=-0.625;}else{if(f[29]<9.5396e+07){r+=0.98224;}else{r+=-0.5;}}}else{if(f[70]<0.999303){if(f[15]<1.16753e+07){r+=-0.954545;}else{r+=0.2;}}else{r+=0.789474;}}}}}else{if(f[14]<1.40188e+11){if(f[14]<1.40158e+11){if(f[9]<2.61122e+08){if(f[28]<1.39556e+11){if(f[4]<1.07294e+07){r+=0.0769231;}else{r+=0.918919;}}else{if(f[25]<1.22836e+09){r+=-0.972573;}else{r+=0.555556;}}}else{if(f[64]<1.97712){if(f[28]<1.40058e+11){r+=0.772727;}else{r+=-0.265306;}}else{if(f[28]<1.39588e+11){r+=0.555556;}else{r+=-0.936709;}}}}else{if(f[81]<2.59836){if(f[0]<1.51501e+11){if(f[39]<2.66519e+08){r+=0.984556;}else{r+=-0.481481;}}else{r+=-0.916667;}}else{if(f[41]<1.45368e+09){if(f[39]<3.10943e+08){r+=-0.0795455;}else{r+=-0.756592;}}else{if(f[28]<1.40002e+11){r+=0.732394;}else{r+=-0.822222;}}}}}else{if(f[67]<1.26457){if(f[41]<3.55e+08){if(f[39]<3.44889e+08){if(f[0]<1.41165e+11){r+=-0.957055;}else{r+=-0;}}else{if(f[11]<1.21627e+08){r+=0.878788;}else{r+=-0.5;}}}else{if(f[40]<1.4741e+08){if(f[35]<1.2369e+08){r+=-0.647059;}else{r+=0.666667;}}else{if(f[17]<1.87101e+08){r+=-0.987103;}else{r+=-0.0909091;}}}}else{if(f[28]<1.40572e+11){r+=-0.862069;}else{if(f[28]<1.4074e+11){r+=0.851852;}else{r+=-0.692308;}}}}}}else{if(f[28]<1.43587e+11){if(f[36]<8.65194e+07){if(f[23]<5.78157e+07){if(f[42]<0.999319){r+=0.666667;}else{if(f[70]<0.999334){if(f[54]<0.415643){r+=-0.384615;}else{r+=-0.991118;}}else{if(f[21]<2.02206e+07){r+=0.6;}else{r+=-0.555556;}}}}else{if(f[0]<1.40261e+11){if(f[37]<7.98068e+07){if(f[0]<1.40143e+11){r+=-0.5;}else{r+=0.864865;}}else{if(f[81]<4.20409){r+=-0.72973;}else{r+=0.0819672;}}}else{if(f[68]<0.151513){if(f[14]<1.40125e+11){r+=-0.75;}else{r+=0.48;}}else{if(f[28]<1.39994e+11){r+=-0.581818;}else{r+=-0.975904;}}}}}else{if(f[14]<1.38992e+11){if(f[58]<1.12498){r+=0.684211;}else{r+=-0.538462;}}else{if(f[49]<0.227367){if(f[0]<1.4024e+11){r+=0.571429;}else{r+=-0.777778;}}else{if(f[37]<1.10869e+08){if(f[14]<1.40002e+11){r+=-0.906977;}else{r+=-0.0322581;}}else{if(f[13]<4.74078e+08){r+=-0.941722;}else{r+=-0.997569;}}}}}}else{if(f[15]<1.17388e+09){r+=0.956522;}else{r+=0.2;}}}}}else{if(f[13]<3.32563e+08){if(f[1]<9.13941e+06){if(f[28]<1.7759e+11){r+=-0.875;}else{r+=0.983122;}}else{if(f[83]<1.52894){if(f[13]<2.39493e+08){r+=-0.727273;}else{if(f[0]<1.40249e+11){r+=0.25;}else{r+=0.913043;}}}else{if(f[1]<1.66615e+07){if(f[27]<2.02476e+08){r+=-0.833333;}else{if(f[0]<1.40246e+11){r+=-0;}else{r+=0.777778;}}}else{if(f[70]<1.01433){if(f[9]<1.3163e+08){r+=0.666667;}else{r+=-0.8;}}else{if(f[28]<1.40174e+11){r+=-0;}else{if(f[23]<5.15677e+08){r+=-0.992467;}else{if(f[0]<1.40875e+11){r+=-0;}else{r+=-0.6;}}}}}}}}else{if(f[67]<0.204213){if(f[28]<1.44662e+11){if(f[37]<2.05904e+08){if(f[4]<6.72233e+07){r+=-0;}else{r+=0.636364;}}else{if(f[2]<2.66027e+08){if(f[52]<1.09232){if(f[0]<1.42217e+11){r+=-0.555556;}else{r+=-0;}}else{if(f[9]<2.08331e+09){r+=-0.982906;}else{r+=-0.25;}}}else{r+=-0;}}}else{if(f[69]<0.124163){if(f[0]<1.40229e+11){r+=-0.882353;}else{if(f[14]<1.43757e+11){if(f[28]<1.44716e+11){r+=0.25;}else{r+=0.958621;}}else{r+=-0.111111;}}}else{if(f[14]<1.40171e+11){if(f[0]<1.4026e+11){r+=0.666667;}else{r+=-0;}}else{if(f[66]<0.366996){r+=-0.929825;}else{if(f[6]<1.41691e+08){r+=0.25;}else{r+=-0.5;}}}}}}else{if(f[28]<1.45265e+11){if(f[14]<1.40752e+11){if(f[0]<1.40235e+11){r+=-0.911111;}else{r+=0.25;}}else{if(f[6]<4.55152e+08){if(f[7]<3.98972e+07){if(f[56]<0.986608){r+=-0.833333;}else{if(f[4]<2.84837e+06){r+=0.25;}else{r+=0.888889;}}}else{if(f[41]<2.7247e+10){if(f[67]<0.243069){r+=0.66537;}else{r+=0.952762;}}else{r+=-0.6;}}}else{if(f[4]<2.64055e+08){if(f[48]<1.46835){if(f[5]<3.16713e+08){r+=-0.884615;}else{r+=0.2;}}else{if(f[83]<2.60911){r+=-0.4;}else{r+=0.764706;}}}else{if(f[6]<1.04065e+09){if(f[32]<3.00968e+07){r+=0.25;}else{r+=0.942857;}}else{if(f[5]<6.73468e+08){r+=-0.684211;}else{r+=0.5;}}}}}}else{if(f[0]<1.40309e+11){if(f[0]<1.40236e+11){r+=-0;}else{r+=0.978723;}}else{if(f[40]<4.60886e+09){if(f[23]<2.44087e+09){if(f[54]<0.730165){r+=-0;}else{if(f[2]<4.52279e+08){r+=-0.97561;}else{r+=-0.25;}}}else{if(f[28]<1.45463e+11){r+=0.764706;}else{r+=-0.666667;}}}else{if(f[28]<1.45612e+11){r+=0.891892;}else{if(f[9]<3.28771e+09){r+=-0.5;}else{r+=0.25;}}}}}}}}}}else{if(f[28]<1.40827e+11){if(f[14]<1.3986e+11){if(f[14]<1.39536e+11){if(f[28]<1.40057e+11){if(f[27]<1.1619e+08){if(f[28]<1.39814e+11){r+=0.978142;}else{if(f[28]<1.39962e+11){if(f[55]<1.02521){if(f[0]<1.40821e+11){r+=0.777778;}else{r+=-0;}}else{if(f[4]<5.03255e+07){r+=-0.878788;}else{r+=-0.25;}}}else{if(f[27]<1.16014e+08){r+=-0.993056;}else{r+=-0.25;}}}}else{if(f[37]<7.23345e+07){if(f[13]<8.27443e+07){r+=-0.25;}else{r+=-0.990244;}}else{if(f[28]<1.39688e+11){if(f[83]<0.924029){if(f[69]<0.852971){if(f[0]<1.42422e+11){r+=-0.636364;}else{r+=0.636364;}}else{if(f[9]<3.75795e+10){r+=0.990033;}else{r+=-0.25;}}}else{if(f[0]<1.40174e+11){r+=0.76;}else{if(f[0]<1.42145e+11){r+=-0.994318;}else{r+=0.25;}}}}else{if(f[54]<0.155277){if(f[15]<4.86386e+07){r+=0.25;}else{r+=-0.983193;}}else{if(f[9]<2.09981e+08){if(f[37]<1.8018e+08){r+=0.910551;}else{r+=-0.20307;}}else{if(f[39]<2.55059e+10){r+=0.961823;}else{r+=-0.6;}}}}}}}else{if(f[0]<1.40207e+11){if(f[56]<0.993068){if(f[1]<1.84153e+09){if(f[13]<1.07166e+08){r+=0.4;}else{if(f[13]<1.26939e+08){if(f[14]<1.3914e+11){r+=-0.878788;}else{r+=0.555556;}}else{if(f[9]<2.88021e+07){r+=-0.25;}else{r+=-0.996635;}}}}else{if(f[1]<4.99517e+09){r+=0.5;}else{r+=-0;}}}else{if(f[28]<1.40147e+11){if(f[66]<0.723137){if(f[40]<1.8849e+08){if(f[9]<1.99599e+08){r+=0.925926;}else{r+=-0.25;}}else{r+=-0.76;}}else{if(f[64]<0.706397){if(f[0]<1.40166e+11){r+=-0;}else{r+=-0.5;}}else{if(f[28]<1.40144e+11){r+=-0.971014;}else{r+=-0.384615;}}}}else{if(f[13]<1.30332e+09){if(f[28]<1.40154e+11){if(f[9]<1.81289e+08){r+=0.62963;}else{r+=-0.5625;}}else{if(f[41]<1.61487e+09){r+=0.984657;}else{r+=-0.272727;}}}else{if(f[8]<3.96056e+08){r+=-0.9;}else{if(f[6]<2.50624e+08){r+=0.666667;}else{r+=-0.272727;}}}}}}else{if(f[14]<1.38912e+11){if(f[28]<1.40114e+11){if(f[27]<1.9589e+08){if(f[55]<0.89805){if(f[0]<1.40546e+11){r+=-0.666667;}else{r+=0.363636;}}else{if(f[15]<1.50098e+07){r+=-0;}else{r+=-0.928571;}}}else{if(f[0]<1.40561e+11){if(f[13]<2.06091e+08){r+=-0.138686;}else{r+=0.795455;}}else{if(f[18]<9.1999e+07){r+=0.890084;}else{r+=-0.333333;}}}}else{if(f[14]<1.28922e+11){if(f[0]<1.4508e+11){r+=0.25;}else{r+=0.921569;}}else{if(f[28]<1.40123e+11){if(f[13]<2.30778e+08){r+=-0.789474;}else{r+=0.517241;}}else{if(f[22]<4.91661e+10){r+=-0.982822;}else{r+=0.555556;}}}}}else{if(f[28]<1.40715e+11){if(f[0]<1.40226e+11){if(f[28]<1.40148e+11){if(f[17]<1.00737e+08){r+=-0.969925;}else{r+=-0.25;}}else{if(f[13]<1.85586e+08){r+=0.793103;}else{r+=-0.898734;}}}else{if(f[1]<1.39891e+09){if(f[42]<1.00583){r+=-0.2;}else{r+=-0.996667;}}else{if(f[23]<2.39322e+10){r+=0.692308;}else{r+=-0.25;}}}}else{if(f[42]<1.02998){r+=0.984127;}else{if(f[0]<1.43658e+11){r+=-0.25;}else{r+=-0.987055;}}}}}}}else{if(f[25]<7.09566e+07){if(f[40]<7.52313e+07){if(f[14]<1.39664e+11){if(f[70]<1.00522){if(f[14]<1.39653e+11){if(f[0]<1.42642e+11){if(f[52]<0.853436){r+=-0.111111;}else{r+=-0.932432;}}else{r+=0.5;}}else{if(f[28]<1.40549e+11){if(f[69]<1.19527){r+=-0.882353;}else{r+=0.0909091;}}else{if(f[0]<1.40464e+11){r+=-0;}else{r+=0.833333;}}}}else{r+=0.924528;}}else{if(f[25]<6.2752e+07){r+=-0.906977;}else{if(f[0]<1.40171e+11){r+=-0.25;}else{if(f[39]<9.23833e+07){if(f[28]<1.3945e+11){r+=0.0769231;}else{r+=0.87395;}}else{r+=-0.25;}}}}}else{if(f[28]<1.40515e+11){if(f[28]<1.3964e+11){r+=0.684211;}else{if(f[52]<0.772189){r+=0.25;}else{if(f[32]<5.24447e+07){r+=-0.991304;}else{r+=-0;}}}}else{if(f[14]<1.3966e+11){r+=-0.684211;}else{if(f[30]<9.11812e+06){r+=-0.5;}else{if(f[16]<3.5322e+07){r+=0.904762;}else{r+=0.25;}}}}}}else{if(f[63]<0.319782){if(f[22]<8.15038e+07){r+=-0.989691;}else{if(f[14]<1.39733e+11){if(f[54]<1.1121){if(f[6]<7.80669e+06){r+=0.25;}else{r+=0.955556;}}else{if(f[58]<0.302032){r+=-0.555556;}else{r+=0.4;}}}else{if(f[46]<2.00497){r+=-0.636364;}else{r+=0.333333;}}}}else{if(f[36]<9.86031e+08){if(f[67]<1.00019){if(f[42]<1.00665){if(f[66]<0.34709){if(f[28]<1.3975e+11){r+=0.918367;}else{r+=-0.923304;}}else{if(f[25]<1.0749e+09){r+=0.843882;}else{r+=-0.747368;}}}else{if(f[0]<1.42932e+11){if(f[23]<4.07409e+07){r+=0.725806;}else{r+=-0.992266;}}else{if(f[70]<1.40178e+11){r+=0.859109;}else{r+=-0.677419;}}}}else{if(f[54]<1.71881){if(f[23]<1.21624e+08){if(f[42]<1.006){r+=0.915224;}else{r+=0.0352505;}}else{if(f[69]<0.793201){r+=-0.535714;}else{r+=0.977175;}}}else{if(f[0]<1.40582e+11){if(f[27]<9.21792e+07){r+=-0;}else{r+=0.932886;}}else{if(f[0]<1.43009e+11){r+=-0.968;}else{r+=0.904762;}}}}}else{if(f[0]<1.43153e+11){if(f[64]<0.434136){if(f[52]<1.04009){r+=0.818182;}else{r+=-0.25;}}else{if(f[28]<1.39676e+11){if(f[1]<9.95868e+07){r+=0.555556;}else{r+=-0.25;}}else{if(f[63]<3.7165){r+=-0.983193;}else{r+=-0.0909091;}}}}else{r+=0.909091;}}}}}}else{if(f[28]<1.3982e+11){if(f[42]<1.00903){if(f[14]<1.39888e+11){if(f[28]<1.39606e+11){if(f[0]<1.40167e+11){r+=-0;}else{if(f[9]<8.78754e+08){r+=0.9819;}else{r+=0.25;}}}else{r+=-0.851852;}}else{if(f[41]<1.83331e+08){if(f[6]<7.34562e+07){if(f[28]<1.39812e+11){if(f[29]<3.17742e+08){if(f[14]<1.39906e+11){r+=-0.411765;}else{r+=-0.995857;}}else{r+=-0;}}else{if(f[41]<1.52636e+08){if(f[69]<4.51314){r+=-0.973087;}else{r+=-0.285714;}}else{if(f[14]<1.40326e+11){r+=0.422222;}else{r+=-0.692308;}}}}else{if(f[28]<1.398e+11){if(f[56]<1.00287){r+=0.111111;}else{if(f[69]<6.12593){r+=-0.981567;}else{r+=-0.285714;}}}else{if(f[14]<1.4037e+11){if(f[0]<1.39917e+11){r+=-0.5;}else{r+=0.675676;}}else{r+=-0.789474;}}}}else{if(f[28]<1.398e+11){if(f[14]<1.40086e+11){if(f[0]<1.40027e+11){r+=-0.809524;}else{if(f[39]<3.68068e+09){r+=0.941176;}else{r+=-0.111111;}}}else{if(f[13]<3.24335e+10){if(f[56]<1.00281){r+=-0.0909091;}else{r+=-0.988649;}}else{if(f[11]<4.1414e+10){r+=-0.25;}else{r+=0.666667;}}}}else{if(f[14]<1.40318e+11){if(f[14]<1.40036e+11){r+=-0.925926;}else{if(f[14]<1.40306e+11){r+=0.961538;}else{r+=0.518519;}}}else{if(f[29]<1.10883e+08){if(f[14]<1.40323e+11){r+=-0.142857;}else{r+=-0.913876;}}else{if(f[1]<1.23182e+08){r+=0.6;}else{r+=-0;}}}}}}}else{if(f[42]<1.01104){if(f[40]<9.94844e+07){r+=0.25;}else{r+=0.991837;}}else{if(f[5]<5.53546e+06){r+=-0;}else{r+=-0.948718;}}}}else{if(f[14]<1.40058e+11){if(f[41]<1.05585e+08){if(f[83]<2.31073){if(f[41]<1.00286e+08){if(f[14]<1.40055e+11){if(f[49]<0.156508){r+=0.25;}else{if(f[73]<47.957){r+=-0.990382;}else{r+=-0;}}}else{if(f[36]<3.14106e+07){if(f[0]<1.40055e+11){r+=0.757009;}else{r+=-0.888889;}}else{if(f[13]<2.09622e+08){r+=-0.956522;}else{r+=-0.25;}}}}else{if(f[14]<1.4002e+11){if(f[70]<0.999749){if(f[28]<1.39939e+11){r+=-0.75;}else{r+=0.966805;}}else{if(f[21]<1.34113e+07){r+=0.25;}else{r+=-0.972789;}}}else{if(f[0]<1.39971e+11){if(f[9]<5.04157e+07){r+=0.692308;}else{r+=-0.833333;}}else{if(f[39]<4.78877e+07){r+=-0;}else{r+=-0.98252;}}}}}else{if(f[55]<0.46354){if(f[11]<6.97336e+06){r+=-0.75;}else{if(f[0]<1.40068e+11){if(f[22]<1.09541e+08){r+=0.981132;}else{r+=0.25;}}else{r+=0.25;}}}else{r+=-0.789474;}}}else{if(f[0]<1.40027e+11){if(f[13]<4.7675e+07){if(f[11]<7.23477e+06){r+=-0.866667;}else{if(f[13]<4.47103e+07){if(f[27]<7.85752e+07){r+=0.2;}else{r+=0.971564;}}else{if(f[0]<1.39996e+11){r+=-0.6;}else{r+=0.692308;}}}}else{if(f[81]<0.40331){if(f[14]<1.40051e+11){if(f[40]<1.34499e+09){r+=-0.971014;}else{r+=0.285714;}}else{if(f[9]<9.03539e+07){r+=0.981132;}else{r+=-0.789474;}}}else{if(f[77]<0.324057){if(f[28]<1.40029e+11){r+=-0.882353;}else{r+=0.318182;}}else{if(f[21]<9.27551e+06){r+=0.0526316;}else{r+=-0.979155;}}}}}else{if(f[28]<1.40015e+11){if(f[14]<1.39992e+11){if(f[41]<1.2467e+08){if(f[28]<1.39993e+11){r+=-0.5;}else{r+=0.976331;}}else{if(f[35]<4.54928e+08){r+=-0.99421;}else{r+=-0.142857;}}}else{if(f[69]<1.57382){if(f[14]<1.4002e+11){r+=-0.504418;}else{r+=0.528535;}}else{if(f[23]<2.81895e+07){r+=-0.577358;}else{r+=0.929467;}}}}else{if(f[70]<1.00168){if(f[31]<1.25119e+08){if(f[0]<1.40098e+11){r+=-0.524862;}else{r+=-0.962824;}}else{if(f[42]<1.0011){r+=0.941704;}else{r+=-0.44;}}}else{if(f[56]<0.996101){if(f[81]<1.5623){r+=-0.904762;}else{r+=0.25;}}else{if(f[0]<1.40096e+11){r+=0.998552;}else{r+=-0;}}}}}}}else{if(f[0]<1.39763e+11){if(f[28]<1.3997e+11){if(f[42]<0.992565){if(f[0]<1.39658e+11){if(f[0]<1.39329e+11){r+=-0.733333;}else{r+=0.998641;}}else{r+=-0.897436;}}else{if(f[69]<0.690818){r+=0.555556;}else{if(f[1]<1.39498e+08){r+=-0.984211;}else{if(f[7]<4.85861e+08){r+=0.454545;}else{r+=-0.692308;}}}}}else{if(f[0]<1.39741e+11){if(f[28]<1.3998e+11){if(f[14]<1.40528e+11){r+=-0.96875;}else{r+=0.913043;}}else{if(f[12]<3.39302e+10){if(f[6]<2.2993e+09){r+=-0.993618;}else{r+=-0.294118;}}else{r+=0.2;}}}else{if(f[11]<1.19856e+08){if(f[23]<2.20797e+07){if(f[29]<6.13942e+06){r+=-0.764706;}else{r+=0.166667;}}else{if(f[71]<0.972983){r+=0.79661;}else{r+=-0.555556;}}}else{if(f[15]<2.49372e+08){if(f[27]<2.32483e+08){r+=-0.142857;}else{r+=-0.940594;}}else{r+=0.555556;}}}}}else{if(f[0]<1.40247e+11){if(f[69]<1.86461){if(f[69]<1.07295){if(f[54]<0.527229){if(f[41]<9.93185e+07){r+=-0.85;}else{r+=0.910572;}}else{if(f[25]<6.75792e+07){r+=-0.171766;}else{r+=0.30019;}}}else{if(f[28]<1.40165e+11){if(f[41]<9.03834e+07){r+=0.350268;}else{r+=0.762836;}}else{if(f[14]<1.40452e+11){r+=-0.541226;}else{r+=0.506823;}}}}else{if(f[0]<1.40071e+11){if(f[28]<1.40622e+11){if(f[39]<8.08778e+06){r+=-0.834586;}else{r+=0.49214;}}else{if(f[36]<1.94464e+08){r+=-0.997692;}else{r+=0.384615;}}}else{if(f[37]<2.00592e+07){if(f[14]<1.40358e+11){r+=0.759618;}else{r+=-0.523511;}}else{if(f[28]<1.39935e+11){r+=0.522318;}else{r+=-0.80754;}}}}}else{if(f[14]<1.41172e+11){if(f[28]<1.40004e+11){if(f[28]<1.39954e+11){if(f[28]<1.39849e+11){r+=0.497475;}else{r+=-0.741909;}}else{if(f[41]<7.90228e+07){r+=-0.896216;}else{r+=0.613783;}}}else{if(f[0]<1.41007e+11){if(f[81]<5.68888){r+=-0.978424;}else{r+=0.980165;}}else{if(f[14]<1.40504e+11){r+=0.0541406;}else{r+=-0.827633;}}}}else{if(f[28]<1.40148e+11){if(f[35]<1.22714e+07){if(f[0]<1.40347e+11){r+=-0.988506;}else{r+=0.993148;}}else{if(f[28]<1.40132e+11){r+=-0.953417;}else{r+=0.45961;}}}else{if(f[14]<1.41505e+11){if(f[13]<1.21622e+09){r+=0.989022;}else{r+=-0.691542;}}else{if(f[52]<0.544281){r+=0.0952381;}else{r+=-0.981148;}}}}}}}}}}else{if(f[28]<1.41469e+11){if(f[14]<1.46452e+11){if(f[0]<1.40698e+11){if(f[14]<1.45814e+11){if(f[14]<1.45551e+11){if(f[23]<9.25258e+09){if(f[28]<1.41462e+11){if(f[0]<1.40692e+11){if(f[29]<5.50693e+08){r+=-0.999332;}else{r+=-0.384615;}}else{if(f[14]<1.40343e+11){r+=0.571429;}else{r+=-0.928571;}}}else{if(f[0]<1.39488e+11){r+=0.733333;}else{r+=-0.878788;}}}else{if(f[36]<1.70759e+09){r+=0.684211;}else{if(f[6]<4.5189e+07){r+=0.5;}else{if(f[29]<1.82808e+09){r+=-0.861386;}else{r+=-0;}}}}}else{if(f[13]<1.83785e+08){if(f[13]<1.80035e+08){r+=-0.975309;}else{r+=-0.25;}}else{if(f[28]<1.41247e+11){r+=-0.636364;}else{r+=0.935484;}}}}else{if(f[40]<6.8732e+07){if(f[28]<1.41264e+11){if(f[31]<6.83234e+06){r+=-0.977359;}else{r+=-0.25;}}else{if(f[0]<1.39502e+11){r+=-0.714286;}else{r+=0.625;}}}else{if(f[28]<1.41238e+11){r+=-0.875;}else{if(f[81]<0.583996){if(f[29]<1.03295e+07){if(f[41]<1.0288e+08){r+=-0;}else{r+=0.972973;}}else{r+=-0.25;}}else{if(f[0]<1.39513e+11){r+=-0.692308;}else{r+=0.25;}}}}}}else{if(f[0]<1.40808e+11){if(f[42]<1.00254){if(f[14]<1.40376e+11){r+=-0.111111;}else{if(f[27]<9.71134e+07){r+=-0.25;}else{r+=-0.991701;}}}else{if(f[70]<1.00283){if(f[8]<5.56486e+07){r+=-0.636364;}else{if(f[2]<2.57178e+08){if(f[28]<1.4084e+11){r+=-0;}else{r+=0.998328;}}else{r+=-0.25;}}}else{r+=-0.857143;}}}else{if(f[28]<1.40839e+11){if(f[0]<1.42848e+11){if(f[0]<1.40809e+11){r+=-0.25;}else{r+=-0.986159;}}else{if(f[14]<1.39477e+11){r+=-0.958333;}else{if(f[0]<1.43581e+11){r+=0.25;}else{r+=0.990588;}}}}else{if(f[0]<1.40822e+11){if(f[14]<1.40447e+11){if(f[11]<1.42427e+08){r+=0.978495;}else{r+=-0.6;}}else{if(f[7]<1.13042e+09){r+=-0.987382;}else{r+=-0.25;}}}else{if(f[0]<1.42818e+11){if(f[0]<1.4084e+11){if(f[14]<1.40496e+11){r+=0.633803;}else{r+=-0.985965;}}else{if(f[1]<3.02614e+09){r+=-0.998555;}else{r+=0.0526316;}}}else{if(f[14]<1.39456e+11){if(f[14]<1.38545e+11){r+=0.5;}else{r+=-0.994667;}}else{if(f[71]<0.319958){r+=-0.647059;}else{r+=0.941606;}}}}}}}}else{if(f[28]<1.41228e+11){r+=-0.764706;}else{if(f[0]<1.39638e+11){r+=0.99922;}else{r+=0.25;}}}}else{if(f[28]<1.42334e+11){if(f[9]<2.47308e+08){if(f[56]<0.98202){r+=-0.996561;}else{if(f[40]<1.91573e+08){if(f[22]<5.67721e+07){r+=-0.979058;}else{if(f[27]<7.23944e+07){if(f[0]<1.39322e+11){r+=0.909091;}else{if(f[0]<1.40257e+11){r+=-0.963303;}else{r+=0.789474;}}}else{if(f[52]<1.8521){if(f[68]<0.504877){r+=-0.140351;}else{r+=0.932798;}}else{if(f[28]<1.41837e+11){r+=-0.895833;}else{r+=0.773585;}}}}}else{if(f[0]<1.39332e+11){if(f[0]<1.39048e+11){r+=0.25;}else{r+=0.957447;}}else{if(f[28]<1.42038e+11){r+=-0.990909;}else{if(f[67]<0.700381){r+=-0.2;}else{r+=0.733333;}}}}}}else{if(f[0]<1.39358e+11){if(f[0]<1.3898e+11){if(f[0]<1.38898e+11){r+=-0.75;}else{r+=-0;}}else{if(f[40]<9.37062e+08){if(f[0]<1.39339e+11){if(f[14]<1.40843e+11){r+=0.989529;}else{r+=0.25;}}else{r+=0.272727;}}else{r+=-0.25;}}}else{if(f[28]<1.4203e+11){if(f[1]<2.59931e+09){r+=-0.997859;}else{if(f[0]<1.42001e+11){r+=-0;}else{r+=-0.5;}}}else{if(f[36]<1.78497e+09){if(f[0]<1.40869e+11){r+=0.823529;}else{r+=-0.333333;}}else{r+=-0.714286;}}}}}else{if(f[27]<1.92433e+08){if(f[56]<0.77364){if(f[42]<1.00093){r+=-0.996113;}else{r+=0.992157;}}else{if(f[41]<2.16415e+08){r+=-0.999107;}else{if(f[28]<1.42715e+11){if(f[0]<1.40621e+11){r+=0.966387;}else{r+=-0.636364;}}else{r+=-0.947368;}}}}else{if(f[0]<1.40235e+11){if(f[70]<1.01722){if(f[0]<1.40089e+11){r+=-0.733333;}else{r+=0.851852;}}else{if(f[56]<0.742127){if(f[0]<1.4013e+11){r+=-0.714286;}else{r+=0.5;}}else{if(f[29]<1.18969e+10){if(f[28]<1.42469e+11){r+=-0.25;}else{r+=-0.995365;}}else{r+=-0.111111;}}}}else{if(f[42]<1.00598){if(f[30]<256284){if(f[1]<6.62915e+06){r+=-0.25;}else{r+=-0.961165;}}else{if(f[6]<1.62129e+09){if(f[28]<2.27541e+11){if(f[25]<3.74016e+10){r+=0.985408;}else{r+=-0.25;}}else{r+=-0.666667;}}else{r+=-0.571429;}}}else{if(f[14]<1.41946e+11){if(f[5]<3.03228e+09){r+=-0.986928;}else{r+=-0.25;}}else{if(f[0]<1.44873e+11){r+=0.6;}else{r+=-0.25;}}}}}}}}}} return r;}
    static double p7(double[] f){double r=0;if(f[28]<1.39443e+11){if(f[28]<1.3942e+11){if(f[10]<5.33391e+10){if(f[57]<0.543661){if(f[42]<1.0093){if(f[1]<1.41473e+09){r+=-0.683347;}else{r+=-0.0426156;}}else{if(f[42]<1.01073){if(f[27]<2.26472e+08){if(f[57]<0.375163){r+=0.607451;}else{if(f[1]<4.21877e+07){r+=0.407747;}else{r+=1.29386;}}}else{r+=-0.487663;}}else{if(f[14]<1.32618e+11){r+=0.153932;}else{r+=-0.672573;}}}}else{if(f[14]<1.38427e+11){if(f[14]<1.38424e+11){if(f[82]<1.01481){r+=-0.678189;}else{if(f[82]<1.01603){r+=0.307253;}else{r+=-0.588758;}}}else{r+=0.495967;}}else{if(f[20]<4.61237e+09){if(f[72]<44.9728){r+=-0.68277;}else{r+=-0.258359;}}else{r+=0.032606;}}}}else{if(f[77]<1.36065){if(f[51]<0.913367){if(f[28]<1.3899e+11){r+=-0.453244;}else{r+=-0.0296361;}}else{r+=0.569579;}}else{r+=-0.645738;}}}else{if(f[84]<1.15){if(f[28]<1.39435e+11){r+=-0.709216;}else{if(f[14]<1.39996e+11){r+=-1.45733;}else{r+=-0.652321;}}}else{if(f[14]<1.39883e+11){if(f[42]<1.00178){if(f[3]<1.34729e+07){r+=-0.0111791;}else{r+=-0.851956;}}else{if(f[14]<1.39592e+11){r+=-0.124851;}else{if(f[28]<1.39428e+11){if(f[14]<1.39782e+11){r+=0.407511;}else{r+=1.49335;}}else{if(f[41]<8.66965e+07){r+=0.0975456;}else{r+=0.737773;}}}}}else{if(f[51]<1.59352){r+=-0.673275;}else{if(f[54]<1.65706){r+=0.0820888;}else{r+=0.621845;}}}}}}else{if(f[70]<0.959368){if(f[1]<7.92723e+09){if(f[42]<1.48165){if(f[14]<1.40498e+11){if(f[14]<1.40099e+11){r+=-0.681051;}else{if(f[84]<2.15){if(f[69]<0.537704){r+=-0.819568;}else{if(f[28]<1.3982e+11){r+=-0.582315;}else{r+=-1.01894;}}}else{if(f[28]<1.40004e+11){r+=-1.39029;}else{r+=-1.01518;}}}}else{r+=-0.706113;}}else{if(f[42]<1.87963){if(f[28]<1.39957e+11){if(f[69]<0.537305){r+=-0.68797;}else{if(f[72]<0.0220687){r+=-0.0806727;}else{r+=-0.737814;}}}else{r+=-0.974899;}}else{r+=0.593415;}}}else{if(f[84]<2.15){if(f[69]<3.30944){r+=-0.681153;}else{r+=0.0305638;}}else{if(f[25]<1.08504e+08){r+=-0.0754998;}else{if(f[67]<1.0012){r+=0.717206;}else{r+=0.919332;}}}}}else{if(f[41]<8.58967e+09){if(f[28]<1.40413e+11){if(f[84]<0.5){if(f[14]<1.40013e+11){if(f[28]<1.39832e+11){if(f[42]<1.00207){r+=-0.706194;}else{if(f[14]<1.38951e+11){r+=-0.58704;}else{if(f[37]<1.8042e+08){if(f[69]<0.536597){if(f[42]<1.00407){r+=0.6956;}else{r+=1.14817;}}else{r+=0.698019;}}else{if(f[9]<2.10425e+08){r+=1.04489;}else{r+=0.702249;}}}}}else{if(f[14]<1.40001e+11){if(f[28]<1.39843e+11){if(f[77]<1.33075){if(f[13]<3.06595e+08){r+=0.11961;}else{r+=0.657077;}}else{r+=-0.661341;}}else{if(f[14]<1.39999e+11){if(f[77]<0.118929){if(f[70]<0.999098){r+=-0.523697;}else{r+=0.442824;}}else{if(f[28]<1.39847e+11){r+=-0.382027;}else{r+=-0.686586;}}}else{if(f[70]<0.999177){r+=-0.675101;}else{if(f[31]<8.70105e+06){r+=0.714584;}else{r+=-0.248787;}}}}}else{if(f[28]<1.40146e+11){if(f[12]<8.78498e+07){r+=0.209845;}else{if(f[28]<1.40146e+11){if(f[66]<0.291388){r+=-0.125255;}else{r+=-0.685344;}}else{r+=-0.131436;}}}else{if(f[70]<0.99916){r+=-0.578928;}else{if(f[69]<0.496605){if(f[42]<1.00149){r+=-0.468162;}else{r+=0.713421;}}else{r+=1.59155;}}}}}}else{if(f[70]<0.998518){if(f[70]<0.991517){if(f[14]<1.40058e+11){r+=-0.685015;}else{r+=-1.01383;}}else{if(f[28]<1.39822e+11){r+=-0.684555;}else{if(f[28]<1.40004e+11){if(f[69]<0.521602){if(f[41]<1.45331e+09){r+=-0.72039;}else{r+=-0.959685;}}else{r+=-1.30344;}}else{r+=-0.716162;}}}}else{if(f[69]<1.07332){if(f[14]<1.40668e+11){if(f[14]<1.40195e+11){if(f[63]<4.06795){if(f[14]<1.40017e+11){r+=-0.192913;}else{r+=-0.747675;}}else{r+=0.3896;}}else{if(f[25]<6.75815e+07){r+=-0.925541;}else{if(f[69]<0.536374){r+=-0.79753;}else{r+=-1.16656;}}}}else{r+=-0.683277;}}else{if(f[14]<1.40058e+11){r+=-0.681122;}else{if(f[69]<1.86417){if(f[41]<9.04777e+07){if(f[70]<1.00227){r+=-1.1991;}else{r+=-0.632608;}}else{r+=-1.53021;}}else{if(f[42]<0.999604){if(f[72]<0.0210322){r+=-0.675227;}else{r+=-1.35578;}}else{r+=-0.718091;}}}}}}}else{if(f[41]<9.10899e+07){if(f[14]<1.40132e+11){if(f[14]<1.39865e+11){if(f[42]<1.00673){if(f[84]<1.15){if(f[28]<1.40058e+11){r+=-1.37738;}else{r+=-0.657846;}}else{if(f[52]<1.37857){if(f[40]<6.28186e+07){r+=-0.191245;}else{r+=0.62401;}}else{if(f[28]<1.40167e+11){r+=-0.526056;}else{r+=0.496257;}}}}else{if(f[33]<5.28277e+07){r+=-0.71047;}else{r+=-0.141556;}}}else{if(f[28]<1.40063e+11){if(f[28]<1.40049e+11){if(f[70]<0.994969){r+=0.384144;}else{if(f[81]<3.63052){r+=-0.695577;}else{r+=0.144112;}}}else{if(f[84]<1.5){r+=-0.582178;}else{if(f[1]<2.71951e+07){r+=0.840516;}else{r+=-0.0317649;}}}}else{if(f[84]<1.15){if(f[14]<1.40055e+11){r+=-0.685886;}else{if(f[27]<1.6301e+08){r+=-1.08003;}else{r+=-0.734784;}}}else{if(f[69]<1.50346){if(f[84]<1.85){r+=0.288804;}else{r+=-0.928242;}}else{if(f[42]<1.00043){r+=-0.806869;}else{r+=-0.669173;}}}}}}else{if(f[42]<0.99175){if(f[14]<1.41513e+11){if(f[28]<1.40256e+11){r+=-0.68503;}else{r+=-0.111953;}}else{if(f[9]<2.22777e+07){if(f[26]<5.64391e+07){r+=0.662848;}else{r+=3.46435e-05;}}else{if(f[66]<1.01015){if(f[1]<1.85383e+07){r+=0.690544;}else{r+=-0.419313;}}else{if(f[1]<1.05011e+08){r+=-0.672096;}else{r+=0.109891;}}}}}else{if(f[28]<1.4002e+11){if(f[84]<2.15){if(f[14]<1.40307e+11){if(f[20]<2.54705e+07){r+=-0.751325;}else{r+=0.0629673;}}else{if(f[36]<5.76687e+07){r+=0.573492;}else{r+=-0.670127;}}}else{if(f[14]<1.40149e+11){if(f[84]<2.8){r+=-0.992926;}else{r+=0.78595;}}else{if(f[42]<0.998332){r+=-1.19575;}else{r+=-0.587089;}}}}else{if(f[84]<2.5){if(f[84]<1.5){if(f[70]<1.00063){r+=0.547197;}else{r+=-0.951892;}}else{if(f[28]<1.40146e+11){r+=0.823881;}else{r+=-0.740756;}}}else{if(f[40]<6.37465e+07){if(f[28]<1.40148e+11){r+=-0.772249;}else{r+=-1.08322;}}else{if(f[28]<1.40144e+11){r+=0.694365;}else{r+=-0.923221;}}}}}}}else{if(f[69]<1.11378){if(f[84]<2.85){if(f[37]<1.05132e+09){if(f[84]<2.5){if(f[31]<2.74544e+08){if(f[84]<0.85){r+=-0.464374;}else{r+=0.0453129;}}else{if(f[1]<7.77326e+07){r+=0.690916;}else{r+=-0.426358;}}}else{if(f[42]<0.999907){if(f[14]<1.40108e+11){r+=0.890023;}else{r+=-0.904991;}}else{if(f[69]<0.53785){r+=-0.698453;}else{r+=-0.886726;}}}}else{if(f[77]<1.59689){if(f[23]<4.34844e+09){if(f[84]<1.5){r+=-0.801492;}else{r+=0.00922495;}}else{if(f[3]<1.01967e+08){r+=0.459323;}else{r+=-0.208964;}}}else{if(f[26]<6.38856e+08){if(f[59]<0.00542843){r+=0.65209;}else{r+=-0.661729;}}else{if(f[14]<1.40058e+11){r+=-0.574309;}else{r+=-1.15608;}}}}}else{if(f[9]<2.54449e+08){if(f[14]<1.40021e+11){if(f[14]<1.39433e+11){if(f[13]<1.70635e+08){r+=0.400275;}else{r+=-0.387469;}}else{if(f[14]<1.4002e+11){r+=-0.712774;}else{r+=-0.237577;}}}else{if(f[14]<1.40827e+11){if(f[54]<1.31961){r+=0.415431;}else{r+=-0.0598455;}}else{if(f[37]<3.59084e+08){r+=-0.669615;}else{r+=0.477653;}}}}else{if(f[42]<0.996668){if(f[28]<1.3998e+11){if(f[28]<1.39875e+11){r+=-0.647271;}else{r+=0.602095;}}else{if(f[52]<0.607723){r+=-0.0836944;}else{r+=-0.730781;}}}else{if(f[28]<1.39804e+11){if(f[14]<1.39873e+11){r+=0.704262;}else{r+=-0.66634;}}else{if(f[21]<5.2919e+08){r+=0.645108;}else{r+=0.0365599;}}}}}}else{if(f[84]<3.5){if(f[28]<1.40143e+11){if(f[55]<0.878638){if(f[14]<1.40094e+11){if(f[28]<1.39953e+11){r+=0.549066;}else{r+=-0.0203993;}}else{if(f[25]<8.49956e+07){r+=0.474629;}else{r+=-0.399587;}}}else{if(f[14]<1.40055e+11){if(f[14]<1.40025e+11){r+=0.336264;}else{r+=-0.687804;}}else{if(f[28]<1.39819e+11){r+=-0.46707;}else{r+=0.542168;}}}}else{if(f[54]<0.337515){if(f[70]<1.00053){if(f[70]<0.993595){r+=0.572881;}else{r+=-0.74114;}}else{if(f[14]<1.40598e+11){r+=0.695647;}else{r+=-0.928608;}}}else{if(f[14]<1.40512e+11){if(f[84]<2.85){r+=-0.531941;}else{r+=0.245736;}}else{if(f[84]<1.15){r+=0.6237;}else{r+=-0.494365;}}}}}else{if(f[14]<1.40178e+11){if(f[14]<1.40034e+11){if(f[38]<1.05024e+08){if(f[70]<0.999086){r+=0.816482;}else{r+=-0.691945;}}else{if(f[66]<0.806419){r+=-0.0885641;}else{r+=-0.621127;}}}else{if(f[70]<1.00129){if(f[14]<1.40058e+11){r+=-0.692688;}else{r+=-1.25812;}}else{if(f[18]<5.51165e+07){r+=1.13184;}else{r+=-0.243256;}}}}else{if(f[28]<1.4015e+11){if(f[66]<0.629244){r+=0.730882;}else{if(f[70]<0.998919){r+=-0.700212;}else{r+=-0.213549;}}}else{if(f[14]<1.40695e+11){if(f[70]<0.999094){r+=-0.440922;}else{r+=0.74535;}}else{r+=-0.680845;}}}}}}}}else{if(f[13]<1.40889e+08){if(f[9]<2.32431e+08){if(f[14]<1.40835e+11){if(f[14]<1.39523e+11){if(f[29]<1.15801e+08){if(f[14]<1.39513e+11){r+=-0.684343;}else{r+=-0.20852;}}else{r+=-0.141037;}}else{if(f[38]<5.91745e+07){if(f[28]<1.40561e+11){if(f[14]<1.40085e+11){if(f[55]<0.616683){r+=0.809769;}else{r+=-0.520623;}}else{if(f[84]<6.60001){r+=-0.853071;}else{r+=0.822392;}}}else{if(f[63]<4.89324){if(f[14]<1.40507e+11){r+=0.729283;}else{r+=0.0144179;}}else{if(f[28]<1.64429e+11){r+=-0.661412;}else{r+=0.604773;}}}}else{if(f[84]<3.15){if(f[84]<1.15){if(f[42]<1.01824){r+=-0.389817;}else{r+=0.696242;}}else{if(f[69]<0.470622){r+=-0.34918;}else{r+=0.536559;}}}else{if(f[42]<0.998505){if(f[82]<1.46634){r+=-1.54583;}else{r+=-0.746068;}}else{r+=-0.70466;}}}}}else{if(f[70]<0.997179){if(f[14]<1.41172e+11){r+=1.4663;}else{if(f[69]<0.425263){r+=1.5511;}else{r+=0.687944;}}}else{if(f[14]<1.4699e+11){if(f[14]<1.40861e+11){if(f[42]<0.996175){if(f[28]<1.4141e+11){r+=-0.617283;}else{r+=0.675523;}}else{r+=-0.678126;}}else{if(f[14]<1.46501e+11){if(f[14]<1.40869e+11){r+=-0.53432;}else{r+=-0.684124;}}else{if(f[13]<1.13208e+08){r+=-0.681675;}else{r+=0.627144;}}}}else{if(f[7]<6.77899e+07){r+=-0.675685;}else{r+=0.680282;}}}}}else{if(f[84]<0.5){if(f[42]<0.995628){if(f[29]<3.33158e+07){if(f[14]<1.46233e+11){r+=0.805255;}else{r+=0.677795;}}else{r+=-0.437113;}}else{r+=-0.822987;}}else{if(f[51]<0.61829){if(f[63]<1.9565){if(f[40]<1.62339e+08){r+=-0.515466;}else{r+=0.269465;}}else{if(f[84]<1.85){r+=-0.476592;}else{if(f[42]<1.00038){r+=0.682838;}else{if(f[36]<1.52101e+08){r+=1.27782;}else{r+=0.445507;}}}}}else{if(f[40]<5.51307e+07){if(f[28]<1.40561e+11){r+=-0.8044;}else{if(f[14]<1.40502e+11){if(f[84]<1.15){r+=-0.743626;}else{r+=0.806231;}}else{if(f[28]<1.40591e+11){r+=0.52192;}else{r+=-0.591106;}}}}else{if(f[81]<1.90013){if(f[70]<1.01734){if(f[11]<7.27145e+07){r+=-0.248134;}else{r+=-0.657778;}}else{if(f[42]<0.996663){r+=0.490104;}else{r+=-0.653682;}}}else{if(f[41]<2.98707e+08){if(f[67]<0.391675){r+=-0.0986731;}else{r+=0.649953;}}else{if(f[53]<0.665205){r+=0.0465374;}else{r+=-0.682982;}}}}}}}}else{if(f[84]<0.5){if(f[14]<1.39668e+11){if(f[28]<1.42034e+11){if(f[14]<1.39584e+11){r+=-0.68729;}else{if(f[25]<9.7173e+08){if(f[38]<1.35847e+08){r+=-0.402207;}else{if(f[42]<1.00705){r+=-1.52655;}else{r+=-0.462624;}}}else{r+=-0.68977;}}}else{if(f[14]<1.39298e+11){r+=-0.675536;}else{r+=0.631771;}}}else{if(f[28]<1.40675e+11){if(f[14]<1.4039e+11){if(f[67]<2.56274){if(f[15]<3.11957e+08){if(f[14]<1.39691e+11){r+=-1.04708;}else{r+=-0.691058;}}else{r+=0.0547665;}}else{r+=0.669096;}}else{if(f[42]<0.997203){if(f[14]<1.41178e+11){if(f[69]<0.32932){r+=1.14519;}else{r+=0.755578;}}else{if(f[82]<2.29913){r+=-0.909271;}else{r+=-0.0590217;}}}else{if(f[28]<1.40579e+11){if(f[75]<0.608053){r+=-0.400317;}else{r+=0.508814;}}else{r+=-0.781413;}}}}else{if(f[14]<1.42011e+11){if(f[42]<1.00132){if(f[52]<0.315071){if(f[28]<1.40947e+11){r+=0.692509;}else{r+=-0.581751;}}else{if(f[28]<1.40691e+11){r+=-0.471292;}else{r+=-0.686497;}}}else{if(f[28]<1.41904e+11){r+=-0.701614;}else{r+=0.689045;}}}else{if(f[9]<4.95266e+07){r+=-0.696069;}else{if(f[70]<0.996131){r+=-0.357932;}else{if(f[82]<0.238537){r+=-0.0325396;}else{r+=0.69251;}}}}}}}else{if(f[70]<0.982602){if(f[14]<1.40043e+11){if(f[41]<3.54316e+08){if(f[1]<2.94658e+08){r+=0.690258;}else{r+=0.0984142;}}else{r+=0.927153;}}else{if(f[23]<4.37083e+10){r+=-0.777432;}else{r+=0.544751;}}}else{if(f[28]<1.41852e+11){if(f[84]<4.85){if(f[55]<2.34981){if(f[27]<1.1403e+08){if(f[42]<1.00179){r+=-0.661222;}else{r+=-0.0716875;}}else{if(f[70]<1.01557){r+=-0.67851;}else{r+=-0.199551;}}}else{if(f[70]<1.00235){if(f[70]<1.00033){r+=-0.637149;}else{r+=0.556506;}}else{if(f[28]<1.40453e+11){r+=0.31437;}else{r+=-0.680497;}}}}else{if(f[70]<0.999223){if(f[14]<1.41172e+11){r+=1.58314;}else{if(f[41]<7.29693e+08){r+=0.692977;}else{r+=1.19916;}}}else{r+=-0.691079;}}}else{if(f[28]<1.4265e+11){if(f[42]<0.99719){if(f[14]<1.41197e+11){if(f[42]<0.997107){r+=0.697771;}else{r+=0.0473067;}}else{r+=-0.426639;}}else{if(f[70]<1.01805){if(f[52]<1.26081){r+=-0.688234;}else{r+=-0.933747;}}else{r+=0.20193;}}}else{if(f[13]<1.60877e+09){if(f[28]<1.4267e+11){if(f[42]<0.997866){r+=0.531714;}else{r+=-0.616373;}}else{r+=-0.689211;}}else{if(f[28]<1.45245e+11){if(f[14]<1.41142e+11){r+=-0.659517;}else{r+=-1.28593;}}else{if(f[7]<8.08005e+09){r+=-0.704431;}else{r+=-0.106024;}}}}}}}}}}else{if(f[14]<1.42573e+11){if(f[13]<1.67201e+10){if(f[35]<6.00861e+07){if(f[14]<1.40089e+11){if(f[21]<1.88506e+07){r+=0.00658826;}else{if(f[81]<14.4179){if(f[28]<1.39649e+11){r+=-0.158519;}else{r+=-0.688713;}}else{r+=-0.079907;}}}else{if(f[84]<0.5){if(f[14]<1.42052e+11){r+=-0.689897;}else{r+=0.388408;}}else{if(f[84]<1.5){if(f[14]<1.40764e+11){if(f[28]<1.39902e+11){r+=-0.512081;}else{if(f[14]<1.40152e+11){r+=0.00475316;}else{r+=0.692037;}}}else{r+=-0.627144;}}else{if(f[14]<1.40297e+11){if(f[84]<3.15){if(f[34]<3.54643e+06){r+=0.320304;}else{r+=-0.727537;}}else{if(f[28]<1.39838e+11){r+=-0.507433;}else{r+=0.767953;}}}else{r+=-0.707639;}}}}}else{if(f[28]<1.446e+11){if(f[14]<1.24393e+11){r+=0.549968;}else{if(f[52]<0.00154844){if(f[12]<331416){r+=-0.15099;}else{r+=0.659551;}}else{if(f[28]<1.43834e+11){if(f[38]<3.59762e+08){r+=0.476259;}else{if(f[66]<8.73586){r+=-0.678539;}else{r+=0.28421;}}}else{if(f[84]<0.8){r+=0.651792;}else{if(f[67]<0.203452){r+=-0.673783;}else{r+=-1.05855;}}}}}}else{if(f[84]<1){if(f[45]<5.36807){r+=0.680675;}else{r+=0.105201;}}else{r+=-0.543355;}}}}else{if(f[40]<2.27889e+10){if(f[14]<1.40241e+11){if(f[14]<1.39936e+11){if(f[84]<1.5){r+=-0.666644;}else{if(f[28]<1.40602e+11){if(f[48]<1.28767){if(f[26]<3.21223e+10){r+=0.973274;}else{r+=0.264506;}}else{r+=-0.210762;}}else{r+=-0.500108;}}}else{if(f[35]<2.37139e+07){r+=-0.519877;}else{if(f[1]<8.85848e+08){if(f[84]<2.15){if(f[70]<1.00083){r+=0.102282;}else{r+=0.796853;}}else{if(f[26]<1.58674e+10){r+=0.868953;}else{r+=1.34053;}}}else{r+=-0.108747;}}}}else{if(f[10]<7.22112e+10){if(f[84]<1.15){if(f[9]<2.98684e+10){if(f[30]<3.63412e+07){r+=-0.226464;}else{if(f[38]<4.82735e+09){r+=0.165914;}else{r+=0.754069;}}}else{r+=-0.591641;}}else{if(f[12]<4.58385e+10){if(f[28]<1.39857e+11){r+=-0.031122;}else{r+=-0.699763;}}else{r+=0.0314021;}}}else{if(f[35]<1.11048e+10){r+=-0.13222;}else{r+=0.98676;}}}}else{if(f[67]<0.469001){if(f[58]<0.0141421){r+=0.0331184;}else{r+=-0.801354;}}else{if(f[1]<4.19438e+09){if(f[6]<2.5596e+09){if(f[23]<7.29093e+09){if(f[14]<1.40649e+11){if(f[69]<1.20199){r+=0.747042;}else{r+=0.0889394;}}else{r+=-0.370353;}}else{if(f[10]<7.78578e+10){if(f[12]<3.6123e+10){r+=-0.716782;}else{r+=-0.161556;}}else{if(f[27]<5.0592e+10){r+=0.691045;}else{r+=-0.260865;}}}}else{if(f[41]<2.75465e+10){r+=-0.136383;}else{if(f[8]<3.3633e+10){r+=0.93219;}else{r+=0.237992;}}}}else{if(f[19]<4.6002e+08){r+=-0.154212;}else{r+=-0.731852;}}}}}}else{if(f[84]<0.5){if(f[1]<5.38339e+09){r+=0.704764;}else{r+=-0.109208;}}else{if(f[36]<1.16921e+08){r+=-0.680875;}else{if(f[11]<4.9348e+10){r+=-1.0012;}else{r+=-0.135294;}}}}}}} return r;}
    static double p8(double[] f){double r=0;if(f[28]<1.39442e+11){if(f[56]<1.00179){if(f[28]<1.39216e+11){if(f[56]<1.00134){r+=-0.639996;}else{r+=0.161434;}}else{if(f[36]<1.44232e+08){if(f[67]<1.43519){r+=0.685711;}else{r+=0.103643;}}else{r+=-0.438884;}}}else{if(f[10]<3.57921e+10){if(f[55]<0.296498){if(f[84]<1.15){r+=-0.577953;}else{if(f[14]<1.40207e+11){r+=0.748367;}else{r+=-0.459711;}}}else{if(f[28]<1.39418e+11){if(f[42]<1.00945){r+=-0.592982;}else{if(f[42]<1.01067){if(f[84]<2.15){r+=-0.51717;}else{if(f[71]<1.68496){if(f[57]<0.545233){r+=0.706553;}else{r+=2.27686;}}else{r+=0.629908;}}}else{r+=-0.591293;}}}else{if(f[84]<1.15){r+=-0.626922;}else{if(f[42]<1.00286){if(f[22]<4.31466e+08){r+=-0.612458;}else{r+=-0.0732246;}}else{if(f[53]<0.95307){if(f[0]<1.41624e+11){r+=0.149018;}else{r+=-0.354889;}}else{r+=0.649316;}}}}}}else{if(f[56]<1.00987){if(f[34]<1.6977e+09){r+=0.781456;}else{r+=0.0932891;}}else{if(f[71]<0.773735){r+=-0.560582;}else{if(f[50]<0.788133){r+=-0.334635;}else{if(f[65]<1.05585){r+=-0.0113118;}else{r+=0.725151;}}}}}}}else{if(f[0]<1.45705e+11){if(f[0]<1.42562e+11){if(f[28]<1.40709e+11){if(f[0]<1.40096e+11){if(f[14]<1.4011e+11){if(f[28]<1.40001e+11){if(f[14]<1.39654e+11){if(f[28]<1.39886e+11){if(f[14]<1.3902e+11){r+=-0.573714;}else{if(f[3]<2.64662e+08){if(f[80]<2.66952){r+=0.620545;}else{r+=1.05379;}}else{r+=-0.0152641;}}}else{if(f[9]<2.27513e+08){r+=-0.593566;}else{r+=-0.864374;}}}else{if(f[0]<1.40075e+11){if(f[14]<1.40058e+11){if(f[29]<3.50589e+08){if(f[41]<8.02299e+07){r+=-0.595442;}else{r+=-0.718374;}}else{if(f[42]<1.0001){r+=-0.297435;}else{r+=0.551522;}}}else{if(f[57]<0.099159){if(f[83]<1.27452){r+=0.677488;}else{r+=-0.305635;}}else{if(f[26]<2.60069e+08){r+=-1.0411;}else{r+=-1.37232;}}}}else{if(f[35]<1.32184e+08){r+=-0.631533;}else{if(f[53]<1.87049){if(f[80]<0.60165){r+=-0.0240577;}else{r+=0.631337;}}else{r+=-0.393045;}}}}}else{if(f[84]<2.85){if(f[21]<1.78423e+08){if(f[41]<7.17366e+07){if(f[83]<2.17363){if(f[0]<1.4007e+11){r+=-0.643545;}else{r+=-0.810783;}}else{if(f[0]<1.40001e+11){r+=-0.468844;}else{r+=0.711084;}}}else{if(f[84]<0.5){r+=-0.60742;}else{if(f[82]<15.8636){r+=0.604937;}else{r+=-0.649544;}}}}else{if(f[82]<2.46655){if(f[84]<2.5){if(f[78]<0.436863){r+=0.195489;}else{r+=-0.688505;}}else{if(f[78]<0.340133){r+=-0.572525;}else{r+=0.605578;}}}else{if(f[28]<1.40185e+11){if(f[8]<4.5959e+08){r+=0.379149;}else{r+=-0.434175;}}else{if(f[84]<1.15){r+=-0.469747;}else{r+=0.615437;}}}}}else{if(f[28]<1.40165e+11){if(f[14]<1.40058e+11){if(f[42]<1.00351){if(f[14]<1.40023e+11){r+=-0.632676;}else{r+=-0.749487;}}else{if(f[9]<2.27227e+08){r+=0.659349;}else{r+=-0.409851;}}}else{if(f[67]<1.04526){r+=-0.771073;}else{if(f[84]<3.15){r+=-1.09145;}else{r+=-1.72725;}}}}else{if(f[0]<1.39969e+11){if(f[36]<2.39363e+07){if(f[14]<1.39202e+11){r+=-0.460826;}else{r+=-0.978089;}}else{if(f[14]<1.40068e+11){r+=-0.623767;}else{r+=0.341732;}}}else{if(f[28]<1.40356e+11){if(f[26]<6.25519e+07){r+=-0.662378;}else{r+=0.636189;}}else{r+=-0.573473;}}}}}}else{if(f[14]<1.40212e+11){if(f[64]<0.324119){if(f[56]<1.00013){if(f[26]<7.25017e+08){if(f[31]<1.09688e+08){r+=-0.708816;}else{r+=-0.140022;}}else{if(f[43]<2.03904){if(f[65]<0.370314){r+=0.726397;}else{r+=0.119164;}}else{if(f[5]<1.11373e+08){r+=-0.0736857;}else{r+=-0.414343;}}}}else{if(f[56]<1.00099){if(f[80]<3.94184){if(f[84]<1){r+=-0.476675;}else{r+=0.705349;}}else{r+=-0.457947;}}else{if(f[28]<1.39989e+11){r+=0.124669;}else{r+=-0.696571;}}}}else{if(f[39]<4.99295e+08){if(f[41]<1.01916e+08){if(f[51]<0.666843){if(f[41]<8.43684e+07){r+=-0.629817;}else{r+=1.21671;}}else{if(f[21]<9.46107e+06){r+=-0.141984;}else{r+=-0.60835;}}}else{if(f[26]<1.44885e+08){if(f[0]<1.4006e+11){r+=-0.757225;}else{r+=-1.03629;}}else{if(f[84]<2.15){r+=-0.6599;}else{r+=-1.25274;}}}}else{if(f[14]<1.40126e+11){if(f[38]<1.41265e+09){if(f[55]<0.906564){r+=0.694964;}else{r+=1.20089;}}else{if(f[41]<2.22642e+09){r+=-0.469285;}else{r+=0.11667;}}}else{if(f[37]<1.64708e+09){if(f[36]<6.23771e+08){r+=-0.671415;}else{r+=-0.0404937;}}else{if(f[0]<1.39967e+11){r+=-0.293207;}else{r+=0.743551;}}}}}}else{if(f[0]<1.39786e+11){if(f[84]<2.85){if(f[84]<0.5){if(f[56]<1.00062){r+=0.816846;}else{r+=-0.592779;}}else{if(f[18]<7.78687e+08){if(f[41]<9.10782e+07){r+=-0.581936;}else{r+=-0.670255;}}else{if(f[56]<1.0019){r+=0.516854;}else{r+=-0.249552;}}}}else{if(f[28]<1.40052e+11){if(f[28]<1.39977e+11){if(f[54]<1.31985){r+=0.624637;}else{r+=0.696614;}}else{r+=1.30098;}}else{if(f[14]<1.40827e+11){r+=-0.692139;}else{r+=-0.596923;}}}}else{if(f[28]<1.40557e+11){if(f[84]<0.5){if(f[28]<1.40529e+11){if(f[0]<1.40047e+11){r+=-0.776771;}else{r+=-0.564779;}}else{if(f[0]<1.39924e+11){r+=0.616276;}else{r+=-0.286329;}}}else{if(f[28]<1.40403e+11){if(f[67]<0.477028){r+=-0.42952;}else{r+=0.40485;}}else{if(f[34]<2.42945e+08){r+=-0.895706;}else{r+=0.527281;}}}}else{if(f[84]<0.5){if(f[14]<1.40277e+11){r+=-0.617325;}else{if(f[13]<1.40888e+08){r+=0.815816;}else{r+=0.638394;}}}else{if(f[1]<3.72635e+09){if(f[28]<1.40621e+11){r+=-0.869827;}else{r+=-0.60973;}}else{r+=0.320954;}}}}}}}else{if(f[14]<1.40161e+11){if(f[28]<1.40002e+11){if(f[84]<2.85){if(f[28]<1.39948e+11){if(f[14]<1.40071e+11){if(f[42]<1.006){if(f[84]<0.85){r+=-0.365406;}else{r+=0.625918;}}else{if(f[84]<1.85){r+=-0.815173;}else{r+=0.368939;}}}else{if(f[0]<1.41534e+11){if(f[0]<1.40139e+11){r+=0.182919;}else{r+=-0.738552;}}else{if(f[84]<2.15){r+=-0.329321;}else{r+=1.20358;}}}}else{if(f[14]<1.38973e+11){if(f[14]<1.3888e+11){if(f[14]<1.38849e+11){r+=-0.778715;}else{r+=-0.445149;}}else{if(f[84]<2.15){r+=0.686459;}else{r+=-0.67981;}}}else{if(f[14]<1.40155e+11){if(f[14]<1.40139e+11){r+=-0.698106;}else{r+=-1.29177;}}else{if(f[39]<7.69764e+07){r+=-0.390125;}else{r+=0.827725;}}}}}else{if(f[80]<1.25705){if(f[0]<1.40123e+11){if(f[84]<3.15){if(f[67]<1.74483){r+=-1.14594;}else{r+=-0.705202;}}else{r+=0.586593;}}else{if(f[80]<1.18628){if(f[39]<4.3569e+08){r+=0.651601;}else{r+=-0.464658;}}else{if(f[10]<1.52826e+08){r+=0.64923;}else{r+=-0.654878;}}}}else{if(f[55]<0.676619){if(f[14]<1.39069e+11){r+=0.692288;}else{if(f[12]<3.24148e+08){r+=-0.725939;}else{r+=0.0519037;}}}else{if(f[38]<1.88259e+08){if(f[65]<0.67502){r+=-0.545738;}else{r+=0.673305;}}else{if(f[13]<1.70127e+08){r+=-1.02343;}else{r+=-0.676464;}}}}}}else{if(f[84]<2.85){if(f[26]<3.37061e+08){if(f[14]<1.40091e+11){if(f[84]<1.15){if(f[41]<6.54962e+08){r+=-0.716626;}else{r+=-0.0469325;}}else{if(f[84]<2.15){r+=0.0250665;}else{r+=-0.647089;}}}else{if(f[0]<1.41597e+11){if(f[0]<1.40161e+11){r+=-0.789294;}else{r+=-0.59894;}}else{if(f[84]<1.8){r+=-0.811337;}else{r+=0.941135;}}}}else{if(f[14]<1.40058e+11){if(f[0]<1.401e+11){if(f[25]<4.01033e+08){r+=0.755775;}else{r+=-0.515357;}}else{if(f[14]<1.38998e+11){r+=-0.291249;}else{r+=-0.603887;}}}else{if(f[84]<2.15){if(f[55]<1.24077){r+=-1.02904;}else{r+=-0.741592;}}else{if(f[0]<1.41455e+11){r+=-1.23147;}else{r+=0.903575;}}}}}else{if(f[12]<7.26226e+08){if(f[41]<2.23068e+08){if(f[42]<1.00046){if(f[55]<0.878136){r+=-0.670162;}else{r+=-0.983405;}}else{if(f[28]<1.40023e+11){r+=0.0424398;}else{r+=-0.6443;}}}else{if(f[28]<1.4014e+11){if(f[14]<1.38865e+11){r+=-0.834381;}else{r+=0.458916;}}else{if(f[42]<1.00055){r+=0.532102;}else{r+=-0.641199;}}}}else{if(f[14]<1.40017e+11){if(f[26]<2.21504e+08){r+=0.333622;}else{if(f[17]<7.82245e+07){r+=-0.618388;}else{r+=-0.284591;}}}else{if(f[55]<0.863996){if(f[83]<0.699792){r+=1.71756;}else{r+=0.876871;}}else{if(f[21]<5.29174e+08){r+=0.684514;}else{r+=0.853246;}}}}}}}else{if(f[28]<1.3981e+11){if(f[42]<1.00929){if(f[56]<1.00279){if(f[13]<3.57624e+08){r+=0.684497;}else{r+=-0.189516;}}else{if(f[41]<3.83484e+07){if(f[28]<1.39772e+11){r+=-0.338069;}else{r+=0.779905;}}else{if(f[10]<4.03061e+10){if(f[28]<1.39806e+11){r+=-0.649792;}else{r+=-0.392602;}}else{if(f[14]<1.40797e+11){r+=0.560284;}else{r+=-0.339268;}}}}}else{if(f[1]<6.97422e+09){r+=0.699988;}else{r+=-0.305216;}}}else{if(f[84]<4.65){if(f[28]<1.4038e+11){if(f[39]<2.23436e+08){if(f[14]<1.40234e+11){if(f[25]<6.41414e+07){r+=-0.650194;}else{r+=0.73752;}}else{if(f[37]<2.26078e+07){r+=0.488201;}else{r+=-0.279624;}}}else{if(f[0]<1.4123e+11){if(f[84]<1.15){r+=0.1884;}else{r+=-0.682106;}}else{if(f[84]<1.5){r+=-1.33127;}else{r+=0.691501;}}}}else{if(f[28]<1.40593e+11){if(f[84]<3.15){if(f[83]<144.072){r+=-0.708564;}else{r+=-1.38478;}}else{if(f[0]<1.40274e+11){r+=-1.27606;}else{r+=-0.722242;}}}else{if(f[29]<2.22484e+07){if(f[42]<1.00337){r+=-0.512026;}else{r+=0.931083;}}else{if(f[39]<5.9163e+07){r+=0.34652;}else{r+=-0.593446;}}}}}else{if(f[0]<1.40237e+11){if(f[56]<1.00344){if(f[14]<1.40167e+11){if(f[0]<1.40145e+11){r+=-1.09741;}else{r+=0.517638;}}else{if(f[8]<7.41596e+06){r+=0.949039;}else{r+=0.716009;}}}else{if(f[52]<0.452723){r+=0.400396;}else{r+=-0.679511;}}}else{if(f[28]<1.40645e+11){if(f[38]<5.23272e+08){if(f[14]<1.40234e+11){r+=0.435543;}else{r+=-0.734977;}}else{if(f[10]<5.0518e+08){r+=0.566702;}else{r+=-0.481878;}}}else{if(f[13]<2.11928e+09){if(f[14]<1.40735e+11){r+=-0.473127;}else{r+=0.615434;}}else{if(f[18]<9.89253e+07){r+=-0.554212;}else{r+=0.0586105;}}}}}}}}}else{if(f[28]<1.41479e+11){if(f[14]<1.46481e+11){if(f[0]<1.40693e+11){if(f[14]<1.45798e+11){if(f[28]<1.41461e+11){if(f[30]<3.54733e+08){if(f[14]<1.45541e+11){if(f[17]<2.24491e+09){r+=-0.602075;}else{r+=0.268248;}}else{if(f[84]<2){r+=0.547054;}else{r+=-0.58496;}}}else{if(f[52]<0.603905){if(f[84]<4.60001){r+=-0.256756;}else{r+=1.04599;}}else{if(f[50]<0.722968){r+=0.186635;}else{r+=-0.494894;}}}}else{if(f[0]<1.39275e+11){if(f[51]<1.33757){r+=0.622181;}else{r+=0.0203675;}}else{r+=-0.57948;}}}else{if(f[84]<2){r+=0.609955;}else{r+=-0.609482;}}}else{if(f[0]<1.40819e+11){if(f[84]<1.85){r+=-0.608387;}else{if(f[84]<2.15){if(f[83]<2.41827){if(f[83]<0.718784){r+=0.731843;}else{r+=0.63486;}}else{if(f[33]<1.4109e+08){r+=1.43207;}else{r+=0.440531;}}}else{r+=-0.575939;}}}else{if(f[84]<4.85){if(f[0]<1.40834e+11){if(f[84]<1.85){r+=-0.591259;}else{if(f[84]<2.15){r+=0.758995;}else{r+=-0.61334;}}}else{if(f[1]<2.62927e+09){if(f[13]<1.40894e+08){r+=-0.67057;}else{r+=-0.605838;}}else{if(f[50]<1.29894){r+=0.507979;}else{r+=-0.459867;}}}}else{if(f[1]<3.12653e+08){r+=0.603644;}else{r+=-0.324708;}}}}}else{if(f[84]<1){r+=0.592507;}else{r+=-0.595565;}}}else{if(f[0]<1.39329e+11){if(f[28]<1.42038e+11){if(f[84]<2.15){r+=-0.898214;}else{if(f[83]<1.96298){if(f[42]<0.987112){r+=-0.247301;}else{if(f[41]<1.45808e+08){r+=0.618837;}else{r+=0.714195;}}}else{if(f[55]<0.764116){r+=0.625395;}else{if(f[52]<1.13814){r+=0.33955;}else{r+=1.5129;}}}}}else{if(f[14]<1.46987e+11){r+=-0.593775;}else{r+=-0.112242;}}}else{if(f[84]<1.5){if(f[28]<1.42027e+11){if(f[41]<8.69272e+07){r+=0.36681;}else{r+=-0.602772;}}else{if(f[0]<1.41315e+11){if(f[0]<1.40235e+11){if(f[3]<2.76339e+07){if(f[18]<1.91935e+07){r+=-0.615436;}else{r+=-0.200677;}}else{if(f[80]<0.364884){r+=-0.402141;}else{r+=0.415168;}}}else{if(f[14]<1.40859e+11){if(f[80]<0.0122221){r+=-0.489265;}else{r+=0.58035;}}else{if(f[56]<0.982573){r+=-0.596083;}else{r+=0.563878;}}}}else{if(f[28]<1.43851e+11){if(f[84]<2.60001){if(f[18]<1.78435e+08){r+=-0.592074;}else{r+=-0.0508024;}}else{if(f[4]<3.07481e+07){r+=0.450457;}else{r+=0.0297301;}}}else{if(f[13]<6.11319e+08){r+=-0.398577;}else{r+=0.603139;}}}}}else{if(f[0]<1.39526e+11){if(f[84]<2.5){if(f[12]<1.20962e+08){if(f[26]<6.03056e+07){r+=-1.11844;}else{if(f[40]<1.42182e+08){r+=-2.32975;}else{r+=-1.21834;}}}else{if(f[80]<0.555164){r+=-0.586883;}else{if(f[40]<1.88181e+08){r+=-1.01904;}else{r+=-0.691146;}}}}else{if(f[84]<3.35){r+=0.637273;}else{if(f[9]<2.55454e+08){if(f[40]<1.91573e+08){r+=-0.796778;}else{r+=-0.566908;}}else{r+=-0.599633;}}}}else{if(f[14]<1.40082e+11){if(f[38]<1.64596e+08){if(f[65]<1.13984){if(f[56]<0.982027){r+=-0.740559;}else{r+=-1.1645;}}else{r+=-0.633251;}}else{r+=-0.614088;}}else{if(f[0]<1.42261e+11){if(f[10]<5.50326e+10){r+=-0.599985;}else{r+=-0.0642646;}}else{if(f[67]<0.205441){r+=-0.570573;}else{r+=-0.807348;}}}}}}}}}else{if(f[84]<7.40001){if(f[56]<1.00805){if(f[40]<6.06179e+07){if(f[14]<1.41178e+11){r+=-0.629538;}else{r+=-0.103973;}}else{if(f[28]<1.45302e+11){if(f[28]<1.40378e+11){if(f[26]<5.82494e+07){r+=-0.510373;}else{if(f[8]<3.29297e+07){r+=-0.431356;}else{if(f[67]<5.59899){if(f[67]<0.853466){r+=0.744585;}else{r+=0.675987;}}else{r+=-0.277987;}}}}else{if(f[28]<1.40668e+11){if(f[31]<5.53423e+07){if(f[67]<0.946967){r+=-0.0908276;}else{r+=-0.614175;}}else{r+=0.5074;}}else{if(f[84]<2.15){if(f[56]<0.98593){if(f[84]<0.85){r+=0.589514;}else{r+=-0.576276;}}else{if(f[1]<2.63395e+09){r+=0.593436;}else{r+=-0.272709;}}}else{r+=-0.564281;}}}}else{if(f[84]<1){r+=0.597793;}else{r+=-0.610987;}}}}else{if(f[28]<1.39935e+11){r+=0.99164;}else{if(f[0]<1.42887e+11){if(f[80]<0.829367){r+=-0.511037;}else{r+=0.326798;}}else{if(f[28]<1.40132e+11){if(f[41]<9.14283e+07){r+=-0.827732;}else{r+=-0.657318;}}else{if(f[14]<1.41772e+11){r+=-1.29355;}else{if(f[20]<4.30828e+09){r+=-0.721848;}else{r+=-0.0619137;}}}}}}}else{if(f[42]<1.03035){r+=-0.742031;}else{r+=-0.591052;}}}}else{if(f[5]<8.19722e+08){if(f[41]<4.81024e+08){if(f[14]<1.39711e+11){r+=0.180428;}else{r+=-0.689368;}}else{if(f[77]<0.322178){if(f[14]<1.39856e+11){r+=0.603911;}else{r+=-0.3708;}}else{r+=-0.602001;}}}else{if(f[34]<5.04092e+08){if(f[39]<1.3379e+08){r+=-0.311836;}else{r+=0.630871;}}else{r+=-0.578361;}}}} return r;}
    static double p9(double[] f){double r=0;if(f[69]<0.48539){if(f[84]<1.15){if(f[14]<1.41943e+11){if(f[73]<39.3026){if(f[84]<0.85){if(f[0]<1.40105e+11){if(f[0]<1.40072e+11){if(f[14]<1.40417e+11){if(f[28]<1.39879e+11){if(f[42]<1.00215){if(f[14]<1.40349e+11){r+=-0.921569;}else{r+=0.25;}}else{if(f[14]<1.38799e+11){r+=-0.6;}else{r+=0.97351;}}}else{if(f[71]<0.0428904){if(f[5]<5.46238e+07){r+=0.84;}else{r+=-0.6;}}else{if(f[43]<6.83527){if(f[0]<1.40068e+11){r+=-0.989965;}else{r+=-0.3;}}else{if(f[36]<2.18438e+07){r+=0.578947;}else{r+=-0.914286;}}}}}else{if(f[56]<0.999665){if(f[28]<1.40709e+11){if(f[0]<1.39839e+11){r+=0.99;}else{r+=0.25;}}else{r+=-0.714286;}}else{if(f[14]<1.40471e+11){r+=-0.25;}else{if(f[56]<1.0001){r+=-0.25;}else{r+=-0.965812;}}}}}else{if(f[37]<2.30221e+07){r+=-0.945946;}else{if(f[13]<3.46224e+08){if(f[56]<0.999802){if(f[12]<6.59125e+07){r+=0.25;}else{r+=0.993092;}}else{if(f[0]<1.40088e+11){r+=0.25;}else{r+=-0.5;}}}else{if(f[32]<3.44678e+06){r+=0.4;}else{if(f[17]<4.23297e+07){r+=-0.789474;}else{r+=-0;}}}}}}else{if(f[28]<1.42891e+11){if(f[83]<1.23537){if(f[0]<1.42783e+11){if(f[71]<0.0861151){r+=0.809524;}else{if(f[27]<1.05609e+09){if(f[0]<1.40127e+11){r+=-0.25;}else{r+=-0.988024;}}else{if(f[1]<5.09697e+07){r+=0.25;}else{r+=-0.555556;}}}}else{if(f[56]<1.00799){r+=0.960784;}else{r+=-0.666667;}}}else{if(f[21]<3.99234e+08){if(f[63]<12.3609){r+=-0.992385;}else{r+=-0.25;}}else{if(f[41]<7.84861e+08){if(f[79]<0.491538){r+=-0.75;}else{if(f[10]<2.12605e+08){r+=-0.2;}else{r+=0.909091;}}}else{if(f[28]<1.39852e+11){if(f[10]<1.8099e+09){r+=0.25;}else{r+=-0.692308;}}else{r+=-0.98961;}}}}}else{if(f[40]<2.2781e+10){if(f[0]<1.43319e+11){r+=0.983122;}else{r+=0.25;}}else{r+=-0.828571;}}}}else{if(f[28]<1.39953e+11){if(f[0]<1.40492e+11){r+=-0.997655;}else{r+=-0.25;}}else{if(f[56]<1.00018){if(f[1]<4.08671e+08){if(f[5]<277141){r+=-0.25;}else{r+=-0.995192;}}else{r+=-0.25;}}else{if(f[27]<6.27477e+07){if(f[0]<1.40322e+11){r+=-0.977654;}else{r+=-0;}}else{if(f[28]<1.40421e+11){if(f[42]<0.990087){r+=-0.5;}else{if(f[30]<1.83997e+07){if(f[23]<4.52647e+06){r+=0.25;}else{r+=0.99375;}}else{if(f[25]<4.61708e+08){r+=-0.5;}else{r+=0.75;}}}}else{r+=-0.764706;}}}}}}else{if(f[0]<1.40129e+11){if(f[0]<1.40063e+11){r+=-0.25;}else{r+=0.99895;}}else{if(f[28]<1.46012e+11){if(f[84]<0.85){if(f[0]<2.43477e+11){if(f[48]<5.3161){r+=-0.994444;}else{r+=-0.25;}}else{r+=0.555556;}}else{r+=0.6;}}else{r+=0.875;}}}}else{if(f[70]<0.997805){if(f[63]<2.14261){r+=-0.809524;}else{r+=-0;}}else{if(f[10]<4.45739e+10){if(f[28]<1.41254e+11){r+=-0;}else{r+=0.999217;}}else{r+=-0.6;}}}}else{if(f[83]<2.4495){if(f[23]<9.65569e+07){if(f[37]<1.11918e+07){if(f[56]<1.00194){if(f[56]<1.00118){r+=-0.931035;}else{if(f[84]<1.5){r+=-0.84;}else{if(f[0]<1.39825e+11){r+=0.25;}else{r+=0.988439;}}}}else{r+=-0.983193;}}else{if(f[27]<7.2776e+08){if(f[67]<0.70716){if(f[22]<9.79328e+07){if(f[31]<1.47053e+06){if(f[0]<1.40228e+11){if(f[84]<12){if(f[21]<4.38288e+07){r+=-0.8967;}else{r+=-0.283582;}}else{if(f[0]<1.4017e+11){r+=-0.958333;}else{r+=0.927711;}}}else{r+=-0.992606;}}else{if(f[78]<0.0784712){r+=-0;}else{if(f[33]<124307){r+=-0.2;}else{if(f[23]<8.71108e+07){r+=-0.975267;}else{r+=-0.844875;}}}}}else{if(f[28]<1.40496e+11){if(f[0]<1.4023e+11){if(f[0]<1.40025e+11){r+=-0.92;}else{if(f[56]<1.00113){r+=-0.333333;}else{r+=0.777778;}}}else{if(f[16]<4.25592e+07){r+=-0.966851;}else{r+=-0.25;}}}else{if(f[14]<1.40234e+11){if(f[7]<1.45364e+08){r+=0.25;}else{r+=-0.826087;}}else{r+=0.818182;}}}}else{if(f[42]<0.997476){if(f[0]<1.39833e+11){if(f[28]<1.40186e+11){r+=0.25;}else{r+=-0.972789;}}else{if(f[70]<1.00129){r+=-0.891892;}else{if(f[84]<2.15){if(f[9]<6.76237e+07){r+=-0.789474;}else{r+=0.777778;}}else{if(f[14]<1.40542e+11){r+=-0.25;}else{r+=0.95082;}}}}}else{if(f[0]<1.40119e+11){if(f[84]<2.15){if(f[27]<1.9233e+08){if(f[12]<8.03939e+07){r+=0.555556;}else{r+=-0.555556;}}else{r+=0.95;}}else{r+=-0.968;}}else{if(f[84]<3.15){if(f[13]<8.83229e+08){if(f[72]<7.70916){r+=-0.981435;}else{r+=-0;}}else{r+=0.111111;}}else{if(f[16]<3.65849e+06){if(f[0]<1.40397e+11){r+=0.555556;}else{r+=-0;}}else{r+=-0.75;}}}}}}else{if(f[14]<1.39833e+11){if(f[0]<1.27709e+11){r+=0.25;}else{r+=0.913043;}}else{if(f[12]<2.22052e+09){if(f[45]<0.295513){if(f[5]<4.7749e+06){r+=-0.25;}else{r+=0.5;}}else{if(f[9]<1.83999e+08){r+=-0.933333;}else{r+=-0.25;}}}else{if(f[1]<4.74005e+06){r+=-0;}else{r+=0.6;}}}}}}else{if(f[14]<1.40996e+11){if(f[0]<1.40033e+11){if(f[28]<1.41873e+11){if(f[84]<12){if(f[53]<0.500262){r+=-0;}else{if(f[22]<6.21393e+09){if(f[49]<5.91876){if(f[70]<1.0172){r+=-0.988845;}else{r+=-0.25;}}else{r+=-0.25;}}else{if(f[0]<1.39524e+11){r+=-0.6;}else{r+=-0;}}}}else{if(f[14]<1.40433e+11){r+=-0.974843;}else{if(f[42]<0.994972){r+=-0.833333;}else{r+=0.906977;}}}}else{if(f[0]<1.39976e+11){r+=0.777778;}else{r+=-0.789474;}}}else{if(f[14]<1.40192e+11){if(f[9]<2.7903e+08){if(f[0]<1.40311e+11){if(f[28]<1.40167e+11){if(f[28]<1.39861e+11){if(f[27]<2.39459e+08){r+=0.951219;}else{r+=-0.5;}}else{if(f[33]<4.27341e+07){r+=-0.921569;}else{r+=0.111111;}}}else{if(f[56]<0.998){if(f[28]<1.40513e+11){r+=0.984615;}else{r+=-0.555556;}}else{r+=-0.636364;}}}else{if(f[28]<1.39563e+11){r+=0.950617;}else{if(f[83]<0.967005){if(f[29]<1.84428e+07){r+=-0.647059;}else{r+=0.869565;}}else{if(f[14]<1.24023e+11){r+=0.733333;}else{r+=-0.951772;}}}}}else{if(f[14]<1.39217e+11){if(f[84]<1.5){r+=-0.904762;}else{if(f[25]<1.11263e+08){r+=-0.733333;}else{if(f[0]<1.40264e+11){r+=-0.384615;}else{r+=0.966346;}}}}else{if(f[27]<1.83536e+08){if(f[56]<0.998838){if(f[51]<2.26205){r+=-0.655172;}else{r+=0.6875;}}else{if(f[29]<1.31594e+07){r+=-0;}else{r+=0.974249;}}}else{if(f[82]<1.57887){if(f[84]<2.15){r+=-0.618321;}else{r+=0.432099;}}else{if(f[56]<1.00207){r+=-0.896774;}else{r+=-0.103448;}}}}}}else{if(f[56]<1.00437){if(f[53]<3.66994){if(f[84]<2.15){if(f[23]<3.69522e+08){if(f[83]<1.7324){r+=0.941467;}else{r+=0.562457;}}else{if(f[70]<1.00186){r+=0.188406;}else{r+=-0.86;}}}else{if(f[67]<0.366706){if(f[0]<1.40133e+11){r+=0.0769231;}else{r+=-0.97235;}}else{if(f[28]<1.40389e+11){r+=0.681063;}else{r+=-0.870968;}}}}else{if(f[14]<1.40236e+11){if(f[82]<2.25814){r+=0.684211;}else{r+=-0.555556;}}else{if(f[56]<1.00337){if(f[42]<0.998394){r+=-0.222222;}else{r+=-0.947712;}}else{r+=0.0909091;}}}}else{if(f[14]<1.40228e+11){if(f[12]<4.29102e+08){r+=0.555556;}else{r+=-0.25;}}else{if(f[17]<2.22091e+06){r+=-0.25;}else{if(f[43]<0.303895){r+=-0.25;}else{r+=-0.986301;}}}}}}}else{if(f[84]<4.85){if(f[1]<8.40459e+06){r+=-0.25;}else{r+=-0.997096;}}else{if(f[37]<2.19394e+08){if(f[51]<1.99408){if(f[0]<1.40094e+11){r+=0.25;}else{r+=0.938462;}}else{r+=-0.25;}}else{if(f[9]<1.77167e+08){r+=0.111111;}else{if(f[4]<7.74949e+06){r+=-0.25;}else{r+=-0.972603;}}}}}}}else{if(f[41]<5.3159e+08){if(f[55]<0.42792){if(f[84]<1.85){if(f[0]<1.40216e+11){r+=0.979381;}else{r+=-0;}}else{r+=-0.733333;}}else{if(f[0]<1.4002e+11){if(f[14]<1.39804e+11){if(f[70]<0.999407){r+=0.625;}else{if(f[84]<2.5){r+=-0.967213;}else{if(f[0]<1.3935e+11){r+=0.733333;}else{r+=-0.25;}}}}else{if(f[78]<0.366889){if(f[84]<2.65){r+=0.571429;}else{r+=-0.789474;}}else{if(f[49]<0.287004){r+=-0.25;}else{if(f[66]<1.82827){r+=-0.993115;}else{r+=-0.25;}}}}}else{if(f[0]<1.40232e+11){if(f[28]<1.40338e+11){if(f[7]<1.06494e+08){if(f[39]<4.23169e+08){if(f[0]<1.40226e+11){if(f[40]<3.93572e+08){r+=-0.977848;}else{r+=-0.25;}}else{if(f[31]<2.02321e+06){r+=-0.642857;}else{r+=0.555556;}}}else{if(f[5]<4.50579e+06){r+=-0.25;}else{r+=0.5;}}}else{if(f[66]<0.705781){if(f[10]<1.34867e+08){r+=-0.6;}else{if(f[1]<1.99558e+07){r+=0.25;}else{r+=0.971831;}}}else{if(f[28]<1.40161e+11){if(f[45]<4.79306){r+=-0.882353;}else{r+=-0;}}else{r+=0.7;}}}}else{if(f[14]<1.40617e+11){if(f[56]<0.996951){r+=-0.913043;}else{if(f[0]<1.40086e+11){if(f[13]<1.18701e+08){r+=0.952941;}else{r+=-0.857143;}}else{if(f[84]<2.15){r+=0.372197;}else{r+=0.940928;}}}}else{if(f[84]<2.15){r+=-0.972028;}else{if(f[1]<2.47082e+06){r+=-0.777778;}else{r+=0.911111;}}}}}else{if(f[37]<1.85181e+08){if(f[12]<1.09106e+08){if(f[56]<1.00135){if(f[25]<1.13168e+08){if(f[69]<0.471376){r+=-0.981722;}else{r+=-0.4;}}else{if(f[84]<1.5){r+=0.625;}else{r+=-0.727273;}}}else{if(f[56]<1.00149){if(f[84]<5.40001){r+=-0.84;}else{r+=0.954545;}}else{if(f[9]<1.30471e+08){r+=-0.985729;}else{r+=0.589744;}}}}else{if(f[0]<1.41593e+11){if(f[39]<3.33719e+08){if(f[71]<2.78564){r+=-0.966102;}else{r+=-0.111111;}}else{r+=0.4;}}else{if(f[0]<1.41915e+11){r+=0.961905;}else{r+=-0.789474;}}}}else{if(f[42]<1.00207){if(f[0]<1.41112e+11){r+=-0.944444;}else{r+=-0.25;}}else{if(f[0]<1.41098e+11){if(f[84]<4.85){if(f[14]<1.38881e+11){r+=-0.25;}else{r+=0.926829;}}else{r+=-0.5;}}else{if(f[67]<0.751758){if(f[8]<2.38982e+08){r+=-0.897436;}else{r+=-0.25;}}else{if(f[10]<3.3709e+08){r+=0.764706;}else{r+=-0;}}}}}}}}}else{if(f[41]<8.36812e+08){if(f[23]<9.76746e+07){if(f[82]<9.60851){if(f[28]<1.40345e+11){if(f[4]<4.15686e+07){r+=-0.995921;}else{r+=-0.25;}}else{if(f[56]<1.00123){r+=-0.989362;}else{if(f[84]<2.15){r+=-0.666667;}else{r+=0.828571;}}}}else{if(f[2]<5.09415e+06){r+=-0.6;}else{r+=0.666667;}}}else{if(f[84]<12){if(f[70]<0.996539){if(f[70]<0.993883){if(f[8]<2.37765e+08){if(f[0]<1.4041e+11){r+=0.333333;}else{if(f[28]<1.40147e+11){r+=-0.976359;}else{r+=-0;}}}else{if(f[14]<1.40025e+11){if(f[64]<1.27563){r+=0.74359;}else{r+=-0.6;}}else{r+=-0.916667;}}}else{if(f[42]<1.00726){if(f[12]<2.30407e+08){if(f[2]<3.90402e+06){r+=-0;}else{r+=0.928854;}}else{r+=-0.692308;}}else{if(f[14]<1.39187e+11){if(f[0]<1.40702e+11){r+=0.862069;}else{r+=-0;}}else{if(f[0]<1.40987e+11){r+=-0.866667;}else{r+=-0;}}}}}else{if(f[0]<1.41572e+11){if(f[63]<0.146078){r+=0.5;}else{if(f[0]<1.39296e+11){r+=0.111111;}else{if(f[14]<1.39637e+11){r+=-0.448276;}else{r+=-0.996021;}}}}else{r+=0.625;}}}else{if(f[14]<1.40974e+11){if(f[56]<0.994417){if(f[6]<3.74812e+07){r+=0.714286;}else{r+=-0;}}else{if(f[14]<1.40469e+11){r+=-0.989011;}else{if(f[0]<1.40251e+11){r+=0.777778;}else{r+=-0.870968;}}}}else{r+=0.986014;}}}}else{if(f[4]<26586.4){if(f[0]<1.40202e+11){r+=-0.666667;}else{r+=0.714286;}}else{if(f[40]<2.71943e+08){if(f[28]<1.40497e+11){r+=-0.875;}else{if(f[25]<1.12384e+08){r+=-0;}else{r+=0.777778;}}}else{if(f[14]<1.40871e+11){if(f[14]<1.38992e+11){if(f[9]<2.51187e+08){r+=-0.980296;}else{if(f[27]<3.9322e+09){r+=0.870968;}else{r+=-0.25;}}}else{if(f[13]<1.53174e+10){if(f[17]<1.41166e+09){r+=-0.995153;}else{r+=-0.25;}}else{if(f[0]<1.40167e+11){r+=-0;}else{r+=-0.6;}}}}else{if(f[84]<12){if(f[14]<1.40923e+11){if(f[28]<1.40995e+11){r+=-0.99287;}else{if(f[0]<1.41997e+11){r+=0.882353;}else{r+=-0.974026;}}}else{if(f[40]<2.98749e+08){r+=-0.25;}else{r+=-0.998046;}}}else{if(f[70]<1.00249){if(f[48]<0.074201){r+=-0.25;}else{r+=-0.993464;}}else{if(f[12]<6.48837e+08){if(f[0]<1.39807e+11){r+=0.25;}else{r+=0.984314;}}else{r+=-0.875;}}}}}}}}}}}else{if(f[28]<1.39431e+11){if(f[10]<3.31037e+10){if(f[28]<1.39417e+11){if(f[71]<1.8334){if(f[53]<0.0614534){r+=-0.25;}else{r+=-0.996809;}}else{if(f[84]<2.15){if(f[1]<2.87245e+08){if(f[28]<1.39415e+11){r+=-0.994688;}else{r+=-0.25;}}else{r+=-0.25;}}else{if(f[42]<1.00865){if(f[1]<1.16138e+08){r+=-0.931035;}else{r+=-0;}}else{if(f[0]<1.41714e+11){r+=0.928571;}else{r+=0.25;}}}}}else{if(f[14]<1.39671e+11){r+=0.818182;}else{if(f[55]<0.276313){if(f[84]<1){r+=-0.764706;}else{r+=0.777778;}}else{if(f[17]<2.47124e+06){r+=0.111111;}else{if(f[64]<0.621636){if(f[28]<1.39424e+11){r+=-0.733333;}else{r+=0.25;}}else{if(f[72]<4.50122){if(f[18]<2.82509e+06){r+=-0.25;}else{if(f[76]<0.100478){r+=-0.25;}else{if(f[5]<8.50467e+07){r+=-0.948148;}else{r+=-0.384615;}}}}else{if(f[5]<1.47744e+07){r+=-0;}else{r+=-0.555556;}}}}}}}}else{if(f[83]<0.801469){if(f[57]<3.6617){r+=-0.870968;}else{r+=0.2;}}else{if(f[79]<1.08253){if(f[66]<0.904681){r+=-0.25;}else{if(f[70]<1.00113){r+=0.789474;}else{r+=-0;}}}else{r+=-0.538462;}}}}else{if(f[67]<6.01597){if(f[14]<1.39858e+11){if(f[14]<1.39528e+11){if(f[28]<1.40242e+11){if(f[28]<1.40017e+11){if(f[28]<1.39689e+11){if(f[83]<0.936034){if(f[7]<4.52775e+07){r+=-0.777778;}else{if(f[69]<0.923379){if(f[1]<4.84492e+07){r+=-0.666667;}else{r+=0.777778;}}else{if(f[1]<6.9747e+08){r+=0.985949;}else{r+=-0.25;}}}}else{if(f[0]<1.40185e+11){r+=0.666667;}else{if(f[0]<1.43234e+11){r+=-0.995227;}else{r+=0.333333;}}}}else{if(f[79]<1.21553){if(f[37]<7.27987e+07){r+=-0.973154;}else{if(f[27]<1.1404e+08){if(f[0]<1.40509e+11){r+=0.966102;}else{r+=-0.85034;}}else{if(f[66]<0.287988){r+=-0.846154;}else{r+=0.947407;}}}}else{if(f[84]<1.85){if(f[8]<4.52651e+07){r+=-0.942857;}else{if(f[83]<1.77518){r+=0.968064;}else{r+=-0.05;}}}else{if(f[55]<0.87224){if(f[15]<4.94682e+07){r+=0.709924;}else{r+=-0.658537;}}else{if(f[84]<3.35){r+=-0.934919;}else{r+=0.727273;}}}}}}else{if(f[84]<1.5){if(f[0]<1.3798e+11){if(f[0]<1.42906e+10){r+=0.25;}else{r+=0.967213;}}else{if(f[82]<3.59703e+08){if(f[2]<9.44491e+08){if(f[82]<3.36617e+08){r+=-0.997927;}else{r+=-0.272727;}}else{r+=0.454545;}}else{r+=0.6;}}}else{if(f[84]<2.15){if(f[28]<1.40154e+11){if(f[14]<1.39311e+11){if(f[57]<43.9541){r+=0.977769;}else{r+=-0.6;}}else{if(f[28]<1.40144e+11){r+=-0.932382;}else{r+=0.127517;}}}else{if(f[40]<1.21244e+09){if(f[0]<1.40635e+11){r+=0.98683;}else{r+=0.166667;}}else{if(f[29]<7.16561e+07){r+=-0.884615;}else{r+=0.538462;}}}}else{if(f[9]<2.53071e+08){if(f[55]<0.81944){if(f[84]<3.5){r+=-0.88;}else{r+=0.571429;}}else{if(f[9]<2.36866e+08){r+=-0.996683;}else{r+=-0.473684;}}}else{if(f[78]<0.416613){if(f[0]<1.41204e+11){r+=-0.846154;}else{r+=0.25;}}else{if(f[84]<2.85){r+=-0.764706;}else{r+=0.973404;}}}}}}}else{if(f[14]<1.39523e+11){if(f[28]<1.40255e+11){if(f[84]<7.40001){r+=-0.981221;}else{r+=0.943662;}}else{if(f[14]<9.37242e+10){if(f[0]<1.42542e+11){r+=-0.809524;}else{r+=0.777778;}}else{if(f[42]<1.00134){if(f[0]<1.39271e+11){if(f[1]<5.28381e+09){r+=-0.967213;}else{r+=-0.25;}}else{r+=0.885714;}}else{if(f[70]<0.983354){if(f[14]<1.39504e+11){r+=-0.971901;}else{r+=0.78125;}}else{if(f[74]<137.971){r+=-0.995075;}else{r+=0.111111;}}}}}}else{if(f[69]<1.20962){if(f[0]<1.42086e+11){r+=-0.987692;}else{r+=0.25;}}else{if(f[68]<1.76223){if(f[67]<1.09397){r+=-0.555556;}else{if(f[9]<2.34541e+08){r+=0.965517;}else{r+=-0;}}}else{if(f[0]<1.41547e+11){r+=-0.733333;}else{r+=-0;}}}}}}else{if(f[22]<4.42054e+08){if(f[66]<0.736207){if(f[84]<2.85){if(f[56]<0.993291){if(f[56]<0.993239){if(f[0]<1.42571e+11){r+=-0.991058;}else{r+=-0.25;}}else{r+=-0.111111;}}else{if(f[66]<0.372674){if(f[30]<1.65563e+07){if(f[0]<1.40249e+11){r+=0.826087;}else{r+=-0.980237;}}else{if(f[0]<1.40523e+11){r+=0.849462;}else{r+=-0.555556;}}}else{if(f[25]<6.83603e+07){if(f[41]<9.88676e+07){r+=0.555556;}else{r+=-0.886792;}}else{if(f[0]<1.40871e+11){r+=0.915544;}else{r+=-0.111111;}}}}}else{if(f[42]<0.996976){if(f[22]<9.06983e+07){r+=-0.733333;}else{r+=0.921569;}}else{if(f[28]<1.41896e+11){r+=-0.996864;}else{r+=-0;}}}}else{if(f[84]<3.15){if(f[63]<0.322555){if(f[22]<8.36121e+07){r+=-0.982759;}else{if(f[0]<1.39684e+11){r+=0.909091;}else{r+=0.25;}}}else{if(f[28]<1.42204e+11){if(f[70]<1.40179e+11){if(f[9]<2.91319e+08){r+=0.881276;}else{r+=0.472032;}}else{if(f[14]<1.3968e+11){r+=-0.97619;}else{r+=0.133333;}}}else{r+=-0.975758;}}}else{if(f[36]<2.61353e+08){r+=-0.995968;}else{r+=0.2;}}}}else{if(f[28]<1.39791e+11){if(f[15]<5.77426e+08){if(f[28]<1.39461e+11){if(f[14]<1.39816e+11){r+=-0.8;}else{r+=0.862069;}}else{if(f[55]<1.94123){if(f[8]<3.10511e+09){r+=0.974093;}else{r+=0.25;}}else{r+=0.272727;}}}else{r+=-0.692308;}}else{if(f[42]<1.02805){if(f[55]<0.252857){if(f[0]<1.39688e+11){r+=0.961165;}else{r+=-0.8;}}else{if(f[70]<1.01831){if(f[84]<1.5){if(f[23]<3.90384e+08){r+=0.235294;}else{r+=-0.784615;}}else{if(f[15]<3.94933e+08){r+=-0.994192;}else{r+=-0.28;}}}else{if(f[33]<1.09535e+08){r+=0.870968;}else{r+=-0.555556;}}}}else{if(f[28]<1.40062e+11){r+=-0.333333;}else{r+=0.961165;}}}}}}else{if(f[0]<1.39813e+11){if(f[84]<0.5){if(f[28]<1.40545e+11){r+=-0.995785;}else{if(f[27]<7.88295e+08){if(f[56]<0.998648){r+=-0.833333;}else{if(f[1]<4.16055e+08){r+=0.999416;}else{r+=0.25;}}}else{if(f[70]<1.00646){if(f[0]<1.39806e+11){r+=0.913043;}else{r+=0.25;}}else{if(f[14]<1.4415e+11){r+=-0.954545;}else{r+=0.555556;}}}}}else{if(f[28]<1.41436e+11){if(f[40]<1.53841e+07){if(f[0]<1.39751e+11){if(f[71]<0.00463014){r+=-0;}else{if(f[29]<1.96878e+07){r+=-0.987988;}else{if(f[25]<3.43123e+07){r+=-0.6;}else{r+=0.25;}}}}else{if(f[84]<1.35){r+=-0.956989;}else{if(f[14]<1.40148e+11){r+=-0.5;}else{r+=0.982906;}}}}else{if(f[0]<1.39784e+11){if(f[27]<2.10833e+10){if(f[0]<1.39742e+11){if(f[43]<13.9577){r+=-0.998872;}else{r+=-0.80303;}}else{if(f[39]<5.83455e+07){r+=-0.186992;}else{r+=-0.98349;}}}else{if(f[66]<3.35236){if(f[29]<1.27153e+09){r+=-0.833333;}else{r+=0.111111;}}else{r+=0.5;}}}else{if(f[28]<1.40668e+11){if(f[28]<1.40064e+11){if(f[82]<0.2391){r+=-0.230769;}else{r+=-0.951515;}}else{if(f[84]<1.85){r+=-0.484536;}else{r+=0.761589;}}}else{if(f[10]<2.7217e+09){r+=-0.998815;}else{r+=-0.25;}}}}}else{if(f[70]<1.01978){if(f[15]<7.64946e+08){r+=0.999282;}else{r+=-0.25;}}else{if(f[45]<0.496616){r+=-0;}else{r+=-0.973154;}}}}}else{if(f[0]<1.42933e+11){if(f[14]<1.39991e+11){if(f[28]<1.39604e+11){if(f[12]<2.88928e+08){if(f[0]<1.40231e+11){r+=0.25;}else{if(f[28]<1.39435e+11){r+=0.25;}else{r+=0.985348;}}}else{r+=-0.5;}}else{if(f[14]<1.39988e+11){if(f[28]<1.77991e+11){if(f[27]<2.30292e+10){if(f[82]<4.6282){r+=-0.990599;}else{r+=-0.272727;}}else{if(f[0]<1.40491e+11){r+=-0.333333;}else{r+=0.6;}}}else{r+=0.636364;}}else{if(f[28]<1.39999e+11){if(f[0]<1.40134e+11){r+=-0.996161;}else{r+=-0.25;}}else{if(f[28]<1.40014e+11){r+=0.983539;}else{if(f[71]<6.69373){r+=-0.979167;}else{r+=-0;}}}}}}else{if(f[28]<1.40714e+11){if(f[0]<1.40089e+11){if(f[28]<1.4e+11){if(f[14]<1.4012e+11){if(f[0]<1.40077e+11){r+=-0.98608;}else{r+=0.496241;}}else{if(f[84]<2.15){r+=0.788847;}else{r+=-0.970658;}}}else{if(f[84]<1.15){if(f[14]<1.40471e+11){r+=-0.272955;}else{r+=0.801673;}}else{if(f[84]<3.15){r+=0.769358;}else{r+=-0.99827;}}}}else{if(f[14]<1.40144e+11){if(f[84]<1.5){if(f[31]<3.57228e+08){r+=-0.981809;}else{r+=0.857924;}}else{if(f[28]<1.40004e+11){r+=0.401855;}else{r+=-0.44935;}}}else{if(f[28]<1.39809e+11){if(f[42]<1.00856){r+=-0.97653;}else{r+=0.982183;}}else{if(f[28]<1.40411e+11){r+=0.456933;}else{r+=-0.200761;}}}}}else{if(f[0]<1.40187e+11){if(f[37]<2.26706e+07){if(f[84]<2.5){r+=-0.928571;}else{if(f[0]<1.40033e+11){r+=-0.25;}else{r+=0.987097;}}}else{if(f[0]<1.40174e+11){if(f[37]<1.1963e+10){r+=-0.990773;}else{r+=0.428571;}}else{if(f[84]<6.00001){r+=-0.964158;}else{r+=0.95;}}}}else{if(f[28]<1.40917e+11){if(f[84]<4.5){if(f[66]<0.609059){r+=-0.0318471;}else{r+=-0.987186;}}else{r+=0.979798;}}else{if(f[14]<1.41e+11){if(f[22]<1.48581e+08){r+=0.65782;}else{r+=-0.322968;}}else{if(f[70]<1.01258){r+=-0.974394;}else{r+=0.868778;}}}}}}}else{if(f[84]<0.5){if(f[28]<1.42695e+11){r+=-0.538462;}else{r+=0.966942;}}else{if(f[70]<0.980519){if(f[63]<23.4506){if(f[43]<11.7394){if(f[56]<0.9954){r+=-0;}else{if(f[70]<0.980348){r+=-0.998063;}else{r+=-0.588235;}}}else{if(f[84]<2.65){r+=-0.971014;}else{r+=0.851852;}}}else{r+=0.4;}}else{if(f[84]<0.85){if(f[14]<1.41314e+11){r+=0.857143;}else{r+=-0;}}else{if(f[36]<3.6818e+10){r+=-0.869565;}else{r+=-0.111111;}}}}}}}}else{if(f[28]<1.81243e+11){if(f[68]<7.66731){if(f[28]<1.39913e+11){if(f[84]<2.15){if(f[56]<1.00442){if(f[0]<1.4025e+11){if(f[56]<1.00154){r+=-0;}else{if(f[15]<1.24038e+08){r+=0.993162;}else{r+=0.25;}}}else{if(f[19]<2.69591e+07){r+=-0.6;}else{r+=0.636364;}}}else{r+=-0.714286;}}else{r+=-0.984615;}}else{if(f[68]<4.27133){if(f[0]<1.39914e+11){if(f[84]<7.40001){if(f[68]<4.05757){if(f[68]<2.0498){r+=-0.2;}else{r+=-0.846154;}}else{if(f[0]<1.3909e+11){r+=0.555556;}else{r+=-0;}}}else{if(f[0]<1.3982e+11){r+=-0;}else{r+=0.955556;}}}else{if(f[1]<7.53625e+06){if(f[0]<1.40107e+11){if(f[4]<1.06417e+07){r+=-0.888889;}else{r+=0.636364;}}else{if(f[0]<1.40312e+11){if(f[0]<1.40227e+11){r+=0.916667;}else{r+=0.25;}}else{r+=-0.6;}}}else{if(f[0]<1.42063e+11){if(f[56]<1.00524){if(f[39]<1.17494e+09){r+=-0.866667;}else{r+=-0.142857;}}else{if(f[55]<0.813671){r+=-0.25;}else{r+=0.636364;}}}else{if(f[0]<1.43588e+11){r+=0.714286;}else{r+=-0;}}}}}else{if(f[74]<87.188){if(f[0]<1.39906e+11){if(f[70]<1.0055){if(f[56]<1.00174){if(f[0]<1.39821e+11){r+=0.25;}else{r+=0.943662;}}else{r+=-0.6;}}else{if(f[36]<2.48214e+08){if(f[43]<0.498464){r+=-0.2;}else{r+=-0.97546;}}else{if(f[68]<5.25895){r+=0.636364;}else{r+=-0.555556;}}}}else{if(f[10]<5.59461e+09){if(f[5]<219358){if(f[14]<1.40316e+11){r+=-0.5;}else{r+=0.25;}}else{if(f[0]<1.41525e+11){r+=-0.984221;}else{r+=-0.5625;}}}else{if(f[59]<3.48138){r+=-0.727273;}else{if(f[1]<1.75443e+08){r+=-0;}else{r+=0.6;}}}}}else{r+=0.7;}}}}else{if(f[13]<5.49372e+09){if(f[71]<0.392777){if(f[64]<1.06988){if(f[40]<2.03212e+07){r+=0.538462;}else{r+=-0.818182;}}else{if(f[66]<3.73325){if(f[28]<1.40113e+11){if(f[7]<6.86823e+07){r+=0.666667;}else{r+=-0;}}else{r+=-0.904762;}}else{if(f[68]<8.59502){if(f[28]<1.3994e+11){if(f[14]<1.4039e+11){r+=0.111111;}else{r+=0.885714;}}else{if(f[8]<2.65227e+09){r+=-0.982906;}else{r+=-0.25;}}}else{if(f[41]<5.89953e+09){r+=-0.995841;}else{r+=-0;}}}}}else{if(f[28]<1.39914e+11){if(f[84]<2.15){if(f[0]<1.40235e+11){if(f[6]<2.53864e+06){r+=0.25;}else{r+=0.96;}}else{r+=-0.818182;}}else{r+=-0.846154;}}else{if(f[37]<2.20026e+10){if(f[28]<1.40115e+11){if(f[83]<0.419901){if(f[70]<0.99946){r+=-0.833333;}else{r+=0.622642;}}else{if(f[49]<1.06616){r+=-0.98556;}else{r+=-0;}}}else{if(f[8]<224731){r+=0.5;}else{if(f[27]<5.98239e+10){r+=-0.984515;}else{r+=-0;}}}}else{r+=0.4;}}}}else{if(f[55]<0.813125){if(f[84]<0.5){if(f[6]<1.09648e+07){r+=0.25;}else{r+=0.888889;}}else{if(f[27]<1.48961e+10){r+=0.6;}else{r+=-0.666667;}}}else{if(f[18]<2.37904e+08){if(f[30]<4.47856e+07){r+=-0.931035;}else{r+=-0.25;}}else{r+=0.25;}}}}}else{if(f[14]<1.40084e+11){r+=0.989529;}else{r+=-0.981818;}}}}} return r;}
    static double p10(double[] f){double r=0;if(f[69]<0.47824){if(f[84]<1.15){if(f[14]<1.4201e+11){if(f[59]<0.0140546){if(f[38]<1.21517e+10){if(f[14]<1.40096e+11){if(f[61]<0.267425){r+=0.683475;}else{r+=0.122901;}}else{r+=-0.0543096;}}else{if(f[28]<1.46012e+11){r+=-0.681426;}else{r+=0.562458;}}}else{if(f[84]<0.85){if(f[0]<1.40104e+11){if(f[0]<1.40071e+11){if(f[14]<1.40442e+11){if(f[28]<1.39932e+11){if(f[28]<1.39621e+11){r+=-0.673369;}else{if(f[21]<2.63606e+07){r+=-0.482;}else{r+=0.672395;}}}else{if(f[71]<0.0841321){if(f[20]<4.35146e+06){r+=0.718745;}else{r+=-0.582791;}}else{if(f[0]<1.3123e+11){r+=0.608603;}else{if(f[0]<1.40067e+11){r+=-0.668999;}else{r+=-0.0983247;}}}}}else{if(f[56]<0.999702){if(f[12]<5.29056e+08){if(f[0]<1.3975e+11){r+=0.187743;}else{r+=0.679917;}}else{r+=-0.563913;}}else{r+=-0.674779;}}}else{if(f[56]<0.998316){if(f[57]<1.44005){if(f[1]<3.53373e+07){r+=0.0455491;}else{r+=0.191875;}}else{r+=-0.680716;}}else{if(f[14]<1.40073e+11){if(f[12]<5.76205e+08){if(f[0]<1.40101e+11){if(f[29]<1.3907e+06){r+=0.173834;}else{r+=0.682459;}}else{r+=0.173595;}}else{r+=-0.343784;}}else{r+=-0.580403;}}}}else{if(f[28]<1.42763e+11){if(f[83]<1.24021){if(f[14]<1.41135e+11){if(f[27]<8.88544e+08){if(f[33]<1.09013e+08){r+=-0.679341;}else{r+=0.0542991;}}else{if(f[43]<5.17921){r+=-0.538764;}else{r+=0.646697;}}}else{if(f[56]<1.00795){r+=0.662984;}else{r+=-0.546344;}}}else{if(f[0]<2.43484e+11){if(f[21]<2.89872e+08){if(f[42]<0.993067){if(f[1]<1.08043e+08){r+=0.315354;}else{r+=-0.520474;}}else{if(f[71]<5.8419e+06){r+=-0.679512;}else{r+=-0.142074;}}}else{if(f[0]<1.42795e+11){if(f[26]<1.57421e+08){r+=-0.221873;}else{r+=-0.676942;}}else{if(f[28]<1.40158e+11){r+=-0.324835;}else{r+=0.908239;}}}}else{r+=0.531807;}}}else{if(f[25]<1.96587e+10){r+=0.671184;}else{r+=-0.600141;}}}}else{if(f[28]<1.39954e+11){if(f[36]<3.56639e+06){r+=-0.141318;}else{if(f[2]<2.44304e+08){r+=-0.685594;}else{r+=-0.12674;}}}else{if(f[56]<1.00011){if(f[59]<0.109897){r+=0.0944131;}else{if(f[28]<1.40115e+11){r+=-0.141898;}else{r+=-0.682232;}}}else{if(f[27]<6.96104e+07){if(f[0]<1.40374e+11){r+=-0.680872;}else{r+=0.516611;}}else{if(f[28]<1.4042e+11){if(f[42]<0.990587){r+=-0.141061;}else{if(f[77]<0.135755){r+=0.0313056;}else{if(f[36]<6.01756e+08){r+=0.67616;}else{r+=0.118963;}}}}else{r+=-0.551111;}}}}}}}else{if(f[54]<6.45817){if(f[28]<1.41224e+11){r+=-0.184674;}else{if(f[28]<1.41304e+11){if(f[1]<6.73712e+07){r+=0.523196;}else{r+=-0.200539;}}else{r+=0.683681;}}}else{r+=-0.559949;}}}else{if(f[83]<2.42531){if(f[23]<9.82239e+07){if(f[39]<5.43203e+07){if(f[28]<1.40299e+11){if(f[34]<6.77435e+06){if(f[56]<1.00176){if(f[28]<1.4005e+11){r+=-0.595388;}else{r+=0.346866;}}else{if(f[6]<5.18215e+07){r+=-0.678058;}else{r+=-0.206616;}}}else{if(f[56]<1.00185){if(f[27]<4.60393e+07){r+=0.123368;}else{r+=0.716918;}}else{r+=-0.552317;}}}else{if(f[84]<2.15){if(f[50]<0.731451){r+=-0.106651;}else{r+=-0.647046;}}else{if(f[28]<1.40413e+11){r+=0.714194;}else{r+=-0.393391;}}}}else{if(f[56]<1.00324){if(f[14]<1.40564e+11){if(f[56]<1.00197){if(f[27]<6.45612e+08){if(f[28]<1.39936e+11){if(f[84]<1.85){if(f[14]<1.40062e+11){r+=0.736886;}else{r+=-0.5518;}}else{r+=-0.665488;}}else{if(f[52]<0.753683){if(f[8]<6.13786e+07){r+=-0.537622;}else{r+=0.814524;}}else{if(f[56]<1.00164){r+=-0.662019;}else{r+=-0.0571959;}}}}else{if(f[14]<1.39831e+11){r+=0.674923;}else{if(f[79]<1.52816){r+=-0.629631;}else{if(f[64]<0.699336){r+=-0.565546;}else{r+=0.739416;}}}}}else{if(f[27]<1.85903e+08){if(f[60]<0.453967){r+=-0.340908;}else{if(f[57]<3.64758){if(f[31]<1.45083e+06){r+=0.692386;}else{r+=1.44888;}}else{r+=-0.1184;}}}else{if(f[55]<1.6545){r+=-0.572881;}else{r+=0.00842038;}}}}else{if(f[0]<1.40384e+11){if(f[0]<1.40017e+11){r+=-0.530012;}else{if(f[84]<1.8){if(f[8]<8.2778e+07){r+=-0.544831;}else{r+=0.700652;}}else{if(f[67]<0.698273){if(f[28]<1.40386e+11){r+=1.26585;}else{r+=0.68348;}}else{r+=0.641373;}}}}else{if(f[42]<1.00037){r+=-0.732821;}else{r+=-0.135753;}}}}else{r+=-0.690893;}}}else{if(f[14]<1.40996e+11){if(f[0]<1.40033e+11){if(f[28]<1.40429e+11){if(f[10]<7.5331e+09){if(f[30]<165176){r+=-0.0974271;}else{r+=-0.675552;}}else{if(f[84]<2.5){if(f[84]<1.85){r+=-0.108679;}else{r+=-0.437619;}}else{r+=0.553234;}}}else{if(f[28]<1.40628e+11){if(f[0]<1.39772e+11){r+=-0.64258;}else{if(f[34]<7.421e+07){r+=0.792437;}else{r+=0.0790079;}}}else{if(f[0]<1.39363e+11){if(f[0]<1.39119e+11){r+=-0.256335;}else{r+=0.580262;}}else{r+=-0.686494;}}}}else{if(f[14]<1.40783e+11){if(f[56]<1.00123){if(f[83]<0.985305){if(f[42]<1.0023){if(f[84]<2.5){if(f[20]<7.77874e+07){r+=-0.75212;}else{r+=0.0897864;}}else{if(f[3]<5.77088e+06){r+=-0.0627793;}else{r+=0.661181;}}}else{if(f[83]<0.61895){if(f[14]<1.39309e+11){r+=0.292882;}else{r+=-0.617719;}}else{if(f[25]<8.2067e+07){r+=-0.0604722;}else{r+=0.673873;}}}}else{if(f[36]<2.14733e+08){if(f[0]<1.40196e+11){if(f[56]<0.997629){r+=0.467113;}else{r+=-0.451859;}}else{if(f[42]<1.00118){r+=-1.23966;}else{r+=-0.626356;}}}else{if(f[26]<1.61517e+08){if(f[51]<1.15152){r+=-0.228345;}else{r+=0.470888;}}else{if(f[14]<1.29454e+11){r+=0.477139;}else{r+=-0.509109;}}}}}else{if(f[56]<1.00414){if(f[84]<2.5){if(f[84]<1.5){if(f[14]<1.40033e+11){r+=0.715478;}else{r+=-1.27082;}}else{if(f[14]<1.40385e+11){r+=0.27844;}else{r+=0.714555;}}}else{if(f[42]<1.00035){r+=0.407699;}else{r+=-0.80308;}}}else{if(f[28]<1.39926e+11){if(f[42]<1.00865){if(f[1]<1.97327e+08){r+=-0.71457;}else{r+=-0.143993;}}else{if(f[17]<2.9684e+07){r+=0.711355;}else{r+=-0.186284;}}}else{if(f[21]<2.03035e+08){r+=0.832424;}else{r+=0.191892;}}}}}else{if(f[27]<1.58046e+08){r+=-0.681199;}else{if(f[12]<1.03307e+08){r+=-0.851444;}else{if(f[28]<1.41014e+11){r+=-0.499413;}else{if(f[13]<1.13633e+09){r+=0.717753;}else{r+=0.121738;}}}}}}}else{if(f[84]<12){if(f[30]<1.13904e+09){if(f[50]<0.448011){if(f[57]<4.03414){r+=-0.41866;}else{r+=0.349651;}}else{if(f[12]<9.09789e+07){if(f[84]<4.5){r+=-0.646303;}else{r+=0.441803;}}else{if(f[67]<2.26679){r+=-0.683739;}else{r+=-0.126836;}}}}else{r+=0.31557;}}else{if(f[9]<1.8096e+08){if(f[52]<1.26148){r+=0.701836;}else{r+=0.136925;}}else{if(f[44]<1.66552){r+=-0.131034;}else{r+=-0.577485;}}}}}}else{if(f[41]<5.50934e+08){if(f[55]<0.442927){if(f[84]<1.85){r+=0.67269;}else{r+=-0.523166;}}else{if(f[0]<1.40025e+11){if(f[78]<0.471485){if(f[65]<0.73238){if(f[1]<1.28298e+07){r+=0.118858;}else{r+=0.734571;}}else{if(f[60]<0.544071){r+=0.194152;}else{r+=-0.601732;}}}else{if(f[14]<1.38632e+11){if(f[14]<1.29306e+11){r+=-0.459511;}else{r+=0.561561;}}else{if(f[28]<1.41388e+11){if(f[66]<1.68048){if(f[63]<2.28435){if(f[29]<888858){r+=-0.142387;}else{r+=-0.678169;}}else{if(f[14]<1.40319e+11){r+=0.214463;}else{r+=-0.546684;}}}else{if(f[1]<1.0362e+07){r+=-0.490686;}else{r+=0.3142;}}}else{if(f[1]<1.99067e+07){r+=-0.594832;}else{if(f[0]<1.39241e+11){r+=0.716711;}else{r+=-0.226531;}}}}}}else{if(f[0]<1.40235e+11){if(f[28]<1.40346e+11){if(f[8]<1.12563e+08){if(f[55]<0.854986){if(f[0]<1.40222e+11){if(f[3]<6.84948e+06){r+=-0.573547;}else{r+=0.0842927;}}else{r+=0.728689;}}else{if(f[39]<3.97235e+08){if(f[4]<236708){r+=-0.179826;}else{r+=-0.653005;}}else{r+=0.0238055;}}}else{if(f[66]<0.631559){if(f[10]<2.52993e+08){r+=0.695239;}else{r+=-0.247076;}}else{if(f[28]<1.40161e+11){if(f[39]<2.51938e+08){r+=-0.783418;}else{r+=0.41691;}}else{r+=0.503477;}}}}else{if(f[14]<1.40612e+11){if(f[14]<1.40511e+11){if(f[41]<3.2273e+08){r+=0.600825;}else{r+=-0.790963;}}else{if(f[14]<1.40513e+11){r+=0.822773;}else{if(f[84]<2.15){r+=-1.04006;}else{r+=0.607434;}}}}else{if(f[84]<2.15){if(f[14]<1.40617e+11){r+=-1.07224;}else{r+=-0.668417;}}else{if(f[1]<1.05793e+06){r+=-0.513355;}else{r+=0.658167;}}}}}else{if(f[37]<1.82214e+08){if(f[12]<1.17616e+08){if(f[39]<2.95911e+08){if(f[69]<0.425674){if(f[38]<2.70481e+08){r+=-0.680054;}else{r+=0.0408622;}}else{if(f[23]<1.25937e+08){r+=-0.540875;}else{r+=0.243337;}}}else{if(f[42]<1.0039){r+=0.637568;}else{r+=-0.118946;}}}else{if(f[0]<1.41709e+11){if(f[12]<1.18399e+08){r+=0.0403043;}else{if(f[6]<8.39756e+07){r+=-0.722305;}else{r+=-0.227012;}}}else{if(f[0]<1.41906e+11){r+=0.668202;}else{r+=-0.561201;}}}}else{if(f[42]<1.0021){r+=-0.65629;}else{if(f[52]<1.00197){if(f[42]<1.00841){r+=0.539656;}else{if(f[9]<2.53477e+08){r+=-0.840479;}else{r+=0.235814;}}}else{if(f[14]<1.3887e+11){r+=-0.469325;}else{if(f[25]<7.40342e+07){r+=-0.254125;}else{r+=0.63035;}}}}}}}}}else{if(f[41]<7.97006e+08){if(f[37]<1.14817e+08){if(f[28]<1.40356e+11){if(f[50]<3.23543){if(f[83]<6.54657){r+=-0.681861;}else{if(f[0]<1.40101e+11){r+=0.259654;}else{r+=-0.538994;}}}else{if(f[34]<9.32982e+06){r+=-0.512944;}else{r+=0.379438;}}}else{if(f[56]<1.00124){if(f[0]<1.41604e+11){if(f[52]<0.737155){r+=-0.190952;}else{r+=-0.675547;}}else{r+=0.507894;}}else{if(f[84]<1.8){r+=-0.388493;}else{r+=0.605578;}}}}else{if(f[42]<1.00234){if(f[84]<12){if(f[12]<8.08854e+07){if(f[53]<0.985029){if(f[13]<1.16238e+08){r+=0.0773942;}else{r+=0.498552;}}else{r+=-0.468599;}}else{if(f[14]<1.39666e+11){r+=-0.0241267;}else{if(f[55]<0.508008){if(f[18]<3.05612e+07){r+=-0.419019;}else{r+=0.214772;}}else{if(f[27]<3.37212e+08){r+=-0.684838;}else{r+=-0.356569;}}}}}else{if(f[14]<1.40403e+11){if(f[68]<0.118159){r+=0.0867575;}else{r+=-0.676766;}}else{if(f[0]<1.40244e+11){if(f[46]<6.73832){r+=0.680257;}else{r+=0.0769153;}}else{r+=-0.474407;}}}}else{if(f[14]<1.40728e+11){if(f[0]<1.40793e+11){if(f[13]<2.60877e+08){if(f[56]<0.991232){r+=-0.553766;}else{if(f[36]<1.47865e+08){r+=-0.29916;}else{r+=0.661968;}}}else{if(f[53]<1.17752){r+=-0.797004;}else{r+=-0.0706942;}}}else{if(f[66]<1.03962){if(f[43]<4.28184){if(f[60]<0.130072){r+=-0.0149898;}else{r+=-0.688042;}}else{r+=0.0565614;}}else{if(f[36]<1.98962e+08){r+=-0.524433;}else{if(f[2]<3.98394e+07){r+=0.678693;}else{r+=0.0756016;}}}}}else{if(f[0]<1.41973e+11){if(f[14]<1.40944e+11){r+=0.699985;}else{r+=0.0764117;}}else{r+=-0.602856;}}}}}else{if(f[7]<641133){if(f[0]<1.40202e+11){r+=-0.569199;}else{r+=0.678491;}}else{if(f[28]<1.40589e+11){if(f[14]<1.38992e+11){if(f[9]<2.2737e+08){r+=-0.673318;}else{if(f[4]<3.54696e+07){r+=0.635646;}else{r+=0.00224736;}}}else{if(f[15]<1.59067e+09){if(f[68]<2.02163){r+=-0.679715;}else{r+=-0.0502686;}}else{r+=0.0965035;}}}else{if(f[84]<12){if(f[13]<1.27154e+08){if(f[25]<9.80412e+07){r+=-0.437788;}else{if(f[6]<2.3238e+07){r+=0.884528;}else{r+=0.233902;}}}else{if(f[0]<1.39262e+11){if(f[2]<5.08009e+07){r+=0.342137;}else{r+=-0.37258;}}else{if(f[8]<1.89956e+07){r+=0.0872073;}else{if(f[13]<1.47022e+08){r+=-0.312775;}else{r+=-0.673015;}}}}}else{if(f[42]<0.994609){if(f[12]<6.48837e+08){r+=0.673538;}else{r+=-0.619075;}}else{if(f[29]<1.09118e+07){r+=0.47839;}else{if(f[42]<0.995493){if(f[0]<1.40388e+11){r+=0.767322;}else{r+=-0.60325;}}else{r+=-0.680212;}}}}}}}}}}}else{if(f[28]<1.39442e+11){if(f[56]<1.00186){if(f[41]<9.90812e+07){r+=0.698423;}else{if(f[56]<1.00117){if(f[42]<1.18277){r+=-0.652742;}else{r+=0.165656;}}else{if(f[78]<1.00884){r+=-0.532106;}else{r+=0.696272;}}}}else{if(f[55]<0.276313){if(f[84]<1.15){if(f[27]<7.31341e+08){r+=-0.0988557;}else{r+=-0.752755;}}else{if(f[14]<1.40164e+11){r+=0.700345;}else{r+=-0.564973;}}}else{if(f[38]<6.07546e+10){if(f[9]<2.12886e+08){if(f[28]<1.39431e+11){if(f[28]<1.39429e+11){r+=-0.684073;}else{if(f[74]<0.569672){r+=0.472361;}else{if(f[65]<0.860195){r+=0.0647338;}else{if(f[1]<6.14766e+07){r+=-0.710745;}else{r+=-0.156932;}}}}}else{if(f[84]<0.85){if(f[0]<1.40736e+11){r+=-1.5542;}else{r+=-0.537931;}}else{if(f[14]<1.39908e+11){r+=0.561587;}else{r+=-0.708055;}}}}else{if(f[27]<1.77907e+08){if(f[42]<1.00705){if(f[84]<1.15){if(f[28]<1.39432e+11){r+=-0.673459;}else{r+=-1.04414;}}else{if(f[14]<1.40061e+11){r+=0.721339;}else{r+=-0.668917;}}}else{if(f[71]<1.79491){r+=1.45295;}else{r+=0.612432;}}}else{if(f[12]<1.05666e+08){if(f[84]<1.15){r+=-0.669366;}else{if(f[14]<1.40116e+11){r+=0.708573;}else{r+=-0.470712;}}}else{if(f[25]<6.07526e+10){r+=-0.683423;}else{r+=-0.139031;}}}}}else{if(f[65]<0.865059){r+=-0.426373;}else{if(f[12]<5.37096e+10){r+=0.499785;}else{r+=-0.137593;}}}}}}else{if(f[68]<6.88305){if(f[14]<1.39854e+11){if(f[14]<1.39536e+11){if(f[28]<1.40244e+11){if(f[28]<1.40019e+11){if(f[28]<1.39695e+11){if(f[42]<1.00547){r+=0.691572;}else{if(f[0]<1.43799e+11){if(f[28]<1.39689e+11){r+=-0.687961;}else{r+=-1.42018;}}else{if(f[68]<1.2281){r+=0.659487;}else{r+=0.123919;}}}}else{if(f[79]<1.17221){if(f[37]<7.23931e+07){r+=-0.671229;}else{if(f[25]<4.96806e+09){if(f[27]<1.02117e+08){r+=-0.377865;}else{r+=0.638809;}}else{if(f[53]<0.691336){r+=-0.174249;}else{r+=-1.47295;}}}}else{if(f[84]<1.85){if(f[9]<6.58006e+07){r+=-0.661428;}else{if(f[25]<5.55091e+09){r+=0.65367;}else{r+=-0.825904;}}}else{if(f[9]<1.24992e+08){if(f[9]<7.5025e+07){r+=-0.729542;}else{r+=0.629407;}}else{if(f[9]<2.56283e+08){r+=-0.745724;}else{r+=0.658195;}}}}}}else{if(f[84]<1.5){if(f[0]<1.3798e+11){if(f[2]<4.91658e+06){r+=0.131961;}else{r+=0.678458;}}else{if(f[57]<14.2475){if(f[2]<7.96696e+08){if(f[78]<2.09748e+07){r+=-0.684364;}else{r+=-0.166264;}}else{r+=0.298986;}}else{if(f[14]<1.38229e+11){r+=-0.370528;}else{r+=0.637043;}}}}else{if(f[84]<2.15){if(f[56]<0.995481){if(f[39]<8.71871e+07){if(f[28]<1.40157e+11){r+=-0.653544;}else{r+=0.646672;}}else{if(f[14]<1.39368e+11){r+=0.65807;}else{r+=0.368718;}}}else{r+=-0.693158;}}else{if(f[9]<2.50325e+08){if(f[55]<0.829296){if(f[26]<1.35657e+08){r+=0.433565;}else{r+=-0.682496;}}else{if(f[9]<2.40379e+08){r+=-0.687051;}else{r+=-0.308385;}}}else{if(f[28]<1.40091e+11){if(f[84]<2.85){r+=-0.61354;}else{r+=0.671001;}}else{r+=-0.849681;}}}}}}else{if(f[0]<1.42937e+11){if(f[42]<1.00155){if(f[0]<1.39332e+11){if(f[83]<1.7123){r+=-0.668571;}else{r+=-0.130771;}}else{r+=0.693366;}}else{if(f[14]<1.39528e+11){if(f[28]<1.40277e+11){if(f[42]<1.00761){if(f[28]<1.40255e+11){r+=0.622783;}else{r+=1.59188;}}else{r+=-0.681266;}}else{if(f[9]<4.29912e+10){if(f[74]<78.2504){r+=-0.684947;}else{r+=-0.149056;}}else{if(f[46]<0.629976){r+=0.362686;}else{r+=-0.426147;}}}}else{if(f[36]<2.12718e+08){if(f[84]<3.35){if(f[66]<0.736517){r+=-0.604668;}else{r+=-1.59785;}}else{r+=-0.565139;}}else{r+=-0.680285;}}}}else{if(f[14]<1.39512e+11){if(f[29]<2.13419e+08){if(f[78]<0.198785){r+=0.568033;}else{if(f[14]<1.39051e+11){r+=0.3038;}else{if(f[14]<1.39509e+11){r+=-0.682329;}else{r+=-0.164479;}}}}else{if(f[0]<1.45898e+11){r+=1.10275;}else{r+=-0.0508832;}}}else{if(f[23]<1.28597e+08){r+=-0.374848;}else{r+=0.692772;}}}}}else{if(f[38]<1.4848e+08){if(f[84]<3.15){if(f[65]<0.646267){if(f[0]<1.40761e+11){if(f[25]<6.89281e+07){if(f[66]<0.853947){r+=-0.705066;}else{r+=0.114112;}}else{if(f[84]<1){if(f[39]<9.58683e+07){r+=-0.909847;}else{r+=0.302802;}}else{if(f[66]<0.735167){r+=0.0318559;}else{r+=0.670258;}}}}else{if(f[63]<0.322954){r+=-0.67301;}else{if(f[66]<0.734939){r+=-0.571361;}else{r+=-1.56647;}}}}else{if(f[25]<6.9121e+07){if(f[69]<0.865751){if(f[42]<0.996685){r+=0.504243;}else{if(f[28]<1.3973e+11){r+=0.441982;}else{r+=-1.29719;}}}else{if(f[26]<6.10712e+07){if(f[0]<1.39288e+11){r+=0.491426;}else{r+=-1.30106;}}else{if(f[10]<2.87397e+08){r+=0.399009;}else{r+=-1.00451;}}}}else{if(f[0]<1.43565e+11){if(f[0]<1.40848e+11){if(f[56]<0.993193){r+=0.496055;}else{r+=0.67748;}}else{if(f[14]<1.39614e+11){r+=-1.32387;}else{r+=0.372026;}}}else{if(f[7]<3.01848e+07){r+=-0.982662;}else{if(f[9]<2.89461e+08){r+=0.707477;}else{r+=0.818558;}}}}}}else{r+=-0.678652;}}else{if(f[28]<1.40822e+11){if(f[84]<2.85){if(f[66]<0.322292){if(f[36]<2.46719e+08){if(f[0]<1.4028e+11){r+=0.415952;}else{if(f[27]<1.16952e+08){r+=-0.197129;}else{r+=-0.69182;}}}else{if(f[66]<0.17844){r+=-0.241347;}else{r+=0.72533;}}}else{if(f[0]<1.40625e+11){if(f[84]<1.15){if(f[14]<1.39631e+11){r+=0.703212;}else{r+=-1.23813;}}else{if(f[36]<1.74715e+09){r+=0.709707;}else{r+=-0.431778;}}}else{if(f[0]<1.43316e+11){if(f[8]<3.75033e+08){r+=-1.44757;}else{r+=-0.751089;}}else{if(f[28]<1.40062e+11){r+=-0.872488;}else{r+=0.706132;}}}}}else{if(f[66]<0.736252){r+=-0.682844;}else{if(f[36]<2.48288e+08){if(f[14]<1.39825e+11){if(f[65]<2.07311){r+=-1.54959;}else{r+=-0.450741;}}else{r+=-0.278286;}}else{if(f[23]<3.64137e+08){if(f[23]<2.30238e+08){r+=-0.628621;}else{r+=0.668119;}}else{if(f[69]<3.60329){r+=-0.722267;}else{r+=0.0877784;}}}}}}else{if(f[28]<1.41873e+11){if(f[84]<1.5){if(f[0]<1.41626e+11){if(f[66]<0.737879){r+=-0.570086;}else{if(f[36]<2.95494e+08){r+=-1.65745;}else{r+=-0.763494;}}}else{r+=0.680395;}}else{if(f[0]<1.39355e+11){r+=0.625023;}else{if(f[0]<1.43046e+11){if(f[66]<0.737094){r+=-0.68273;}else{r+=-0.8677;}}else{if(f[55]<1.21069){r+=0.63729;}else{r+=-0.0559489;}}}}}else{if(f[84]<2.65){if(f[56]<0.983261){if(f[64]<0.593293){r+=-0.135984;}else{r+=-0.674411;}}else{if(f[38]<2.14629e+08){if(f[66]<0.738127){r+=-0.548616;}else{r+=-1.55763;}}else{r+=-0.656762;}}}else{if(f[84]<3.35){r+=0.712025;}else{r+=-0.660942;}}}}}}}else{if(f[0]<1.3996e+11){if(f[66]<1.96448){if(f[28]<1.41437e+11){if(f[39]<2.08495e+09){if(f[66]<1.52288){if(f[0]<1.39917e+11){if(f[38]<1.65932e+07){if(f[84]<1.5){r+=-0.762447;}else{r+=0.217248;}}else{if(f[14]<1.40579e+11){r+=-0.613492;}else{r+=-0.685857;}}}else{if(f[28]<1.40105e+11){if(f[84]<0.85){r+=0.908139;}else{r+=-0.740519;}}else{if(f[28]<1.4056e+11){r+=-1.19564;}else{r+=-0.674663;}}}}else{if(f[42]<0.993772){if(f[14]<1.45718e+11){if(f[14]<1.40145e+11){r+=0.619942;}else{r+=-0.685714;}}else{r+=0.646275;}}else{if(f[28]<1.40074e+11){if(f[56]<1.00342){r+=-0.682389;}else{r+=0.523212;}}else{if(f[0]<1.39902e+11){r+=0.37073;}else{r+=-1.00736;}}}}}else{if(f[0]<1.39773e+11){if(f[29]<1.18052e+09){if(f[0]<1.39769e+11){if(f[72]<0.067438){r+=-0.150128;}else{r+=-0.660003;}}else{r+=-0.0680978;}}else{if(f[36]<3.41126e+10){if(f[8]<7.38722e+09){r+=0.00940387;}else{r+=0.777957;}}else{r+=-0.483807;}}}else{if(f[28]<1.40534e+11){if(f[84]<1.5){if(f[53]<0.993593){r+=0.125478;}else{r+=-0.876049;}}else{if(f[53]<1.00659){r+=-1.11027;}else{r+=0.670403;}}}else{if(f[28]<1.41089e+11){if(f[14]<1.41166e+11){r+=0.732297;}else{r+=-0.249408;}}else{r+=-0.614011;}}}}}else{if(f[42]<0.985709){if(f[10]<3.11326e+09){r+=-0.683408;}else{r+=0.0668881;}}else{if(f[7]<4.85468e+09){r+=0.684;}else{r+=0.0171159;}}}}else{if(f[84]<2.5){if(f[14]<1.40221e+11){if(f[84]<2.15){if(f[0]<1.39957e+11){if(f[12]<8.23634e+09){if(f[64]<9.17264){r+=-0.690165;}else{r+=-0.277722;}}else{r+=0.102611;}}else{if(f[9]<2.9478e+07){r+=-0.420284;}else{r+=0.637424;}}}else{r+=0.431221;}}else{if(f[0]<1.39912e+11){if(f[28]<1.40074e+11){if(f[84]<1.5){if(f[56]<1.0032){r+=-0.808688;}else{r+=0.131232;}}else{if(f[42]<0.995829){r+=-0.667552;}else{r+=0.711503;}}}else{if(f[29]<2.72026e+07){if(f[0]<1.39448e+11){r+=-0.585563;}else{r+=0.540824;}}else{if(f[34]<3.91524e+07){r+=-0.519195;}else{r+=0.262535;}}}}else{if(f[28]<1.40185e+11){if(f[84]<4.60001){if(f[7]<1.35428e+09){r+=0.657173;}else{r+=-0.454142;}}else{r+=-0.650846;}}else{if(f[84]<4.60001){if(f[28]<1.40578e+11){r+=-1.04725;}else{r+=-0.66294;}}else{if(f[67]<1.81464){r+=-0.5092;}else{r+=0.814667;}}}}}}else{if(f[42]<0.995214){if(f[14]<1.40159e+11){if(f[1]<1.05631e+09){r+=0.813342;}else{r+=-0.21822;}}else{r+=-0.685538;}}else{if(f[28]<1.40195e+11){r+=-0.677945;}else{if(f[28]<1.40717e+11){r+=-1.55828;}else{r+=-0.589471;}}}}}}else{if(f[84]<0.5){if(f[28]<1.54186e+11){if(f[0]<1.4173e+11){if(f[14]<1.40285e+11){if(f[0]<1.40089e+11){r+=-0.876691;}else{if(f[14]<1.40144e+11){r+=-0.687151;}else{if(f[28]<1.39782e+11){r+=-0.68264;}else{r+=-1.24397;}}}}else{if(f[26]<1.27896e+08){if(f[56]<1.00247){if(f[0]<1.40067e+11){r+=-0.611228;}else{r+=-1.25816;}}else{if(f[56]<1.01251){r+=-0.746688;}else{r+=-1.21719;}}}else{if(f[28]<1.40666e+11){if(f[0]<1.40235e+11){r+=-1.32122;}else{r+=-0.826886;}}else{if(f[2]<7.79254e+08){r+=-0.718358;}else{r+=0.409823;}}}}}else{if(f[28]<1.42623e+11){if(f[42]<1.00012){r+=-0.13556;}else{r+=-0.635258;}}else{if(f[28]<1.4465e+11){r+=0.885513;}else{r+=0.700772;}}}}else{if(f[8]<3.39419e+08){if(f[21]<1.01077e+08){r+=0.773589;}else{if(f[67]<5.6836){r+=1.07737;}else{r+=0.311711;}}}else{r+=1.15382;}}}else{if(f[41]<9.25316e+07){if(f[14]<1.40126e+11){if(f[83]<2.44715){if(f[84]<1.85){if(f[41]<8.99671e+07){if(f[27]<7.72754e+07){r+=-0.947082;}else{r+=-0.684679;}}else{if(f[14]<1.40056e+11){r+=-0.830164;}else{r+=0.0767569;}}}else{if(f[67]<1.07167){if(f[84]<2.5){r+=-1.25327;}else{r+=-0.767065;}}else{if(f[0]<1.40077e+11){r+=-0.678385;}else{r+=-0.826782;}}}}else{if(f[56]<0.997649){if(f[0]<1.40106e+11){r+=0.706349;}else{r+=-0.402024;}}else{if(f[2]<3.49865e+06){r+=-0.190807;}else{r+=-0.933026;}}}}else{if(f[9]<1.65497e+08){if(f[84]<1.5){if(f[14]<1.4029e+11){if(f[79]<0.287717){r+=0.381479;}else{r+=-0.784429;}}else{if(f[37]<1.52063e+07){r+=0.687832;}else{r+=0.224298;}}}else{if(f[14]<1.40422e+11){if(f[84]<1.85){r+=-1.41661;}else{r+=0.330034;}}else{if(f[56]<1.03566){r+=-0.997227;}else{r+=0.61458;}}}}else{if(f[0]<1.4098e+11){if(f[56]<0.995334){if(f[14]<1.40947e+11){r+=0.708764;}else{r+=-0.511117;}}else{if(f[0]<1.40208e+11){r+=-0.572227;}else{r+=-1.00766;}}}else{if(f[84]<1.5){if(f[28]<1.40582e+11){r+=-0.153327;}else{r+=0.992304;}}else{if(f[14]<1.40283e+11){r+=0.793321;}else{r+=-0.996989;}}}}}}else{if(f[0]<1.40247e+11){if(f[0]<1.40161e+11){if(f[0]<1.40096e+11){if(f[28]<1.40685e+11){if(f[28]<1.39998e+11){r+=-0.354001;}else{r+=0.444129;}}else{if(f[38]<5.20393e+07){r+=0.772965;}else{r+=-0.684626;}}}else{if(f[26]<1.27915e+09){if(f[14]<1.40561e+11){r+=-0.520143;}else{r+=0.567459;}}else{if(f[84]<2.85){r+=0.0746131;}else{r+=1.27778;}}}}else{if(f[84]<1.5){if(f[28]<1.40004e+11){if(f[67]<1.13337){r+=0.581443;}else{r+=-0.791807;}}else{if(f[28]<1.42024e+11){r+=-0.927491;}else{r+=0.317333;}}}else{if(f[14]<1.40021e+11){if(f[84]<2.85){r+=-1.03996;}else{r+=0.440502;}}else{if(f[84]<1.85){r+=0.83168;}else{r+=0.405239;}}}}}else{if(f[84]<3.15){if(f[14]<1.40156e+11){if(f[56]<1.00191){if(f[0]<1.41549e+11){r+=-0.923597;}else{r+=-0.370272;}}else{if(f[39]<9.98588e+08){r+=0.648025;}else{r+=-0.804715;}}}else{if(f[55]<0.736339){if(f[66]<0.400993){r+=0.321475;}else{r+=-0.67393;}}else{if(f[8]<2.71713e+08){r+=0.194912;}else{r+=-0.544552;}}}}else{if(f[56]<1.00101){if(f[14]<1.40144e+11){r+=-0.791777;}else{if(f[8]<1.44802e+08){r+=-1.24633;}else{r+=-0.844127;}}}else{if(f[28]<1.39954e+11){r+=-0.680755;}else{if(f[28]<1.40369e+11){r+=0.724477;}else{r+=1.09638;}}}}}}}}}}else{if(f[28]<1.80984e+11){if(f[28]<1.40118e+11){if(f[57]<4.88029){if(f[39]<9.05394e+06){if(f[79]<0.468908){r+=-0.721299;}else{r+=0.465505;}}else{if(f[84]<2.15){if(f[53]<0.136698){if(f[41]<1.37693e+08){r+=0.241632;}else{r+=-0.607989;}}else{if(f[0]<1.4017e+11){if(f[53]<1.89588){if(f[78]<1.59342){r+=0.73263;}else{r+=-0.0865887;}}else{if(f[7]<6.38533e+07){r+=0.425763;}else{r+=-0.601034;}}}else{if(f[28]<1.39629e+11){if(f[71]<0.581319){r+=0.104353;}else{r+=0.634081;}}else{if(f[4]<1.97907e+06){r+=0.148165;}else{r+=-0.658037;}}}}}else{if(f[28]<1.40042e+11){if(f[1]<4.30965e+06){r+=-0.087059;}else{r+=-0.730355;}}else{if(f[56]<1.00185){if(f[42]<0.999621){if(f[26]<1.79012e+08){r+=0.0502652;}else{r+=0.860412;}}else{r+=-0.453102;}}else{r+=-0.6888;}}}}}else{if(f[0]<1.39919e+11){if(f[21]<4.83935e+07){r+=0.6103;}else{r+=-0.456989;}}else{if(f[13]<6.07611e+09){r+=-0.684923;}else{r+=-0.0780025;}}}}else{if(f[67]<3.60968){if(f[28]<1.40412e+11){if(f[15]<1.53147e+08){r+=-1.24662;}else{r+=-0.214889;}}else{if(f[28]<1.40713e+11){r+=-0.904546;}else{if(f[12]<1.5857e+09){r+=-0.67659;}else{r+=-0.143557;}}}}else{if(f[37]<6.5486e+08){if(f[7]<89878.3){r+=0.507567;}else{if(f[84]<12){if(f[38]<1.29122e+09){if(f[9]<5.77105e+09){r+=-0.684013;}else{if(f[1]<7.71744e+08){r+=0.119777;}else{r+=-0.471751;}}}else{r+=-0.144426;}}else{if(f[20]<2.37926e+07){r+=-0.676328;}else{if(f[19]<1.1899e+07){r+=1.04622;}else{if(f[20]<9.05855e+07){r+=-0.541258;}else{r+=0.426853;}}}}}}else{if(f[69]<4.06265){if(f[0]<1.41661e+11){r+=-0.754732;}else{r+=-0.20111;}}else{if(f[20]<1.95516e+08){if(f[51]<0.52481){r+=0.101465;}else{r+=-0.570478;}}else{if(f[6]<6.15549e+08){if(f[25]<4.97397e+10){r+=0.806192;}else{r+=0.218463;}}else{r+=-0.118294;}}}}}}}else{if(f[14]<1.40122e+11){r+=0.719085;}else{r+=-0.672177;}}}}} return r;}
    static double p11(double[] f){double r=0;if(f[68]<0.493536){if(f[14]<1.42267e+11){if(f[41]<6.13101e+08){if(f[36]<1.35603e+08){if(f[28]<1.40255e+11){if(f[31]<7.27064e+07){if(f[28]<1.39998e+11){if(f[0]<1.40229e+11){if(f[28]<1.39936e+11){if(f[42]<0.999606){if(f[42]<0.999483){if(f[48]<4.15395){r+=-0.617039;}else{r+=-0.166748;}}else{r+=-0.0539848;}}else{if(f[28]<1.39475e+11){r+=-0.472464;}else{if(f[23]<4.40985e+07){r+=0.0764502;}else{r+=0.646015;}}}}else{if(f[27]<3.21191e+08){if(f[14]<1.40172e+11){r+=-0.645738;}else{if(f[67]<0.438503){r+=-0.65354;}else{r+=0.492864;}}}else{if(f[34]<2.6103e+06){r+=0.153574;}else{r+=0.55222;}}}}else{if(f[42]<0.997894){if(f[28]<1.39934e+11){r+=-0.538692;}else{r+=0.619537;}}else{if(f[23]<1.94329e+08){if(f[22]<3.24342e+06){if(f[0]<1.40388e+11){r+=-0.437849;}else{r+=0.555227;}}else{if(f[66]<0.180004){r+=-0.756539;}else{r+=-0.562067;}}}else{if(f[70]<0.993359){r+=-0.490802;}else{if(f[56]<1.00095){r+=-0.503831;}else{r+=0.369386;}}}}}}else{if(f[38]<9.14321e+07){if(f[28]<1.40031e+11){if(f[68]<0.383768){if(f[41]<3.11613e+08){if(f[27]<3.49072e+07){r+=-0.59122;}else{r+=-1.07337;}}else{if(f[27]<1.81995e+08){r+=-0.638633;}else{r+=-1.25511;}}}else{if(f[42]<1.00029){if(f[27]<3.44654e+07){r+=-0.48868;}else{r+=-1.33722;}}else{if(f[31]<4.38511e+06){r+=-0.583894;}else{r+=-0.167646;}}}}else{if(f[28]<1.40132e+11){if(f[70]<0.999519){r+=-0.629554;}else{if(f[28]<1.40075e+11){r+=-0.62571;}else{r+=0.38698;}}}else{if(f[10]<9.08328e+07){if(f[70]<0.998663){r+=-1.17207;}else{r+=-0.689376;}}else{if(f[0]<1.39867e+11){r+=-0.480234;}else{r+=-1.40366;}}}}}else{if(f[27]<2.86225e+08){if(f[70]<0.992472){if(f[42]<1.01211){if(f[40]<1.68144e+08){r+=-0.337232;}else{r+=0.547135;}}else{r+=-0.576022;}}else{if(f[28]<1.40035e+11){if(f[38]<1.08772e+08){r+=-0.737574;}else{r+=-0.609642;}}else{if(f[28]<1.40138e+11){r+=-0.0409209;}else{r+=-0.639162;}}}}else{if(f[14]<1.40155e+11){if(f[64]<0.719823){r+=-0.126381;}else{r+=-0.530731;}}else{if(f[53]<0.812832){r+=-0.50454;}else{r+=-1.7031;}}}}}}else{if(f[0]<1.40065e+11){r+=-0.264596;}else{if(f[74]<6.9701){r+=0.150925;}else{r+=0.584854;}}}}else{if(f[70]<1.00251){if(f[14]<1.40284e+11){if(f[14]<1.40063e+11){if(f[42]<1.00581){r+=0.673895;}else{r+=-0.574636;}}else{if(f[23]<8.54351e+07){r+=-0.631286;}else{if(f[28]<1.40805e+11){r+=-0.321187;}else{r+=0.38839;}}}}else{if(f[28]<1.40578e+11){if(f[28]<1.40413e+11){if(f[0]<1.40152e+11){if(f[28]<1.404e+11){if(f[27]<2.58834e+08){r+=0.59731;}else{r+=-0.248045;}}else{r+=-0.341356;}}else{if(f[28]<1.40404e+11){if(f[76]<0.255777){r+=-0.50647;}else{r+=0.0586909;}}else{if(f[14]<1.40621e+11){r+=0.557327;}else{r+=-0.264735;}}}}else{if(f[14]<1.40585e+11){r+=-0.656929;}else{if(f[3]<3.12461e+06){r+=-0.323388;}else{r+=0.148075;}}}}else{if(f[42]<1.00013){if(f[0]<1.40843e+11){r+=-0.541843;}else{r+=0.460114;}}else{r+=0.732329;}}}}else{if(f[57]<0.0362242){r+=0.567358;}else{if(f[42]<0.996387){if(f[56]<0.995766){r+=0.61871;}else{if(f[52]<0.834102){if(f[14]<1.40431e+11){r+=-0.497794;}else{r+=0.533671;}}else{if(f[42]<0.992682){if(f[0]<1.39843e+11){r+=-0.404052;}else{r+=0.352375;}}else{if(f[55]<1.86431){r+=-0.586672;}else{r+=-0.00334235;}}}}}else{if(f[54]<0.478828){if(f[67]<0.34215){r+=-0.394524;}else{r+=0.463526;}}else{if(f[41]<1.44311e+08){if(f[38]<1.47768e+08){if(f[13]<5.70786e+07){r+=-0.546518;}else{r+=-1.51181;}}else{r+=-0.670509;}}else{if(f[0]<1.39361e+11){if(f[4]<1.61379e+07){r+=-0.235945;}else{r+=0.458272;}}else{if(f[13]<7.33161e+07){r+=0.0958834;}else{r+=-0.61653;}}}}}}}}}else{if(f[14]<1.40615e+11){if(f[0]<1.40225e+11){if(f[27]<7.93229e+07){if(f[28]<1.40399e+11){if(f[14]<1.40075e+11){if(f[43]<0.344064){r+=0.00117196;}else{if(f[70]<0.99973){r+=-0.145016;}else{r+=-0.59245;}}}else{r+=-0.715981;}}else{if(f[0]<1.39982e+11){r+=-0.412117;}else{r+=0.575388;}}}else{if(f[28]<1.4063e+11){if(f[28]<1.39463e+11){if(f[54]<1.01985){r+=-0.149072;}else{r+=-0.615142;}}else{if(f[28]<1.39883e+11){if(f[59]<0.0174381){if(f[0]<1.40077e+11){r+=-0.548981;}else{r+=0.451163;}}else{if(f[56]<1.00347){r+=0.64314;}else{r+=-0.472708;}}}else{if(f[52]<1.67303){if(f[0]<1.40012e+11){r+=-0.19183;}else{r+=0.467018;}}else{if(f[39]<5.15044e+08){r+=-0.362004;}else{r+=-1.12123;}}}}}else{if(f[42]<0.996324){r+=0.348272;}else{r+=-0.60106;}}}}else{if(f[14]<1.4044e+11){if(f[66]<0.231257){if(f[28]<1.40492e+11){if(f[28]<1.39881e+11){if(f[56]<1.0029){if(f[81]<2.79932){r+=0.654757;}else{r+=-0.617508;}}else{if(f[28]<1.39812e+11){r+=-0.60716;}else{r+=-0.969165;}}}else{if(f[14]<1.40418e+11){if(f[14]<1.40156e+11){r+=-0.663813;}else{r+=-1.21714;}}else{if(f[11]<2.27717e+08){r+=-0.709489;}else{r+=0.330147;}}}}else{if(f[50]<0.880739){r+=-0.363046;}else{if(f[7]<2.34319e+08){r+=0.670278;}else{r+=0.224306;}}}}else{if(f[36]<2.12128e+08){if(f[70]<0.999857){if(f[28]<1.39895e+11){if(f[56]<1.00318){r+=0.284516;}else{r+=-0.602161;}}else{if(f[70]<0.991452){r+=-0.157198;}else{r+=-0.635262;}}}else{if(f[70]<1.00242){if(f[28]<1.40514e+11){r+=0.0690492;}else{r+=0.726866;}}else{if(f[28]<1.63032e+11){r+=-0.761068;}else{r+=-0.161679;}}}}else{if(f[51]<1.11737){if(f[42]<1.00682){if(f[28]<1.40004e+11){r+=0.0074778;}else{r+=0.637291;}}else{if(f[11]<2.27182e+08){r+=-0.758468;}else{r+=0.00400376;}}}else{if(f[26]<1.77032e+08){if(f[42]<1.00239){r+=-0.22729;}else{r+=0.476394;}}else{if(f[42]<1.00653){r+=0.121015;}else{r+=-0.915634;}}}}}}else{if(f[28]<1.39935e+11){if(f[28]<1.3981e+11){r+=-0.645614;}else{r+=-1.04467;}}else{if(f[81]<4.44177){if(f[70]<0.992411){if(f[42]<1.00656){if(f[14]<1.40474e+11){r+=0.529065;}else{r+=0.731343;}}else{if(f[81]<4.03327){r+=0.298689;}else{r+=-0.905714;}}}else{if(f[31]<7.94738e+06){r+=0.210834;}else{if(f[46]<0.389354){r+=-0.21889;}else{r+=-0.794854;}}}}else{if(f[14]<1.40487e+11){if(f[28]<1.39985e+11){if(f[13]<2.22752e+08){r+=-1.18098;}else{r+=-0.354716;}}else{if(f[28]<1.39999e+11){r+=-0.195032;}else{r+=0.511203;}}}else{if(f[67]<0.25257){r+=-0.136023;}else{if(f[1]<8.88972e+07){r+=0.711249;}else{r+=0.187877;}}}}}}}}else{if(f[66]<0.267213){if(f[8]<2.71668e+08){if(f[28]<1.3999e+11){r+=-1.43192;}else{r+=-0.262882;}}else{r+=-0.948744;}}else{if(f[42]<1.00535){if(f[70]<1.01343){if(f[14]<1.4062e+11){if(f[13]<1.4035e+08){r+=-0.311391;}else{if(f[59]<0.892183){r+=-0.0458437;}else{r+=0.400031;}}}else{if(f[42]<1.00306){if(f[68]<0.316196){if(f[9]<1.4415e+08){r+=0.0167447;}else{r+=-0.503186;}}else{if(f[22]<7.12105e+07){r+=-0.14888;}else{r+=-0.597488;}}}else{r+=-1.04532;}}}else{if(f[14]<1.41065e+11){r+=0.614566;}else{r+=-0.305657;}}}else{if(f[28]<1.40163e+11){if(f[51]<1.35527){r+=-0.116218;}else{r+=-0.726549;}}else{if(f[28]<1.41298e+11){r+=0.687171;}else{r+=0.0808845;}}}}}}}else{if(f[73]<69.1004){if(f[81]<1.66966){if(f[14]<1.40297e+11){if(f[78]<0.688868){if(f[70]<1.32611){if(f[39]<1.36067e+08){if(f[56]<1.00086){if(f[40]<2.12707e+08){r+=-0.080488;}else{r+=-0.570409;}}else{if(f[13]<3.00986e+08){r+=0.0775027;}else{r+=0.615238;}}}else{if(f[22]<7.95e+09){if(f[14]<1.39264e+11){if(f[56]<0.993209){r+=-0.552677;}else{r+=-0.056319;}}else{if(f[41]<6.23345e+08){r+=-0.0788788;}else{r+=-0.594727;}}}else{if(f[27]<7.45074e+08){r+=0.378989;}else{r+=-0.446468;}}}}else{r+=0.375784;}}else{if(f[56]<1.00089){if(f[70]<0.981967){if(f[23]<1.0554e+08){r+=-0.144215;}else{r+=0.61312;}}else{if(f[7]<1.62602e+08){if(f[0]<1.39604e+11){r+=0.544235;}else{if(f[38]<6.44081e+07){r+=-0.0640888;}else{r+=-0.555117;}}}else{if(f[14]<1.39031e+11){if(f[50]<1.24539){r+=-0.366517;}else{r+=0.510803;}}else{if(f[27]<1.80455e+09){r+=-0.399691;}else{r+=0.24777;}}}}}else{if(f[39]<1.92049e+08){if(f[14]<1.40192e+11){r+=0.643092;}else{r+=-0.425402;}}else{if(f[42]<1.00334){if(f[52]<2.19776){if(f[52]<0.986687){r+=0.038067;}else{r+=-0.873958;}}else{if(f[56]<1.00288){r+=0.314923;}else{r+=-0.345985;}}}else{if(f[56]<1.00327){if(f[1]<3.14414e+07){r+=0.0807193;}else{r+=0.64502;}}else{r+=-0.327522;}}}}}}else{if(f[14]<1.40297e+11){if(f[0]<1.40051e+11){r+=0.559987;}else{r+=0.849231;}}else{if(f[56]<0.999323){if(f[68]<0.43573){if(f[70]<1.00327){if(f[28]<1.41282e+11){if(f[28]<1.40594e+11){r+=-0.582582;}else{r+=0.672878;}}else{if(f[21]<2.44778e+08){r+=0.182442;}else{r+=-0.555601;}}}else{if(f[14]<1.41598e+11){if(f[28]<1.40635e+11){r+=-0.0165052;}else{r+=-0.583844;}}else{if(f[13]<2.40795e+09){r+=0.53033;}else{r+=-0.0277068;}}}}else{if(f[23]<3.07304e+08){if(f[14]<1.41052e+11){if(f[39]<5.80435e+08){r+=0.57423;}else{r+=-0.33963;}}else{r+=-0.410014;}}else{if(f[63]<2.19179){if(f[22]<3.02076e+08){r+=-0.0330358;}else{r+=-0.539796;}}else{r+=0.279273;}}}}else{if(f[55]<2.55233){if(f[56]<1.00199){if(f[23]<4.01299e+09){if(f[27]<1.85547e+08){r+=-0.0157428;}else{r+=-0.603738;}}else{if(f[0]<1.39874e+11){r+=0.367878;}else{r+=-0.475365;}}}else{if(f[55]<0.509284){if(f[20]<1.9572e+07){r+=0.367397;}else{r+=-0.397074;}}else{if(f[49]<0.575163){r+=-0.227791;}else{r+=-0.591028;}}}}else{if(f[0]<1.412e+11){if(f[34]<1.06959e+08){if(f[41]<7.23947e+08){r+=0.114681;}else{r+=-0.560901;}}else{if(f[56]<1.00332){r+=0.359358;}else{r+=-0.437504;}}}else{if(f[28]<1.39919e+11){r+=-0.505419;}else{if(f[81]<1.26895){r+=0.690504;}else{r+=-0.130971;}}}}}}}}else{if(f[14]<1.40146e+11){if(f[56]<0.865552){if(f[57]<15.4356){r+=0.571103;}else{r+=-0.186349;}}else{if(f[13]<1.87304e+08){if(f[28]<1.39606e+11){r+=0.655444;}else{if(f[51]<0.13117){if(f[54]<0.0904407){r+=-0.0761186;}else{r+=0.845515;}}else{if(f[42]<1.00043){if(f[42]<1.00007){r+=-0.529701;}else{r+=0.351384;}}else{if(f[17]<1.10237e+08){r+=-0.564588;}else{r+=0.0356587;}}}}}else{if(f[41]<7.40642e+08){if(f[0]<1.40097e+11){if(f[0]<1.40008e+11){if(f[28]<1.39863e+11){r+=0.371876;}else{r+=-0.371185;}}else{if(f[74]<5.04691){r+=0.779529;}else{r+=0.173129;}}}else{if(f[36]<2.89313e+08){if(f[29]<7.48052e+07){r+=-0.588144;}else{r+=0.0791617;}}else{if(f[38]<7.15453e+08){r+=0.306303;}else{r+=-0.341773;}}}}else{if(f[28]<1.41878e+11){if(f[14]<1.38988e+11){if(f[14]<1.38825e+11){r+=-0.58723;}else{r+=0.263011;}}else{if(f[0]<1.3948e+11){r+=0.108004;}else{r+=-0.590914;}}}else{r+=0.0789692;}}}}}else{if(f[14]<1.40185e+11){if(f[39]<2.55356e+08){if(f[26]<1.07129e+08){if(f[0]<1.40216e+11){r+=-0.565415;}else{r+=-0.152218;}}else{if(f[66]<0.532085){r+=-0.414313;}else{if(f[28]<1.40007e+11){if(f[0]<1.41251e+11){r+=0.612583;}else{r+=-0.229027;}}else{r+=-0.289693;}}}}else{if(f[81]<5.8025){if(f[41]<9.42164e+08){if(f[27]<3.2315e+08){if(f[40]<5.00297e+08){r+=-0.24912;}else{r+=-0.574914;}}else{if(f[40]<4.91869e+08){r+=0.107752;}else{r+=-0.891515;}}}else{if(f[39]<3.34826e+08){if(f[66]<0.467798){r+=-0.46765;}else{r+=0.261702;}}else{if(f[20]<8.44902e+06){r+=-0.408676;}else{r+=-0.59248;}}}}else{if(f[29]<1.09641e+07){if(f[70]<0.998771){if(f[28]<1.40001e+11){r+=0.406118;}else{r+=-0.164479;}}else{if(f[5]<1.91345e+06){r+=0.0280102;}else{r+=-0.428303;}}}else{if(f[74]<0.392955){r+=-0.151501;}else{r+=-0.57301;}}}}}else{if(f[38]<2.31687e+08){if(f[27]<5.17719e+08){if(f[28]<1.40383e+11){if(f[13]<1.15911e+08){if(f[7]<7.26831e+07){r+=-0.465581;}else{r+=0.451299;}}else{if(f[41]<4.78178e+09){r+=-0.59749;}else{r+=-0.0759321;}}}else{if(f[56]<1.00056){if(f[70]<0.998567){r+=0.585441;}else{r+=-0.590111;}}else{if(f[11]<7.0559e+07){r+=-0.168055;}else{r+=0.569708;}}}}else{if(f[41]<9.88924e+08){if(f[0]<1.40306e+11){r+=-2.07105;}else{r+=-0.421004;}}else{if(f[72]<0.0952416){if(f[80]<1.49308){r+=0.179273;}else{r+=-0.306203;}}else{if(f[41]<1.53558e+09){r+=-0.76265;}else{r+=-0.555046;}}}}}else{if(f[70]<0.991386){if(f[28]<1.39954e+11){if(f[68]<0.0498943){if(f[13]<3.26181e+08){r+=0.421314;}else{r+=-0.376073;}}else{if(f[15]<1.39788e+08){r+=-0.593782;}else{r+=-0.180936;}}}else{if(f[0]<1.41361e+11){if(f[14]<1.40361e+11){r+=-0.392198;}else{r+=0.555508;}}else{if(f[81]<3.6339){r+=0.22424;}else{r+=-0.485319;}}}}else{if(f[0]<1.40214e+11){if(f[0]<1.3981e+11){r+=-0.593297;}else{if(f[14]<1.41011e+11){r+=-0.327806;}else{r+=0.423525;}}}else{if(f[0]<1.42375e+11){if(f[40]<3.19043e+08){r+=0.304711;}else{r+=-0.580298;}}else{if(f[66]<0.299693){r+=-0.440745;}else{r+=0.349467;}}}}}}}}}else{if(f[14]<1.40054e+11){if(f[0]<1.40069e+11){if(f[80]<5.36115){if(f[49]<1.09024){r+=-0.263691;}else{r+=0.202848;}}else{if(f[41]<6.2891e+08){r+=-0.155785;}else{r+=-0.576896;}}}else{if(f[38]<7.20569e+09){if(f[1]<2.17352e+07){r+=0.587265;}else{r+=-0.00070217;}}else{if(f[28]<1.65314e+11){r+=-0.55412;}else{r+=0.350017;}}}}else{if(f[38]<5.69968e+08){if(f[68]<0.310812){if(f[53]<1.22496){r+=0.475209;}else{r+=-0.114611;}}else{r+=-0.406248;}}else{r+=-0.586212;}}}}}else{if(f[27]<3.5641e+08){if(f[66]<1.77142){r+=-0.596917;}else{r+=-0.0575986;}}else{if(f[67]<0.201834){if(f[39]<1.52783e+09){if(f[56]<0.986562){if(f[26]<3.86142e+08){r+=-0.376034;}else{r+=-0.00152381;}}else{if(f[39]<1.41114e+09){r+=0.526515;}else{r+=0.0408706;}}}else{if(f[13]<1.79797e+09){r+=-0.018193;}else{if(f[17]<1.92236e+08){if(f[34]<1.14462e+07){r+=-0.0778814;}else{r+=-0.571009;}}else{r+=-0.0778814;}}}}else{if(f[28]<1.45271e+11){if(f[34]<1.92085e+08){if(f[56]<0.986136){if(f[7]<1.2201e+08){if(f[13]<1.59722e+09){r+=0.22782;}else{if(f[51]<1.43166){r+=-0.0780233;}else{r+=-0.519661;}}}else{if(f[28]<1.44454e+11){r+=-0.0335139;}else{if(f[26]<3.17175e+08){r+=0.0448997;}else{if(f[80]<1.98933){r+=0.563859;}else{r+=0.0943562;}}}}}else{if(f[70]<0.999){r+=-0.167605;}else{if(f[20]<4.17686e+08){if(f[39]<4.35819e+09){if(f[34]<1.64835e+08){if(f[39]<6.32578e+08){r+=0.24867;}else{r+=0.574804;}}else{if(f[19]<1.23311e+08){r+=-0.0773189;}else{r+=0.395331;}}}else{r+=0.132969;}}else{if(f[8]<1.58663e+09){if(f[4]<9.92611e+07){r+=-0.365194;}else{r+=0.0749136;}}else{r+=0.485013;}}}}}else{if(f[5]<2.36807e+08){if(f[20]<3.10082e+08){r+=0.180988;}else{r+=-0.508938;}}else{if(f[6]<1.01423e+09){r+=0.401107;}else{r+=-0.161253;}}}}else{if(f[40]<4.68992e+09){if(f[23]<1.85276e+09){if(f[0]<1.42431e+11){r+=-0.136955;}else{r+=-0.544874;}}else{r+=-0.0563876;}}else{if(f[5]<1.32492e+08){r+=0.337518;}else{r+=0.0701714;}}}}}}}else{if(f[28]<1.39438e+11){if(f[28]<1.39424e+11){if(f[10]<3.76391e+10){if(f[56]<1.00196){if(f[56]<1.00149){if(f[28]<1.39361e+11){if(f[14]<1.38576e+11){r+=-0.580816;}else{r+=-0.140869;}}else{r+=0.177529;}}else{if(f[64]<0.99931){r+=0.836149;}else{r+=0.0628283;}}}else{if(f[63]<0.658076){if(f[14]<1.40311e+11){if(f[14]<1.40195e+11){if(f[5]<9.81246e+07){r+=-0.539168;}else{r+=0.0500605;}}else{if(f[81]<0.519739){r+=-0.483003;}else{if(f[0]<1.41624e+11){r+=-0.136885;}else{r+=0.803699;}}}}else{r+=-0.590315;}}else{if(f[54]<0.0876564){r+=-0.0668467;}else{if(f[74]<30.039){r+=-0.591471;}else{r+=-0.139693;}}}}}else{if(f[51]<0.928775){if(f[29]<5.48047e+09){r+=-0.5522;}else{r+=0.247721;}}else{if(f[53]<0.91942){r+=-0.212706;}else{if(f[44]<0.843673){r+=0.06055;}else{r+=0.615455;}}}}}else{if(f[14]<1.40339e+11){if(f[14]<1.39837e+11){if(f[0]<1.4003e+11){r+=0.524986;}else{if(f[14]<1.39825e+11){r+=-0.713889;}else{if(f[1]<4.66709e+07){r+=-0.472794;}else{if(f[39]<8.57624e+07){r+=-0.242924;}else{r+=0.702538;}}}}}else{if(f[11]<8.3086e+07){r+=-0.232861;}else{if(f[39]<3.35946e+08){if(f[49]<0.794466){r+=0.148351;}else{r+=0.898967;}}else{r+=-0.0622611;}}}}else{r+=-0.608393;}}}else{if(f[68]<6.89335){if(f[28]<1.40143e+11){if(f[0]<1.45762e+11){if(f[41]<9.11486e+07){if(f[14]<1.40132e+11){if(f[28]<1.39778e+11){if(f[39]<7.03363e+07){if(f[74]<1.11046){if(f[6]<1.36438e+07){r+=-0.100819;}else{r+=-0.984913;}}else{r+=0.361379;}}else{if(f[28]<1.39445e+11){if(f[27]<1.12015e+08){r+=-0.696198;}else{r+=0.463234;}}else{if(f[29]<6.37858e+07){if(f[33]<2.45084e+06){r+=0.221116;}else{r+=0.607182;}}else{if(f[28]<1.39462e+11){r+=-0.558614;}else{r+=0.423564;}}}}}else{if(f[28]<1.40134e+11){if(f[0]<1.39727e+11){if(f[14]<1.39594e+11){r+=0.548497;}else{r+=-0.484454;}}else{if(f[0]<1.40048e+11){if(f[41]<8.82749e+07){r+=-0.591836;}else{r+=-0.232653;}}else{if(f[27]<7.06949e+07){r+=-0.71129;}else{r+=-0.627517;}}}}else{if(f[0]<1.40153e+11){if(f[39]<8.47586e+07){r+=-0.617733;}else{r+=-0.127923;}}else{if(f[14]<1.39442e+11){if(f[39]<8.77192e+07){r+=-0.566648;}else{r+=-0.0911821;}}else{if(f[41]<8.99584e+07){r+=2.00858;}else{r+=1.18128;}}}}}}else{if(f[36]<7.16249e+07){if(f[28]<1.40029e+11){if(f[52]<1.4393){if(f[51]<1.58093){if(f[0]<1.40332e+11){r+=-0.26326;}else{r+=0.69639;}}else{if(f[38]<1.29893e+07){r+=-0.45904;}else{r+=0.553165;}}}else{if(f[0]<1.39998e+11){if(f[56]<1.00346){r+=-0.533691;}else{r+=0.312258;}}else{if(f[41]<8.04009e+07){r+=-0.627967;}else{r+=0.21075;}}}}else{if(f[42]<0.997997){if(f[28]<1.40076e+11){if(f[14]<1.40317e+11){r+=0.226952;}else{r+=-0.593276;}}else{if(f[56]<1.00217){r+=0.35594;}else{r+=-0.771536;}}}else{if(f[0]<1.40196e+11){if(f[40]<8.7512e+07){r+=0.676256;}else{r+=-0.36145;}}else{if(f[56]<1.00184){r+=0.458016;}else{r+=-0.612367;}}}}}else{if(f[28]<1.40128e+11){if(f[78]<0.3066){if(f[70]<0.997539){r+=-0.662694;}else{if(f[60]<4.08478){r+=0.46053;}else{r+=-0.0374281;}}}else{if(f[56]<1.00157){if(f[14]<1.40167e+11){r+=0.159196;}else{r+=0.666028;}}else{if(f[70]<0.998434){r+=-0.778059;}else{r+=-0.557421;}}}}else{if(f[14]<1.41247e+11){if(f[42]<0.997129){r+=-0.250153;}else{r+=0.742551;}}else{if(f[57]<3.027){r+=-0.862648;}else{r+=-0.212376;}}}}}}else{if(f[13]<8.73407e+07){if(f[56]<1.01013){if(f[39]<3.6431e+07){if(f[14]<1.40208e+11){r+=-0.70174;}else{if(f[27]<6.30857e+07){r+=0.0195187;}else{r+=0.671149;}}}else{if(f[41]<5.3471e+08){if(f[9]<2.67388e+07){if(f[28]<1.40021e+11){r+=-0.781712;}else{r+=-1.21226;}}else{if(f[31]<2.27682e+06){r+=-0.22166;}else{r+=-0.676376;}}}else{r+=0.911462;}}}else{if(f[0]<1.40458e+11){if(f[0]<1.40365e+11){r+=-0.515847;}else{r+=0.499523;}}else{if(f[28]<1.39809e+11){r+=-0.649482;}else{if(f[41]<1.97127e+08){if(f[55]<0.741502){r+=-0.70008;}else{r+=-1.43235;}}else{r+=-0.38366;}}}}}else{if(f[68]<1.07753){if(f[28]<1.40012e+11){if(f[42]<1.00897){if(f[8]<7.87668e+07){if(f[14]<1.40022e+11){r+=-0.555173;}else{r+=0.403503;}}else{if(f[14]<1.40628e+11){r+=-0.0235947;}else{r+=-0.876419;}}}else{if(f[10]<2.38726e+08){if(f[38]<1.91383e+08){r+=0.406111;}else{r+=-0.823073;}}else{if(f[14]<1.40357e+11){r+=0.60843;}else{r+=-0.70236;}}}}else{if(f[10]<2.17541e+08){if(f[28]<1.40111e+11){if(f[27]<5.04059e+08){r+=-0.488833;}else{r+=-1.84267;}}else{if(f[42]<1.00078){r+=0.24236;}else{r+=-0.594768;}}}else{if(f[14]<1.40628e+11){if(f[14]<1.40145e+11){r+=0.00162894;}else{r+=0.477961;}}else{if(f[0]<1.40216e+11){r+=-0.4988;}else{r+=-0.949271;}}}}}else{if(f[0]<1.40087e+11){if(f[35]<3.49579e+07){if(f[14]<1.39991e+11){if(f[68]<2.01213){r+=-0.662818;}else{r+=0.326017;}}else{if(f[0]<1.39976e+11){r+=0.863647;}else{r+=0.514301;}}}else{if(f[28]<1.39833e+11){if(f[14]<1.39786e+11){r+=0.598778;}else{r+=-0.547806;}}else{if(f[66]<0.745935){r+=0.593196;}else{r+=-0.449805;}}}}else{if(f[0]<1.40142e+11){if(f[26]<3.79711e+08){if(f[56]<1.0015){r+=-0.734096;}else{r+=0.847162;}}else{if(f[14]<1.4009e+11){r+=0.758128;}else{r+=-0.561955;}}}else{if(f[14]<1.40346e+11){if(f[70]<0.999393){r+=0.238554;}else{r+=0.709417;}}else{if(f[0]<1.40479e+11){r+=0.0470027;}else{r+=-0.930607;}}}}}}}}else{if(f[55]<0.735628){r+=-0.598219;}else{r+=-0.679971;}}}else{if(f[0]<1.41674e+11){if(f[28]<1.41479e+11){if(f[81]<3.72556){if(f[14]<1.41224e+11){if(f[28]<1.41006e+11){if(f[0]<1.39254e+11){if(f[28]<1.40213e+11){if(f[56]<0.995109){r+=-0.35029;}else{r+=0.568449;}}else{if(f[81]<1.65429){r+=-0.583269;}else{r+=0.275168;}}}else{if(f[13]<1.24681e+08){if(f[41]<9.56909e+07){r+=-0.202164;}else{r+=0.22953;}}else{if(f[68]<1.02778){r+=-0.209395;}else{r+=-0.491782;}}}}else{if(f[70]<0.997755){if(f[26]<3.07901e+09){if(f[22]<1.48593e+08){r+=0.755742;}else{r+=1.22161;}}else{r+=-0.470979;}}else{if(f[0]<1.40163e+11){if(f[14]<1.39858e+11){r+=-0.817361;}else{r+=-0.598474;}}else{if(f[42]<1.00265){r+=-0.976727;}else{r+=-0.571129;}}}}}else{if(f[56]<1.00354){if(f[28]<1.41081e+11){r+=-0.721087;}else{if(f[13]<7.42523e+09){r+=-0.606717;}else{r+=0.192742;}}}else{if(f[0]<1.39479e+11){if(f[14]<1.45703e+11){r+=-0.594907;}else{r+=0.623048;}}else{if(f[41]<3.08391e+08){if(f[28]<1.40244e+11){r+=-0.840928;}else{r+=0.606022;}}else{if(f[42]<0.998618){r+=-0.712554;}else{r+=0.388912;}}}}}}else{if(f[28]<1.40217e+11){if(f[14]<1.39827e+11){if(f[14]<1.39679e+11){if(f[28]<1.40173e+11){if(f[63]<1.24503){r+=-0.582742;}else{r+=0.746747;}}else{if(f[17]<8.30255e+06){r+=-0.670919;}else{r+=-0.13613;}}}else{if(f[26]<2.34028e+08){r+=-1.31747;}else{if(f[64]<1.87667){r+=0.635329;}else{r+=-0.282033;}}}}else{if(f[59]<0.0207759){if(f[55]<0.704403){if(f[0]<1.40073e+11){r+=-0.099135;}else{r+=0.719481;}}else{if(f[57]<0.0408285){r+=0.0637008;}else{r+=-0.451297;}}}else{if(f[51]<0.486106){if(f[0]<1.40088e+11){r+=-0.999949;}else{r+=-0.611761;}}else{if(f[14]<1.39858e+11){r+=-1.10391;}else{r+=-0.633711;}}}}}else{if(f[56]<0.993483){if(f[0]<1.39914e+11){r+=-0.0623363;}else{r+=-0.591815;}}else{if(f[9]<6.10581e+08){if(f[55]<0.00380564){if(f[28]<1.40536e+11){r+=-0.845564;}else{r+=0.762065;}}else{if(f[42]<0.990328){r+=-0.589027;}else{r+=0.664563;}}}else{if(f[53]<0.314524){if(f[68]<0.813846){r+=0.58762;}else{r+=-0.251518;}}else{if(f[35]<4.43342e+08){r+=0.159882;}else{r+=-0.603142;}}}}}}}else{if(f[56]<0.982044){if(f[57]<0.167651){if(f[56]<0.89902){if(f[0]<1.40287e+11){if(f[22]<1.48695e+08){r+=0.612516;}else{r+=0.717859;}}else{r+=-0.0705183;}}else{if(f[28]<1.49095e+11){if(f[28]<1.46877e+11){r+=-0.578321;}else{if(f[0]<1.40187e+11){r+=-0.381767;}else{r+=-1.37438;}}}else{if(f[0]<1.39982e+11){r+=-0.557148;}else{r+=-0.0794935;}}}}else{if(f[66]<2.52633){if(f[51]<1.311){if(f[36]<1.8058e+07){if(f[0]<1.40233e+11){r+=-0.472965;}else{r+=0.495288;}}else{if(f[14]<1.39994e+11){r+=-0.592513;}else{r+=-0.667;}}}else{if(f[14]<1.39991e+11){r+=-0.619884;}else{if(f[23]<1.29572e+07){r+=-1.33126;}else{r+=-0.837807;}}}}else{if(f[28]<1.76651e+11){if(f[1]<2.22963e+08){r+=-0.638315;}else{r+=0.0017058;}}else{if(f[14]<1.40579e+11){r+=0.60385;}else{r+=-0.379292;}}}}}else{if(f[52]<1.68503){if(f[14]<1.40978e+11){if(f[28]<1.42025e+11){if(f[0]<1.39346e+11){if(f[0]<1.39325e+11){r+=0.611238;}else{r+=0.134773;}}else{if(f[38]<7.49268e+07){r+=0.71061;}else{r+=-1.01063;}}}else{if(f[56]<0.986993){if(f[52]<1.38413){r+=0.625733;}else{r+=-0.410551;}}else{if(f[22]<1.48598e+08){r+=0.694848;}else{r+=0.961787;}}}}else{if(f[56]<0.992777){if(f[0]<1.40104e+11){r+=-0.389103;}else{if(f[81]<1.32627){r+=0.661665;}else{r+=0.165055;}}}else{if(f[14]<1.41e+11){if(f[22]<1.48632e+08){r+=-1.62385;}else{r+=-0.882183;}}else{if(f[10]<1.02649e+10){r+=-0.64153;}else{r+=0.0021899;}}}}}else{if(f[0]<1.39357e+11){if(f[0]<1.37538e+11){r+=-0.566499;}else{r+=0.612704;}}else{if(f[0]<1.40032e+11){if(f[41]<9.67545e+07){if(f[38]<1.48404e+08){r+=-1.20284;}else{r+=-0.700862;}}else{if(f[0]<1.39562e+11){r+=-0.819222;}else{r+=-0.612704;}}}else{if(f[70]<1.01024){if(f[14]<1.39772e+11){r+=0.268335;}else{r+=-0.618507;}}else{if(f[64]<0.621822){r+=-0.373417;}else{r+=0.66541;}}}}}}}}else{if(f[42]<1.02958){if(f[56]<0.989431){if(f[28]<1.43826e+11){if(f[14]<1.41e+11){if(f[22]<1.48488e+08){r+=-1.5542;}else{if(f[28]<1.41723e+11){r+=-0.567994;}else{r+=-0.837986;}}}else{r+=-0.715332;}}else{if(f[20]<6.42098e+08){if(f[28]<1.4543e+11){if(f[20]<4.46056e+08){r+=0.579024;}else{if(f[19]<1.44714e+08){r+=-0.376401;}else{r+=0.467996;}}}else{r+=-0.122969;}}else{if(f[19]<3.83624e+08){r+=-0.673911;}else{if(f[5]<3.95045e+08){r+=-0.175354;}else{if(f[55]<1.5832){r+=0.519947;}else{r+=0.0689689;}}}}}}else{if(f[36]<1.77107e+08){if(f[14]<1.40776e+11){if(f[42]<1.02766){if(f[28]<1.40249e+11){r+=0.827717;}else{if(f[70]<0.993896){r+=-0.632653;}else{r+=0.716738;}}}else{if(f[28]<1.40651e+11){r+=-0.629489;}else{r+=0.603269;}}}else{if(f[14]<1.41312e+11){if(f[55]<0.736461){if(f[40]<6.98056e+07){r+=0.280074;}else{r+=1.08915;}}else{if(f[8]<2.71632e+08){r+=0.746958;}else{r+=1.02627;}}}else{if(f[28]<1.42779e+11){if(f[33]<2.60373e+07){r+=-0.690971;}else{r+=0.0339479;}}else{r+=0.527339;}}}}else{if(f[14]<1.40065e+11){if(f[42]<1.02687){if(f[13]<4.56803e+09){if(f[22]<4.41641e+08){r+=-0.69122;}else{r+=-0.581408;}}else{r+=0.0488288;}}else{if(f[66]<3.80217){r+=0.609432;}else{r+=-0.0420122;}}}else{if(f[56]<1.00777){if(f[13]<1.16644e+09){if(f[8]<2.73083e+08){r+=0.787093;}else{r+=1.07571;}}else{if(f[56]<0.99869){r+=0.572071;}else{r+=-0.0174628;}}}else{if(f[14]<1.41347e+11){r+=-0.143133;}else{r+=-0.66492;}}}}}}else{if(f[28]<1.40194e+11){if(f[13]<4.71389e+08){r+=0.0794666;}else{r+=0.625306;}}else{if(f[14]<1.39069e+11){if(f[0]<1.46308e+11){r+=0.677262;}else{r+=-0.19801;}}else{if(f[42]<1.02986){if(f[28]<1.40693e+11){if(f[0]<1.43616e+11){r+=-0.50041;}else{r+=-0.120508;}}else{if(f[9]<1.47151e+08){r+=0.934951;}else{r+=0.203408;}}}else{if(f[28]<1.40308e+11){r+=-0.752573;}else{if(f[15]<3.21676e+08){r+=-0.603351;}else{if(f[14]<1.40229e+11){r+=0.271516;}else{r+=-0.451385;}}}}}}}}}}else{if(f[28]<1.80868e+11){if(f[28]<1.40119e+11){if(f[57]<5.53722){if(f[42]<0.997629){if(f[42]<0.996279){r+=-0.540724;}else{if(f[0]<1.39998e+11){if(f[28]<1.39847e+11){if(f[64]<2.56963){r+=-0.242862;}else{r+=0.228797;}}else{if(f[38]<3.44681e+07){if(f[62]<0.676186){r+=0.1009;}else{r+=0.63469;}}else{r+=-0.0330961;}}}else{if(f[9]<4.16258e+07){if(f[14]<1.40394e+11){if(f[54]<1.2182){r+=-0.518142;}else{r+=-0.0626862;}}else{if(f[41]<3.05442e+07){r+=0.41783;}else{r+=-0.328941;}}}else{if(f[8]<5.16748e+07){if(f[41]<3.47239e+07){r+=0.61513;}else{r+=0.103702;}}else{r+=-0.123199;}}}}}else{if(f[41]<1.05108e+08){if(f[20]<3.86421e+07){if(f[56]<1.00367){r+=-0.648593;}else{if(f[42]<0.998626){r+=0.172938;}else{r+=-0.472601;}}}else{r+=-0.0567345;}}else{if(f[54]<0.152775){if(f[39]<7.13369e+07){r+=0.254258;}else{if(f[28]<1.39947e+11){r+=0.188901;}else{r+=-0.583424;}}}else{if(f[56]<1.00059){if(f[42]<1.00915){if(f[49]<2.04438){r+=0.580242;}else{r+=-0.0683579;}}else{r+=-0.411375;}}else{if(f[42]<1.00025){if(f[4]<8.83955e+06){r+=-0.747381;}else{r+=0.0644132;}}else{if(f[75]<1.16049){r+=0.470891;}else{r+=-0.481137;}}}}}}}else{if(f[63]<0.967999){r+=0.539196;}else{if(f[68]<7.05553){r+=0.0771199;}else{if(f[15]<2.49962e+10){r+=-0.593299;}else{r+=-0.14207;}}}}}else{if(f[38]<1.27915e+09){if(f[66]<1.80746){if(f[27]<3.04146e+09){r+=-0.664502;}else{r+=-0.128028;}}else{if(f[41]<1.50835e+09){if(f[7]<326448){r+=0.123482;}else{if(f[56]<1.0073){if(f[39]<1.68523e+09){if(f[28]<1.4012e+11){r+=-0.115368;}else{r+=-0.590165;}}else{r+=-0.129372;}}else{if(f[41]<1.94961e+08){if(f[9]<2.74825e+07){r+=-0.063386;}else{r+=-0.568073;}}else{r+=0.865636;}}}}else{if(f[14]<1.40443e+11){r+=-0.464983;}else{r+=0.284415;}}}}else{if(f[26]<6.94081e+09){if(f[43]<0.191728){r+=-0.0305989;}else{r+=-0.582988;}}else{if(f[28]<1.40471e+11){if(f[9]<8.91475e+08){r+=0.304556;}else{r+=-0.636102;}}else{if(f[59]<1.191){r+=-0.288199;}else{if(f[17]<1.02222e+08){r+=0.19651;}else{r+=0.729557;}}}}}}}else{if(f[14]<1.40148e+11){r+=0.616642;}else{r+=-0.574641;}}}}} return r;}
    static double p12(double[] f){double r=0;if(f[28]<1.39505e+11){if(f[56]<1.00169){if(f[14]<1.39494e+11){if(f[57]<11.6538){if(f[0]<1.40808e+11){if(f[9]<6.26031e+10){r+=-0.980392;}else{r+=-0.25;}}else{if(f[56]<0.972802){r+=0.25;}else{r+=-0.636364;}}}else{if(f[28]<1.29784e+11){r+=-0.75;}else{r+=0.647059;}}}else{if(f[83]<0.871784){r+=0.99458;}else{r+=0.111111;}}}else{if(f[10]<3.31037e+10){if(f[51]<0.00406406){r+=0.333333;}else{if(f[27]<2.93146e+10){r+=-0.999639;}else{if(f[0]<1.40294e+11){r+=-0;}else{r+=-0.5;}}}}else{if(f[56]<1.01193){if(f[51]<1.12579){r+=0.0909091;}else{r+=0.777778;}}else{if(f[16]<4.6093e+09){if(f[71]<1.28013){r+=-0.789474;}else{if(f[6]<1.4114e+09){r+=-0.25;}else{r+=0.25;}}}else{if(f[10]<7.16082e+10){r+=0.6;}else{r+=-0;}}}}}}else{if(f[69]<0.486492){if(f[83]<8.13187){if(f[14]<1.42242e+11){if(f[83]<1.69652){if(f[14]<1.40205e+11){if(f[39]<7.11405e+07){if(f[84]<1.85){if(f[57]<2.41593){if(f[27]<7.65373e+07){if(f[0]<1.40555e+11){r+=0.995794;}else{r+=-0;}}else{if(f[84]<1.5){r+=-0.555556;}else{r+=0.538462;}}}else{r+=-0.777778;}}else{if(f[23]<4.01168e+07){r+=-0.960784;}else{r+=0.272727;}}}else{if(f[28]<1.39884e+11){if(f[42]<0.99984){if(f[6]<5.09931e+07){r+=-0.846154;}else{r+=0.692308;}}else{if(f[71]<4.77599){if(f[28]<1.39872e+11){if(f[27]<2.98485e+08){r+=0.982379;}else{r+=0.466667;}}else{if(f[0]<1.40858e+11){r+=-0.454545;}else{r+=0.714286;}}}else{r+=-0.714286;}}}else{if(f[27]<6.66756e+08){if(f[1]<1.25865e+08){if(f[25]<2.02277e+07){if(f[28]<1.40089e+11){r+=0.956332;}else{r+=-0.989446;}}else{if(f[14]<1.39438e+11){r+=-0.524096;}else{r+=-0.908546;}}}else{if(f[41]<3.03999e+08){if(f[41]<2.56316e+08){r+=-0.962617;}else{r+=-0.333333;}}else{if(f[28]<1.40071e+11){r+=0.839196;}else{r+=-0.296296;}}}}else{if(f[21]<4.04888e+07){if(f[56]<0.99807){if(f[0]<1.26745e+11){r+=0.25;}else{r+=0.974684;}}else{if(f[84]<3){r+=-0.625;}else{r+=0.692308;}}}else{if(f[56]<0.920502){r+=0.933333;}else{if(f[84]<2.85){r+=-0.786408;}else{r+=0.365854;}}}}}}}else{if(f[27]<8.50052e+07){if(f[27]<7.14317e+07){if(f[37]<4.11186e+06){r+=0.4;}else{if(f[21]<1.38969e+08){if(f[15]<1.2167e+06){r+=-0;}else{if(f[51]<4.25285){r+=-0.942478;}else{r+=-0.444444;}}}else{if(f[57]<2.99629){r+=0.5;}else{r+=-0.555556;}}}}else{if(f[12]<7.61727e+07){if(f[0]<1.39392e+11){r+=-0.636364;}else{if(f[68]<0.678353){r+=0.911111;}else{r+=-0.5;}}}else{if(f[14]<1.40358e+11){r+=0.0909091;}else{if(f[10]<3.65766e+07){r+=-0;}else{if(f[49]<0.880113){r+=-0;}else{r+=-0.9375;}}}}}}else{if(f[28]<1.39824e+11){if(f[0]<1.40192e+11){r+=-0.25;}else{r+=-0.98806;}}else{if(f[12]<5.87228e+08){if(f[8]<8.23238e+07){if(f[40]<1.16098e+08){if(f[56]<1.00125){r+=-0.555556;}else{r+=0.912621;}}else{if(f[28]<1.39985e+11){r+=0.781818;}else{r+=-0.72332;}}}else{if(f[14]<1.4101e+11){if(f[23]<2.49762e+08){r+=0.911407;}else{r+=0.226667;}}else{if(f[84]<4.5){r+=-0.897959;}else{r+=0.6;}}}}else{if(f[14]<1.40234e+11){if(f[49]<3.23425){r+=0.92;}else{r+=-0.25;}}else{if(f[28]<1.40116e+11){if(f[77]<0.55812){r+=-0.677419;}else{r+=0.205479;}}else{if(f[14]<1.4198e+11){r+=-0.804196;}else{r+=0.5;}}}}}}}}else{if(f[66]<2.04724){if(f[12]<1.71381e+08){if(f[82]<7.97725){if(f[42]<1.01382){if(f[0]<1.41597e+11){if(f[28]<1.4069e+11){if(f[0]<1.39507e+11){r+=0.502268;}else{r+=-0.517215;}}else{if(f[70]<1.00299){r+=0.241158;}else{r+=-0.975706;}}}else{if(f[14]<1.41014e+11){if(f[67]<0.279216){r+=-0.84;}else{r+=0.990083;}}else{if(f[1]<1.41054e+08){r+=-0.967213;}else{r+=-0.25;}}}}else{if(f[10]<2.39075e+08){r+=-0.997227;}else{if(f[69]<0.353012){if(f[31]<4.05444e+07){r+=-0.851852;}else{r+=-0;}}else{if(f[8]<2.05163e+08){r+=-0.6;}else{r+=0.913669;}}}}}else{if(f[14]<1.40179e+11){if(f[29]<2.40154e+07){if(f[28]<1.40185e+11){if(f[84]<1.15){r+=0.868132;}else{r+=-0.617647;}}else{if(f[4]<1.3831e+07){r+=0.988604;}else{r+=0.111111;}}}else{if(f[39]<5.79743e+08){if(f[0]<1.39957e+11){r+=-0.5;}else{r+=-0;}}else{r+=-0.932203;}}}else{if(f[82]<11.4513){if(f[13]<1.28958e+08){if(f[1]<1.82596e+07){r+=-0.625;}else{r+=0.6;}}else{if(f[0]<1.39905e+11){r+=-0.230769;}else{r+=-0.937853;}}}else{if(f[70]<1.00271){if(f[48]<1.57493){r+=0.75;}else{r+=0.111111;}}else{r+=-0.272727;}}}}}else{if(f[83]<2.27106){if(f[0]<1.41258e+11){if(f[84]<2.85){if(f[51]<1.26052){if(f[27]<1.75409e+08){r+=-0.0896552;}else{r+=-0.77373;}}else{if(f[27]<1.08179e+10){r+=-0.937622;}else{r+=0.5;}}}else{if(f[28]<1.40011e+11){if(f[51]<0.861767){r+=-0.68;}else{r+=0.683453;}}else{if(f[39]<1.4377e+09){r+=-0.836576;}else{r+=0.652174;}}}}else{if(f[84]<2.15){if(f[42]<1.01007){if(f[56]<1.00437){r+=0.974068;}else{r+=-0.902439;}}else{if(f[14]<1.30999e+11){r+=0.764706;}else{r+=-0.917808;}}}else{if(f[14]<1.39137e+11){r+=0.764706;}else{r+=-0.980861;}}}}else{if(f[28]<1.438e+11){if(f[13]<5.27751e+08){if(f[84]<1.15){if(f[84]<0.85){r+=-0.884993;}else{r+=-0.119463;}}else{if(f[0]<1.39491e+11){r+=-0.538951;}else{r+=-0.929595;}}}else{if(f[0]<1.4008e+11){if(f[84]<2.85){r+=-0.96139;}else{r+=-0.364444;}}else{if(f[49]<0.228095){r+=-0.111111;}else{r+=-0.99342;}}}}else{if(f[84]<0.8){if(f[25]<1.86358e+10){r+=0.978261;}else{r+=-0.636364;}}else{r+=-0.980392;}}}}}else{if(f[0]<1.4026e+11){if(f[70]<1.00199){if(f[28]<1.40379e+11){if(f[28]<1.40147e+11){if(f[84]<1.15){if(f[28]<1.40002e+11){r+=0.555556;}else{r+=-0.5;}}else{if(f[42]<0.999543){r+=-0;}else{r+=-0.897436;}}}else{if(f[0]<1.39986e+11){if(f[84]<2.7){r+=-0.424242;}else{r+=0.888889;}}else{if(f[28]<1.40182e+11){r+=0.693182;}else{r+=0.983213;}}}}else{if(f[84]<6.60001){r+=-0.959184;}else{r+=0.944444;}}}else{if(f[84]<1.15){if(f[14]<1.40471e+11){r+=-0.647059;}else{if(f[32]<1.05027e+07){if(f[28]<1.40388e+11){r+=0;}else{r+=0.909091;}}else{r+=0.0909091;}}}else{if(f[64]<0.987889){if(f[6]<3.26943e+07){r+=-0.5;}else{r+=0.555556;}}else{if(f[83]<1.72291){r+=0.111111;}else{if(f[67]<2.3455){r+=-0.953704;}else{r+=-0.2;}}}}}}else{if(f[28]<1.43375e+11){if(f[41]<5.75801e+08){if(f[0]<1.4069e+11){r+=-0.818182;}else{if(f[28]<1.40579e+11){if(f[15]<7.85252e+06){r+=0.25;}else{r+=-0.764706;}}else{r+=0.809524;}}}else{if(f[55]<0.39416){r+=-0;}else{if(f[28]<1.39999e+11){if(f[84]<1.15){r+=0.466667;}else{r+=-0.909091;}}else{if(f[0]<1.40262e+11){r+=-0.333333;}else{r+=-0.990868;}}}}}else{if(f[84]<1){r+=0.8;}else{r+=-0.6;}}}}}}else{if(f[84]<1.15){if(f[55]<6.00848){if(f[7]<1.3189e+10){r+=0.999194;}else{r+=0.25;}}else{r+=-0.5;}}else{r+=-0.99824;}}}else{if(f[0]<1.40043e+11){if(f[28]<1.4018e+11){if(f[84]<3.5){if(f[12]<9.43453e+07){r+=0.6;}else{if(f[28]<1.40177e+11){if(f[82]<18.3787){if(f[7]<1.50097e+07){r+=-0.25;}else{if(f[58]<5.15881){r+=-0.961977;}else{r+=-0.25;}}}else{r+=-0;}}else{if(f[16]<4.05944e+06){if(f[15]<1.84751e+06){r+=0.25;}else{if(f[6]<6.84086e+06){if(f[0]<1.39987e+11){r+=-0.846154;}else{r+=-0.3;}}else{if(f[53]<0.722204){r+=0.5;}else{r+=-0.5;}}}}else{if(f[12]<1.61172e+08){if(f[51]<0.908625){r+=0.764706;}else{r+=-0.25;}}else{r+=-0.764706;}}}}}else{if(f[28]<1.40164e+11){r+=-0.84;}else{r+=0.923077;}}}else{if(f[70]<1.00333){if(f[0]<1.39989e+11){if(f[84]<3.35){if(f[69]<0.148642){if(f[25]<7.48866e+07){if(f[12]<1.20297e+08){r+=-0.918367;}else{r+=-0.0909091;}}else{if(f[14]<1.40094e+11){if(f[32]<3.06313e+06){r+=-0.0769231;}else{r+=-0.789474;}}else{if(f[49]<0.786627){r+=-0.111111;}else{r+=0.733333;}}}}else{if(f[53]<1.00218){if(f[68]<0.183869){if(f[0]<1.39984e+11){r+=-0.111111;}else{r+=0.477876;}}else{if(f[71]<0.777136){r+=0.4;}else{r+=-0.333333;}}}else{if(f[66]<0.530701){if(f[74]<3.90152){r+=-0.6;}else{r+=0.333333;}}else{if(f[28]<1.40181e+11){r+=-0.555556;}else{r+=0.44;}}}}}else{r+=0.99376;}}else{if(f[28]<1.40182e+11){if(f[84]<2.5){if(f[5]<2.05772e+06){if(f[16]<3.24001e+06){r+=0.733333;}else{r+=-0;}}else{if(f[51]<0.771718){r+=0.666667;}else{if(f[5]<7.5723e+06){r+=-0.515152;}else{r+=0.5;}}}}else{r+=0.943662;}}else{if(f[0]<1.39993e+11){if(f[84]<2.5){if(f[82]<15.6748){if(f[14]<1.40094e+11){r+=0.357143;}else{r+=0.814255;}}else{if(f[78]<1.01035){r+=0.75;}else{r+=-0.52;}}}else{r+=0.995876;}}else{if(f[12]<5.52774e+08){r+=0.995554;}else{r+=-0;}}}}}else{if(f[70]<1.00568){if(f[17]<9.4655e+06){r+=0.333333;}else{r+=-0.636364;}}else{r+=-0.97351;}}}}else{if(f[36]<1.26954e+08){if(f[84]<1.15){if(f[84]<0.85){if(f[70]<0.999183){if(f[0]<2.43481e+11){r+=-0.975155;}else{r+=0.555556;}}else{if(f[0]<1.40186e+11){r+=-0.666667;}else{r+=0.959596;}}}else{if(f[28]<1.39837e+11){r+=-0.764706;}else{r+=0.990805;}}}else{if(f[0]<2.43482e+11){if(f[33]<3.67141e+07){if(f[25]<7.45503e+08){if(f[55]<0.228361){r+=-0.272727;}else{if(f[82]<3.63141){if(f[0]<1.40623e+11){r+=-0.555556;}else{r+=-0;}}else{if(f[71]<4.54406e+06){r+=-0.995365;}else{r+=-0.25;}}}}else{r+=-0.111111;}}else{r+=0.25;}}else{r+=0.733333;}}}else{if(f[28]<1.98924e+11){if(f[7]<5.63097e+06){if(f[84]<1.85){r+=0.76;}else{r+=-0.6;}}else{if(f[41]<6.56305e+08){r+=0.692308;}else{if(f[64]<7.45257){if(f[39]<2.70958e+08){r+=-0.111111;}else{if(f[29]<3.80788e+06){if(f[8]<1.05647e+08){r+=-0.833333;}else{r+=0.272727;}}else{r+=-0.994267;}}}else{if(f[5]<4.61431e+07){r+=0.555556;}else{r+=-0.5;}}}}}else{if(f[0]<1.40263e+11){r+=0.909091;}else{r+=-0.25;}}}}}}else{if(f[67]<6.01752){if(f[0]<1.39262e+11){if(f[14]<1.4085e+11){if(f[14]<1.3949e+11){if(f[71]<0.00691518){if(f[28]<1.40178e+11){r+=0.769231;}else{r+=-0.5;}}else{if(f[28]<1.40178e+11){if(f[12]<3.57199e+08){r+=0.6;}else{r+=-0.818182;}}else{if(f[7]<4.95255e+10){r+=-0.991471;}else{r+=-0.25;}}}}else{if(f[56]<0.997826){if(f[25]<6.1844e+07){if(f[0]<1.39179e+11){r+=-0.25;}else{r+=-0.923077;}}else{if(f[22]<1.03032e+10){if(f[39]<5.802e+07){r+=-0.75;}else{if(f[28]<1.4007e+11){r+=-0.714286;}else{if(f[14]<1.40815e+11){r+=0.984471;}else{r+=0.750636;}}}}else{r+=-0.764706;}}}else{if(f[14]<1.40216e+11){if(f[56]<0.998696){r+=-0.75;}else{if(f[0]<1.38954e+11){r+=0.8;}else{r+=-0;}}}else{if(f[10]<5.65794e+10){r+=-0.979487;}else{r+=-0.25;}}}}}else{if(f[56]<0.996441){if(f[0]<1.39145e+11){r+=-0;}else{r+=0.826087;}}else{if(f[29]<1.39129e+09){if(f[19]<3.83145e+09){r+=-0.998631;}else{r+=-0;}}else{r+=-0.0909091;}}}}else{if(f[28]<1.40121e+11){if(f[0]<1.45794e+11){if(f[84]<0.85){if(f[28]<1.40007e+11){if(f[41]<1.15728e+08){if(f[84]<0.5){if(f[14]<1.39643e+11){if(f[9]<7.50161e+07){r+=-0.959184;}else{r+=0.98892;}}else{if(f[82]<1.13568){r+=-0.984836;}else{r+=-0.532258;}}}else{if(f[56]<1.00309){r+=-0.995713;}else{r+=0.99765;}}}else{if(f[0]<1.40255e+11){if(f[51]<0.75093){if(f[42]<1.00087){r+=-0.936977;}else{r+=0.768595;}}else{if(f[41]<5.8766e+08){r+=0.932841;}else{r+=-0.316597;}}}else{if(f[39]<4.68915e+08){if(f[68]<0.255063){r+=-0.25;}else{r+=-0.995512;}}else{if(f[53]<1.1804){r+=-0.915493;}else{r+=0.204819;}}}}}else{if(f[41]<7.28221e+07){if(f[0]<1.40032e+11){r+=-0.998137;}else{if(f[84]<0.60001){r+=-0.985915;}else{if(f[42]<1.00052){r+=0.994725;}else{r+=-0.934426;}}}}else{if(f[14]<1.40291e+11){if(f[12]<1.98452e+10){if(f[28]<1.40008e+11){r+=-0.366667;}else{r+=-0.995691;}}else{if(f[25]<9.61413e+09){r+=0.764706;}else{r+=-0.6;}}}else{if(f[70]<0.999003){if(f[34]<8.81453e+07){r+=-0.994987;}else{r+=0.0769231;}}else{if(f[0]<1.40058e+11){r+=-0.981308;}else{r+=0.92;}}}}}}else{if(f[14]<1.39082e+11){if(f[10]<2.2428e+08){if(f[55]<0.953786){if(f[83]<1.28398){if(f[28]<1.39965e+11){r+=0.125;}else{r+=-0.904762;}}else{if(f[14]<1.38842e+11){r+=-0.863014;}else{r+=0.942153;}}}else{if(f[70]<0.986123){if(f[0]<1.422e+11){r+=0.25;}else{r+=0.945205;}}else{if(f[84]<3.5){r+=-0.900574;}else{r+=0.837838;}}}}else{if(f[0]<1.40244e+11){if(f[56]<0.99123){if(f[29]<1.28744e+07){r+=-0.25;}else{r+=-0.925926;}}else{r+=0.4;}}else{if(f[67]<0.328411){if(f[0]<1.40462e+11){r+=-0.891892;}else{r+=0.6;}}else{if(f[28]<1.40061e+11){r+=0.97391;}else{r+=0.727103;}}}}}else{if(f[9]<1.16052e+08){if(f[14]<1.40098e+11){if(f[0]<1.40081e+11){if(f[28]<1.4e+11){r+=-0.580379;}else{r+=0.844337;}}else{if(f[28]<1.39939e+11){r+=0.622307;}else{r+=-0.980253;}}}else{if(f[28]<1.39809e+11){if(f[28]<1.39801e+11){r+=-0.989514;}else{r+=-0.609195;}}else{if(f[68]<1.07131){r+=0.323582;}else{r+=0.721973;}}}}else{if(f[40]<4.25603e+08){if(f[0]<1.39894e+11){if(f[14]<1.39627e+11){r+=0.988961;}else{r+=-0.224059;}}else{if(f[78]<0.355015){r+=0.106758;}else{r+=-0.696805;}}}else{if(f[84]<1.5){if(f[66]<0.995353){r+=-0.95716;}else{r+=-0.234043;}}else{if(f[56]<1.0047){r+=0.523803;}else{r+=-0.809129;}}}}}}}else{if(f[14]<1.36191e+11){r+=0.25;}else{if(f[0]<1.46435e+11){if(f[0]<1.46141e+11){r+=-0.636364;}else{r+=-0;}}else{r+=-0.999471;}}}}else{if(f[82]<0.935047){if(f[0]<1.42575e+11){if(f[84]<0.85){if(f[56]<1.00715){if(f[28]<1.42048e+11){if(f[40]<5.45355e+07){if(f[28]<1.4052e+11){r+=-0.697703;}else{r+=0.786329;}}else{if(f[14]<1.3944e+11){r+=-0.238579;}else{r+=-0.81964;}}}else{if(f[0]<1.40296e+11){if(f[0]<1.40056e+11){r+=-0.955056;}else{r+=0.997149;}}else{if(f[56]<0.982987){r+=-0.988439;}else{r+=0.940298;}}}}else{if(f[77]<0.644271){if(f[84]<1.40001){if(f[28]<1.41488e+11){r+=0.995202;}else{r+=-0.733333;}}else{r+=-0.826087;}}else{if(f[14]<1.41467e+11){r+=0.764706;}else{if(f[14]<1.43828e+11){r+=-0.968;}else{r+=-0;}}}}}else{if(f[28]<1.40144e+11){if(f[69]<1.78826){if(f[14]<1.40059e+11){if(f[10]<5.57987e+07){r+=0.555556;}else{r+=-0.976735;}}else{if(f[0]<1.40201e+11){r+=0.787007;}else{r+=-0.99376;}}}else{if(f[0]<1.40076e+11){if(f[84]<1.15){r+=-0.977612;}else{r+=0.604478;}}else{if(f[0]<1.40143e+11){r+=-0.994383;}else{r+=0.144654;}}}}else{if(f[28]<1.41846e+11){if(f[0]<1.40705e+11){if(f[37]<1.72439e+09){r+=-0.88493;}else{r+=0.270227;}}else{if(f[28]<1.40593e+11){r+=-0.991581;}else{r+=0.00440599;}}}else{if(f[14]<1.40891e+11){if(f[42]<0.997501){r+=0.929988;}else{r+=-0.99441;}}else{r+=-0.999093;}}}}}else{if(f[14]<1.39506e+11){if(f[84]<0.5){if(f[0]<1.7153e+11){r+=-0.995704;}else{r+=0.5;}}else{if(f[28]<1.40872e+11){if(f[0]<1.43001e+11){r+=0.25;}else{r+=0.92;}}else{r+=-0.333333;}}}else{if(f[56]<0.997312){if(f[84]<2.65){if(f[9]<3.92412e+10){if(f[77]<4.64104){r+=0.995888;}else{r+=-0.5;}}else{r+=-0.5;}}else{r+=-0.866667;}}else{if(f[32]<6.24662e+08){r+=-0.983051;}else{if(f[0]<1.43977e+11){r+=0.6;}else{r+=-0;}}}}}}else{if(f[14]<1.39542e+11){if(f[28]<1.40244e+11){if(f[84]<12){if(f[57]<6.85189){if(f[14]<1.38043e+11){r+=0.6;}else{if(f[0]<1.39802e+11){r+=-0;}else{r+=-0.950591;}}}else{if(f[84]<2.65){r+=0.826087;}else{r+=-0.636364;}}}else{if(f[28]<1.40152e+11){if(f[77]<1.05169){if(f[8]<1.30067e+08){r+=0.0769231;}else{r+=-0.802198;}}else{if(f[75]<0.569854){r+=-0.5;}else{r+=0.809524;}}}else{if(f[70]<0.999377){r+=-0.833333;}else{if(f[25]<3.49849e+08){r+=0.988173;}else{r+=-0.272727;}}}}}else{if(f[42]<1.00235){r+=0.977012;}else{if(f[28]<1.40256e+11){if(f[0]<1.40533e+11){r+=0.914894;}else{r+=-0.969697;}}else{if(f[14]<1.01113e+11){r+=0.5;}else{if(f[18]<2.75123e+09){r+=-0.993634;}else{r+=0.333333;}}}}}}else{if(f[84]<1.5){if(f[14]<1.39841e+11){if(f[10]<1.50958e+08){if(f[41]<8.73667e+07){if(f[0]<1.40598e+11){r+=-0.986667;}else{r+=-0.25;}}else{if(f[25]<7.74766e+07){r+=-0.283333;}else{r+=0.971014;}}}else{if(f[22]<6.16197e+07){r+=0.935484;}else{if(f[0]<1.42408e+11){r+=-0.991379;}else{r+=0.84;}}}}else{if(f[42]<0.997465){if(f[56]<1.00132){if(f[56]<0.980988){r+=-0.985075;}else{r+=0.654372;}}else{if(f[14]<1.4342e+11){r+=-0.982524;}else{r+=0.959596;}}}else{if(f[67]<2.15732){if(f[28]<1.43444e+11){r+=-0.894573;}else{r+=0.694118;}}else{if(f[84]<1.15){r+=-0.721154;}else{r+=0.579208;}}}}}else{if(f[82]<2.99309){if(f[42]<0.993959){if(f[17]<6.1794e+08){if(f[79]<4.99096){r+=-0.992652;}else{r+=-0.411765;}}else{if(f[8]<1.16645e+10){r+=0.5;}else{r+=-0.636364;}}}else{if(f[14]<1.40571e+11){if(f[28]<1.41024e+11){r+=0.132521;}else{r+=-0.800722;}}else{if(f[12]<3.67117e+08){r+=0.798577;}else{r+=-0.688591;}}}}else{if(f[84]<2.85){if(f[56]<1.00414){if(f[53]<0.167844){r+=0.157432;}else{r+=0.951508;}}else{if(f[29]<4.66466e+08){r+=-0.974468;}else{r+=0.375;}}}else{if(f[42]<0.998697){r+=-0.99466;}else{if(f[14]<1.40073e+11){r+=-0.870968;}else{r+=0.898551;}}}}}}}}}}else{if(f[28]<1.90876e+11){if(f[28]<1.40114e+11){if(f[84]<0.85){if(f[28]<1.39931e+11){if(f[6]<1.14807e+06){r+=-0;}else{if(f[0]<1.40137e+11){r+=0.991803;}else{r+=0.25;}}}else{if(f[14]<1.40326e+11){r+=-0.882353;}else{r+=-0.25;}}}else{if(f[84]<2.15){if(f[84]<1.85){if(f[64]<1.72946){if(f[51]<1.4392){if(f[13]<4.71146e+08){if(f[0]<1.40095e+11){r+=0.25;}else{r+=-0.555556;}}else{r+=0.891892;}}else{if(f[25]<7.56355e+07){r+=-0;}else{r+=-0.733333;}}}else{if(f[0]<1.39909e+11){if(f[0]<1.39795e+11){r+=-0;}else{r+=0.733333;}}else{if(f[68]<2.29064){r+=0.333333;}else{if(f[0]<1.44156e+11){r+=-0.973289;}else{r+=0.2;}}}}}else{if(f[14]<1.40112e+11){r+=-0.666667;}else{if(f[0]<1.40168e+11){if(f[1]<1.31968e+08){r+=0.987578;}else{r+=0.25;}}else{r+=-0.25;}}}}else{if(f[28]<1.40084e+11){if(f[70]<0.999505){if(f[28]<1.4006e+11){r+=-0.997476;}else{if(f[0]<1.40167e+11){r+=-0;}else{r+=-0.555556;}}}else{if(f[41]<9.76604e+07){if(f[28]<1.40069e+11){r+=-0.953488;}else{r+=-0.25;}}else{if(f[84]<3.15){r+=0.833333;}else{r+=-0.636364;}}}}else{if(f[39]<2.80354e+07){r+=0.8125;}else{r+=-0.454545;}}}}}else{if(f[68]<4.27008){if(f[84]<3.5){if(f[40]<7.44143e+08){if(f[0]<1.39308e+11){if(f[0]<1.39053e+11){r+=-0;}else{r+=0.5;}}else{if(f[27]<5.09697e+08){if(f[51]<0.813762){r+=-0.733333;}else{if(f[0]<1.40315e+11){r+=-0;}else{r+=0.5;}}}else{r+=-0.947368;}}}else{if(f[53]<0.261191){if(f[28]<1.40184e+11){if(f[25]<3.74337e+09){r+=-0;}else{r+=-0.692308;}}else{if(f[13]<7.34101e+08){r+=-0;}else{r+=0.9;}}}else{if(f[41]<1.69065e+09){if(f[21]<6.78772e+09){if(f[0]<1.40071e+11){r+=-0;}else{r+=0.75;}}else{r+=-0.636364;}}else{r+=-0.851852;}}}}else{if(f[56]<1.00083){if(f[14]<1.40072e+11){r+=-0.5;}else{if(f[0]<1.40227e+11){r+=0.97561;}else{r+=0.25;}}}else{if(f[4]<3.08531e+07){r+=-0.809524;}else{r+=-0;}}}}else{if(f[14]<1.40009e+11){if(f[28]<1.63546e+11){if(f[66]<3.78301){if(f[1]<4.77104e+08){r+=0.733333;}else{r+=-0.5;}}else{if(f[12]<6.80341e+09){if(f[42]<0.997062){r+=-0.25;}else{r+=-0.980861;}}else{if(f[15]<2.40092e+08){r+=-0.5;}else{r+=0.555556;}}}}else{r+=0.960396;}}else{if(f[2]<48333.3){r+=0.73913;}else{if(f[10]<4.78778e+09){if(f[69]<4.03027){if(f[84]<3.5){if(f[9]<6.09861e+08){r+=-0.942308;}else{r+=-0.047619;}}else{if(f[28]<1.40162e+11){r+=-0.5;}else{r+=0.8;}}}else{if(f[0]<1.42369e+11){if(f[68]<4.58306){r+=-0.542857;}else{r+=-0.992439;}}else{if(f[27]<5.1512e+10){r+=-0.714286;}else{r+=0.555556;}}}}else{if(f[84]<3.2){if(f[57]<1.04817){r+=0.0909091;}else{if(f[18]<7.44084e+08){r+=-0.8;}else{r+=-0;}}}else{if(f[6]<2.57441e+08){r+=0.833333;}else{r+=-0;}}}}}}}}else{if(f[0]<1.40062e+11){r+=-0.8;}else{r+=0.985915;}}}}} return r;}
    static double p13(double[] f){double r=0;if(f[28]<1.39505e+11){if(f[56]<1.00186){if(f[14]<1.39491e+11){if(f[43]<7.16963){if(f[56]<1.00109){if(f[18]<8.92983e+09){if(f[6]<2.00604e+06){r+=-0.185133;}else{r+=-0.68809;}}else{r+=-0.129697;}}else{r+=0.315644;}}else{if(f[56]<0.996473){r+=0.595327;}else{r+=-0.624665;}}}else{if(f[14]<1.39666e+11){if(f[0]<1.39989e+11){r+=0.685886;}else{r+=0.13552;}}else{r+=-0.468421;}}}else{if(f[10]<3.76391e+10){if(f[53]<0.0882194){r+=0.0499157;}else{r+=-0.683764;}}else{if(f[8]<3.35815e+10){if(f[36]<1.68314e+10){if(f[26]<3.77048e+10){r+=0.132371;}else{r+=-0.576346;}}else{if(f[0]<1.3948e+11){if(f[14]<1.40147e+11){r+=0.411044;}else{r+=-0.251617;}}else{r+=0.738137;}}}else{r+=-0.533801;}}}}else{if(f[0]<1.4026e+11){if(f[14]<1.40803e+11){if(f[28]<1.40629e+11){if(f[14]<1.39837e+11){if(f[56]<0.993114){if(f[84]<3.35){if(f[0]<7.05995e+10){r+=0.48344;}else{if(f[79]<2.79747e+07){if(f[28]<1.40121e+11){if(f[77]<3.48376){if(f[9]<3.05205e+08){r+=-0.702709;}else{r+=-0.586338;}}else{r+=-0.0974343;}}else{if(f[83]<0.850901){r+=-0.864698;}else{if(f[7]<2.05718e+09){r+=-0.707613;}else{r+=-0.136476;}}}}else{if(f[84]<2.65){r+=0.567377;}else{r+=-0.659978;}}}}else{if(f[12]<2.90679e+10){r+=0.824229;}else{r+=-0.954246;}}}else{if(f[26]<6.07532e+07){if(f[23]<1.12727e+08){if(f[69]<0.486986){r+=-0.716943;}else{r+=-0.933138;}}else{if(f[56]<0.994487){r+=0.843701;}else{r+=-0.592034;}}}else{if(f[84]<2.85){if(f[0]<1.40188e+11){if(f[41]<1.10133e+08){if(f[9]<1.16053e+08){if(f[14]<1.39677e+11){r+=1.37999;}else{r+=-0.61294;}}else{if(f[28]<1.40157e+11){r+=0.614944;}else{r+=1.05725;}}}else{if(f[36]<1.77214e+09){if(f[69]<0.245447){r+=-0.214167;}else{r+=0.740499;}}else{if(f[26]<1.0466e+10){r+=-0.877078;}else{r+=0.527085;}}}}else{if(f[28]<1.40152e+11){if(f[28]<1.39835e+11){r+=0.716353;}else{if(f[28]<1.40145e+11){r+=-0.716118;}else{r+=-0.144422;}}}else{if(f[41]<9.9724e+07){r+=1.0294;}else{if(f[28]<1.40158e+11){r+=-0.258661;}else{r+=0.732591;}}}}}else{if(f[15]<5.0632e+08){if(f[12]<3.31869e+08){if(f[54]<2.49823){r+=-0.836875;}else{r+=-0.676444;}}else{if(f[14]<1.39493e+11){r+=-0.654596;}else{if(f[0]<1.3905e+11){r+=-1.67836;}else{r+=-0.928092;}}}}else{if(f[29]<8.71921e+08){r+=0.608445;}else{r+=-0.0181877;}}}}}}else{if(f[14]<1.40052e+11){if(f[28]<1.40013e+11){if(f[0]<1.40059e+11){if(f[84]<4.85){if(f[28]<1.40003e+11){if(f[26]<7.37403e+08){if(f[84]<0.85){r+=-0.704735;}else{r+=-0.793592;}}else{if(f[84]<0.5){r+=-0.70845;}else{r+=-1.25416;}}}else{if(f[36]<7.74107e+07){r+=0.71414;}else{if(f[38]<4.03333e+09){r+=-0.860661;}else{r+=0.407182;}}}}else{r+=-1.64609;}}else{if(f[14]<1.39991e+11){if(f[28]<1.40001e+11){if(f[69]<0.486709){r+=-0.698846;}else{if(f[29]<1.35402e+08){r+=-0.796123;}else{r+=-0.00209855;}}}else{if(f[0]<1.40106e+11){r+=0.704102;}else{r+=-0.632336;}}}else{if(f[84]<2.5){if(f[41]<9.99395e+07){if(f[28]<1.39998e+11){r+=-0.72999;}else{r+=0.634218;}}else{if(f[83]<1.71329){r+=0.667316;}else{r+=-0.603265;}}}else{if(f[1]<1.1489e+08){if(f[0]<1.40132e+11){r+=-0.783531;}else{r+=-0.687253;}}else{r+=0.598917;}}}}}else{if(f[0]<1.40236e+11){if(f[28]<1.40386e+11){if(f[21]<8.43146e+06){if(f[0]<1.3999e+11){if(f[83]<1.69758){r+=0.709158;}else{r+=1.33632;}}else{if(f[3]<784348){r+=-0.206557;}else{r+=-0.716055;}}}else{if(f[0]<1.39825e+11){if(f[14]<1.39878e+11){r+=0.730291;}else{r+=-0.588993;}}else{if(f[13]<4.06759e+07){r+=0.028985;}else{r+=-0.743468;}}}}else{if(f[29]<2.56378e+08){if(f[78]<0.767761){if(f[52]<1.01931){r+=-0.0431542;}else{r+=-0.448894;}}else{if(f[54]<0.475748){r+=0.734532;}else{r+=1.04635;}}}else{if(f[83]<10.2539){r+=-0.693193;}else{r+=0.196769;}}}}else{if(f[84]<0.65){if(f[83]<8.127){if(f[66]<2.06985){if(f[26]<1.61804e+08){r+=1.32587;}else{r+=1.64285;}}else{if(f[69]<0.486813){r+=0.708066;}else{r+=1.19506;}}}else{if(f[9]<2.26573e+07){r+=-0.454461;}else{r+=0.660614;}}}else{if(f[6]<318876){r+=-0.102615;}else{r+=-0.79219;}}}}}else{if(f[84]<1.85){if(f[0]<1.40049e+11){if(f[0]<1.39773e+11){if(f[84]<1.15){if(f[56]<1.0015){if(f[14]<1.40248e+11){r+=-0.798029;}else{r+=1.02618;}}else{if(f[1]<3.3852e+08){r+=-0.837007;}else{r+=-0.128743;}}}else{if(f[12]<2.56531e+10){if(f[42]<0.992554){r+=-0.693246;}else{r+=-0.878424;}}else{r+=0.767685;}}}else{if(f[25]<1.79309e+08){if(f[84]<1.15){if(f[14]<1.4036e+11){r+=-0.922594;}else{r+=0.514437;}}else{if(f[28]<1.40035e+11){r+=-0.885303;}else{r+=0.656251;}}}else{if(f[78]<1.15074){if(f[28]<1.40206e+11){r+=0.288247;}else{r+=-0.333146;}}else{if(f[42]<0.998923){r+=-1.12961;}else{r+=-0.510123;}}}}}else{if(f[14]<1.40301e+11){if(f[0]<1.4016e+11){if(f[28]<1.39965e+11){if(f[14]<1.40092e+11){r+=0.7982;}else{r+=-0.529936;}}else{if(f[13]<7.814e+07){r+=0.101366;}else{r+=-0.802418;}}}else{if(f[14]<1.40187e+11){if(f[39]<2.77496e+08){r+=0.633039;}else{r+=-0.548981;}}else{if(f[9]<3.5569e+07){r+=-1.01905;}else{r+=-0.688984;}}}}else{if(f[0]<1.40166e+11){if(f[84]<1.15){if(f[84]<0.85){r+=-0.901143;}else{r+=0.910678;}}else{if(f[28]<1.40032e+11){r+=-1.44145;}else{r+=-0.666925;}}}else{if(f[50]<7.33231){if(f[28]<1.39881e+11){r+=-1.33323;}else{r+=-0.599337;}}else{r+=1.29283;}}}}}else{if(f[37]<3.14566e+07){if(f[0]<1.40122e+11){if(f[84]<2.85){if(f[79]<0.799846){if(f[56]<1.0022){r+=0.496968;}else{r+=-1.08779;}}else{if(f[10]<6.90157e+07){r+=-0.72306;}else{r+=0.262522;}}}else{if(f[28]<1.40561e+11){if(f[84]<3.5){r+=-0.960822;}else{r+=-0.603997;}}else{if(f[13]<2.47049e+08){r+=0.833759;}else{r+=-0.567403;}}}}else{if(f[14]<1.40124e+11){if(f[41]<2.97319e+08){if(f[14]<1.40098e+11){r+=-0.685425;}else{r+=-0.794386;}}else{if(f[84]<2.85){r+=-0.7963;}else{r+=0.739231;}}}else{if(f[65]<0.618334){if(f[14]<1.40565e+11){r+=-1.0418;}else{r+=0.938172;}}else{if(f[28]<1.40414e+11){r+=0.654863;}else{r+=-0.671064;}}}}}else{if(f[84]<2.5){if(f[33]<1.47176e+07){if(f[83]<2.34226){if(f[68]<0.567506){r+=-0.61004;}else{r+=0.537596;}}else{if(f[71]<0.627586){r+=-0.376508;}else{r+=-0.910975;}}}else{if(f[79]<0.881169){if(f[28]<1.40381e+11){r+=0.704169;}else{r+=-0.813812;}}else{if(f[36]<1.14245e+08){r+=0.700995;}else{r+=-1.10297;}}}}else{if(f[0]<1.4024e+11){if(f[14]<1.40062e+11){if(f[84]<3.15){r+=0.586857;}else{r+=-0.86332;}}else{if(f[23]<3.04879e+07){r+=0.188335;}else{r+=0.649501;}}}else{if(f[28]<1.39983e+11){if(f[28]<1.3979e+11){r+=-0.778719;}else{r+=0.78201;}}else{if(f[10]<1.64647e+09){r+=-1.13747;}else{r+=0.445573;}}}}}}}}}else{if(f[0]<1.3933e+11){if(f[84]<2.15){if(f[14]<1.40731e+11){if(f[38]<6.77572e+10){r+=-0.709639;}else{r+=0.0953503;}}else{if(f[12]<2.8303e+08){r+=-1.66321;}else{r+=-0.578248;}}}else{if(f[37]<7.40946e+07){r+=-0.706088;}else{if(f[69]<0.509889){if(f[1]<1.17488e+08){r+=1.5836;}else{r+=0.299885;}}else{if(f[28]<1.41196e+11){r+=-0.5238;}else{if(f[0]<1.39262e+11){r+=0.687442;}else{if(f[68]<0.753435){if(f[83]<0.697989){r+=0.682737;}else{r+=1.43307;}}else{if(f[84]<3.35){r+=0.756765;}else{r+=-0.680378;}}}}}}}}else{if(f[28]<1.77349e+11){if(f[0]<1.40247e+11){if(f[38]<4.80051e+07){if(f[14]<1.40172e+11){if(f[84]<2.5){r+=-0.557614;}else{r+=1.68077;}}else{r+=-0.693113;}}else{if(f[28]<1.42029e+11){if(f[37]<1.88459e+07){r+=0.947662;}else{if(f[28]<1.41879e+11){if(f[0]<1.39353e+11){r+=-0.364748;}else{r+=-0.723799;}}else{if(f[41]<9.91219e+07){r+=-1.34116;}else{r+=-0.758525;}}}}else{if(f[14]<1.40661e+11){if(f[29]<5.09359e+09){if(f[12]<5.81104e+07){r+=-1.03214;}else{r+=-0.70672;}}else{r+=-0.0526021;}}else{r+=0.702498;}}}}else{if(f[14]<1.40722e+11){if(f[84]<6.60001){r+=-0.691068;}else{if(f[12]<1.03046e+08){r+=-1.36068;}else{r+=-0.519942;}}}else{if(f[25]<2.68421e+08){r+=0.695698;}else{r+=-0.483686;}}}}else{if(f[14]<1.40102e+11){if(f[69]<0.474074){if(f[41]<1.88327e+09){r+=1.55401;}else{r+=0.715415;}}else{r+=0.686438;}}else{r+=-0.664797;}}}}}else{if(f[84]<0.5){if(f[28]<1.40529e+11){if(f[29]<8.25973e+08){if(f[69]<0.340927){if(f[83]<1.68731){r+=-0.657937;}else{if(f[12]<1.68014e+08){r+=-1.23119;}else{r+=-0.657262;}}}else{if(f[79]<5.52039){r+=-0.714871;}else{r+=0.251261;}}}else{if(f[68]<1.02096){r+=0.783851;}else{r+=-0.220691;}}}else{if(f[68]<1.41726){if(f[10]<1.25177e+08){if(f[69]<0.963885){if(f[56]<1.00146){if(f[0]<1.40025e+11){r+=-0.629305;}else{r+=0.65762;}}else{if(f[0]<1.39697e+11){r+=0.627906;}else{r+=1.06976;}}}else{if(f[0]<1.40087e+11){if(f[14]<1.45817e+11){r+=-0.714053;}else{r+=0.397953;}}else{r+=0.60108;}}}else{if(f[56]<1.00182){if(f[28]<1.41761e+11){if(f[38]<2.85616e+10){r+=-0.720688;}else{r+=0.515436;}}else{if(f[50]<3.5667){r+=0.700517;}else{r+=0.164615;}}}else{if(f[69]<0.36384){if(f[12]<6.2341e+08){if(f[12]<1.71385e+08){r+=1.32347;}else{r+=1.66435;}}else{r+=-0.557044;}}else{if(f[13]<1.85478e+09){if(f[69]<0.467284){if(f[0]<1.3967e+11){r+=0.418266;}else{r+=1.2817;}}else{if(f[23]<1.66302e+09){r+=0.690971;}else{r+=0.130234;}}}else{if(f[48]<1.06211){r+=0.459495;}else{r+=-0.650888;}}}}}}else{if(f[56]<1.02424){if(f[56]<0.992973){if(f[1]<1.44225e+08){r+=0.673326;}else{r+=-0.352652;}}else{if(f[29]<5.16089e+08){r+=-0.718998;}else{if(f[0]<1.40076e+11){r+=-0.180595;}else{r+=0.622814;}}}}else{if(f[67]<5.96966){r+=0.674639;}else{r+=1.13182;}}}}}else{if(f[14]<1.40843e+11){if(f[37]<9.01234e+07){r+=-0.697878;}else{if(f[84]<2.15){if(f[28]<1.41702e+11){if(f[0]<1.39358e+11){r+=-1.50022;}else{r+=-0.713563;}}else{r+=0.701038;}}else{if(f[42]<0.990887){if(f[10]<6.89585e+08){if(f[28]<1.39987e+11){if(f[69]<0.490656){r+=0.802236;}else{r+=1.06121;}}else{if(f[0]<1.39224e+11){r+=0.713401;}else{r+=-0.262336;}}}else{if(f[54]<1.00853){r+=-0.0972967;}else{r+=-0.535292;}}}else{r+=-0.655486;}}}}else{if(f[42]<0.995432){if(f[28]<1.39866e+11){if(f[9]<1.15744e+08){r+=-0.692982;}else{if(f[10]<8.93827e+08){r+=-0.888135;}else{r+=-0.0996431;}}}else{if(f[28]<1.3998e+11){if(f[14]<1.41006e+11){r+=0.884467;}else{r+=-0.673856;}}else{if(f[83]<1.69384){if(f[23]<4.37425e+10){if(f[28]<1.40144e+11){if(f[39]<1.73845e+07){r+=-1.04396;}else{r+=-0.723855;}}else{if(f[0]<1.39204e+11){r+=-0.638158;}else{r+=-0.702429;}}}else{r+=0.12642;}}else{if(f[27]<1.44449e+08){if(f[0]<1.3938e+11){if(f[69]<0.488655){r+=-1.22464;}else{r+=-0.566957;}}else{r+=-0.73559;}}else{if(f[51]<0.533006){if(f[69]<0.486144){r+=-0.682225;}else{r+=-0.990672;}}else{if(f[12]<1.71421e+08){r+=-0.771914;}else{r+=-0.717522;}}}}}}}else{if(f[12]<3.91996e+08){r+=0.654686;}else{r+=-0.574082;}}}}}}else{if(f[36]<7.41066e+06){if(f[0]<1.40328e+11){if(f[14]<1.41284e+11){if(f[28]<1.89929e+11){if(f[28]<1.39846e+11){if(f[84]<2.65){if(f[28]<1.39817e+11){r+=-0.687949;}else{if(f[84]<1.65){r+=-1.15088;}else{if(f[0]<1.40275e+11){r+=0.641694;}else{r+=0.151121;}}}}else{r+=0.731861;}}else{if(f[66]<2.58669){if(f[14]<1.40415e+11){r+=-0.962624;}else{if(f[69]<1.80926){r+=-0.701328;}else{if(f[28]<1.40144e+11){r+=-1.01314;}else{r+=-0.646274;}}}}else{if(f[59]<2.94461){r+=-0.0369505;}else{r+=-0.6851;}}}}else{r+=0.629685;}}else{if(f[57]<13.9887){r+=0.645259;}else{r+=-0.248131;}}}else{if(f[0]<1.41074e+11){if(f[79]<0.0489504){if(f[19]<3.37301e+07){r+=-0.812493;}else{r+=-0.212555;}}else{if(f[84]<1.65){if(f[84]<0.85){if(f[14]<1.40085e+11){r+=-0.470644;}else{if(f[78]<2.44193){r+=0.689046;}else{r+=1.28704;}}}else{if(f[14]<1.401e+11){r+=1.62749;}else{if(f[83]<1.69479){if(f[68]<1.07139){r+=0.86051;}else{if(f[0]<1.40749e+11){r+=0.733962;}else{r+=0.94463;}}}else{r+=1.16851;}}}}else{r+=-0.633883;}}}else{if(f[0]<1.41086e+11){if(f[7]<1.00839e+07){r+=-0.396261;}else{r+=0.0444517;}}else{r+=-0.700908;}}}}else{if(f[14]<1.39067e+11){if(f[10]<2.31061e+08){if(f[55]<0.916373){if(f[83]<3.41093){if(f[83]<1.33204){if(f[84]<3.5){if(f[56]<0.992787){if(f[41]<2.19895e+08){r+=-0.71775;}else{if(f[14]<1.38777e+11){r+=-0.509704;}else{r+=0.532716;}}}else{if(f[12]<9.9347e+07){r+=0.490917;}else{r+=-0.508433;}}}else{r+=0.724332;}}else{if(f[14]<1.38836e+11){r+=-0.674992;}else{if(f[12]<1.4256e+08){if(f[13]<9.4181e+07){r+=-0.826018;}else{if(f[17]<5.78567e+07){r+=0.686866;}else{r+=-0.423161;}}}else{if(f[69]<0.624074){r+=-0.781218;}else{r+=0.151932;}}}}}else{if(f[42]<1.01317){r+=0.584866;}else{if(f[69]<0.767807){r+=-0.690001;}else{if(f[14]<1.3886e+11){r+=-0.603209;}else{r+=0.406871;}}}}}else{if(f[84]<3.5){if(f[0]<1.42246e+11){if(f[42]<1.01119){if(f[55]<1.06147){if(f[12]<9.818e+07){r+=0.996748;}else{r+=0.204335;}}else{r+=-0.570861;}}else{if(f[52]<0.735737){if(f[61]<2.3809){if(f[36]<1.47008e+08){r+=0.950042;}else{r+=-0.030687;}}else{r+=-0.604268;}}else{if(f[55]<1.02417){if(f[56]<0.993046){r+=-0.659574;}else{r+=0.29068;}}else{if(f[84]<2.85){r+=-0.701015;}else{r+=-0.803114;}}}}}else{if(f[12]<1.87544e+08){r+=0.641157;}else{r+=-0.116262;}}}else{if(f[21]<2.08857e+08){r+=0.70733;}else{r+=0.155279;}}}}else{if(f[28]<1.40112e+11){if(f[67]<0.184535){if(f[33]<2.07112e+07){r+=-0.757414;}else{r+=-0.085473;}}else{if(f[25]<5.55091e+09){if(f[28]<1.40059e+11){if(f[9]<2.28503e+08){if(f[0]<1.42292e+11){if(f[39]<1.19952e+08){r+=0.546436;}else{r+=-0.862887;}}else{r+=0.684409;}}else{if(f[69]<0.48646){if(f[65]<1.55834){r+=1.03563;}else{r+=-0.630353;}}else{if(f[67]<0.347232){r+=-0.0189958;}else{r+=0.678425;}}}}else{if(f[84]<2.35){if(f[69]<0.354343){r+=-0.202263;}else{r+=0.740897;}}else{if(f[84]<3.35){if(f[14]<1.38879e+11){r+=-1.37761;}else{r+=-0.238703;}}else{r+=0.644073;}}}}else{if(f[38]<1.43564e+09){r+=0.196437;}else{if(f[67]<3.09126){r+=-1.25266;}else{if(f[0]<1.40278e+11){r+=-0.632207;}else{r+=-0.0956825;}}}}}}else{if(f[42]<1.14792){if(f[28]<1.40132e+11){if(f[84]<2.5){if(f[0]<1.41538e+11){r+=0.882467;}else{r+=0.248965;}}else{if(f[28]<1.40121e+11){r+=-1.24968;}else{r+=-0.575436;}}}else{if(f[6]<1.61295e+09){if(f[65]<6.36604){r+=-0.709909;}else{r+=-0.0992265;}}else{r+=0.272322;}}}else{r+=0.692571;}}}}else{if(f[28]<1.4061e+11){if(f[28]<1.40019e+11){if(f[28]<1.39955e+11){if(f[56]<0.995858){if(f[9]<2.25692e+08){if(f[38]<1.87333e+08){if(f[0]<1.41834e+11){if(f[27]<1.53448e+08){r+=-0.107178;}else{r+=0.968832;}}else{r+=-0.673366;}}else{if(f[13]<7.49936e+08){r+=-0.749536;}else{r+=-1.12209;}}}else{if(f[39]<2.5401e+08){if(f[14]<1.39082e+11){if(f[10]<2.39859e+08){r+=0.0608724;}else{r+=0.731918;}}else{if(f[69]<0.486052){r+=0.851221;}else{r+=1.475;}}}else{if(f[12]<1.6334e+08){r+=0.485208;}else{if(f[71]<0.329882){r+=0.158823;}else{r+=-0.793904;}}}}}else{if(f[0]<1.40301e+11){if(f[84]<2.65){if(f[28]<1.39813e+11){if(f[23]<2.94318e+08){r+=-0.695284;}else{r+=-0.897191;}}else{if(f[84]<2.15){r+=-0.952247;}else{r+=0.0219616;}}}else{if(f[28]<1.39859e+11){if(f[9]<1.20355e+08){r+=0.771293;}else{r+=1.04483;}}else{r+=-0.740277;}}}else{if(f[42]<0.996621){if(f[9]<1.16044e+08){if(f[69]<0.486573){r+=-0.769087;}else{r+=-1.18801;}}else{r+=-0.748823;}}else{if(f[13]<3.2086e+10){if(f[28]<1.39942e+11){r+=-0.755087;}else{r+=-0.674725;}}else{if(f[27]<3.57153e+10){r+=0.777541;}else{r+=-0.415057;}}}}}}else{if(f[84]<2.85){if(f[14]<1.40157e+11){if(f[14]<1.40133e+11){if(f[8]<9.91802e+06){r+=0.687978;}else{if(f[13]<1.6592e+10){r+=-0.738983;}else{r+=0.518623;}}}else{if(f[84]<1.15){if(f[84]<0.85){r+=-0.687865;}else{r+=0.530001;}}else{if(f[9]<1.15957e+08){r+=-1.24422;}else{r+=-0.941098;}}}}else{if(f[0]<1.4142e+11){if(f[68]<0.315363){if(f[39]<5.07458e+08){r+=-0.566996;}else{r+=0.391418;}}else{if(f[84]<1.85){r+=0.836925;}else{r+=-0.753985;}}}else{if(f[38]<1.5595e+09){if(f[0]<1.42753e+11){r+=-0.867453;}else{r+=-0.686102;}}else{if(f[42]<1.01061){r+=0.582249;}else{r+=-0.540275;}}}}}else{if(f[66]<0.784686){if(f[27]<1.55055e+08){r+=-0.712153;}else{if(f[68]<0.596483){if(f[42]<1.01219){r+=1.15024;}else{r+=-0.341488;}}else{if(f[7]<1.9227e+08){r+=-0.682999;}else{r+=-0.0683446;}}}}else{if(f[83]<1.48){if(f[28]<1.40014e+11){if(f[29]<5.8733e+07){r+=0.809203;}else{r+=-0.0724112;}}else{if(f[9]<2.77114e+08){r+=-0.796838;}else{r+=0.895415;}}}else{if(f[25]<2.39769e+08){if(f[14]<1.39083e+11){r+=0.680817;}else{r+=1.41426;}}else{r+=-0.555488;}}}}}}else{if(f[43]<138.108){if(f[42]<0.994845){if(f[84]<1.15){if(f[1]<5.5221e+08){if(f[83]<1.04733){r+=0.688568;}else{r+=1.23124;}}else{r+=-0.448324;}}else{if(f[1]<2.44663e+09){r+=-0.707989;}else{r+=0.651512;}}}else{if(f[25]<1.15215e+08){if(f[84]<1.85){if(f[28]<1.40596e+11){if(f[0]<1.40262e+11){r+=0.292952;}else{r+=-0.728207;}}else{if(f[39]<6.15905e+07){r+=0.94465;}else{r+=-0.74771;}}}else{if(f[14]<1.39542e+11){if(f[84]<2.35){r+=-0.790339;}else{r+=-0.69807;}}else{if(f[83]<0.675609){r+=-0.735538;}else{r+=-0.920253;}}}}else{if(f[23]<2.98504e+07){if(f[14]<1.39895e+11){if(f[84]<1){r+=0.675071;}else{r+=0.921048;}}else{if(f[0]<1.40265e+11){r+=-0.107413;}else{r+=-0.886209;}}}else{if(f[28]<1.40042e+11){if(f[42]<1.0054){r+=-0.791619;}else{r+=0.312377;}}else{if(f[28]<1.40598e+11){r+=-0.710178;}else{r+=-0.357454;}}}}}}else{if(f[28]<1.40192e+11){if(f[28]<1.40075e+11){r+=-0.740573;}else{if(f[84]<2.5){if(f[28]<1.40092e+11){r+=1.30061;}else{if(f[15]<921082){r+=-0.735113;}else{r+=0.696405;}}}else{r+=-0.711246;}}}else{if(f[27]<5.74656e+07){r+=-0.763247;}else{r+=-0.127471;}}}}}else{if(f[0]<1.42563e+11){if(f[84]<4.85){if(f[84]<2.15){if(f[23]<1.7794e+08){if(f[84]<1.85){if(f[0]<1.40315e+11){if(f[28]<1.42019e+11){r+=-0.741501;}else{r+=0.709018;}}else{if(f[39]<4.56704e+07){r+=0.982052;}else{r+=-0.626015;}}}else{if(f[56]<0.992554){if(f[0]<1.42133e+11){r+=-0.698756;}else{r+=-1.37676;}}else{if(f[42]<1.00229){r+=-0.78964;}else{r+=0.841205;}}}}else{if(f[28]<1.4364e+11){if(f[39]<5.20292e+07){if(f[42]<1.00217){r+=-0.652061;}else{r+=0.988194;}}else{if(f[39]<2.36716e+10){r+=-0.669234;}else{r+=0.800682;}}}else{if(f[84]<1.2){if(f[27]<2.22519e+08){r+=-0.686595;}else{r+=0.694293;}}else{if(f[9]<9.90813e+09){r+=-0.683945;}else{r+=-0.0973027;}}}}}else{if(f[52]<0.375157){if(f[42]<0.997059){if(f[14]<1.41083e+11){r+=0.690064;}else{r+=-0.432756;}}else{r+=-0.685345;}}else{if(f[41]<2.16916e+08){if(f[0]<1.40617e+11){if(f[28]<1.41743e+11){r+=-0.811569;}else{r+=0.470471;}}else{if(f[83]<0.884476){r+=-0.982773;}else{r+=-1.20769;}}}else{if(f[0]<1.40302e+11){if(f[84]<6.60001){r+=-0.712626;}else{r+=0.610037;}}else{if(f[84]<3.5){r+=-0.726767;}else{r+=-1.26248;}}}}}}else{if(f[83]<1.01522){if(f[71]<0.0927876){r+=0.256504;}else{if(f[12]<1.0466e+08){r+=0.827022;}else{r+=0.970797;}}}else{if(f[20]<6.78062e+08){if(f[69]<0.478412){r+=1.01069;}else{if(f[12]<1.0439e+08){r+=0.730327;}else{r+=0.805073;}}}else{r+=-0.440442;}}}}else{if(f[84]<2.15){if(f[28]<1.45323e+11){if(f[14]<1.3948e+11){if(f[15]<1.04741e+08){r+=-0.637662;}else{r+=0.606257;}}else{if(f[1]<2.63199e+09){if(f[56]<1.00001){if(f[12]<7.65087e+07){r+=0.78271;}else{r+=0.680432;}}else{if(f[7]<1.95144e+09){r+=-0.583964;}else{r+=0.580939;}}}else{if(f[84]<0.5){if(f[2]<1.4582e+09){r+=0.484102;}else{r+=-0.312032;}}else{if(f[4]<2.80977e+09){r+=-0.812283;}else{r+=-0.13465;}}}}}else{if(f[84]<0.85){if(f[29]<1.42651e+08){r+=-0.249158;}else{r+=0.572133;}}else{r+=-0.689982;}}}else{r+=-0.687364;}}}}}}} return r;}
    static double p14(double[] f){double r=0;if(f[28]<1.39505e+11){if(f[56]<1.00196){if(f[14]<1.39494e+11){if(f[56]<1.00152){if(f[3]<7.39214e+09){if(f[43]<5.22843){if(f[53]<0.476396){r+=-0.148929;}else{r+=-0.590507;}}else{r+=-0.0401052;}}else{if(f[28]<1.3517e+11){r+=-0.421564;}else{r+=0.410268;}}}else{r+=0.367702;}}else{if(f[14]<1.39685e+11){if(f[1]<1.10916e+08){if(f[11]<7.68127e+07){r+=0.130275;}else{r+=0.592544;}}else{r+=0.0933249;}}else{r+=-0.457768;}}}else{if(f[10]<3.56987e+10){if(f[54]<0.0892123){r+=-0.0892819;}else{if(f[9]<9.85384e+09){r+=-0.592614;}else{if(f[72]<1.4775){r+=-0.481813;}else{r+=0.419269;}}}}else{if(f[78]<0.701332){r+=-0.345896;}else{if(f[8]<8.41524e+09){r+=-0.337817;}else{if(f[43]<1.38555){r+=0.75944;}else{if(f[40]<2.72718e+10){r+=0.481091;}else{r+=-0.299911;}}}}}}}else{if(f[0]<1.40252e+11){if(f[28]<1.40675e+11){if(f[14]<1.39839e+11){if(f[14]<1.39133e+11){if(f[48]<47.5962){if(f[56]<0.993854){if(f[0]<1.40242e+11){if(f[74]<24.232){if(f[9]<3.89346e+10){r+=-0.61538;}else{r+=-0.115155;}}else{r+=0.045949;}}else{if(f[42]<1.01007){r+=0.648119;}else{if(f[72]<3.02801e+06){if(f[60]<0.332701){r+=-0.140797;}else{r+=-0.59744;}}else{if(f[62]<6.07472){r+=-0.47696;}else{r+=0.389901;}}}}}else{r+=0.263655;}}else{r+=0.493079;}}else{if(f[27]<7.78654e+07){if(f[54]<1.31059){if(f[28]<1.40163e+11){if(f[1]<6.55567e+07){r+=-0.655055;}else{r+=-0.0234185;}}else{if(f[14]<1.39676e+11){r+=0.643017;}else{r+=-0.45774;}}}else{if(f[63]<1.39317){if(f[18]<5.81637e+07){if(f[64]<1.26278){r+=-0.658244;}else{r+=-0.0859458;}}else{r+=-0.0934629;}}else{if(f[80]<0.788244){r+=-0.596625;}else{if(f[38]<1.37381e+08){r+=0.379543;}else{r+=-0.190675;}}}}}else{if(f[56]<0.992773){if(f[15]<1.01319e+08){r+=-0.584821;}else{if(f[0]<1.39989e+11){r+=0.315759;}else{r+=-0.300772;}}}else{if(f[0]<1.40157e+11){if(f[35]<8.62644e+08){if(f[41]<1.77985e+09){if(f[54]<1.69618){if(f[39]<6.58838e+07){r+=-0.824848;}else{r+=0.644326;}}else{if(f[27]<9.69342e+07){r+=-0.366497;}else{r+=0.496744;}}}else{if(f[13]<9.88768e+08){if(f[55]<0.832818){r+=0.495921;}else{r+=-0.688592;}}else{if(f[29]<1.04883e+08){r+=0.656847;}else{r+=-0.308178;}}}}else{if(f[14]<1.39601e+11){if(f[7]<4.45134e+09){r+=0.603839;}else{r+=-0.275135;}}else{if(f[27]<1.01562e+10){if(f[28]<1.39689e+11){r+=0.39277;}else{r+=-0.726764;}}else{if(f[60]<2.09791){r+=0.47017;}else{r+=-0.227438;}}}}}else{if(f[28]<1.40149e+11){if(f[28]<1.39898e+11){r+=0.606232;}else{if(f[80]<1.28304){if(f[0]<1.40188e+11){r+=-0.97295;}else{r+=-0.594938;}}else{if(f[39]<1.80108e+08){r+=0.700031;}else{r+=-0.684125;}}}}else{if(f[70]<1.4018e+11){if(f[70]<1.40176e+11){if(f[9]<2.17388e+08){r+=0.601113;}else{r+=-0.237514;}}else{if(f[14]<1.39668e+11){r+=-0.483016;}else{r+=0.528587;}}}else{if(f[14]<1.39681e+11){r+=-0.564845;}else{if(f[27]<5.64948e+08){r+=-0.418127;}else{r+=0.248908;}}}}}}}}}else{if(f[14]<1.39992e+11){if(f[51]<4.49384){if(f[49]<0.211669){if(f[1]<5.09509e+06){if(f[70]<0.999972){r+=1.36858;}else{r+=0.65749;}}else{if(f[60]<14.2658){r+=-0.4926;}else{r+=-0.0875307;}}}else{if(f[14]<1.39989e+11){if(f[0]<1.39481e+11){if(f[53]<1.28692){r+=0.580433;}else{r+=-0.317439;}}else{if(f[14]<1.3994e+11){if(f[0]<1.40027e+11){if(f[15]<3.86934e+08){r+=-0.649343;}else{r+=0.310136;}}else{if(f[29]<1.19265e+07){r+=-1.00081;}else{r+=-0.573513;}}}else{if(f[53]<0.110116){r+=0.464996;}else{if(f[28]<1.40001e+11){r+=-0.622498;}else{r+=-0.538137;}}}}}else{if(f[40]<8.20871e+07){if(f[28]<1.39954e+11){r+=-0.615648;}else{r+=0.600836;}}else{if(f[55]<3.81792){if(f[68]<2.47414){if(f[27]<9.05019e+07){r+=-0.0770058;}else{r+=-0.6372;}}else{r+=-0.086779;}}else{r+=0.164023;}}}}}else{if(f[15]<2.48706e+06){r+=0.619624;}else{r+=-0.508677;}}}else{if(f[0]<1.39787e+11){if(f[28]<1.39974e+11){if(f[28]<1.39901e+11){if(f[28]<1.39891e+11){if(f[1]<3.79643e+08){if(f[29]<2.79953e+08){if(f[22]<3.67841e+07){r+=-0.0560711;}else{r+=-0.633067;}}else{r+=0.0301256;}}else{if(f[14]<1.40437e+11){r+=0.634211;}else{r+=-0.312083;}}}else{if(f[23]<1.29069e+08){if(f[0]<1.395e+11){r+=0.655438;}else{r+=-0.143318;}}else{if(f[22]<1.03704e+08){r+=-0.0806516;}else{r+=-0.601404;}}}}else{if(f[67]<0.902149){if(f[41]<4.68446e+09){if(f[14]<1.40463e+11){r+=-0.488781;}else{if(f[9]<1.16053e+08){r+=0.680924;}else{r+=0.777164;}}}else{if(f[0]<1.39393e+11){r+=-0.0692589;}else{r+=-0.571054;}}}else{if(f[55]<2.06983){if(f[80]<0.219806){r+=0.27353;}else{if(f[68]<0.863067){r+=0.109228;}else{r+=-0.663331;}}}else{if(f[14]<1.40511e+11){r+=-0.537174;}else{if(f[14]<1.40817e+11){r+=0.766394;}else{r+=0.211499;}}}}}}else{if(f[63]<0.0654252){if(f[28]<1.40097e+11){if(f[28]<1.40047e+11){r+=-0.25996;}else{if(f[40]<1.52784e+10){r+=0.196792;}else{r+=0.735049;}}}else{if(f[1]<4.12278e+08){r+=-0.752132;}else{r+=0.341462;}}}else{if(f[14]<1.40394e+11){if(f[70]<1.00318){if(f[28]<1.40079e+11){if(f[56]<1.00169){r+=0.263197;}else{r+=-0.622286;}}else{if(f[58]<0.921063){r+=-0.417932;}else{r+=0.480784;}}}else{if(f[52]<0.0748719){if(f[55]<0.225569){r+=-0.887511;}else{r+=-1.94847;}}else{if(f[0]<1.39573e+11){r+=-0.722666;}else{r+=-0.975511;}}}}else{if(f[42]<0.994498){if(f[28]<1.3998e+11){if(f[66]<0.914703){r+=0.556479;}else{r+=-0.582851;}}else{if(f[13]<2.7855e+10){r+=-0.619375;}else{r+=0.13534;}}}else{if(f[63]<3.47497){if(f[36]<1.52152e+07){r+=0.0240572;}else{r+=0.786455;}}else{if(f[10]<7.37147e+07){r+=0.327426;}else{r+=-0.593104;}}}}}}}else{if(f[14]<1.40214e+11){if(f[38]<5.47396e+07){if(f[0]<1.40162e+11){if(f[0]<1.40075e+11){if(f[38]<5.05166e+07){if(f[56]<0.997477){r+=0.532521;}else{r+=-0.647157;}}else{if(f[14]<1.40105e+11){r+=0.318369;}else{r+=-0.504145;}}}else{if(f[14]<1.40178e+11){if(f[0]<1.40117e+11){r+=-0.688071;}else{r+=-0.59268;}}else{if(f[27]<1.28968e+08){r+=0.301563;}else{r+=-0.484729;}}}}else{if(f[28]<1.40146e+11){if(f[1]<1.83173e+07){if(f[52]<1.76103){r+=0.588368;}else{r+=-0.562701;}}else{if(f[52]<1.07212){r+=0.201581;}else{r+=-0.637126;}}}else{r+=-0.639322;}}}else{if(f[0]<1.39982e+11){if(f[52]<1.20895){if(f[63]<1.05606){if(f[27]<2.03084e+09){r+=-0.780316;}else{r+=-0.438332;}}else{if(f[14]<1.40089e+11){r+=0.342208;}else{r+=-0.633801;}}}else{if(f[14]<1.40092e+11){if(f[28]<1.39967e+11){r+=-0.560338;}else{r+=0.625189;}}else{if(f[34]<2.86611e+07){r+=-0.819549;}else{r+=0.306023;}}}}else{if(f[0]<1.40088e+11){if(f[14]<1.40176e+11){if(f[28]<1.39998e+11){r+=-0.61295;}else{r+=0.417202;}}else{if(f[38]<1.70645e+09){r+=-0.768288;}else{r+=0.6836;}}}else{if(f[28]<1.40143e+11){if(f[0]<1.40143e+11){r+=-0.213845;}else{r+=0.207921;}}else{if(f[42]<1.0016){r+=-0.507908;}else{r+=0.70263;}}}}}}else{if(f[28]<1.39821e+11){if(f[14]<1.40317e+11){if(f[28]<1.39788e+11){if(f[0]<1.40054e+11){r+=0.3425;}else{if(f[41]<1.14823e+08){r+=-0.598219;}else{r+=-0.841471;}}}else{if(f[68]<0.62837){if(f[57]<2.28275){r+=-0.575101;}else{r+=0.170636;}}else{if(f[14]<1.4031e+11){r+=0.631293;}else{r+=-0.0692794;}}}}else{if(f[42]<0.99879){if(f[14]<1.40358e+11){if(f[60]<1.64781){r+=-0.393912;}else{r+=0.621574;}}else{if(f[21]<6.28433e+08){r+=-0.609131;}else{r+=1.02022;}}}else{if(f[28]<1.39809e+11){if(f[39]<1.42559e+07){r+=-0.153071;}else{r+=-0.950673;}}else{if(f[38]<5.87544e+07){r+=-2.1461;}else{r+=-0.737396;}}}}}else{if(f[14]<1.40235e+11){if(f[28]<1.40184e+11){if(f[0]<1.40064e+11){if(f[38]<3.4986e+08){r+=-0.552405;}else{r+=0.661629;}}else{if(f[28]<1.40123e+11){r+=0.46464;}else{r+=0.733694;}}}else{if(f[23]<6.82937e+08){if(f[28]<1.40351e+11){r+=-0.816383;}else{r+=-0.613786;}}else{if(f[3]<4.84697e+08){r+=0.626825;}else{r+=0.180883;}}}}else{if(f[14]<1.40279e+11){if(f[66]<1.12939){if(f[28]<1.39881e+11){r+=0.244263;}else{r+=-0.74116;}}else{if(f[28]<1.40177e+11){r+=0.36698;}else{r+=-0.693484;}}}else{if(f[28]<1.40559e+11){if(f[28]<1.40413e+11){r+=0.288757;}else{r+=-0.781922;}}else{if(f[0]<1.4023e+11){r+=0.66779;}else{r+=-0.593941;}}}}}}}}}}else{if(f[0]<1.39314e+11){if(f[28]<1.41439e+11){if(f[78]<3.23465){if(f[46]<0.0874536){r+=-0.1285;}else{r+=-0.62913;}}else{r+=0.213245;}}else{if(f[70]<1.02134){if(f[14]<1.39142e+11){r+=-0.374127;}else{if(f[0]<1.38999e+11){r+=0.0765474;}else{if(f[14]<1.40843e+11){if(f[0]<1.39262e+11){r+=0.596181;}else{if(f[36]<1.57359e+08){if(f[40]<9.11696e+07){r+=0.614615;}else{r+=0.745862;}}else{r+=0.0676465;}}}else{r+=0.835385;}}}}else{if(f[44]<0.322965){r+=-0.185089;}else{r+=-0.602707;}}}}else{if(f[0]<1.40196e+11){if(f[56]<1.03269){if(f[38]<4.22145e+07){if(f[70]<1.0122){r+=0.712391;}else{r+=-0.588987;}}else{if(f[0]<1.4018e+11){if(f[0]<1.39341e+11){if(f[28]<1.41565e+11){r+=-0.618624;}else{if(f[78]<0.754374){if(f[0]<1.39331e+11){r+=0.545503;}else{r+=1.13171;}}else{if(f[36]<1.20988e+08){r+=0.414872;}else{r+=-0.700374;}}}}else{if(f[80]<8.45911){if(f[30]<7.19134e+08){if(f[36]<1.04831e+07){r+=0.256256;}else{r+=-0.602833;}}else{if(f[23]<2.8619e+10){r+=0.983264;}else{r+=-0.198168;}}}else{if(f[81]<8.00507){if(f[17]<6.686e+06){r+=0.00840684;}else{r+=0.536008;}}else{if(f[28]<1.40742e+11){r+=-0.0873963;}else{r+=-0.570353;}}}}}else{if(f[28]<1.42064e+11){if(f[28]<1.40681e+11){r+=0.18386;}else{r+=-0.598991;}}else{if(f[14]<1.40571e+11){r+=-0.568901;}else{r+=0.589746;}}}}}else{if(f[14]<1.5524e+11){if(f[14]<1.46496e+11){if(f[13]<1.82147e+08){if(f[28]<1.4126e+11){r+=-0.585635;}else{if(f[0]<1.39504e+11){if(f[4]<3.06354e+07){r+=-0.469097;}else{r+=-0.0904505;}}else{if(f[11]<9.6324e+07){r+=0.0653503;}else{r+=0.447062;}}}}else{if(f[40]<7.92814e+07){r+=0.0653503;}else{r+=0.561106;}}}else{if(f[28]<1.41233e+11){r+=-0.0128884;}else{r+=0.592765;}}}else{r+=-0.590378;}}}else{if(f[28]<1.42022e+11){if(f[80]<7.22649){r+=-0.603747;}else{if(f[28]<1.40714e+11){r+=0.527634;}else{r+=-0.339206;}}}else{if(f[35]<1.86902e+06){r+=-0.587241;}else{if(f[51]<2.68327){if(f[21]<1.3369e+07){r+=-0.404873;}else{if(f[23]<2.44969e+07){r+=-0.0966295;}else{if(f[53]<3.08374){if(f[1]<3.33919e+08){r+=0.600276;}else{r+=0.0672305;}}else{r+=0.0611113;}}}}else{if(f[68]<1.55396){r+=-0.656624;}else{r+=-0.124978;}}}}}}}}else{if(f[80]<1.21864){if(f[42]<1.02328){if(f[14]<1.39145e+11){if(f[28]<1.40114e+11){if(f[27]<1.19978e+08){if(f[14]<1.38978e+11){if(f[67]<1.13467){if(f[10]<2.31936e+08){r+=-0.603455;}else{r+=-0.936916;}}else{r+=-0.14537;}}else{if(f[68]<0.678874){r+=-0.397901;}else{if(f[32]<3.4594e+07){if(f[55]<0.960631){r+=0.494868;}else{if(f[10]<2.05399e+08){r+=1.23705;}else{r+=0.470238;}}}else{r+=0.109888;}}}}else{if(f[14]<1.38834e+11){if(f[14]<1.38811e+11){if(f[30]<6.98121e+07){if(f[10]<2.24318e+08){r+=-0.571524;}else{if(f[28]<1.40058e+11){r+=-1.37782;}else{r+=-0.388393;}}}else{r+=0.145753;}}else{if(f[13]<2.12599e+08){if(f[75]<0.334084){r+=-0.106089;}else{r+=-0.645736;}}else{if(f[22]<1.9396e+08){r+=-0.204372;}else{if(f[28]<1.40112e+11){r+=0.549953;}else{r+=0.0925499;}}}}}else{if(f[70]<0.98518){if(f[41]<6.34569e+08){if(f[27]<4.65293e+08){r+=0.57181;}else{r+=-0.390401;}}else{if(f[13]<6.80369e+08){r+=-0.639391;}else{r+=-1.07017;}}}else{if(f[9]<2.14325e+08){if(f[38]<1.80759e+08){if(f[40]<1.79891e+08){r+=0.620394;}else{r+=-0.345969;}}else{if(f[41]<2.35574e+08){r+=-0.641285;}else{r+=-0.011831;}}}else{if(f[64]<2.34401){if(f[14]<1.39082e+11){r+=0.593463;}else{r+=0.829437;}}else{r+=-0.902025;}}}}}}else{if(f[28]<1.40136e+11){if(f[27]<2.26214e+08){if(f[1]<3.02246e+07){r+=-0.0863194;}else{r+=-0.65668;}}else{if(f[13]<2.14007e+08){if(f[29]<3.8359e+07){r+=0.0366796;}else{r+=-0.458037;}}else{if(f[15]<4.39476e+07){if(f[33]<2.94376e+07){r+=0.808431;}else{r+=0.228922;}}else{r+=-0.0813611;}}}}else{if(f[41]<8.44562e+09){r+=-0.623765;}else{r+=0.117286;}}}}else{if(f[14]<1.40134e+11){if(f[28]<1.5736e+11){if(f[36]<2.09772e+07){if(f[28]<1.40174e+11){if(f[28]<1.39975e+11){if(f[0]<1.40361e+11){r+=-0.506013;}else{if(f[21]<4.04449e+07){r+=0.651346;}else{r+=1.3216;}}}else{if(f[0]<1.40258e+11){if(f[14]<1.40007e+11){r+=0.466129;}else{r+=-0.229465;}}else{if(f[56]<1.00088){r+=-0.658694;}else{r+=0.112641;}}}}else{if(f[14]<1.39897e+11){if(f[23]<2.98559e+07){if(f[26]<2.79192e+08){r+=0.944535;}else{r+=0.6703;}}else{r+=1.15371;}}else{r+=-0.426225;}}}else{if(f[14]<1.39197e+11){if(f[28]<1.4006e+11){if(f[41]<2.43609e+08){r+=-0.568261;}else{if(f[51]<1.08731){r+=-0.0100006;}else{r+=0.765407;}}}else{r+=-0.624146;}}else{if(f[0]<1.40259e+11){if(f[81]<1.83257){if(f[27]<2.73714e+08){r+=-0.586364;}else{r+=-0.0602981;}}else{if(f[14]<1.40009e+11){r+=0.677955;}else{r+=-0.376659;}}}else{if(f[38]<2.35233e+10){if(f[40]<1.75606e+08){r+=-0.637026;}else{r+=-0.556489;}}else{if(f[28]<1.41045e+11){r+=0.686419;}else{r+=-0.476476;}}}}}}else{r+=0.597845;}}else{if(f[28]<1.39951e+11){if(f[27]<2.21468e+08){if(f[27]<1.7397e+08){if(f[70]<0.996853){if(f[0]<1.4026e+11){r+=-0.868114;}else{if(f[70]<0.996806){r+=-0.626342;}else{r+=0.231179;}}}else{if(f[28]<1.39839e+11){if(f[19]<2.14422e+06){r+=-0.250401;}else{r+=0.547401;}}else{if(f[54]<1.33295){r+=-0.792334;}else{r+=0.23178;}}}}else{if(f[28]<1.39803e+11){if(f[40]<3.80414e+07){r+=-0.0709335;}else{if(f[5]<375525){r+=-0.152104;}else{r+=-0.621926;}}}else{if(f[28]<1.39845e+11){if(f[14]<1.40325e+11){r+=0.354699;}else{r+=-0.440467;}}else{if(f[11]<8.71251e+07){r+=-0.248244;}else{r+=-0.70522;}}}}}else{if(f[28]<1.398e+11){if(f[18]<4.11584e+08){if(f[14]<1.40199e+11){r+=0.0792189;}else{r+=-0.638378;}}else{r+=0.328316;}}else{if(f[0]<1.40528e+11){if(f[28]<1.3985e+11){if(f[29]<4.5033e+06){r+=0.173587;}else{r+=0.566387;}}else{if(f[56]<1.00311){r+=-0.679149;}else{r+=0.0879421;}}}else{if(f[20]<6.88847e+08){if(f[8]<4.76326e+09){r+=-0.657286;}else{r+=0.00225656;}}else{r+=0.625174;}}}}}else{if(f[28]<1.39985e+11){if(f[29]<1.65778e+07){if(f[39]<2.69474e+08){if(f[40]<3.57997e+07){r+=-0.545194;}else{if(f[68]<0.312766){r+=1.26331;}else{r+=0.640765;}}}else{if(f[40]<7.96019e+08){if(f[38]<2.1e+08){r+=-0.69357;}else{r+=0.359535;}}else{if(f[67]<0.892841){r+=0.968217;}else{r+=-0.0738421;}}}}else{if(f[0]<1.4048e+11){if(f[68]<1.04084){r+=-0.00529418;}else{r+=0.804137;}}else{if(f[66]<0.400141){r+=0.488859;}else{if(f[14]<1.40215e+11){r+=0.0518325;}else{r+=-0.671173;}}}}}else{if(f[14]<1.40746e+11){if(f[0]<1.40714e+11){if(f[28]<1.4e+11){if(f[14]<1.40165e+11){r+=-0.637745;}else{r+=0.45008;}}else{if(f[42]<0.996978){r+=0.384927;}else{r+=-0.650301;}}}else{if(f[0]<1.4109e+11){if(f[28]<1.4059e+11){r+=-0.481547;}else{r+=0.633311;}}else{if(f[28]<1.40018e+11){r+=0.0990534;}else{r+=-0.605379;}}}}else{if(f[0]<1.41152e+11){if(f[0]<1.40473e+11){if(f[23]<1.35534e+08){r+=0.503635;}else{r+=-0.468424;}}else{if(f[38]<4.41169e+07){r+=0.677509;}else{r+=-0.660331;}}}else{if(f[56]<1.00583){if(f[56]<0.990392){r+=0.269387;}else{r+=0.593613;}}else{if(f[63]<7.96081){r+=-0.616825;}else{r+=0.233151;}}}}}}}}}else{if(f[14]<1.39798e+11){if(f[28]<1.40842e+11){if(f[70]<0.985702){if(f[41]<8.73385e+07){if(f[14]<1.39542e+11){r+=-0.568907;}else{r+=0.337813;}}else{if(f[26]<5.92792e+07){r+=-0.532561;}else{if(f[14]<1.39515e+11){if(f[41]<1.41031e+08){if(f[70]<0.980348){r+=-0.569087;}else{r+=0.310137;}}else{if(f[8]<3.99125e+07){r+=-0.461377;}else{r+=0.588096;}}}else{if(f[66]<3.55136){if(f[8]<6.94071e+07){r+=0.0702259;}else{r+=0.611138;}}else{r+=-0.108427;}}}}}else{if(f[1]<2.61026e+08){r+=-0.649928;}else{r+=-0.0984667;}}}else{if(f[0]<1.43658e+11){if(f[11]<7.86241e+09){if(f[0]<1.4365e+11){r+=0.593139;}else{r+=0.153369;}}else{if(f[23]<4.2936e+10){r+=-0.589641;}else{r+=-0.133447;}}}else{r+=-0.583381;}}}else{if(f[0]<1.43659e+11){if(f[63]<1.38226){r+=0.438181;}else{r+=-0.302002;}}else{if(f[0]<1.43765e+11){if(f[50]<3.115){r+=-0.319542;}else{r+=-0.0510298;}}else{r+=-0.607398;}}}}}else{if(f[35]<6.55308e+06){if(f[28]<1.40009e+11){if(f[0]<1.40353e+11){r+=-0.376269;}else{if(f[55]<0.849681){r+=0.0921604;}else{r+=0.653396;}}}else{if(f[28]<1.40591e+11){if(f[14]<1.41242e+11){r+=-0.603835;}else{r+=0.0415344;}}else{if(f[0]<1.62539e+11){r+=0.790186;}else{r+=-0.382246;}}}}else{if(f[36]<2.59823e+08){if(f[42]<1.54731){if(f[28]<1.40907e+11){if(f[14]<1.2468e+11){if(f[0]<1.42477e+11){r+=-0.34825;}else{r+=0.584676;}}else{if(f[0]<1.40262e+11){if(f[14]<1.4001e+11){if(f[14]<1.39854e+11){r+=-0.512983;}else{if(f[14]<1.40009e+11){r+=0.757719;}else{r+=0.0753064;}}}else{if(f[28]<1.39875e+11){if(f[41]<4.17762e+08){r+=0.448245;}else{r+=-0.536624;}}else{if(f[41]<7.31766e+08){r+=-0.851159;}else{r+=-0.573363;}}}}else{if(f[27]<1.23914e+08){if(f[28]<1.40598e+11){if(f[14]<1.40296e+11){r+=-0.630464;}else{r+=-0.219328;}}else{if(f[28]<1.40821e+11){r+=0.565827;}else{r+=-0.63848;}}}else{if(f[28]<1.40005e+11){if(f[28]<1.39956e+11){r+=-0.587728;}else{r+=-0.325289;}}else{if(f[56]<1.00282){r+=-0.613933;}else{r+=-0.20623;}}}}}}else{if(f[28]<1.41032e+11){if(f[42]<1.00239){r+=-0.61459;}else{if(f[0]<1.40969e+11){r+=0.697971;}else{r+=-0.420633;}}}else{if(f[0]<1.41622e+11){if(f[29]<1.15369e+08){if(f[0]<1.40285e+11){if(f[28]<1.42099e+11){r+=-0.571787;}else{r+=0.632081;}}else{if(f[28]<1.4104e+11){r+=-0.0703755;}else{r+=-0.609373;}}}else{if(f[56]<0.995661){if(f[52]<1.45854){r+=0.625989;}else{r+=-0.265533;}}else{r+=-0.559827;}}}else{if(f[41]<2.33451e+08){if(f[14]<1.41136e+11){r+=-0.608867;}else{r+=-0.749426;}}else{if(f[67]<0.207728){if(f[39]<1.36465e+09){r+=0.199404;}else{r+=-0.534059;}}else{if(f[56]<0.986224){r+=-0.131207;}else{r+=0.592356;}}}}}}}else{if(f[9]<1.00338e+08){r+=-0.383898;}else{r+=0.593101;}}}else{if(f[70]<0.991631){if(f[0]<1.41393e+11){if(f[56]<1.0046){if(f[14]<1.40296e+11){if(f[55]<0.789834){if(f[40]<3.96906e+08){r+=0.0160846;}else{r+=0.517672;}}else{if(f[66]<0.310294){r+=-0.0970075;}else{r+=-0.590756;}}}else{if(f[40]<4.25627e+08){if(f[28]<1.39955e+11){r+=1.84333;}else{r+=0.951167;}}else{if(f[28]<1.39955e+11){if(f[28]<1.39848e+11){r+=-0.364326;}else{r+=1.08967;}}else{if(f[28]<1.40019e+11){r+=0.634074;}else{r+=0.7534;}}}}}else{if(f[28]<1.39928e+11){if(f[29]<1.0436e+08){r+=-0.614096;}else{r+=-0.0652783;}}else{r+=-0.0839918;}}}else{if(f[14]<1.39043e+11){if(f[81]<5.60226){r+=0.584209;}else{r+=0.0163325;}}else{if(f[7]<1.41272e+10){if(f[54]<2.782){if(f[70]<0.988933){if(f[5]<1.366e+08){r+=-0.643498;}else{r+=-0.241221;}}else{if(f[28]<1.40043e+11){r+=0.246661;}else{r+=-0.589985;}}}else{r+=0.219766;}}else{if(f[58]<0.821695){r+=0.665393;}else{r+=0.149792;}}}}}else{if(f[70]<1.0094){if(f[0]<1.40847e+11){if(f[41]<3.2589e+09){if(f[64]<0.403819){if(f[14]<1.40285e+11){if(f[1]<8.46184e+07){r+=-0.590181;}else{r+=-0.135001;}}else{if(f[64]<0.11319){r+=-0.406638;}else{r+=0.545583;}}}else{if(f[13]<1.18919e+08){if(f[38]<4.05004e+08){r+=-0.0789044;}else{r+=0.654107;}}else{if(f[39]<4.48572e+09){r+=-0.5463;}else{r+=0.407303;}}}}else{if(f[6]<2.15198e+09){r+=-0.606855;}else{r+=0.53165;}}}else{if(f[28]<1.40019e+11){if(f[51]<1.14788){if(f[0]<1.41127e+11){r+=-0.988176;}else{r+=-0.17016;}}else{if(f[36]<4.02334e+08){r+=0.51549;}else{r+=-0.429335;}}}else{if(f[3]<8.51202e+08){if(f[57]<11.9942){if(f[51]<0.432449){r+=-0.509831;}else{r+=-0.59633;}}else{r+=0.178068;}}else{r+=0.419675;}}}}else{if(f[20]<4.44418e+08){if(f[13]<6.81281e+08){if(f[0]<1.40332e+11){r+=0.453231;}else{r+=-0.520868;}}else{if(f[35]<7.54792e+09){if(f[67]<0.212116){if(f[6]<6.25434e+07){r+=0.25487;}else{r+=-0.324308;}}else{if(f[31]<7.38249e+06){r+=0.255322;}else{r+=0.602314;}}}else{r+=-0.191402;}}}else{if(f[19]<3.58958e+08){if(f[56]<0.99074){r+=-0.592647;}else{r+=-0.152438;}}else{r+=0.380067;}}}}}}}}} return r;}
    static double p15(double[] f){double r=0;if(f[28]<1.39437e+11){if(f[56]<1.00184){if(f[14]<1.39533e+11){if(f[58]<10.4954){if(f[56]<1.00112){r+=-0.963964;}else{r+=-0.25;}}else{if(f[14]<1.29261e+11){r+=-0.75;}else{r+=0.733333;}}}else{if(f[1]<8.06499e+07){r+=0.904762;}else{r+=-0.25;}}}else{if(f[10]<3.31037e+10){if(f[28]<1.39426e+11){if(f[71]<1.7998){r+=-0.996824;}else{if(f[42]<1.00946){if(f[10]<1.67755e+10){r+=-0.99583;}else{r+=-0.111111;}}else{if(f[27]<2.20314e+08){if(f[1]<2.22139e+07){r+=-0;}else{r+=0.928571;}}else{r+=-0.960784;}}}}else{if(f[14]<1.40348e+11){if(f[84]<1.15){r+=-0.973856;}else{if(f[42]<1.00248){r+=-0.809524;}else{if(f[12]<3.00623e+08){r+=0.942029;}else{r+=-0.25;}}}}else{r+=-0.988858;}}}else{if(f[56]<1.0118){if(f[77]<1.66907){r+=0.647059;}else{r+=-0.25;}}else{if(f[15]<1.68378e+10){if(f[71]<1.28013){r+=-0.833333;}else{if(f[6]<1.33638e+09){r+=-0.333333;}else{r+=0.25;}}}else{r+=0.4;}}}}}else{if(f[69]<0.548064){if(f[0]<1.40123e+11){if(f[0]<1.39989e+11){if(f[28]<1.39976e+11){if(f[84]<2.85){if(f[14]<1.39716e+11){if(f[0]<1.39675e+11){r+=-0;}else{r+=0.963303;}}else{if(f[23]<3.94389e+07){r+=0.625;}else{r+=-0.997232;}}}else{if(f[14]<1.40433e+11){r+=-0.988372;}else{if(f[0]<1.39694e+11){r+=0.998494;}else{r+=-0.6;}}}}else{if(f[82]<8.33145){if(f[49]<5.09367){if(f[66]<1.33276){if(f[43]<10.0923){if(f[0]<1.37972e+11){if(f[28]<1.40224e+11){if(f[5]<8.06226e+06){r+=0.902439;}else{r+=-0;}}else{r+=-0.733333;}}else{if(f[84]<0.5){if(f[56]<0.996204){r+=-0.996894;}else{r+=0.266862;}}else{if(f[49]<3.67881){r+=-0.97016;}else{r+=-0.445614;}}}}else{if(f[29]<7.63036e+06){if(f[10]<6.46004e+07){r+=-0.5;}else{if(f[0]<1.14137e+11){r+=0.25;}else{r+=0.956044;}}}else{r+=-0.833333;}}}else{if(f[0]<1.39777e+11){if(f[77]<0.986855){if(f[34]<3.36043e+07){if(f[66]<1.34031){r+=-0.25;}else{r+=-0.98995;}}else{if(f[12]<1.33835e+08){r+=0.666667;}else{r+=-0.864865;}}}else{if(f[84]<0.8){if(f[0]<1.39735e+11){r+=-0;}else{r+=0.818182;}}else{r+=-0.777778;}}}else{if(f[84]<0.5){if(f[28]<1.40151e+11){r+=-0.878788;}else{if(f[28]<1.40661e+11){r+=0.979592;}else{r+=-0.84;}}}else{if(f[0]<1.39986e+11){if(f[37]<2.23493e+07){r+=0.2;}else{r+=-0.738019;}}else{if(f[28]<1.40179e+11){r+=-0.76;}else{r+=0.469388;}}}}}}else{if(f[14]<1.4017e+11){if(f[28]<1.39995e+11){if(f[9]<1.03094e+10){r+=-0.866667;}else{r+=0.6;}}else{if(f[84]<2.5){if(f[14]<1.39189e+11){r+=-0.555556;}else{if(f[10]<7.19197e+07){r+=0.25;}else{r+=0.992453;}}}else{r+=-0.833333;}}}else{r+=-0.966387;}}}else{if(f[70]<1.00568){if(f[84]<3.5){if(f[14]<1.40242e+11){if(f[28]<1.4018e+11){if(f[0]<1.39984e+11){if(f[64]<9.79845){r+=-0.952756;}else{r+=-0;}}else{if(f[78]<0.534204){r+=-0.047619;}else{r+=-0.641026;}}}else{if(f[69]<0.148642){if(f[12]<1.15044e+08){r+=-0.873016;}else{r+=-0.155556;}}else{if(f[53]<0.974946){r+=0.335677;}else{r+=-0.279279;}}}}else{if(f[56]<1.00193){if(f[56]<1.00067){r+=-0.0909091;}else{if(f[15]<1.70854e+07){r+=0.25;}else{r+=0.964158;}}}else{r+=-0.809524;}}}else{if(f[27]<2.61238e+08){r+=-0.809524;}else{r+=0.993865;}}}else{r+=-0.992453;}}}}else{if(f[55]<0.864785){if(f[14]<1.40142e+11){if(f[28]<1.40182e+11){if(f[0]<1.4007e+11){if(f[70]<1.00111){if(f[16]<2.06064e+07){if(f[0]<1.40067e+11){if(f[23]<2.67046e+07){r+=-0.25;}else{r+=-0.978142;}}else{if(f[77]<14.195){r+=0.714286;}else{r+=-0.733333;}}}else{r+=0.6;}}else{if(f[84]<2.5){if(f[51]<0.769587){if(f[28]<1.40179e+11){r+=-0;}else{r+=0.764706;}}else{if(f[47]<1.60894){r+=0.0714286;}else{r+=-0.515152;}}}else{r+=0.965217;}}}else{if(f[0]<1.40105e+11){if(f[41]<3.35368e+08){r+=-0;}else{if(f[84]<1.40001){r+=0.993737;}else{r+=-0;}}}else{if(f[1]<2.94381e+07){r+=-0.75;}else{r+=-0;}}}}else{if(f[70]<1.00373){if(f[32]<3.3928e+08){if(f[28]<1.40186e+11){if(f[84]<2.5){if(f[69]<0.147485){r+=-0.473684;}else{r+=0.732283;}}else{if(f[0]<1.40028e+11){r+=0.995392;}else{r+=-0;}}}else{if(f[15]<7.87064e+07){r+=0.99353;}else{r+=0.25;}}}else{if(f[84]<0.5){r+=-0.885714;}else{r+=0.964602;}}}else{r+=-0.714286;}}}else{if(f[14]<1.40211e+11){if(f[10]<4.49357e+07){r+=-0.25;}else{r+=-0.984848;}}else{if(f[28]<1.40405e+11){if(f[84]<0.85){r+=-0.684211;}else{if(f[70]<1.00046){if(f[4]<1.07183e+07){r+=0.4;}else{r+=-0.6;}}else{if(f[28]<1.40398e+11){if(f[29]<7.09304e+07){r+=0.947368;}else{r+=-0;}}else{if(f[75]<0.235716){r+=-0.666667;}else{r+=0.789474;}}}}}else{if(f[14]<1.40532e+11){if(f[0]<1.40056e+11){r+=-0.5;}else{r+=0.25;}}else{r+=-0.942857;}}}}}else{if(f[84]<2.5){if(f[63]<0.479668){if(f[39]<1.36472e+09){if(f[28]<1.40613e+11){if(f[14]<1.4019e+11){if(f[0]<1.40099e+11){if(f[22]<2.71848e+08){r+=0.994461;}else{r+=-0.25;}}else{r+=-0.454545;}}else{if(f[14]<1.40392e+11){r+=-0.833333;}else{if(f[28]<1.404e+11){r+=0.878788;}else{r+=-0.428571;}}}}else{r+=-0.777778;}}else{if(f[79]<0.910981){if(f[78]<1.28437){r+=-0.333333;}else{r+=0.555556;}}else{if(f[8]<2.17763e+09){r+=-0.954023;}else{r+=-0.25;}}}}else{if(f[0]<1.4007e+11){if(f[1]<1.06358e+07){if(f[70]<1.00129){if(f[27]<1.08609e+08){r+=0.5;}else{if(f[82]<0.367993){r+=0.25;}else{r+=-0.989247;}}}else{if(f[82]<1.65201){if(f[33]<265278){r+=0.636364;}else{r+=-0.93921;}}else{if(f[28]<1.40483e+11){r+=0.798561;}else{r+=-0.666667;}}}}else{if(f[67]<0.496507){if(f[56]<0.99458){r+=-0.75;}else{if(f[21]<4.62685e+07){r+=-0.157895;}else{r+=0.91358;}}}else{if(f[28]<1.39858e+11){r+=0.818182;}else{if(f[0]<1.40051e+11){r+=-0.812081;}else{r+=0.133333;}}}}}else{if(f[41]<6.6774e+08){if(f[28]<1.40398e+11){if(f[67]<0.551502){if(f[84]<1.15){r+=0.911111;}else{r+=0.0309278;}}else{if(f[14]<1.40034e+11){r+=-0.789474;}else{r+=0.952995;}}}else{if(f[39]<1.8188e+08){if(f[69]<0.530302){r+=-0.971631;}else{r+=-0.25;}}else{if(f[43]<2.61977){r+=-0.6;}else{r+=0.555556;}}}}else{if(f[42]<0.997811){if(f[28]<1.40402e+11){if(f[1]<8.7156e+07){r+=0.946667;}else{r+=0.25;}}else{if(f[14]<1.40516e+11){r+=0.333333;}else{r+=-0.857143;}}}else{if(f[73]<6.09516){if(f[28]<1.39935e+11){r+=-0;}else{r+=-0.896104;}}else{if(f[31]<1.15866e+08){r+=-0;}else{r+=0.666667;}}}}}}}else{if(f[21]<1.33068e+08){if(f[55]<0.904136){if(f[41]<1.36136e+09){if(f[36]<2.66045e+07){r+=0.25;}else{r+=-0.924528;}}else{if(f[0]<1.40006e+11){r+=0.914894;}else{r+=-0.25;}}}else{if(f[22]<2.45645e+08){if(f[14]<1.39427e+11){if(f[1]<2.43296e+08){r+=0.75;}else{r+=-0.6;}}else{if(f[2]<1.13499e+08){if(f[82]<7.23226){r+=-0.981319;}else{r+=-0.702703;}}else{r+=0.25;}}}else{r+=0.789474;}}}else{if(f[83]<2.66243){if(f[0]<1.40036e+11){if(f[63]<3.28069){r+=-0.777778;}else{r+=0.636364;}}else{if(f[14]<1.39028e+11){r+=-0.625;}else{if(f[70]<1.00133){if(f[71]<16.4958){r+=0.98419;}else{r+=-0.5;}}else{if(f[27]<4.16008e+08){r+=-0.538462;}else{r+=0.333333;}}}}}else{if(f[79]<0.482262){r+=0.73913;}else{if(f[14]<1.39616e+11){r+=0.6;}else{if(f[82]<1.40948){r+=-0;}else{r+=-0.939394;}}}}}}}}}else{if(f[70]<1.00915){if(f[41]<5.52278e+08){if(f[23]<1.24551e+08){if(f[42]<0.998035){if(f[14]<1.40618e+11){if(f[84]<2.15){if(f[12]<1.47338e+07){r+=0.973154;}else{r+=-0.981735;}}else{if(f[56]<1.00115){r+=-0.806452;}else{if(f[28]<1.40165e+11){r+=-0.75;}else{r+=0.992908;}}}}else{if(f[84]<1.15){if(f[84]<0.5){r+=-0.933333;}else{if(f[7]<1.03423e+08){r+=0.978723;}else{r+=-0.25;}}}else{if(f[14]<1.40626e+11){if(f[84]<1.8){r+=-0.916667;}else{r+=0.897436;}}else{if(f[0]<1.40191e+11){if(f[84]<12){r+=-0.75;}else{r+=0.555556;}}else{if(f[14]<1.40628e+11){r+=-0.5;}else{r+=-0.99632;}}}}}}else{if(f[23]<1.06635e+08){if(f[28]<1.39876e+11){if(f[56]<1.00317){if(f[84]<1.5){if(f[84]<1){r+=-0.6;}else{r+=0.957895;}}else{r+=-0.8;}}else{if(f[70]<0.996687){r+=-0.994065;}else{r+=-0;}}}else{if(f[25]<1.0375e+08){if(f[28]<1.39985e+11){if(f[36]<9.60307e+06){r+=0.888889;}else{r+=-0.924067;}}else{if(f[14]<1.41212e+11){r+=-0.974527;}else{r+=0.454545;}}}else{if(f[14]<1.40873e+11){if(f[55]<3.30241){r+=-0.939699;}else{r+=0.333333;}}else{if(f[6]<6.73058e+06){r+=-0.5;}else{r+=0.84;}}}}}else{if(f[0]<1.41604e+11){if(f[84]<2.85){if(f[28]<1.39547e+11){if(f[14]<1.40128e+11){r+=0.865546;}else{r+=-0.970149;}}else{if(f[83]<0.953978){r+=-0.5;}else{r+=-0.964653;}}}else{if(f[56]<1.00049){if(f[14]<1.39156e+11){r+=0.555556;}else{r+=-0.864476;}}else{if(f[56]<1.00406){r+=0.875912;}else{r+=-0.594203;}}}}else{if(f[27]<1.21956e+08){r+=-0.982833;}else{if(f[14]<1.40188e+11){if(f[84]<2.15){r+=-0.555556;}else{r+=0.529412;}}else{if(f[15]<7.81678e+07){r+=0.985611;}else{r+=0.25;}}}}}}}else{if(f[84]<3.5){if(f[42]<1.00257){if(f[0]<1.40156e+11){if(f[28]<1.4039e+11){if(f[66]<0.556371){r+=-0.2;}else{r+=0.972028;}}else{r+=-0.846154;}}else{if(f[0]<1.40179e+11){if(f[84]<2.15){if(f[0]<1.40178e+11){r+=-0.913043;}else{r+=-0.25;}}else{if(f[12]<1.72643e+08){r+=0.764706;}else{r+=-0.555556;}}}else{if(f[14]<1.39903e+11){r+=-0;}else{r+=-0.996109;}}}}else{if(f[14]<1.38919e+11){if(f[14]<1.38875e+11){if(f[12]<4.35032e+08){r+=-0.980738;}else{r+=-0.25;}}else{if(f[9]<2.44462e+08){r+=-0.976331;}else{if(f[25]<1.15647e+08){r+=-0.6;}else{r+=0.910112;}}}}else{if(f[79]<1.19794){if(f[84]<1.15){if(f[14]<1.39663e+11){r+=0.7;}else{r+=-0.813853;}}else{if(f[56]<0.992281){r+=-0.818182;}else{r+=0.736472;}}}else{if(f[28]<1.39679e+11){if(f[0]<1.40178e+11){r+=0.25;}else{r+=0.948052;}}else{if(f[14]<1.40823e+11){r+=-0.931035;}else{r+=0.0833333;}}}}}}else{if(f[28]<1.40962e+11){if(f[10]<2.53297e+08){if(f[28]<1.40153e+11){if(f[14]<1.39222e+11){r+=0.960396;}else{if(f[42]<1.00012){r+=0.636364;}else{r+=-0.907834;}}}else{if(f[42]<1.007){if(f[27]<7.88225e+07){r+=-0.25;}else{r+=0.955335;}}else{r+=-0.666667;}}}else{if(f[56]<1.00327){if(f[25]<1.96689e+08){if(f[63]<2.43396){r+=0.994097;}else{r+=0.384615;}}else{r+=-0.25;}}else{if(f[1]<4.34678e+07){r+=-0.555556;}else{r+=0.25;}}}}else{r+=-0.982063;}}}}else{if(f[84]<0.5){if(f[37]<9.63322e+07){if(f[27]<1.83385e+08){r+=-0.8125;}else{if(f[14]<1.40044e+11){if(f[14]<1.3969e+11){r+=0.25;}else{r+=0.996974;}}else{r+=-0.826087;}}}else{if(f[28]<1.42861e+11){if(f[79]<0.109082){if(f[0]<1.40258e+11){r+=0.909091;}else{r+=0.25;}}else{if(f[29]<4.2632e+06){if(f[36]<1.03195e+08){if(f[15]<4.88666e+06){r+=0.666667;}else{r+=-0.555556;}}else{r+=-0.878788;}}else{if(f[34]<1.5165e+06){if(f[10]<1.14197e+08){r+=0.5;}else{r+=-0.826087;}}else{if(f[28]<1.39851e+11){r+=-0.612903;}else{r+=-0.993631;}}}}}else{r+=0.969697;}}}else{if(f[83]<1.43781){if(f[0]<1.41638e+11){if(f[71]<0.107204){if(f[56]<0.999747){if(f[27]<2.12106e+08){r+=-0;}else{r+=0.904762;}}else{if(f[28]<1.39982e+11){if(f[4]<1.18181e+07){r+=-0.25;}else{r+=0.666667;}}else{r+=-0.666667;}}}else{if(f[28]<1.39884e+11){if(f[56]<1.00306){if(f[37]<2.94774e+08){r+=-0.25;}else{r+=0.954023;}}else{if(f[6]<1.99633e+08){r+=-0.949367;}else{r+=-0.25;}}}else{if(f[70]<0.990464){if(f[1]<4.29436e+07){r+=-0;}else{r+=0.733333;}}else{if(f[14]<1.39713e+11){r+=-0.383333;}else{r+=-0.87234;}}}}}else{if(f[82]<0.774479){if(f[8]<7.5386e+09){if(f[12]<3.78369e+08){r+=-0;}else{r+=-0.785714;}}else{r+=0.6;}}else{if(f[15]<5.71096e+06){r+=-0.555556;}else{if(f[84]<2.5){if(f[28]<1.3976e+11){r+=-0;}else{r+=0.920792;}}else{r+=-0.5;}}}}}else{if(f[41]<6.77625e+08){if(f[23]<1.07943e+08){if(f[28]<1.3953e+11){r+=0.692308;}else{if(f[42]<0.997499){if(f[84]<1.15){r+=0.727273;}else{r+=-0.904762;}}else{r+=-0.99323;}}}else{if(f[0]<1.41535e+11){if(f[84]<3.5){if(f[0]<1.40243e+11){r+=-0.0108696;}else{r+=-0.883954;}}else{if(f[7]<1.18561e+08){r+=-0;}else{r+=0.958763;}}}else{if(f[28]<1.40009e+11){if(f[0]<1.41749e+11){r+=0.5;}else{r+=-0.968504;}}else{if(f[78]<1.67929){r+=0.967638;}else{r+=-0.636364;}}}}}else{if(f[42]<1.61529){if(f[14]<1.25637e+11){if(f[0]<1.42573e+11){r+=-0.846154;}else{r+=0.875;}}else{if(f[4]<21167.9){r+=0.647059;}else{if(f[40]<2.02792e+08){r+=-0.16129;}else{r+=-0.985007;}}}}else{r+=0.878788;}}}}}}else{if(f[84]<1.5){if(f[55]<5.02302){if(f[28]<1.42022e+11){r+=-0.911111;}else{if(f[27]<1.80579e+08){if(f[84]<1){r+=0.976879;}else{if(f[14]<1.40359e+11){r+=-0.951219;}else{r+=-0;}}}else{if(f[51]<5.07245){if(f[14]<1.39922e+11){r+=-0.25;}else{if(f[25]<8.83648e+09){r+=0.998643;}else{if(f[0]<1.41278e+11){r+=0.555556;}else{r+=-0;}}}}else{if(f[0]<1.41051e+11){r+=-0.75;}else{r+=0.636364;}}}}}else{if(f[82]<4.16017e+08){if(f[66]<1.04101){if(f[70]<1.0129){r+=-0.0909091;}else{if(f[1]<8.91542e+07){if(f[28]<1.40175e+11){r+=-0.25;}else{r+=-0.975309;}}else{r+=-0.25;}}}else{r+=0.25;}}else{r+=0.333333;}}}else{if(f[28]<1.40178e+11){if(f[23]<2.7325e+07){r+=-0.25;}else{r+=0.733333;}}else{r+=-0.996444;}}}}}else{if(f[14]<1.39855e+11){if(f[14]<1.39524e+11){if(f[28]<1.40023e+11){if(f[28]<1.39691e+11){if(f[83]<0.936034){if(f[37]<7.66755e+07){r+=-0.789474;}else{if(f[69]<0.852971){if(f[0]<1.42422e+11){r+=-0.6;}else{r+=0.636364;}}else{if(f[1]<3.81727e+09){if(f[30]<2.11883e+06){r+=0.25;}else{if(f[57]<0.250319){r+=0.25;}else{r+=0.981308;}}}else{r+=-0;}}}}else{if(f[0]<1.40186e+11){r+=0.647059;}else{if(f[0]<1.40443e+11){r+=-0.994406;}else{r+=0.25;}}}}else{if(f[9]<7.643e+07){r+=-0.987179;}else{if(f[55]<0.187954){if(f[0]<1.40496e+11){r+=-0.979695;}else{r+=-0.25;}}else{if(f[79]<1.17918){if(f[13]<9.75483e+07){r+=-0.769231;}else{if(f[66]<0.356841){if(f[57]<3.18883){r+=0.636364;}else{r+=-0.818182;}}else{if(f[28]<1.39994e+11){r+=0.964136;}else{r+=0.792415;}}}}else{if(f[84]<2){if(f[23]<1.19353e+10){if(f[83]<2.19799){r+=0.978261;}else{r+=-0.125;}}else{r+=-0.555556;}}else{if(f[55]<0.884309){if(f[84]<2.85){r+=-0.862069;}else{r+=0.842975;}}else{if(f[9]<2.98824e+08){r+=-0.92891;}else{r+=0.642857;}}}}}}}}else{if(f[84]<12){if(f[0]<1.43618e+11){if(f[43]<15.0807){if(f[28]<1.40058e+11){if(f[9]<2.35131e+08){if(f[83]<1.69227){if(f[40]<6.97167e+07){r+=-0.25;}else{r+=-0.992322;}}else{if(f[84]<3.5){r+=-0.4;}else{r+=0.764706;}}}else{if(f[84]<2.85){if(f[8]<2.37125e+09){r+=-0.944444;}else{r+=0.5;}}else{if(f[13]<1.77849e+08){r+=0.52381;}else{r+=0.983193;}}}}else{if(f[0]<1.42966e+11){if(f[53]<0.237357){if(f[0]<1.39811e+11){r+=0.771429;}else{r+=-0.906977;}}else{if(f[42]<0.986726){r+=-0.15;}else{r+=-0.991464;}}}else{if(f[70]<0.979876){r+=-0.960784;}else{if(f[27]<6.54919e+09){r+=0.918367;}else{r+=-0.6;}}}}}else{if(f[28]<1.40178e+11){if(f[14]<1.3886e+11){if(f[0]<1.41258e+11){r+=-0;}else{r+=-0.714286;}}else{if(f[12]<2.46486e+08){if(f[0]<1.40778e+11){r+=-0.5;}else{r+=-0;}}else{if(f[63]<0.715407){r+=0.0909091;}else{r+=0.949275;}}}}else{r+=-0.906977;}}}else{if(f[84]<0.5){if(f[0]<1.7153e+11){r+=-0.8;}else{r+=0.5;}}else{if(f[28]<1.41181e+11){if(f[67]<3.43781){r+=0.989362;}else{r+=-0;}}else{if(f[0]<1.46034e+11){r+=-0;}else{r+=-0.5;}}}}}else{if(f[0]<1.40211e+11){if(f[28]<1.40152e+11){if(f[77]<1.09272){if(f[28]<1.40147e+11){if(f[66]<0.622827){r+=-0.0909091;}else{if(f[42]<1.00535){r+=-0.333333;}else{r+=-0.961538;}}}else{if(f[9]<1.50326e+08){if(f[8]<1.54207e+08){r+=0.75;}else{r+=-0;}}else{if(f[8]<1.32915e+08){r+=0.333333;}else{r+=-0.8125;}}}}else{if(f[66]<0.752388){if(f[10]<2.98155e+08){r+=0.949367;}else{r+=-0.4;}}else{if(f[47]<0.593324){if(f[4]<1.76873e+07){r+=-0.25;}else{r+=0.555556;}}else{r+=-0.846154;}}}}else{if(f[39]<6.19404e+08){if(f[28]<1.40158e+11){if(f[40]<6.69734e+07){if(f[0]<1.40163e+11){r+=-0.666667;}else{r+=0.25;}}else{if(f[13]<9.73506e+07){r+=-0.333333;}else{r+=0.862069;}}}else{if(f[51]<1.69493){if(f[39]<6.8688e+07){r+=0.25;}else{r+=0.995117;}}else{if(f[5]<3.97407e+07){r+=0.5;}else{r+=-0.25;}}}}else{if(f[73]<1.09198){r+=0.4;}else{r+=-0.636364;}}}}else{if(f[70]<0.999744){if(f[0]<1.40247e+11){if(f[28]<1.4016e+11){if(f[28]<1.40148e+11){if(f[12]<1.67385e+08){r+=-0.983471;}else{r+=-0.25;}}else{if(f[28]<1.4015e+11){r+=0.5;}else{r+=-0.428571;}}}else{if(f[8]<1.87559e+08){r+=0.914894;}else{r+=-0;}}}else{if(f[70]<0.999206){r+=-0.998236;}else{r+=0.111111;}}}else{if(f[0]<1.40237e+11){r+=0.875;}else{r+=-0;}}}}}}else{if(f[22]<4.42054e+08){if(f[84]<3.15){if(f[25]<6.89099e+07){if(f[25]<6.4543e+07){if(f[42]<0.996716){if(f[0]<1.38993e+11){r+=-0.5;}else{r+=0.25;}}else{if(f[23]<2.04395e+08){if(f[29]<8.91451e+07){r+=-0.989779;}else{r+=-0.25;}}else{r+=-0.25;}}}else{if(f[14]<1.39658e+11){if(f[0]<1.39616e+11){if(f[7]<1.91755e+08){r+=0.714286;}else{r+=-0;}}else{if(f[40]<5.92798e+07){r+=-0.111111;}else{r+=-0.964912;}}}else{if(f[51]<1.58485){if(f[84]<1){r+=-0.894737;}else{if(f[69]<0.864579){r+=-0.0508475;}else{r+=0.830769;}}}else{if(f[0]<1.39324e+11){r+=0.538462;}else{if(f[10]<1.48766e+08){r+=-0.25;}else{r+=-0.945205;}}}}}}else{if(f[66]<0.734946){if(f[42]<1.00631){if(f[66]<0.385499){if(f[82]<2.32879){if(f[28]<1.40093e+11){r+=-0.989247;}else{r+=0.529412;}}else{if(f[0]<1.40386e+11){r+=0.945205;}else{r+=-0.25;}}}else{if(f[84]<2.85){if(f[28]<1.40831e+11){r+=0.90133;}else{r+=-0.954545;}}else{if(f[28]<1.41056e+11){r+=-0.955556;}else{r+=0.925926;}}}}else{if(f[21]<4.8102e+07){r+=0.956522;}else{if(f[69]<0.560623){if(f[1]<5.23444e+07){r+=-0.5;}else{r+=-0;}}else{if(f[1]<2.766e+08){r+=-0.995935;}else{r+=-0.25;}}}}}else{if(f[28]<1.42203e+11){if(f[70]<1.4018e+11){if(f[27]<5.16135e+08){if(f[84]<0.5){r+=-0.323323;}else{r+=0.82904;}}else{if(f[84]<2.85){r+=0.984825;}else{r+=-0.871795;}}}else{if(f[25]<1.29455e+08){if(f[40]<3.92401e+08){r+=-0.955882;}else{r+=-0.333333;}}else{if(f[84]<2.65){r+=0.862069;}else{r+=-0.764706;}}}}else{r+=-0.985075;}}}}else{if(f[36]<2.61353e+08){r+=-0.996269;}else{r+=-0.0909091;}}}else{if(f[28]<1.39786e+11){if(f[28]<1.39449e+11){if(f[42]<1.0035){r+=0.625;}else{r+=-0.8;}}else{if(f[15]<8.61816e+08){if(f[68]<0.37383){r+=-0;}else{if(f[28]<1.39466e+11){if(f[84]<1){r+=-0.555556;}else{r+=0.846154;}}else{if(f[39]<8.42978e+09){if(f[55]<0.122552){r+=0.25;}else{r+=0.984887;}}else{r+=0.272727;}}}}else{r+=-0.4;}}}else{if(f[70]<0.982695){if(f[15]<7.16419e+08){if(f[1]<3.55926e+08){r+=0.962963;}else{r+=0.25;}}else{r+=-0.4;}}else{if(f[55]<0.252857){if(f[0]<1.39688e+11){r+=0.961165;}else{if(f[1]<6.89859e+06){r+=0.25;}else{r+=-0.833333;}}}else{if(f[70]<1.01831){if(f[84]<1.5){if(f[67]<0.666912){if(f[28]<1.4036e+11){r+=-0.5;}else{r+=0.826087;}}else{if(f[56]<0.983822){r+=-0;}else{r+=-0.80303;}}}else{if(f[15]<4.99183e+08){if(f[55]<0.323115){r+=-0.444444;}else{r+=-0.995607;}}else{if(f[51]<1.1705){r+=-0.5;}else{r+=0.764706;}}}}else{if(f[84]<2.65){r+=-0.692308;}else{r+=0.870968;}}}}}}}}else{if(f[28]<1.40121e+11){if(f[28]<1.39809e+11){if(f[14]<1.39911e+11){if(f[70]<0.997118){if(f[19]<1.94426e+06){r+=0.25;}else{r+=0.981538;}}else{r+=-0.625;}}else{if(f[10]<3.30625e+08){if(f[56]<1.00292){if(f[0]<1.40081e+11){r+=-0.911111;}else{if(f[40]<7.7995e+07){r+=-0.647059;}else{if(f[2]<1.65459e+07){r+=0.875;}else{r+=-0;}}}}else{if(f[28]<1.39801e+11){if(f[17]<3.06973e+08){if(f[14]<1.39919e+11){r+=-0.25;}else{r+=-0.996714;}}else{r+=-0;}}else{if(f[14]<1.40296e+11){if(f[4]<1.69984e+07){r+=0.906977;}else{r+=-0;}}else{if(f[66]<4.30182){if(f[32]<139491){r+=-0.25;}else{r+=-0.980892;}}else{if(f[4]<3.25979e+06){r+=-0;}else{r+=0.5;}}}}}}else{if(f[42]<1.00903){if(f[70]<0.996951){if(f[1]<7.88719e+08){if(f[14]<1.40126e+11){if(f[12]<1.20726e+09){r+=0.5;}else{r+=-0.6;}}else{if(f[64]<0.155674){r+=-0.272727;}else{r+=-0.987993;}}}else{if(f[1]<1.67693e+09){r+=-0;}else{r+=0.666667;}}}else{if(f[70]<0.998279){if(f[84]<1.5){if(f[0]<1.40113e+11){r+=-0;}else{r+=-0.636364;}}else{if(f[76]<0.227856){r+=-0.5;}else{r+=0.896552;}}}else{if(f[12]<3.24752e+10){if(f[49]<0.578515){r+=-0.2;}else{r+=-0.939394;}}else{r+=0.555556;}}}}else{if(f[0]<1.42e+11){if(f[0]<1.41545e+11){r+=0.25;}else{r+=0.990588;}}else{if(f[4]<1.96451e+07){r+=-0;}else{r+=-0.909091;}}}}}}else{if(f[14]<1.3999e+11){if(f[10]<1.61424e+07){if(f[0]<1.3988e+11){r+=-0.636364;}else{r+=0.932203;}}else{if(f[14]<1.39988e+11){if(f[27]<6.4187e+09){if(f[51]<0.174421){if(f[0]<1.3995e+11){r+=-0.5;}else{r+=0.4;}}else{if(f[49]<3.88303){if(f[1]<4.37256e+08){r+=-0.989888;}else{r+=-0;}}else{if(f[7]<6.10817e+07){r+=0.454545;}else{r+=-0.809524;}}}}else{if(f[14]<1.39904e+11){r+=0.666667;}else{if(f[68]<1.58003){r+=-0.692308;}else{r+=-0;}}}}else{if(f[28]<1.39999e+11){r+=-0.995931;}else{if(f[0]<1.40078e+11){r+=0.982833;}else{r+=-0.878788;}}}}}else{if(f[0]<1.40076e+11){if(f[84]<3.15){if(f[28]<1.40001e+11){if(f[14]<1.40358e+11){if(f[56]<1.00294){if(f[28]<1.39998e+11){r+=-0.983246;}else{r+=0.209302;}}else{if(f[84]<2.15){r+=0.84;}else{r+=-0.884058;}}}else{if(f[84]<0.85){if(f[14]<1.40708e+11){r+=0.998828;}else{r+=-0.818182;}}else{if(f[37]<2.9503e+07){r+=-0.834043;}else{r+=0.761487;}}}}else{if(f[84]<0.5){r+=-0.997859;}else{if(f[0]<1.39706e+11){if(f[2]<6.09801e+08){r+=-0.97786;}else{r+=0.5;}}else{if(f[67]<0.614115){r+=-0.340793;}else{r+=0.883845;}}}}}else{if(f[66]<0.206743){r+=0.111111;}else{if(f[0]<1.40068e+11){if(f[1]<1.9571e+09){r+=-0.996491;}else{r+=-0.25;}}else{if(f[28]<1.40012e+11){if(f[1]<6.47347e+07){r+=-0.957895;}else{r+=-0.25;}}else{if(f[1]<1.91988e+07){r+=-0;}else{r+=0.714286;}}}}}}else{if(f[84]<3.15){if(f[37]<2.07343e+07){if(f[84]<0.85){if(f[34]<3.34163e+07){r+=-0.995778;}else{r+=-0.25;}}else{if(f[14]<1.40188e+11){if(f[84]<1.15){r+=0.950617;}else{r+=-0.975;}}else{if(f[28]<1.39956e+11){r+=0.699629;}else{r+=0.962475;}}}}else{if(f[84]<1.5){if(f[73]<43.1493){if(f[0]<1.40085e+11){r+=0.269461;}else{r+=-0.940458;}}else{if(f[0]<1.40097e+11){r+=0.986014;}else{r+=-0.8;}}}else{if(f[10]<2.90749e+08){if(f[14]<1.40178e+11){r+=-0.931558;}else{r+=0.0357406;}}else{if(f[67]<0.537446){r+=-0.948238;}else{r+=0.684007;}}}}}else{if(f[0]<1.40134e+11){if(f[14]<1.40174e+11){if(f[12]<7.25381e+09){if(f[13]<8.89074e+07){r+=-0.0666667;}else{r+=-0.991515;}}else{r+=0.764706;}}else{if(f[56]<1.00239){r+=0.996286;}else{if(f[66]<0.879163){r+=0.636364;}else{r+=-0.975155;}}}}else{if(f[14]<1.40352e+11){if(f[14]<1.40014e+11){if(f[2]<2.94473e+07){r+=-0.977401;}else{r+=-0.25;}}else{if(f[0]<1.40222e+11){r+=0.915849;}else{r+=0.635376;}}}else{if(f[56]<1.00244){if(f[0]<1.40295e+11){r+=0.943662;}else{r+=-0.6;}}else{if(f[0]<1.40149e+11){r+=0.25;}else{r+=-0.99391;}}}}}}}}}else{if(f[83]<0.986214){if(f[28]<1.41466e+11){if(f[0]<1.41693e+11){if(f[39]<4.53824e+07){if(f[84]<0.85){if(f[28]<1.40244e+11){if(f[42]<0.997329){r+=-0.997723;}else{if(f[12]<6.27566e+07){r+=0.888889;}else{r+=-0.789474;}}}else{if(f[42]<0.995916){if(f[28]<1.40645e+11){r+=0.997872;}else{r+=-0.924528;}}else{if(f[70]<1.00163){r+=0.72956;}else{r+=-0.936262;}}}}else{if(f[28]<1.40589e+11){if(f[28]<1.40133e+11){if(f[0]<1.40204e+11){r+=0.591304;}else{r+=-0.972222;}}else{if(f[41]<1.0543e+08){r+=-0.98222;}else{r+=-0.57022;}}}else{if(f[42]<1.00144){if(f[28]<1.4137e+11){r+=-0.992883;}else{r+=-0;}}else{if(f[84]<1.8){r+=0.998259;}else{r+=-0.666667;}}}}}else{if(f[79]<1.90527){if(f[28]<1.40711e+11){if(f[14]<1.40206e+11){if(f[0]<1.40092e+11){r+=-0.259054;}else{r+=-0.97374;}}else{if(f[14]<1.40238e+11){r+=0.464101;}else{r+=-0.575826;}}}else{if(f[0]<1.41584e+11){if(f[41]<1.50482e+10){r+=-0.997887;}else{r+=-0.142857;}}else{if(f[56]<0.999423){r+=0.904762;}else{r+=-0.862069;}}}}else{if(f[84]<2.85){if(f[31]<4.57585e+08){if(f[0]<1.40023e+11){r+=-0.463014;}else{r+=-0.962687;}}else{if(f[84]<1){r+=0.891892;}else{r+=-0.809524;}}}else{if(f[70]<1.00136){r+=0.998535;}else{r+=-0.966102;}}}}}else{if(f[0]<1.42937e+11){if(f[1]<3.10599e+08){if(f[56]<1.00817){if(f[14]<1.40177e+11){if(f[2]<2.07639e+07){r+=-0.878788;}else{r+=-0;}}else{if(f[14]<1.41407e+11){r+=0.995769;}else{r+=-0.692308;}}}else{r+=-0.921569;}}else{if(f[31]<2.18707e+08){if(f[0]<1.42764e+11){if(f[61]<9.37538){r+=-0.947368;}else{r+=-0.333333;}}else{r+=-0;}}else{if(f[22]<2.60364e+10){r+=0.625;}else{r+=-0.666667;}}}}else{if(f[70]<0.980489){if(f[55]<0.684415){r+=0.2;}else{if(f[59]<0.200961){r+=-0;}else{if(f[27]<7.69035e+07){r+=-0.25;}else{r+=-0.96648;}}}}else{if(f[1]<1.38541e+09){if(f[42]<1.01178){r+=-0.25;}else{r+=0.84;}}else{r+=-0.6;}}}}}else{if(f[14]<1.40984e+11){if(f[36]<2.51782e+06){if(f[0]<1.4025e+11){if(f[30]<1.45495e+06){r+=-0.997754;}else{r+=-0.111111;}}else{if(f[84]<0.8){if(f[14]<1.40144e+11){r+=0.942857;}else{r+=-0.789474;}}else{r+=-0.96748;}}}else{if(f[70]<1.01246){if(f[14]<1.40148e+11){if(f[14]<1.4007e+11){r+=-0.5;}else{if(f[9]<3.11356e+07){r+=0.25;}else{r+=0.988981;}}}else{if(f[71]<0.183454){r+=0.25;}else{if(f[1]<1.00594e+09){r+=-0.992537;}else{r+=-0;}}}}else{if(f[42]<1.00207){if(f[25]<5.68753e+07){if(f[28]<1.73135e+11){r+=-0.979167;}else{r+=0.826087;}}else{if(f[84]<3.35){r+=0.988793;}else{r+=-0.971831;}}}else{if(f[71]<11.1678){r+=-0.979592;}else{r+=0.555556;}}}}}else{if(f[84]<0.5){if(f[56]<0.993327){r+=0.985611;}else{if(f[14]<1.43877e+11){r+=-0.980488;}else{r+=0.6;}}}else{r+=-0.998327;}}}}else{if(f[84]<1.5){if(f[14]<1.40425e+11){if(f[57]<0.122638){if(f[14]<1.40058e+11){if(f[71]<11.8609){if(f[28]<1.70645e+11){r+=-0.75;}else{r+=0.692308;}}else{if(f[31]<1.86376e+07){r+=0.25;}else{r+=0.987013;}}}else{r+=-0.857143;}}else{if(f[66]<3.94877){if(f[28]<1.40136e+11){if(f[14]<1.40054e+11){if(f[0]<1.40106e+11){r+=0.975904;}else{r+=-0.75;}}else{if(f[56]<1.00132){r+=-0.969849;}else{r+=0.5;}}}else{if(f[28]<1.73752e+11){if(f[41]<2.18376e+10){r+=-0.979276;}else{r+=0.733333;}}else{r+=0.921569;}}}else{if(f[0]<1.39991e+11){if(f[48]<0.270485){r+=0.0909091;}else{if(f[7]<5.4084e+08){r+=-0.860465;}else{r+=-0.2;}}}else{if(f[14]<1.40127e+11){if(f[0]<1.40701e+11){r+=0.908163;}else{r+=-0.555556;}}else{r+=-0.809524;}}}}}else{if(f[14]<1.40546e+11){if(f[28]<1.40376e+11){if(f[84]<0.85){r+=-0.987461;}else{if(f[0]<1.40113e+11){r+=0.714286;}else{r+=-0.5;}}}else{if(f[28]<1.40657e+11){if(f[70]<0.996494){r+=-0.911111;}else{if(f[28]<1.40381e+11){r+=0.348837;}else{r+=0.934593;}}}else{if(f[6]<1.92812e+08){r+=-0.973856;}else{r+=-0;}}}}else{if(f[70]<1.01247){if(f[0]<1.42318e+11){if(f[14]<1.40556e+11){if(f[28]<1.40461e+11){r+=0.709091;}else{r+=-0.826087;}}else{if(f[14]<1.40568e+11){r+=-0.6;}else{r+=-0.98725;}}}else{if(f[56]<1.00796){if(f[84]<1){r+=0.965398;}else{r+=-0.6;}}else{r+=-0.92;}}}else{if(f[0]<1.40088e+11){r+=-0.981567;}else{r+=0.995086;}}}}}else{if(f[28]<1.40151e+11){if(f[42]<0.999135){if(f[14]<1.40242e+11){if(f[10]<1.04151e+08){r+=-0.84;}else{r+=0.993837;}}else{if(f[84]<1.85){r+=0.692308;}else{if(f[12]<2.29754e+09){if(f[39]<2.05549e+07){r+=-0;}else{r+=-0.970443;}}else{r+=0.333333;}}}}else{if(f[14]<1.40116e+11){if(f[71]<14.6333){if(f[43]<6.22851){r+=-0.992821;}else{r+=-0.25;}}else{r+=-0.111111;}}else{if(f[14]<1.40246e+11){if(f[70]<0.99981){if(f[0]<1.40235e+11){r+=0.979798;}else{r+=0.25;}}else{if(f[0]<1.40138e+11){r+=-0.25;}else{r+=-0.916667;}}}else{r+=-0.994228;}}}}else{if(f[12]<5.83399e+08){if(f[36]<2.97199e+08){if(f[28]<1.42736e+11){if(f[70]<0.999678){if(f[13]<8.58929e+07){r+=-0.846868;}else{r+=0.879661;}}else{if(f[84]<2.5){r+=0.530065;}else{r+=-0.925241;}}}else{if(f[84]<7.40001){r+=-0.995633;}else{r+=0.666667;}}}else{if(f[84]<1.85){if(f[28]<1.40652e+11){if(f[0]<1.40263e+11){r+=0.933054;}else{r+=-0.846154;}}else{if(f[1]<3.32354e+06){r+=-0;}else{r+=-0.983539;}}}else{if(f[66]<1.86057){if(f[84]<4.85){r+=-0.940645;}else{r+=0.269231;}}else{if(f[28]<1.40381e+11){r+=0.915789;}else{r+=-0.376344;}}}}}else{if(f[84]<3.5){if(f[55]<0.476632){if(f[13]<5.97053e+08){if(f[0]<1.40087e+11){r+=0.777778;}else{r+=-0.692308;}}else{if(f[42]<1.00395){r+=-0.995098;}else{r+=-0;}}}else{if(f[28]<1.40382e+11){if(f[0]<1.40088e+11){r+=0.686747;}else{r+=-0.233577;}}else{if(f[49]<0.299896){r+=0.25;}else{r+=-0.962963;}}}}else{if(f[14]<1.40651e+11){if(f[0]<1.39727e+11){if(f[0]<1.39579e+11){r+=-0;}else{r+=-0.866667;}}else{if(f[42]<1.00016){r+=0.932632;}else{r+=-0.75;}}}else{if(f[84]<4.85){if(f[43]<0.238634){r+=-0.25;}else{r+=-0.971223;}}else{if(f[0]<1.4044e+11){r+=-0.5;}else{r+=0.6;}}}}}}}}}}}} return r;}
    static double p16(double[] f){double r=0;if(f[28]<1.39444e+11){if(f[56]<1.00178){if(f[28]<1.39216e+11){if(f[56]<1.0014){r+=-0.677097;}else{r+=-0.096843;}}else{if(f[36]<1.39327e+08){if(f[84]<0.8){r+=-0.440861;}else{r+=0.703468;}}else{r+=-0.505004;}}}else{if(f[55]<0.291289){if(f[84]<1){if(f[57]<9.33998){r+=-0.721435;}else{r+=-0.141631;}}else{if(f[14]<1.40169e+11){if(f[59]<3.51536){r+=0.758242;}else{r+=0.210867;}}else{r+=-0.579427;}}}else{if(f[28]<1.39424e+11){if(f[10]<3.53657e+10){if(f[42]<1.00945){r+=-0.68407;}else{if(f[42]<1.01071){if(f[25]<2.37038e+08){if(f[71]<1.82299){r+=1.49361;}else{r+=0.631183;}}else{r+=-0.611711;}}else{r+=-0.683804;}}}else{if(f[51]<1.13525){if(f[10]<1.01747e+11){r+=-0.692126;}else{r+=0.0540607;}}else{if(f[29]<3.7306e+09){r+=0.557522;}else{r+=-0.394486;}}}}else{if(f[84]<1.15){if(f[28]<1.39438e+11){r+=-0.678316;}else{if(f[14]<1.40004e+11){if(f[68]<0.272786){r+=-0.412116;}else{r+=-1.38309;}}else{r+=-0.639461;}}}else{if(f[42]<1.00273){if(f[42]<0.997897){r+=-0.0301248;}else{r+=-0.701932;}}else{if(f[0]<1.40591e+11){r+=0.732639;}else{if(f[41]<2.70339e+08){r+=0.483433;}else{r+=-0.644498;}}}}}}}}else{if(f[69]<0.52948){if(f[0]<1.40123e+11){if(f[0]<1.39989e+11){if(f[28]<1.3998e+11){if(f[84]<2.85){if(f[42]<1.00175){if(f[23]<3.49124e+07){r+=0.548183;}else{if(f[1]<4.52117e+08){r+=-0.681324;}else{r+=-0.141418;}}}else{if(f[12]<7.39714e+08){r+=0.659896;}else{r+=-0.096388;}}}else{if(f[14]<1.40433e+11){r+=-0.67664;}else{if(f[0]<1.39694e+11){if(f[28]<1.39976e+11){r+=0.682796;}else{r+=1.42263;}}else{r+=-0.476728;}}}}else{if(f[83]<8.25875){if(f[50]<4.43637){if(f[66]<1.31227){if(f[43]<6.21978){if(f[56]<1.00091){if(f[14]<1.40445e+11){if(f[28]<1.41873e+11){r+=-0.644356;}else{r+=1.12994;}}else{if(f[28]<1.40657e+11){r+=0.918835;}else{r+=-0.485791;}}}else{if(f[38]<1.37063e+07){r+=0.987559;}else{if(f[28]<1.39997e+11){r+=0.755637;}else{r+=-0.683835;}}}}else{if(f[29]<7.58092e+06){if(f[14]<1.40099e+11){if(f[19]<5.04683e+06){r+=0.830898;}else{r+=0.0708174;}}else{if(f[9]<3.6576e+07){r+=-0.665353;}else{r+=0.0585133;}}}else{if(f[26]<3.99571e+09){if(f[67]<0.735114){r+=-0.680954;}else{r+=-0.0209431;}}else{r+=0.142236;}}}}else{if(f[28]<1.4064e+11){if(f[84]<1.15){if(f[42]<0.995995){if(f[14]<1.40723e+11){r+=0.693308;}else{r+=-0.564981;}}else{if(f[56]<1.00143){r+=-0.7234;}else{r+=0.182035;}}}else{if(f[0]<1.39825e+11){if(f[28]<1.40551e+11){r+=-0.700792;}else{r+=0.198658;}}else{if(f[79]<0.977012){r+=0.276703;}else{r+=-0.667957;}}}}else{if(f[67]<0.412782){r+=0.203341;}else{r+=-0.688418;}}}}else{if(f[28]<1.40101e+11){if(f[28]<1.39995e+11){if(f[29]<8.46407e+06){r+=-0.561385;}else{r+=-0.038112;}}else{if(f[56]<1.00177){if(f[21]<4.22268e+06){r+=0.700172;}else{if(f[7]<2.46609e+07){r+=1.11038;}else{r+=0.743159;}}}else{r+=-0.37869;}}}else{if(f[27]<5.22616e+08){if(f[13]<4.5786e+06){if(f[56]<1.00027){r+=-0.401008;}else{r+=0.517889;}}else{if(f[65]<0.00137128){r+=0.0324748;}else{r+=-0.715713;}}}else{if(f[28]<1.40184e+11){r+=0.631622;}else{r+=-0.125579;}}}}}else{if(f[0]<1.39983e+11){if(f[0]<1.39981e+11){if(f[51]<0.647482){if(f[84]<1.85){if(f[46]<1.26225){r+=-0.18713;}else{r+=0.603709;}}else{if(f[64]<0.149835){r+=-0.0739146;}else{r+=-0.570485;}}}else{if(f[43]<3.29487){if(f[79]<0.310445){if(f[28]<1.40179e+11){r+=-0.640154;}else{r+=0.181457;}}else{if(f[20]<428362){r+=-0.21642;}else{r+=-0.687791;}}}else{if(f[63]<1.08227){r+=-0.578713;}else{if(f[28]<1.40176e+11){r+=-0.400314;}else{r+=0.633646;}}}}}else{if(f[28]<1.40182e+11){if(f[1]<9.97655e+06){if(f[8]<2.44815e+08){if(f[84]<3.2){r+=-0.70603;}else{r+=-0.127556;}}else{r+=0.00568928;}}else{if(f[14]<1.40093e+11){r+=-0.210728;}else{r+=0.546463;}}}else{if(f[14]<1.40098e+11){r+=0.694862;}else{r+=-0.319952;}}}}else{if(f[84]<3.5){if(f[28]<1.40183e+11){if(f[28]<1.40179e+11){if(f[50]<0.718953){if(f[74]<1.12028){r+=0.598231;}else{r+=-0.450236;}}else{if(f[51]<1.17317){r+=-0.619614;}else{r+=0.319769;}}}else{if(f[77]<0.670758){if(f[25]<9.29579e+07){r+=-0.359637;}else{r+=0.492357;}}else{if(f[54]<0.474761){r+=-0.488191;}else{r+=0.0537375;}}}}else{if(f[53]<0.999451){if(f[69]<0.150268){if(f[34]<4.15776e+06){r+=-0.358431;}else{r+=0.712153;}}else{if(f[60]<0.319959){r+=0.113494;}else{r+=0.630492;}}}else{if(f[14]<1.40097e+11){if(f[28]<1.40186e+11){r+=-0.68988;}else{r+=0.18142;}}else{if(f[52]<1.12397){r+=0.719385;}else{r+=-0.192117;}}}}}else{if(f[28]<1.40153e+11){r+=-0.384147;}else{r+=0.692174;}}}}}}else{if(f[55]<0.872812){if(f[14]<1.40132e+11){if(f[28]<1.40185e+11){if(f[0]<1.40071e+11){if(f[42]<0.999505){if(f[84]<2.5){if(f[77]<0.971){if(f[27]<6.02225e+08){r+=0.546255;}else{r+=0.111184;}}else{if(f[26]<2.09834e+08){r+=0.321216;}else{r+=-0.282227;}}}else{r+=0.679436;}}else{if(f[84]<1){if(f[42]<0.999902){r+=-0.578189;}else{if(f[83]<3.31493){r+=0.151755;}else{r+=0.727924;}}}else{if(f[1]<1.55251e+07){r+=-0.701856;}else{r+=-0.182562;}}}}else{if(f[0]<1.40105e+11){if(f[67]<1.54585){if(f[25]<2.43805e+08){r+=0.684327;}else{r+=0.142213;}}else{r+=-0.0452958;}}else{if(f[3]<7.96665e+06){r+=-0.523325;}else{r+=0.0087144;}}}}else{if(f[32]<3.06314e+08){if(f[28]<1.40673e+11){if(f[0]<1.39992e+11){if(f[84]<2.5){if(f[14]<1.40095e+11){r+=0.013426;}else{r+=0.507575;}}else{r+=0.680568;}}else{if(f[30]<7.16638e+08){if(f[83]<2.34656){r+=0.193187;}else{r+=0.680002;}}else{r+=0.14229;}}}else{r+=-0.470931;}}else{if(f[84]<0.5){r+=-0.729085;}else{r+=0.664005;}}}}else{if(f[14]<1.40212e+11){if(f[19]<1.89126e+07){r+=-0.684631;}else{r+=0.116398;}}else{if(f[28]<1.40404e+11){if(f[84]<0.85){r+=-0.534309;}else{if(f[83]<5.29503){if(f[48]<0.42701){r+=0.152212;}else{if(f[55]<0.530648){r+=0.0989361;}else{r+=0.682322;}}}else{r+=-0.183249;}}}else{if(f[28]<1.40408e+11){r+=-0.115894;}else{r+=-0.656887;}}}}}else{if(f[84]<2.5){if(f[63]<0.458673){if(f[39]<1.98723e+09){if(f[28]<1.40533e+11){if(f[84]<2.15){if(f[0]<1.40099e+11){if(f[2]<2.58948e+07){r+=0.681846;}else{r+=-0.114503;}}else{r+=-0.431723;}}else{if(f[56]<1.00137){r+=-0.716628;}else{r+=0.549919;}}}else{if(f[14]<1.40076e+11){r+=0.527533;}else{r+=-0.665133;}}}else{if(f[66]<0.582852){r+=-0.660784;}else{r+=-0.00623633;}}}else{if(f[67]<1.24728){if(f[0]<1.39996e+11){if(f[28]<1.40182e+11){if(f[52]<1.58773){r+=-0.694415;}else{r+=-0.0293675;}}else{r+=0.403252;}}else{if(f[28]<1.40405e+11){if(f[0]<1.40073e+11){if(f[28]<1.40385e+11){r+=-0.199635;}else{r+=0.705848;}}else{if(f[42]<1.00057){r+=0.600719;}else{r+=-0.261379;}}}else{if(f[14]<1.40108e+11){if(f[84]<1.2){r+=-0.501717;}else{r+=0.70683;}}else{if(f[6]<6.27261e+07){r+=-0.710901;}else{r+=-0.211198;}}}}}else{if(f[26]<1.09173e+08){if(f[0]<1.40086e+11){if(f[3]<3.1562e+06){r+=0.0537847;}else{r+=-0.634692;}}else{if(f[15]<4.77496e+06){r+=0.0714033;}else{r+=0.54894;}}}else{if(f[23]<1.45926e+09){if(f[48]<7.29247e+06){if(f[64]<2.61461){r+=-0.693761;}else{r+=-0.194599;}}else{r+=0.0720976;}}else{r+=0.40988;}}}}}else{if(f[21]<1.38455e+08){if(f[55]<0.969572){if(f[14]<1.4008e+11){if(f[1]<1.39255e+07){r+=-0.67507;}else{r+=-0.0433959;}}else{if(f[42]<0.998893){r+=-0.561984;}else{if(f[55]<0.90322){r+=0.61626;}else{r+=1.25439;}}}}else{if(f[25]<5.4094e+08){if(f[14]<1.39446e+11){if(f[1]<1.53898e+08){r+=0.547764;}else{r+=-0.448025;}}else{if(f[2]<1.07784e+08){if(f[14]<1.40073e+11){r+=-0.689132;}else{r+=-0.603182;}}else{r+=0.109642;}}}else{if(f[14]<1.40077e+11){r+=-0.464781;}else{r+=0.580804;}}}}else{if(f[28]<1.40257e+11){if(f[53]<0.768494){if(f[4]<1.32736e+07){r+=-0.55745;}else{r+=0.0282673;}}else{if(f[56]<0.991537){r+=-0.408162;}else{if(f[59]<0.0187231){r+=-0.40175;}else{if(f[69]<0.157433){r+=-0.234649;}else{r+=0.645259;}}}}}else{if(f[50]<1.10274){r+=0.0206803;}else{r+=-0.694738;}}}}}}}else{if(f[28]<1.43853e+11){if(f[83]<2.55729){if(f[28]<1.42081e+11){if(f[23]<1.18293e+08){if(f[42]<0.998042){if(f[0]<1.40475e+11){if(f[84]<2.65){if(f[84]<0.85){if(f[56]<1.00512){r+=-0.621767;}else{r+=0.00148449;}}else{if(f[41]<5.90759e+08){r+=0.465293;}else{r+=1.33316;}}}else{r+=-0.670859;}}else{r+=-0.677435;}}else{if(f[23]<1.0526e+08){if(f[14]<1.40145e+11){if(f[27]<6.0182e+08){if(f[28]<1.39579e+11){r+=0.409793;}else{r+=-0.678573;}}else{if(f[14]<1.39893e+11){r+=0.722701;}else{r+=-0.53943;}}}else{if(f[84]<3.15){if(f[28]<1.40606e+11){r+=-0.633012;}else{r+=-0.342977;}}else{if(f[28]<1.40174e+11){r+=0.235382;}else{r+=-0.672174;}}}}else{if(f[0]<1.41641e+11){if(f[56]<0.99461){if(f[42]<1.00599){r+=1.21384;}else{r+=-0.182007;}}else{if(f[0]<1.40294e+11){r+=0.0360352;}else{r+=-0.666477;}}}else{if(f[84]<1.85){if(f[0]<1.42553e+11){r+=-0.54257;}else{r+=0.255235;}}else{if(f[25]<9.19541e+07){r+=-0.0208168;}else{r+=0.73002;}}}}}}else{if(f[26]<1.58842e+08){if(f[84]<1.85){if(f[28]<1.39575e+11){if(f[84]<1.15){r+=-0.663689;}else{if(f[14]<1.39577e+11){r+=-0.643644;}else{r+=0.737913;}}}else{if(f[0]<1.42772e+11){if(f[0]<1.40265e+11){r+=-0.318228;}else{r+=-0.700199;}}else{if(f[56]<1.00796){r+=1.16727;}else{r+=-0.671701;}}}}else{if(f[56]<1.00493){if(f[10]<2.49156e+08){if(f[0]<1.41198e+11){r+=-0.355353;}else{r+=0.690443;}}else{if(f[56]<0.991321){r+=-0.666176;}else{r+=0.57837;}}}else{if(f[10]<5.76899e+08){r+=-0.695142;}else{r+=-0.189184;}}}}else{if(f[0]<1.41623e+11){if(f[83]<0.945429){if(f[43]<7.28868){if(f[83]<0.89246){r+=-0.403654;}else{r+=0.317768;}}else{if(f[19]<1.68549e+07){r+=1.05432;}else{r+=0.0414943;}}}else{if(f[14]<1.39149e+11){if(f[79]<0.896162){r+=0.606719;}else{r+=-0.693929;}}else{if(f[68]<0.13434){r+=0.0140915;}else{r+=-0.66513;}}}}else{if(f[14]<1.41282e+11){if(f[42]<1.01199){if(f[14]<1.40061e+11){r+=-0.471264;}else{r+=0.909544;}}else{if(f[14]<1.38939e+11){r+=0.6938;}else{r+=-0.570149;}}}else{r+=-0.679861;}}}}}else{if(f[0]<1.40517e+11){if(f[14]<1.40516e+11){r+=-0.535769;}else{if(f[14]<1.41023e+11){r+=0.683097;}else{r+=-0.125516;}}}else{if(f[66]<1.38007){if(f[71]<2.4279){r+=-0.741235;}else{r+=-0.206467;}}else{r+=0.440956;}}}}else{if(f[84]<0.5){if(f[37]<9.14668e+07){if(f[14]<1.40034e+11){if(f[28]<1.4006e+11){r+=-0.476789;}else{r+=0.683469;}}else{r+=-0.668502;}}else{if(f[65]<7.88793){if(f[65]<1.63391){if(f[64]<3.75445){if(f[17]<2.44793e+08){if(f[32]<433105){r+=-0.173746;}else{r+=-0.685533;}}else{r+=-0.107151;}}else{r+=-0.0323791;}}else{if(f[28]<1.40182e+11){r+=0.924953;}else{if(f[14]<1.41844e+11){r+=-0.65819;}else{r+=0.122963;}}}}else{if(f[1]<1.32642e+07){r+=0.698308;}else{r+=0.00721215;}}}}else{if(f[28]<1.39543e+11){if(f[14]<1.39939e+11){if(f[84]<1){r+=-0.629649;}else{if(f[39]<5.52734e+08){if(f[14]<1.39734e+11){r+=-0.238124;}else{r+=0.75455;}}else{r+=1.27977;}}}else{r+=-0.681541;}}else{if(f[41]<5.45182e+08){if(f[0]<1.4023e+11){if(f[84]<2.15){if(f[39]<3.03609e+08){if(f[50]<3.25131){r+=-0.688463;}else{r+=-0.229529;}}else{if(f[29]<4.74596e+07){r+=0.0586807;}else{r+=-0.191636;}}}else{if(f[28]<1.40152e+11){if(f[72]<0.102191){r+=-0.0980707;}else{r+=-0.663918;}}else{if(f[84]<2.5){r+=0.676663;}else{r+=-0.46375;}}}}else{if(f[12]<1.09104e+08){if(f[84]<3.5){if(f[10]<2.35933e+08){r+=-0.695031;}else{r+=-0.385994;}}else{if(f[33]<7.30978e+06){r+=-0.659521;}else{r+=0.479853;}}}else{if(f[0]<1.41115e+11){if(f[84]<2.85){r+=-0.667376;}else{r+=0.185703;}}else{if(f[0]<1.42058e+11){r+=0.718849;}else{r+=-0.35531;}}}}}else{if(f[0]<2.43485e+11){if(f[67]<0.72254){if(f[7]<835244){if(f[28]<1.4058e+11){r+=-0.213621;}else{r+=0.708027;}}else{if(f[84]<4.85){r+=-0.677234;}else{r+=-0.5616;}}}else{if(f[28]<1.40658e+11){if(f[54]<1.02261){r+=-0.670949;}else{r+=-0.206428;}}else{if(f[0]<1.41154e+11){r+=-0.675971;}else{r+=0.884123;}}}}else{r+=0.57547;}}}}}}else{if(f[84]<1){if(f[25]<1.96587e+10){r+=0.684707;}else{r+=-0.584505;}}else{r+=-0.685049;}}}}else{if(f[14]<1.39852e+11){if(f[14]<1.39531e+11){if(f[28]<1.40034e+11){if(f[37]<7.2902e+07){if(f[9]<8.85287e+07){r+=-0.680624;}else{r+=-1.02507;}}else{if(f[28]<1.3968e+11){if(f[83]<0.932514){if(f[0]<1.40195e+11){r+=0.696278;}else{if(f[0]<1.43935e+11){r+=-0.740693;}else{r+=0.631415;}}}else{if(f[0]<1.40173e+11){r+=0.4197;}else{if(f[1]<1.15002e+08){r+=-0.695962;}else{r+=-0.0448371;}}}}else{if(f[79]<1.25831){if(f[27]<1.43594e+08){if(f[39]<1.14319e+08){if(f[14]<1.38914e+11){r+=-1.03347;}else{if(f[13]<9.4269e+07){r+=-0.55935;}else{r+=0.687787;}}}else{if(f[0]<1.40608e+11){if(f[28]<1.39948e+11){r+=0.62666;}else{r+=-0.160544;}}else{if(f[9]<2.2935e+08){r+=-1.18991;}else{r+=0.131572;}}}}else{if(f[38]<3.85774e+09){if(f[28]<1.39703e+11){if(f[0]<1.40192e+11){r+=0.734066;}else{r+=-0.976876;}}else{if(f[66]<0.304536){r+=-0.488329;}else{r+=0.653132;}}}else{if(f[78]<1.27696){r+=-1.30747;}else{r+=0.360476;}}}}else{if(f[7]<1.5068e+08){if(f[28]<1.39994e+11){if(f[84]<2){if(f[29]<5.76519e+06){r+=-0.227735;}else{r+=0.679489;}}else{if(f[9]<1.2877e+08){r+=0.655787;}else{r+=-0.529109;}}}else{if(f[0]<1.43093e+11){if(f[14]<1.39061e+11){r+=-0.630513;}else{r+=0.085664;}}else{r+=0.589527;}}}else{if(f[28]<1.39896e+11){if(f[30]<6.74122e+07){if(f[84]<2.35){r+=0.654812;}else{r+=0.100709;}}else{if(f[1]<2.19329e+08){r+=-0.0787145;}else{r+=-0.560056;}}}else{if(f[0]<1.4219e+11){if(f[84]<3.35){r+=-0.668929;}else{r+=0.583979;}}else{if(f[20]<1.29496e+07){r+=0.145454;}else{r+=0.631997;}}}}}}}}else{if(f[84]<12){if(f[0]<1.43591e+11){if(f[42]<0.988443){if(f[84]<2.65){if(f[28]<1.40179e+11){r+=0.700731;}else{r+=-0.0802893;}}else{r+=-0.77151;}}else{if(f[28]<1.40065e+11){if(f[84]<2.85){if(f[2]<1.8319e+08){if(f[10]<6.90484e+07){r+=-0.119026;}else{r+=-0.68524;}}else{r+=0.140864;}}else{if(f[36]<1.29478e+08){if(f[0]<1.40722e+11){r+=-0.0434753;}else{r+=-0.605692;}}else{if(f[13]<1.78037e+08){r+=0.0892527;}else{r+=0.678162;}}}}else{if(f[53]<0.237357){if(f[56]<0.993044){r+=-0.68545;}else{r+=0.654093;}}else{if(f[14]<1.39524e+11){if(f[57]<99.7688){r+=-0.678477;}else{r+=0.692362;}}else{if(f[0]<1.43068e+11){r+=-1.20483;}else{r+=0.45145;}}}}}}else{if(f[84]<0.5){if(f[8]<7.6062e+08){r+=-0.660296;}else{r+=0.137552;}}else{if(f[83]<1.59555){if(f[15]<1.45584e+09){r+=0.687073;}else{r+=0.0386206;}}else{r+=-0.360026;}}}}else{if(f[0]<1.40208e+11){if(f[28]<1.40152e+11){if(f[38]<2.35764e+08){if(f[79]<1.02338){if(f[57]<2.89801){if(f[63]<1.94697){r+=-0.706129;}else{r+=-0.254171;}}else{if(f[10]<1.60445e+08){r+=0.362387;}else{r+=-0.55533;}}}else{if(f[26]<8.68122e+07){if(f[34]<1.95093e+07){r+=0.103341;}else{r+=0.732728;}}else{if(f[8]<1.4858e+08){r+=-0.0956939;}else{r+=-0.516885;}}}}else{if(f[12]<1.13996e+08){r+=0.704805;}else{if(f[2]<2.32419e+07){r+=0.0330451;}else{r+=-0.711606;}}}}else{if(f[26]<3.88759e+08){if(f[28]<1.40157e+11){if(f[66]<0.831557){r+=0.654168;}else{if(f[7]<2.03104e+08){r+=0.369878;}else{r+=-0.788095;}}}else{if(f[39]<6.93463e+07){if(f[3]<2.8092e+07){r+=0.434529;}else{r+=-0.118604;}}else{if(f[4]<1.00327e+06){r+=0.141916;}else{r+=0.680181;}}}}else{if(f[55]<0.844225){r+=0.328489;}else{r+=-0.619031;}}}}else{if(f[0]<1.40226e+11){if(f[28]<1.40158e+11){if(f[28]<1.40148e+11){r+=-0.680639;}else{r+=-0.144812;}}else{if(f[27]<8.22245e+07){r+=0.149037;}else{r+=0.671777;}}}else{if(f[42]<1.00592){if(f[28]<1.40144e+11){r+=-0.558834;}else{r+=0.541968;}}else{if(f[57]<9.58893){r+=-0.679657;}else{r+=-0.186639;}}}}}}}else{if(f[38]<1.42354e+08){if(f[84]<3.15){if(f[25]<7.30315e+07){if(f[41]<1.07579e+08){if(f[84]<1){if(f[0]<1.42125e+11){r+=-0.846684;}else{r+=0.369174;}}else{if(f[26]<6.09753e+07){if(f[42]<0.996738){if(f[1]<9.91602e+07){r+=0.679868;}else{r+=-0.123429;}}else{if(f[84]<1.5){r+=0.490303;}else{r+=-0.870369;}}}else{if(f[52]<1.88058){if(f[28]<1.42362e+11){r+=0.643714;}else{r+=-0.559721;}}else{if(f[2]<5.73738e+07){r+=-0.959547;}else{r+=-0.194122;}}}}}else{if(f[0]<1.4105e+11){if(f[23]<9.75666e+07){if(f[7]<1.54583e+07){r+=-0.19483;}else{r+=-0.725611;}}else{if(f[27]<7.7975e+07){r+=-0.684898;}else{if(f[1]<3.46778e+07){r+=-0.102767;}else{r+=0.526667;}}}}else{if(f[25]<6.88408e+07){r+=-0.648724;}else{if(f[66]<0.732152){r+=-0.370883;}else{if(f[28]<1.40892e+11){r+=-1.55636;}else{r+=-0.485273;}}}}}}else{if(f[0]<1.4327e+11){if(f[0]<1.41444e+11){if(f[38]<1.3147e+08){if(f[71]<7.09405){if(f[66]<2.31994){r+=0.66508;}else{r+=-0.323259;}}else{if(f[84]<2.65){r+=0.582293;}else{r+=-0.951452;}}}else{if(f[84]<1.15){if(f[84]<0.5){r+=-0.534309;}else{r+=-1.60632;}}else{if(f[0]<1.40902e+11){r+=0.640669;}else{r+=-0.56945;}}}}else{if(f[13]<1.14749e+09){r+=-1.58979;}else{r+=-0.16494;}}}else{if(f[7]<3.06797e+07){r+=-0.83385;}else{if(f[15]<2.3238e+08){r+=0.714748;}else{r+=0.119204;}}}}}else{r+=-0.677946;}}else{if(f[28]<1.40822e+11){if(f[84]<2.85){if(f[52]<1.33885){if(f[84]<0.5){if(f[0]<1.40249e+11){if(f[38]<1.27383e+09){if(f[27]<4.72067e+08){r+=0.994529;}else{r+=0.668761;}}else{r+=-0.425356;}}else{if(f[66]<0.735106){if(f[42]<1.00629){r+=-1.59622;}else{r+=-0.595916;}}else{r+=-0.844256;}}}else{if(f[10]<9.21503e+08){if(f[41]<1.71127e+09){if(f[14]<1.39729e+11){r+=0.69362;}else{r+=0.511525;}}else{if(f[0]<1.40573e+11){r+=0.648098;}else{r+=-1.13904;}}}else{if(f[64]<0.587632){if(f[42]<1.00851){r+=0.662993;}else{r+=0.0159815;}}else{if(f[0]<1.43026e+11){r+=-0.645654;}else{r+=0.476174;}}}}}else{if(f[42]<1.00679){if(f[66]<0.327727){if(f[28]<1.40087e+11){if(f[28]<1.39704e+11){r+=0.493927;}else{r+=-0.687679;}}else{if(f[25]<1.21823e+08){r+=0.177648;}else{r+=0.608231;}}}else{if(f[28]<1.39472e+11){if(f[84]<1){r+=-1.44144;}else{r+=0.630778;}}else{if(f[39]<7.42644e+08){r+=0.674865;}else{r+=-0.708584;}}}}else{if(f[0]<1.43326e+11){if(f[66]<0.734932){r+=-0.660045;}else{if(f[66]<1.30068){r+=-1.58025;}else{r+=-0.283777;}}}else{r+=0.599328;}}}}else{if(f[26]<1.0879e+10){if(f[28]<1.40298e+11){if(f[14]<1.39553e+11){r+=0.189643;}else{if(f[72]<0.0529574){r+=-0.137574;}else{if(f[15]<3.0866e+08){r+=-0.703818;}else{r+=-0.256535;}}}}else{if(f[66]<0.733849){r+=-0.602311;}else{if(f[27]<6.03327e+08){r+=-1.41633;}else{r+=-0.60345;}}}}else{if(f[64]<0.56948){r+=-0.257601;}else{r+=0.806732;}}}}else{if(f[28]<1.41869e+11){if(f[0]<1.43046e+11){if(f[84]<1.5){if(f[23]<6.37561e+08){if(f[66]<0.734084){r+=-0.636458;}else{if(f[69]<0.575404){r+=-0.447129;}else{r+=-1.61875;}}}else{if(f[0]<1.41034e+11){r+=-0.55181;}else{r+=-0.130808;}}}else{if(f[0]<1.39367e+11){r+=0.66699;}else{if(f[38]<1.90868e+08){if(f[84]<2.5){r+=-1.44448;}else{r+=-0.788552;}}else{if(f[21]<2.90081e+08){r+=-0.853406;}else{r+=-0.69352;}}}}}else{if(f[67]<2.69662){r+=0.699144;}else{r+=-0.0572494;}}}else{if(f[84]<2.85){if(f[28]<1.42184e+11){if(f[39]<2.10431e+08){if(f[66]<0.734222){r+=-0.538293;}else{if(f[7]<4.25375e+08){r+=-1.57056;}else{r+=-0.456133;}}}else{r+=-0.705253;}}else{r+=-0.694476;}}else{if(f[84]<3.35){r+=0.719187;}else{r+=-0.658632;}}}}}}}else{if(f[68]<6.73286){if(f[0]<1.39804e+11){if(f[28]<1.41445e+11){if(f[14]<1.40796e+11){if(f[14]<1.40464e+11){if(f[0]<1.39764e+11){if(f[14]<1.40367e+11){if(f[27]<1.76908e+10){if(f[43]<12.1933){r+=-0.722095;}else{r+=-0.258336;}}else{if(f[59]<2.47422){r+=0.980942;}else{r+=-0.326611;}}}else{if(f[28]<1.4e+11){if(f[38]<9.26074e+07){r+=-0.531293;}else{r+=-1.53284;}}else{if(f[0]<1.39704e+11){r+=-0.801109;}else{r+=0.206124;}}}}else{if(f[28]<1.40073e+11){if(f[84]<1.5){if(f[84]<0.85){r+=-0.646459;}else{r+=-1.44381;}}else{if(f[28]<1.40056e+11){r+=-0.670828;}else{r+=0.272272;}}}else{if(f[84]<1.85){if(f[14]<1.40426e+11){r+=-0.709061;}else{r+=0.565354;}}else{if(f[28]<1.40189e+11){r+=0.700773;}else{r+=-0.648293;}}}}}else{if(f[36]<7.51303e+07){if(f[42]<0.994611){if(f[56]<1.00461){r+=-0.726534;}else{r+=-0.20854;}}else{if(f[12]<2.91607e+08){r+=0.670051;}else{r+=-0.528717;}}}else{if(f[0]<1.39366e+11){if(f[6]<2.12221e+06){r+=-1.14347;}else{if(f[60]<0.494215){r+=-0.112679;}else{r+=-0.678177;}}}else{if(f[28]<1.40646e+11){if(f[29]<2.88233e+08){r+=0.741775;}else{r+=-0.581901;}}else{r+=-0.579419;}}}}}else{if(f[0]<1.39301e+11){if(f[28]<1.4e+11){if(f[28]<1.39784e+11){if(f[14]<1.41309e+11){r+=-0.129847;}else{r+=-0.585833;}}else{r+=-1.53586;}}else{if(f[29]<1.3253e+09){if(f[28]<1.40764e+11){if(f[28]<1.40121e+11){r+=-0.681604;}else{r+=-0.778083;}}else{r+=-0.682313;}}else{r+=0.522188;}}}else{if(f[28]<1.4006e+11){if(f[84]<2.5){if(f[19]<1.92473e+08){r+=-0.685687;}else{r+=-0.0389886;}}else{r+=0.719853;}}else{if(f[83]<0.986103){if(f[67]<0.88645){if(f[39]<4.54179e+07){r+=-0.684511;}else{r+=-0.773034;}}else{if(f[1]<1.85566e+09){r+=-0.69103;}else{r+=-0.108378;}}}else{if(f[84]<12){if(f[67]<0.953253){r+=-0.710042;}else{r+=-0.809105;}}else{if(f[12]<5.33367e+08){r+=-1.28865;}else{r+=-0.59141;}}}}}}}else{if(f[84]<2.15){if(f[14]<1.4698e+11){r+=-0.68549;}else{r+=-0.179265;}}else{if(f[83]<0.986264){if(f[42]<0.987309){r+=-0.227732;}else{if(f[28]<1.41465e+11){r+=1.13523;}else{r+=0.685141;}}}else{if(f[36]<2.96603e+08){if(f[14]<1.40848e+11){r+=0.794345;}else{r+=0.132813;}}else{r+=1.2771;}}}}}else{if(f[0]<1.40087e+11){if(f[28]<1.4063e+11){if(f[28]<1.4e+11){if(f[14]<1.40116e+11){if(f[0]<1.40078e+11){if(f[26]<2.73687e+09){if(f[57]<0.0831615){r+=-0.212737;}else{r+=-0.684209;}}else{if(f[66]<0.981339){r+=1.07226;}else{r+=-0.314179;}}}else{if(f[33]<8.74678e+06){if(f[32]<6.63787e+06){r+=-0.669589;}else{r+=-0.185392;}}else{if(f[14]<1.40015e+11){r+=-0.248524;}else{r+=0.698918;}}}}else{if(f[84]<2.15){if(f[28]<1.39908e+11){if(f[14]<1.4065e+11){r+=0.695575;}else{r+=-0.638197;}}else{if(f[84]<1.85){r+=-1.06264;}else{r+=0.612388;}}}else{if(f[42]<0.998841){if(f[37]<3.07412e+07){r+=-0.711416;}else{r+=-1.17659;}}else{if(f[57]<2.67841){r+=0.861087;}else{r+=0.153587;}}}}}else{if(f[84]<1.15){if(f[14]<1.40466e+11){if(f[14]<1.40102e+11){if(f[84]<0.5){r+=-0.708899;}else{r+=0.601402;}}else{if(f[28]<1.4055e+11){r+=-0.861353;}else{r+=0.658292;}}}else{if(f[28]<1.40377e+11){if(f[84]<0.85){r+=-0.719114;}else{r+=0.464998;}}else{if(f[39]<4.40079e+07){r+=0.56293;}else{r+=0.893584;}}}}else{if(f[84]<3.15){if(f[36]<2.90169e+07){if(f[28]<1.4015e+11){r+=0.671263;}else{r+=-0.277642;}}else{if(f[0]<1.3998e+11){r+=-0.459351;}else{r+=0.48606;}}}else{if(f[41]<1.7456e+09){if(f[83]<3.31994){r+=-0.693602;}else{r+=-0.278565;}}else{if(f[12]<5.88527e+08){r+=1.37091;}else{r+=0.694424;}}}}}}else{if(f[38]<4.04273e+07){if(f[14]<1.40115e+11){r+=-0.685381;}else{r+=0.68694;}}else{if(f[37]<2.0784e+07){if(f[0]<1.40042e+11){r+=0.124937;}else{r+=0.49377;}}else{if(f[83]<0.987617){if(f[42]<0.999254){if(f[34]<1.39568e+08){r+=-0.687648;}else{r+=-0.195559;}}else{if(f[84]<2.5){r+=-1.10349;}else{r+=0.596767;}}}else{if(f[28]<1.40657e+11){if(f[9]<1.48389e+08){r+=-0.124653;}else{r+=0.704708;}}else{if(f[84]<2.5){r+=-0.864347;}else{r+=-0.691081;}}}}}}}else{if(f[14]<1.40106e+11){if(f[28]<1.40005e+11){if(f[39]<6.13235e+07){if(f[36]<4.83319e+06){if(f[41]<6.72944e+07){r+=-0.371884;}else{r+=0.58868;}}else{if(f[84]<6.00001){if(f[78]<0.132494){r+=-0.0345308;}else{r+=-0.694747;}}else{if(f[0]<1.40172e+11){r+=0.556577;}else{r+=0.106162;}}}}else{if(f[84]<1.85){if(f[84]<1.5){if(f[56]<1.00195){r+=-0.592587;}else{r+=0.659555;}}else{if(f[0]<1.4019e+11){r+=0.757362;}else{r+=-0.672049;}}}else{if(f[10]<3.17544e+08){r+=-0.697045;}else{if(f[15]<2.44751e+08){r+=-1.17967;}else{r+=0.33831;}}}}}else{if(f[28]<1.71347e+11){if(f[0]<1.40093e+11){if(f[56]<0.999672){if(f[28]<1.40588e+11){r+=0.908146;}else{r+=-0.52907;}}else{if(f[31]<5.10614e+07){r+=-0.662739;}else{r+=0.538757;}}}else{if(f[14]<1.40103e+11){if(f[84]<2.15){r+=-0.686495;}else{r+=-0.762465;}}else{if(f[84]<3.15){r+=-0.724992;}else{r+=0.676951;}}}}else{r+=0.686274;}}}else{if(f[0]<1.40246e+11){if(f[84]<1.85){if(f[41]<5.1738e+07){if(f[0]<1.40178e+11){if(f[84]<1.35){r+=0.675849;}else{r+=-1.30212;}}else{if(f[79]<0.26271){r+=0.410592;}else{r+=-0.624891;}}}else{if(f[14]<1.4029e+11){if(f[20]<1.59769e+08){r+=-0.712065;}else{r+=-0.018205;}}else{if(f[28]<1.40169e+11){r+=0.0030989;}else{r+=-0.547396;}}}}else{if(f[64]<0.531192){if(f[14]<1.4056e+11){if(f[84]<2.85){r+=-0.814919;}else{r+=-0.0188841;}}else{if(f[42]<0.994441){r+=-0.579292;}else{r+=0.849362;}}}else{if(f[0]<1.4012e+11){if(f[14]<1.40561e+11){r+=-0.483735;}else{r+=0.632254;}}else{if(f[54]<0.451503){r+=-0.0670078;}else{r+=0.60382;}}}}}else{if(f[14]<1.40744e+11){if(f[84]<2.15){if(f[84]<1.5){if(f[14]<1.4045e+11){r+=-0.671684;}else{r+=-0.0297348;}}else{if(f[67]<0.212796){r+=0.2868;}else{r+=-0.827853;}}}else{if(f[14]<1.40347e+11){if(f[83]<1.00423){r+=0.395095;}else{r+=-0.56692;}}else{if(f[42]<1.00643){r+=-0.752166;}else{r+=-1.03712;}}}}else{if(f[28]<1.39954e+11){if(f[28]<1.39809e+11){if(f[20]<1.74318e+09){r+=-0.684316;}else{r+=0.121163;}}else{if(f[36]<7.73847e+06){r+=0.607707;}else{r+=-0.921622;}}}else{if(f[36]<1.7239e+08){if(f[84]<0.5){r+=-0.560058;}else{r+=0.468579;}}else{if(f[84]<1.15){r+=0.419218;}else{r+=-0.633374;}}}}}}}}}else{if(f[56]<0.817301){if(f[14]<1.40122e+11){r+=0.684738;}else{r+=-0.668832;}}else{if(f[65]<0.738995){if(f[83]<0.257084){r+=0.689433;}else{r+=-0.549638;}}else{if(f[28]<1.40115e+11){if(f[57]<5.10456){if(f[84]<2.15){if(f[0]<1.40176e+11){if(f[83]<0.0373434){r+=-0.579855;}else{if(f[56]<1.00341){r+=0.412096;}else{r+=0.764495;}}}else{if(f[41]<1.9984e+08){r+=-0.69298;}else{r+=0.242334;}}}else{if(f[28]<1.40002e+11){if(f[18]<1.43049e+08){r+=-0.725114;}else{r+=-0.186114;}}else{if(f[67]<7.65815){if(f[21]<1.43172e+08){r+=0.595566;}else{r+=-0.358461;}}else{if(f[1]<3.86357e+07){r+=-0.104361;}else{r+=-0.759384;}}}}}else{if(f[79]<0.669255){if(f[52]<5.11448){if(f[0]<1.40018e+11){if(f[28]<1.40079e+11){r+=0.0142735;}else{r+=0.391344;}}else{if(f[14]<1.40019e+11){r+=-0.200551;}else{r+=-0.685616;}}}else{r+=-1.47406;}}else{r+=-1.4633;}}}else{if(f[41]<2.65551e+09){if(f[7]<89878.3){r+=0.537986;}else{if(f[28]<1.40711e+11){if(f[39]<2.20428e+09){if(f[27]<4.70858e+08){r+=-0.557709;}else{r+=-0.781233;}}else{r+=-0.0207326;}}else{if(f[34]<3.54601e+08){if(f[30]<1.63966e+08){r+=-0.685464;}else{r+=-0.122579;}}else{r+=0.216004;}}}}else{if(f[69]<5.2956){r+=-0.352436;}else{if(f[54]<0.190648){r+=-0.0871423;}else{r+=1.14586;}}}}}}}}}} return r;}
    static double p17(double[] f){double r=0;if(f[28]<1.39446e+11){if(f[28]<1.3942e+11){if(f[13]<3.97735e+10){if(f[56]<1.00196){if(f[28]<1.392e+11){if(f[56]<1.00152){r+=-0.578401;}else{r+=0.284962;}}else{if(f[31]<1.47899e+07){if(f[14]<1.38557e+11){r+=0.44419;}else{r+=0.108266;}}else{r+=-0.414444;}}}else{if(f[21]<4.50722e+10){if(f[42]<1.00952){if(f[44]<0.0153638){r+=-0.0698304;}else{if(f[26]<3.3204e+10){r+=-0.592518;}else{if(f[60]<0.551354){r+=0.0158976;}else{r+=-0.399519;}}}}else{if(f[42]<1.01066){if(f[27]<2.35403e+08){r+=0.679302;}else{r+=-0.566038;}}else{if(f[42]<1.0107){r+=-0.0691052;}else{r+=-0.591186;}}}}else{r+=0.109853;}}}else{if(f[32]<1.89841e+09){if(f[66]<0.936339){r+=-0.150382;}else{if(f[0]<1.39573e+11){r+=0.19936;}else{r+=0.691208;}}}else{r+=-0.524201;}}}else{if(f[56]<1.00325){if(f[14]<1.39837e+11){if(f[14]<1.3968e+11){if(f[38]<1.49194e+08){r+=0.565573;}else{r+=-0.142039;}}else{if(f[14]<1.39821e+11){if(f[28]<1.39438e+11){if(f[19]<5.70477e+06){r+=-0.153555;}else{r+=-0.565251;}}else{if(f[73]<2.15151){r+=-0.952133;}else{r+=-0.301275;}}}else{if(f[26]<8.88259e+07){if(f[78]<0.942554){if(f[50]<1.54339){if(f[28]<1.39435e+11){r+=-0.0791048;}else{r+=0.379702;}}else{r+=-0.344722;}}else{if(f[28]<1.39432e+11){r+=-0.153637;}else{r+=-0.600236;}}}else{if(f[80]<0.903388){if(f[75]<0.438351){r+=0.211997;}else{r+=-0.385594;}}else{r+=0.503232;}}}}}else{if(f[70]<0.993301){r+=-0.320629;}else{if(f[41]<1.02673e+08){if(f[50]<1.20593){r+=-0.336726;}else{r+=0.396526;}}else{if(f[28]<1.39426e+11){r+=1.03227;}else{r+=0.562192;}}}}}else{if(f[31]<9.78946e+07){if(f[42]<1.00425){r+=-0.596035;}else{r+=-0.136026;}}else{r+=-0.0551728;}}}}else{if(f[66]<0.384528){if(f[13]<2.31135e+08){if(f[14]<1.40242e+11){if(f[0]<1.40268e+11){if(f[42]<0.998697){if(f[28]<1.39914e+11){r+=0.124757;}else{r+=-0.628535;}}else{if(f[27]<3.18754e+07){r+=-0.590871;}else{if(f[9]<2.19111e+07){if(f[13]<2.61969e+07){r+=0.209462;}else{if(f[56]<1.00211){r+=-0.626254;}else{r+=0.0484655;}}}else{if(f[56]<1.00289){if(f[0]<1.39959e+11){if(f[70]<1.00123){r+=0.494943;}else{r+=-0.683536;}}else{if(f[30]<6.22216e+08){if(f[28]<1.40503e+11){r+=0.549563;}else{r+=-0.268702;}}else{if(f[0]<1.40075e+11){r+=-0.615089;}else{r+=0.585337;}}}}else{r+=-0.621464;}}}}}else{if(f[28]<1.7759e+11){if(f[28]<1.39591e+11){r+=0.696283;}else{if(f[9]<4.20799e+08){if(f[28]<1.39861e+11){if(f[28]<1.3981e+11){r+=-0.469033;}else{r+=0.430436;}}else{if(f[7]<3.86718e+08){if(f[4]<7.15905e+07){if(f[78]<0.460853){r+=-0.263051;}else{r+=-0.604577;}}else{r+=-0.06696;}}else{if(f[26]<1.05317e+08){r+=-0.495652;}else{r+=0.432123;}}}}else{r+=0.557417;}}}else{r+=0.57593;}}}else{if(f[35]<1.14506e+07){if(f[0]<1.40408e+11){if(f[56]<1.00196){if(f[56]<0.999081){r+=-0.468461;}else{if(f[0]<1.40152e+11){if(f[28]<1.40399e+11){if(f[57]<1.32978){r+=0.11525;}else{if(f[19]<1.50797e+06){r+=0.534084;}else{r+=0.910333;}}}else{r+=-0.190006;}}else{if(f[28]<1.40405e+11){if(f[40]<2.38089e+08){if(f[29]<679837){r+=0.00758627;}else{r+=-0.455146;}}else{r+=0.308464;}}else{if(f[26]<5.94479e+07){r+=0.571831;}else{r+=-0.124544;}}}}}else{r+=-0.605854;}}else{if(f[70]<0.994981){r+=-0.459584;}else{if(f[38]<5.78072e+07){r+=0.627357;}else{if(f[11]<3.35005e+07){r+=0.421746;}else{if(f[78]<0.53939){r+=0.449994;}else{r+=1.25665;}}}}}}else{if(f[42]<0.997396){if(f[70]<1.00379){if(f[27]<6.34628e+07){r+=-0.590597;}else{if(f[28]<1.4017e+11){if(f[56]<1.00175){r+=1.49576;}else{if(f[66]<0.240546){r+=-0.335149;}else{if(f[28]<1.39913e+11){r+=0.0766755;}else{r+=0.665686;}}}}else{if(f[0]<1.40018e+11){r+=-0.534504;}else{if(f[72]<0.403182){if(f[17]<3.31465e+06){r+=0.258899;}else{r+=-0.265175;}}else{if(f[53]<0.968231){r+=0.135953;}else{r+=0.702483;}}}}}}else{if(f[28]<1.40153e+11){r+=-0.0648875;}else{if(f[0]<1.3925e+11){r+=-0.0979519;}else{r+=-0.59998;}}}}else{if(f[28]<1.3989e+11){if(f[56]<1.00322){if(f[53]<1.79304){if(f[64]<0.723835){r+=-0.348859;}else{r+=0.064748;}}else{if(f[50]<1.48857){r+=0.00784613;}else{r+=0.678166;}}}else{r+=-0.606771;}}else{if(f[0]<1.40306e+11){if(f[64]<0.488786){if(f[14]<1.40343e+11){r+=-0.651309;}else{if(f[20]<1.01317e+07){r+=-0.0565326;}else{r+=-0.314504;}}}else{if(f[68]<0.432249){r+=-0.655463;}else{r+=-1.00112;}}}else{if(f[39]<6.33072e+07){r+=0.093504;}else{r+=-0.605086;}}}}}}}else{if(f[56]<0.816676){if(f[55]<3.69835){r+=0.591273;}else{r+=-0.34653;}}else{if(f[28]<1.39882e+11){if(f[40]<5.54686e+08){if(f[56]<1.00303){if(f[63]<0.14794){if(f[55]<1.20246){r+=0.027052;}else{r+=-0.400649;}}else{if(f[14]<1.4025e+11){if(f[80]<2.95644){if(f[41]<6.83716e+08){if(f[17]<368308){r+=0.152656;}else{r+=0.616394;}}else{if(f[55]<1.31759){r+=0.0195229;}else{r+=1.00842;}}}else{if(f[64]<0.750872){r+=0.414398;}else{r+=-0.269101;}}}else{if(f[15]<1.98668e+07){if(f[11]<2.11874e+08){r+=-0.14828;}else{r+=-0.598783;}}else{if(f[21]<1.0078e+08){r+=-0.0066759;}else{r+=0.595702;}}}}}else{if(f[42]<0.999469){r+=0.468705;}else{if(f[42]<1.00774){r+=-0.594717;}else{r+=0.31752;}}}}else{if(f[54]<1.90092){if(f[14]<1.40051e+11){if(f[36]<1.01994e+09){if(f[66]<0.160519){r+=0.0343449;}else{r+=0.494291;}}else{if(f[0]<1.40175e+11){r+=-0.148743;}else{r+=-0.553669;}}}else{if(f[68]<0.620504){if(f[41]<4.30488e+08){if(f[14]<1.40499e+11){r+=0.0619913;}else{r+=-0.401066;}}else{if(f[40]<5.58912e+08){if(f[63]<0.881523){r+=-0.362261;}else{r+=-0.0401399;}}else{r+=-0.603347;}}}else{if(f[6]<4.21965e+07){r+=-0.386555;}else{r+=0.144332;}}}}else{if(f[81]<2.87768){if(f[56]<1.00285){if(f[13]<3.71453e+08){if(f[41]<6.63657e+08){r+=0.534163;}else{r+=-0.452031;}}else{if(f[66]<0.251632){if(f[0]<1.40224e+11){r+=1.00693;}else{r+=0.358597;}}else{r+=-0.00754387;}}}else{if(f[8]<6.32515e+08){if(f[63]<1.45625){r+=-0.596802;}else{r+=0.0293886;}}else{r+=0.283606;}}}else{r+=-0.59101;}}}}else{if(f[14]<1.42523e+11){if(f[0]<1.40095e+11){if(f[67]<0.398445){if(f[0]<1.40071e+11){if(f[28]<1.40201e+11){if(f[70]<1.00106){if(f[67]<0.017911){r+=0.493227;}else{r+=-0.591915;}}else{if(f[40]<2.4198e+09){r+=0.350557;}else{r+=-0.387203;}}}else{if(f[0]<1.39828e+11){if(f[52]<0.398196){r+=-0.151679;}else{r+=-0.594932;}}else{if(f[14]<1.40472e+11){r+=-0.565019;}else{r+=0.0491141;}}}}else{if(f[1]<2.91332e+07){if(f[52]<1.48604){r+=0.599449;}else{r+=0.184984;}}else{if(f[0]<1.40093e+11){if(f[30]<2.07799e+07){r+=0.0462243;}else{r+=-0.59901;}}else{r+=0.701224;}}}}else{if(f[0]<1.40002e+11){if(f[39]<4.28448e+08){if(f[28]<1.40205e+11){if(f[54]<1.7939){r+=0.703806;}else{r+=-0.130278;}}else{r+=-0.444792;}}else{if(f[11]<1.70215e+08){r+=-0.0605964;}else{r+=-0.55509;}}}else{if(f[42]<0.99961){if(f[53]<1.44111){r+=0.674502;}else{r+=0.0606948;}}else{if(f[42]<1.00014){if(f[81]<1.29042){r+=0.21358;}else{r+=-0.601852;}}else{if(f[39]<6.25082e+08){r+=0.587754;}else{r+=-0.175613;}}}}}}else{if(f[80]<1.18817){if(f[14]<1.40127e+11){if(f[78]<1.13552){if(f[14]<1.39748e+11){if(f[14]<1.39666e+11){r+=-0.0194102;}else{r+=0.894356;}}else{if(f[27]<1.74341e+09){r+=-0.578309;}else{r+=0.326289;}}}else{if(f[58]<0.0387326){r+=-0.107093;}else{if(f[56]<1.00056){r+=-0.603702;}else{r+=-0.252523;}}}}else{if(f[28]<1.40635e+11){if(f[56]<1.0015){if(f[27]<2.08643e+08){r+=0.561626;}else{r+=-0.711086;}}else{if(f[36]<2.95914e+07){r+=-0.744725;}else{r+=0.618013;}}}else{if(f[14]<1.42106e+11){if(f[6]<1.03157e+09){r+=-0.609779;}else{r+=-0.0527304;}}else{r+=0.118373;}}}}else{if(f[56]<0.887218){r+=0.460787;}else{if(f[35]<9.3057e+06){if(f[14]<1.4047e+11){r+=-0.462445;}else{r+=1.03442;}}else{if(f[41]<6.95716e+08){if(f[28]<1.39888e+11){r+=-0.334518;}else{r+=-0.671284;}}else{if(f[35]<6.0308e+07){r+=-0.490399;}else{r+=-0.593353;}}}}}}}else{if(f[27]<4.5135e+08){r+=-0.594241;}else{if(f[28]<1.45328e+11){if(f[67]<0.189892){if(f[38]<8.43948e+08){if(f[21]<4.31294e+07){r+=-0.228765;}else{r+=0.427799;}}else{r+=-0.473358;}}else{if(f[56]<0.985381){if(f[1]<1.0394e+08){r+=-0.228765;}else{r+=0.0759619;}}else{if(f[14]<1.42591e+11){if(f[27]<1.17095e+09){r+=0.337449;}else{r+=-0.304915;}}else{if(f[13]<3.36913e+09){r+=0.567738;}else{r+=0.120386;}}}}}else{r+=-0.478388;}}}}}}}else{if(f[67]<6.03786){if(f[0]<1.45949e+11){if(f[0]<1.41631e+11){if(f[28]<1.40716e+11){if(f[0]<1.40095e+11){if(f[14]<1.40792e+11){if(f[14]<1.39838e+11){if(f[14]<1.39127e+11){if(f[55]<2.88781){if(f[14]<1.39104e+11){r+=-0.601395;}else{r+=-0.170278;}}else{if(f[1]<3.53803e+08){r+=-0.311373;}else{r+=0.551538;}}}else{if(f[26]<6.23798e+07){if(f[56]<0.995843){r+=0.30463;}else{r+=-0.603862;}}else{if(f[56]<1.00142){r+=0.579894;}else{r+=-1.8703;}}}}else{if(f[14]<1.39991e+11){if(f[56]<0.997703){if(f[8]<9.45211e+07){r+=0.57258;}else{r+=-0.534857;}}else{if(f[14]<1.39987e+11){r+=-0.60405;}else{r+=-0.358381;}}}else{if(f[14]<1.40111e+11){if(f[28]<1.39998e+11){r+=-0.491091;}else{r+=0.413612;}}else{if(f[14]<1.40215e+11){r+=-0.200193;}else{r+=0.223362;}}}}}else{if(f[14]<1.45058e+11){if(f[28]<1.39982e+11){if(f[14]<1.41003e+11){if(f[67]<1.42593){r+=0.601982;}else{r+=-0.130119;}}else{if(f[28]<1.39809e+11){r+=-0.591916;}else{r+=-0.715421;}}}else{if(f[17]<8.86362e+08){if(f[0]<1.39803e+11){r+=-0.616125;}else{r+=-0.80993;}}else{r+=0.179856;}}}else{r+=0.60588;}}}else{if(f[28]<1.39983e+11){if(f[14]<1.4034e+11){if(f[40]<5.83822e+07){if(f[0]<1.40252e+11){if(f[28]<1.39846e+11){r+=0.137139;}else{r+=-0.531849;}}else{if(f[0]<1.40434e+11){r+=-0.59277;}else{r+=-0.0522161;}}}else{if(f[68]<0.499755){if(f[28]<1.39843e+11){r+=0.427532;}else{r+=-0.374485;}}else{if(f[28]<1.3982e+11){r+=0.106258;}else{r+=0.418217;}}}}else{if(f[34]<1.87899e+06){if(f[28]<1.3992e+11){if(f[0]<1.40232e+11){r+=0.268394;}else{r+=-0.705109;}}else{if(f[8]<9.16419e+07){r+=0.679749;}else{r+=-0.43138;}}}else{if(f[0]<1.40227e+11){if(f[28]<1.39823e+11){r+=-0.756706;}else{r+=0.390018;}}else{if(f[10]<3.40472e+07){r+=0.48094;}else{r+=-0.610488;}}}}}else{if(f[14]<1.40177e+11){if(f[14]<1.40033e+11){if(f[0]<1.40557e+11){if(f[8]<4.44406e+07){r+=-0.153665;}else{r+=0.28319;}}else{if(f[28]<1.40043e+11){r+=0.042939;}else{r+=-0.591255;}}}else{if(f[26]<5.57392e+08){if(f[0]<1.41577e+11){r+=-0.633581;}else{r+=-0.0426663;}}else{if(f[38]<3.67406e+08){r+=0.578105;}else{r+=-0.650359;}}}}else{if(f[0]<1.40234e+11){if(f[28]<1.40413e+11){if(f[28]<1.40019e+11){r+=-0.672108;}else{r+=0.469248;}}else{if(f[28]<1.40562e+11){r+=-0.851527;}else{r+=0.319304;}}}else{if(f[28]<1.40599e+11){if(f[36]<1.12053e+07){r+=0.321177;}else{r+=-0.610552;}}else{if(f[0]<1.40984e+11){r+=-0.821797;}else{r+=0.668601;}}}}}}}else{if(f[28]<1.41462e+11){if(f[0]<1.41545e+11){if(f[28]<1.40726e+11){if(f[14]<1.41141e+11){if(f[58]<0.111818){r+=-0.108631;}else{r+=-0.647695;}}else{if(f[42]<0.998349){r+=-0.5061;}else{if(f[13]<1.37111e+08){r+=0.650882;}else{r+=1.00561;}}}}else{if(f[0]<1.40087e+11){if(f[39]<3.8772e+07){r+=0.52342;}else{if(f[14]<1.39855e+11){r+=-0.665735;}else{r+=-0.596829;}}}else{if(f[29]<2.56775e+09){if(f[67]<0.712358){r+=-0.60951;}else{r+=-0.697803;}}else{r+=0.543668;}}}}else{if(f[27]<1.56593e+09){if(f[28]<1.41097e+11){r+=-0.14244;}else{r+=0.723255;}}else{r+=-0.55677;}}}else{if(f[14]<1.40991e+11){if(f[51]<1.65458){if(f[56]<0.98202){if(f[28]<1.74148e+11){if(f[14]<1.40198e+11){r+=-0.610721;}else{r+=-1.05153;}}else{if(f[80]<2.40886){r+=0.592268;}else{r+=0.0755173;}}}else{if(f[28]<1.42024e+11){if(f[0]<1.39342e+11){r+=0.612535;}else{r+=-0.681971;}}else{if(f[18]<1.59805e+08){r+=0.628702;}else{r+=-0.56281;}}}}else{if(f[42]<0.996781){if(f[70]<1.02213){if(f[70]<1.01284){r+=-0.158261;}else{r+=0.623612;}}else{r+=-0.540816;}}else{if(f[28]<1.69501e+11){if(f[56]<0.988436){r+=-0.777997;}else{r+=0.49892;}}else{r+=0.583246;}}}}else{if(f[56]<0.992847){if(f[0]<1.40133e+11){if(f[43]<4.77336){r+=-0.589483;}else{r+=-0.0590544;}}else{if(f[66]<0.774485){r+=0.149591;}else{r+=0.621964;}}}else{if(f[14]<1.41123e+11){if(f[0]<1.40246e+11){r+=-0.564513;}else{r+=-0.780213;}}else{if(f[0]<1.41253e+11){if(f[0]<1.40995e+11){r+=-0.604636;}else{r+=-0.191442;}}else{r+=0.253681;}}}}}}}else{if(f[56]<1.00908){if(f[80]<1.22682){if(f[41]<8.81315e+07){if(f[28]<1.40131e+11){if(f[14]<1.40294e+11){r+=0.735369;}else{if(f[38]<1.45252e+08){if(f[14]<1.41206e+11){r+=-0.669058;}else{r+=-0.81418;}}else{if(f[28]<1.40109e+11){r+=-0.624519;}else{r+=0.661905;}}}}else{if(f[70]<0.980521){if(f[41]<8.71645e+07){if(f[66]<1.00531){r+=0.0460207;}else{r+=-0.570265;}}else{if(f[14]<1.39538e+11){r+=-0.39606;}else{r+=0.462128;}}}else{if(f[38]<1.31099e+08){r+=-0.560368;}else{if(f[14]<1.40162e+11){r+=-0.324894;}else{r+=0.61404;}}}}}else{if(f[0]<1.41903e+11){if(f[14]<1.4012e+11){if(f[28]<1.40016e+11){if(f[21]<7.57102e+07){r+=-0.228122;}else{r+=0.612924;}}else{if(f[28]<1.40208e+11){r+=-0.186267;}else{r+=-0.61783;}}}else{if(f[42]<1.00407){if(f[28]<1.43111e+11){r+=-0.632691;}else{r+=0.469678;}}else{if(f[9]<1.26245e+08){r+=0.471482;}else{r+=0.680179;}}}}else{if(f[0]<1.42546e+11){if(f[28]<1.43619e+11){if(f[56]<0.991746){r+=-0.716012;}else{r+=0.0410771;}}else{if(f[20]<5.33308e+08){r+=0.544865;}else{r+=-0.10277;}}}else{if(f[56]<0.996122){if(f[14]<1.3951e+11){r+=0.345859;}else{r+=0.58992;}}else{if(f[70]<0.980627){r+=-0.539908;}else{r+=0.579671;}}}}}}else{if(f[0]<1.42556e+11){if(f[56]<0.996955){if(f[14]<1.41717e+11){if(f[14]<1.3904e+11){if(f[41]<9.71866e+08){r+=0.528048;}else{r+=-0.530732;}}else{if(f[28]<1.42556e+11){r+=-0.604099;}else{r+=-0.764722;}}}else{if(f[34]<1.54171e+08){if(f[67]<0.232441){r+=-0.251881;}else{r+=0.546137;}}else{if(f[32]<1.32072e+08){r+=-0.48878;}else{r+=0.0842624;}}}}else{if(f[70]<0.993539){if(f[13]<1.02406e+10){r+=-0.599891;}else{r+=0.483919;}}else{if(f[14]<1.411e+11){if(f[41]<7.05652e+08){r+=0.647647;}else{r+=2.03011;}}else{r+=-0.464551;}}}}else{if(f[54]<0.522794){if(f[64]<1.27483){if(f[58]<2.32935){r+=-0.595271;}else{r+=-0.135234;}}else{if(f[63]<1.5406){r+=-0.0195285;}else{if(f[7]<1.64249e+09){r+=0.521101;}else{r+=0.143034;}}}}else{if(f[28]<1.45292e+11){if(f[56]<1.00788){if(f[68]<1.60541){r+=0.562122;}else{r+=-0.211618;}}else{r+=-0.489336;}}else{if(f[3]<1.07796e+08){r+=-0.473685;}else{r+=0.0843828;}}}}}}else{if(f[35]<6.61936e+07){r+=0.469175;}else{if(f[28]<1.39829e+11){r+=-0.594704;}else{r+=-0.838751;}}}}}else{if(f[43]<780.103){if(f[0]<2.6146e+11){if(f[68]<0.660189){r+=-0.601851;}else{if(f[14]<1.40178e+11){if(f[80]<0.247005){r+=-0.00250432;}else{r+=-0.630925;}}else{if(f[0]<1.61824e+11){r+=-0.762984;}else{r+=-0.65786;}}}}else{r+=0.501023;}}else{if(f[14]<1.3981e+11){r+=0.596155;}else{r+=-0.234649;}}}}else{if(f[28]<1.79162e+11){if(f[68]<5.00207){if(f[0]<1.40229e+11){if(f[28]<1.40649e+11){if(f[28]<1.40186e+11){if(f[64]<2.14319){if(f[14]<1.40095e+11){if(f[35]<3.7907e+08){r+=-0.576695;}else{r+=0.389252;}}else{if(f[26]<5.15889e+07){if(f[5]<2.73601e+06){r+=-0.35727;}else{r+=-0.0139435;}}else{if(f[50]<2.58305){r+=0.580303;}else{r+=-0.148583;}}}}else{if(f[28]<1.3986e+11){if(f[0]<1.40029e+11){if(f[27]<1.05744e+08){r+=0.313703;}else{r+=-0.355514;}}else{if(f[14]<1.40392e+11){r+=-0.147786;}else{r+=0.735769;}}}else{if(f[26]<6.40129e+09){if(f[42]<0.997232){r+=0.0170579;}else{r+=-0.522525;}}else{r+=0.341102;}}}}else{if(f[75]<0.0605316){r+=-0.189432;}else{if(f[66]<2.99463){r+=-0.158617;}else{if(f[0]<1.39818e+11){r+=-0.122893;}else{if(f[72]<0.0933346){r+=0.110144;}else{r+=0.784554;}}}}}}else{if(f[0]<1.3927e+11){if(f[0]<1.38911e+11){r+=-0.305619;}else{r+=0.153511;}}else{r+=-0.597853;}}}else{if(f[0]<1.42195e+11){if(f[27]<5.93706e+09){if(f[9]<1.30855e+07){r+=-0.0543776;}else{if(f[68]<4.97046){r+=-0.645354;}else{r+=-0.0868653;}}}else{r+=0.0624487;}}else{if(f[74]<1.59805){r+=0.726226;}else{r+=-0.111264;}}}}else{if(f[8]<1.55995e+07){if(f[14]<1.4038e+11){if(f[33]<2.21655e+06){if(f[64]<2.02888){r+=-0.126901;}else{r+=-0.610335;}}else{r+=0.110644;}}else{if(f[14]<1.40484e+11){if(f[41]<2.65972e+07){if(f[70]<0.998788){if(f[9]<3.59818e+07){r+=0.973282;}else{r+=0.542083;}}else{if(f[28]<1.39847e+11){r+=-0.304227;}else{if(f[42]<0.997248){r+=0.548251;}else{r+=0.0414034;}}}}else{if(f[38]<1.35947e+07){if(f[0]<1.39988e+11){r+=0.397631;}else{if(f[22]<6.1931e+06){r+=0.108999;}else{r+=-0.487004;}}}else{if(f[9]<3.55024e+07){r+=0.701666;}else{r+=0.128637;}}}}else{r+=-0.469633;}}}else{if(f[13]<5.99234e+09){if(f[64]<1.98823){if(f[23]<6.31783e+07){if(f[28]<1.40121e+11){if(f[67]<9.17494){if(f[32]<4.18651e+06){r+=0.488854;}else{r+=-0.105579;}}else{r+=-0.421801;}}else{r+=-0.479762;}}else{if(f[52]<0.119103){r+=0.363259;}else{if(f[68]<5.99203){if(f[0]<1.39904e+11){r+=0.437278;}else{r+=-0.332004;}}else{if(f[13]<2.378e+09){r+=-0.580088;}else{r+=-0.224846;}}}}}else{if(f[66]<3.60779){if(f[14]<1.40424e+11){if(f[5]<4.5298e+06){r+=-0.205192;}else{if(f[28]<1.39859e+11){r+=0.129055;}else{r+=0.511067;}}}else{r+=-0.519555;}}else{if(f[28]<1.40114e+11){if(f[63]<10.5251){if(f[0]<1.40284e+11){r+=0.0954675;}else{r+=-0.610789;}}else{if(f[28]<1.4009e+11){r+=-0.595986;}else{r+=-0.0278835;}}}else{if(f[36]<6.71184e+08){if(f[70]<0.996859){r+=-0.68541;}else{r+=-0.603948;}}else{if(f[70]<1.00254){r+=-0.363692;}else{r+=0.400898;}}}}}}else{if(f[54]<0.401485){if(f[28]<1.40853e+11){if(f[35]<1.7605e+08){r+=0.180146;}else{r+=0.755069;}}else{r+=-0.371094;}}else{if(f[68]<7.88736){if(f[80]<0.167931){r+=0.578176;}else{r+=-0.390226;}}else{if(f[5]<2.82273e+06){r+=0.0906231;}else{if(f[0]<1.39232e+11){r+=-0.127188;}else{r+=-0.607289;}}}}}}}}else{if(f[14]<1.40148e+11){r+=0.592089;}else{r+=-0.573727;}}}}} return r;}
    static double p18(double[] f){double r=0;if(f[28]<1.39441e+11){if(f[28]<1.39426e+11){if(f[10]<3.28954e+10){if(f[56]<0.996267){if(f[34]<707683){r+=0.714286;}else{r+=-0.925926;}}else{if(f[71]<1.79984){r+=-0.997005;}else{if(f[0]<1.41642e+11){r+=-0.994875;}else{if(f[53]<1.76062){if(f[43]<1.73263){r+=-0.97561;}else{r+=-0.25;}}else{if(f[0]<1.42101e+11){r+=0.928571;}else{r+=-0.692308;}}}}}}else{if(f[42]<1.0138){if(f[17]<1.63097e+09){if(f[22]<1.30514e+10){if(f[40]<2.82829e+10){r+=0.714286;}else{r+=-0.272727;}}else{r+=-0.647059;}}else{if(f[67]<1.39028){r+=0.777778;}else{r+=-0;}}}else{if(f[17]<9.55611e+09){r+=-0.777778;}else{r+=-0;}}}}else{if(f[14]<1.40044e+11){if(f[84]<1){r+=-0.978378;}else{if(f[0]<1.40119e+11){if(f[15]<8.4463e+07){r+=-0.8;}else{r+=0.25;}}else{if(f[0]<1.40387e+11){r+=0.963303;}else{r+=-0.25;}}}}else{if(f[42]<1.00704){r+=-0.992172;}else{r+=0.555556;}}}}else{if(f[68]<7.06146){if(f[79]<1.2525){if(f[42]<0.953817){if(f[84]<2.85){if(f[28]<1.43679e+11){if(f[28]<1.39992e+11){r+=-0.272727;}else{if(f[0]<2.77548e+07){r+=-0;}else{if(f[29]<5.69092e+07){if(f[39]<4.07554e+07){r+=0.25;}else{r+=0.999727;}}else{if(f[5]<3.37361e+08){r+=0.5;}else{r+=-0;}}}}}else{if(f[0]<1.28543e+11){r+=-0.968;}else{r+=-0.25;}}}else{r+=-0.995745;}}else{if(f[28]<1.41856e+11){if(f[28]<1.40997e+11){if(f[41]<9.32729e+07){if(f[14]<1.40138e+11){if(f[68]<1.26305){if(f[10]<7.29487e+07){if(f[14]<1.40056e+11){if(f[21]<4.18735e+06){r+=-0;}else{r+=-0.993958;}}else{if(f[84]<1.15){r+=-0.973723;}else{r+=0.581454;}}}else{if(f[84]<1.15){if(f[28]<1.39781e+11){r+=-0.344828;}else{r+=-0.995181;}}else{if(f[9]<1.7488e+08){r+=0.582169;}else{r+=-0.774775;}}}}else{if(f[28]<1.39625e+11){if(f[84]<1){r+=-0.6;}else{if(f[0]<1.40174e+11){r+=-0.25;}else{r+=0.972028;}}}else{if(f[69]<1.58617){if(f[41]<8.98416e+07){r+=-0.793716;}else{r+=-0.0119205;}}else{if(f[0]<1.40049e+11){r+=-0.894019;}else{r+=-0.990408;}}}}}else{if(f[0]<1.39742e+11){if(f[71]<0.00381932){r+=-0.111111;}else{if(f[42]<0.995869){if(f[0]<1.39736e+11){r+=-0.997821;}else{r+=-0.384615;}}else{if(f[1]<4.71137e+07){r+=-0.714286;}else{r+=0.25;}}}}else{if(f[36]<5.36671e+07){if(f[84]<0.5){if(f[28]<1.40026e+11){r+=-0.837526;}else{r+=0.340072;}}else{if(f[28]<1.40362e+11){r+=0.689887;}else{r+=-0.318095;}}}else{if(f[84]<1.5){if(f[28]<1.40127e+11){r+=-0.884567;}else{r+=0.411231;}}else{if(f[0]<1.39905e+11){r+=0.753351;}else{r+=-0.907439;}}}}}}else{if(f[28]<1.40207e+11){if(f[0]<1.40089e+11){if(f[14]<1.40165e+11){if(f[28]<1.4e+11){if(f[42]<1.0025){r+=-0.979478;}else{r+=0.971947;}}else{if(f[29]<2.06291e+07){r+=0.804277;}else{r+=-0.716144;}}}else{if(f[28]<1.39967e+11){if(f[56]<1.00692){r+=0.692272;}else{r+=-0.987362;}}else{if(f[66]<1.97991){r+=-0.877935;}else{r+=0.19244;}}}}else{if(f[0]<1.40145e+11){if(f[14]<1.39861e+11){if(f[14]<1.39024e+11){r+=-0.956044;}else{r+=0.899396;}}else{if(f[25]<2.72874e+08){r+=-0.913547;}else{r+=-0.361063;}}}else{if(f[14]<1.40348e+11){if(f[84]<0.85){r+=-0.590809;}else{r+=0.395595;}}else{if(f[84]<1.15){r+=0.474847;}else{r+=-0.894247;}}}}}else{if(f[12]<1.28981e+08){if(f[14]<1.40269e+11){if(f[70]<0.980855){if(f[84]<1.15){r+=-0.993186;}else{r+=0.998944;}}else{if(f[39]<2.21086e+08){r+=-0.753082;}else{r+=0.372855;}}}else{if(f[42]<0.991692){if(f[41]<6.54033e+08){r+=-0.997067;}else{r+=-0.344828;}}else{if(f[84]<4.65){r+=0.370455;}else{r+=0.930734;}}}}else{if(f[55]<2.43678){if(f[84]<4.85){if(f[70]<0.983472){r+=0.636582;}else{r+=-0.740458;}}else{if(f[0]<1.4098e+11){r+=-0.692308;}else{r+=0.992593;}}}else{if(f[28]<1.40892e+11){if(f[82]<0.777451){r+=-0.76087;}else{r+=-0.141243;}}else{if(f[42]<1.00192){r+=-0.789474;}else{r+=0.992212;}}}}}}}else{if(f[0]<1.41525e+11){if(f[0]<1.39318e+11){if(f[28]<1.41458e+11){if(f[28]<1.41438e+11){if(f[10]<4.74708e+10){r+=-0.997195;}else{r+=-0.25;}}else{if(f[0]<1.39095e+11){r+=-0.666667;}else{r+=0.555556;}}}else{if(f[0]<1.3892e+11){r+=-0.764706;}else{if(f[14]<1.39721e+11){r+=-0.25;}else{r+=0.999116;}}}}else{if(f[39]<4.22689e+07){if(f[0]<1.40035e+11){r+=0.25;}else{r+=0.987578;}}else{if(f[14]<1.45817e+11){if(f[28]<1.41011e+11){if(f[42]<1.00205){r+=-0.993506;}else{r+=0.840637;}}else{if(f[70]<1.01775){r+=-0.990736;}else{r+=0.6;}}}else{if(f[84]<2){r+=0.980769;}else{r+=-0.986014;}}}}}else{if(f[14]<1.41125e+11){if(f[56]<0.992261){r+=-0.538462;}else{if(f[28]<1.41044e+11){r+=-0;}else{if(f[36]<6.92585e+07){r+=0.25;}else{r+=0.99926;}}}}else{if(f[33]<7.23964e+08){r+=-0.973154;}else{r+=-0;}}}}}else{if(f[56]<0.981991){if(f[57]<0.246745){if(f[70]<1.21374){if(f[5]<1.41573e+08){r+=-0.966942;}else{r+=-0;}}else{if(f[0]<1.40659e+11){r+=0.996756;}else{r+=-0.25;}}}else{if(f[27]<2.12944e+08){if(f[67]<8.87148){if(f[37]<1.89028e+07){if(f[28]<1.66784e+11){r+=-0.979695;}else{r+=0.8;}}else{r+=-0.998982;}}else{if(f[0]<1.40191e+11){r+=-0.636364;}else{r+=0.5;}}}else{if(f[84]<0.5){if(f[14]<1.39507e+11){r+=-0.8;}else{if(f[13]<1.53247e+08){r+=-0.25;}else{if(f[10]<1.2218e+10){r+=0.982684;}else{r+=0.25;}}}}else{if(f[6]<3.70325e+09){r+=-0.989501;}else{r+=-0.111111;}}}}}else{if(f[70]<1.00496){if(f[84]<0.5){if(f[14]<1.42319e+11){r+=-0.25;}else{r+=0.878788;}}else{if(f[42]<0.999027){r+=-0.25;}else{r+=-0.994652;}}}else{if(f[84]<3.15){if(f[84]<2.65){if(f[28]<1.42027e+11){if(f[0]<1.39322e+11){r+=-0.25;}else{r+=-0.996476;}}else{if(f[84]<1.5){if(f[0]<1.40053e+11){r+=-0.884615;}else{r+=0.988962;}}else{r+=-0.990338;}}}else{if(f[6]<5.52625e+08){r+=0.999753;}else{r+=-0.5;}}}else{r+=-0.982222;}}}}}}else{if(f[41]<7.23036e+09){if(f[42]<1.00934){if(f[41]<1.82342e+09){if(f[0]<1.40039e+11){if(f[0]<1.39504e+11){if(f[14]<1.40843e+11){if(f[53]<0.620019){if(f[0]<1.39294e+11){if(f[12]<9.12857e+06){r+=-0.951807;}else{r+=0.657143;}}else{if(f[12]<2.7119e+08){r+=-0.986842;}else{r+=-0;}}}else{if(f[84]<1.5){if(f[14]<1.39942e+11){r+=0.943262;}else{r+=-0.98524;}}else{if(f[25]<6.06079e+07){r+=-0.813953;}else{r+=0.984228;}}}}else{if(f[28]<1.4e+11){if(f[14]<1.41131e+11){r+=0.911111;}else{r+=-0.6;}}else{if(f[56]<0.996663){r+=0.5;}else{r+=-0.99807;}}}}else{if(f[82]<6.99675){if(f[79]<4.83095){if(f[51]<0.234232){if(f[28]<1.40005e+11){r+=-0.921569;}else{r+=0.845238;}}else{if(f[28]<1.39764e+11){r+=0.247788;}else{r+=-0.915724;}}}else{if(f[55]<0.55423){if(f[84]<2.65){r+=0.951134;}else{r+=-0.913043;}}else{if(f[6]<2.71492e+07){r+=-0.933884;}else{r+=0.25;}}}}else{if(f[37]<6.31975e+07){if(f[12]<1.46747e+07){r+=-0.92;}else{if(f[0]<1.39792e+11){r+=-0.764706;}else{r+=0.985714;}}}else{if(f[67]<0.599297){if(f[72]<0.215446){r+=-0;}else{r+=-0.981395;}}else{if(f[70]<1.00176){r+=0.73913;}else{r+=-0.454545;}}}}}}else{if(f[25]<4.48597e+08){if(f[84]<2.15){if(f[36]<2.08631e+08){if(f[0]<1.40261e+11){if(f[67]<1.01047){r+=0.073492;}else{r+=0.629219;}}else{if(f[84]<0.85){r+=-0.917199;}else{r+=-0.0526786;}}}else{if(f[0]<1.40069e+11){if(f[14]<1.39557e+11){r+=0.951807;}else{r+=-0.908873;}}else{if(f[70]<1.00355){r+=0.689793;}else{r+=-0.540984;}}}}else{if(f[27]<5.87054e+08){if(f[14]<1.4124e+11){if(f[0]<1.40148e+11){r+=-0.108511;}else{r+=-0.866157;}}else{if(f[84]<4.65){r+=-0.960396;}else{r+=0.862069;}}}else{if(f[70]<1.4018e+11){if(f[0]<1.40195e+11){r+=0.736913;}else{r+=-0.789474;}}else{if(f[84]<2.65){r+=0.6;}else{r+=-0.975;}}}}}else{if(f[84]<2.85){if(f[28]<1.44205e+11){if(f[42]<0.998487){if(f[84]<4.60001){r+=-0.375;}else{r+=0.940171;}}else{if(f[0]<1.41227e+11){r+=-0.970554;}else{r+=0.641791;}}}else{if(f[12]<6.47067e+08){r+=-0;}else{r+=0.987013;}}}else{if(f[42]<1.00093){if(f[14]<1.40046e+11){r+=-0.555556;}else{if(f[28]<1.40033e+11){r+=-0.25;}else{r+=0.993428;}}}else{if(f[12]<2.22335e+09){r+=-0.99618;}else{r+=0.25;}}}}}}else{if(f[84]<2.15){if(f[28]<1.43843e+11){if(f[82]<8.61544){if(f[70]<0.992619){if(f[84]<1.5){if(f[1]<2.8912e+07){r+=-0;}else{r+=-0.891892;}}else{r+=0.971223;}}else{if(f[72]<69.9052){if(f[13]<2.32324e+08){r+=0.631579;}else{r+=-0.92786;}}else{if(f[0]<1.40093e+11){r+=0.906977;}else{r+=0.25;}}}}else{if(f[82]<16.1865){if(f[14]<1.40189e+11){if(f[0]<1.39989e+11){r+=-0.157556;}else{r+=0.805293;}}else{if(f[0]<1.40984e+11){r+=-0.91453;}else{r+=0.2;}}}else{if(f[28]<1.4e+11){if(f[84]<1.15){r+=0.684211;}else{r+=-0.473684;}}else{if(f[21]<3.00467e+08){r+=-0.867021;}else{r+=0.6;}}}}}else{if(f[84]<0.8){if(f[0]<1.40237e+11){r+=0.25;}else{r+=0.99807;}}else{r+=-0.968992;}}}else{if(f[14]<1.40052e+11){if(f[10]<3.97341e+09){if(f[83]<0.684858){r+=0.111111;}else{if(f[17]<5.47741e+07){r+=-0.996143;}else{r+=-0;}}}else{if(f[84]<2.85){r+=-0.647059;}else{if(f[8]<2.35308e+09){r+=0.916667;}else{r+=-0;}}}}else{if(f[66]<0.241058){if(f[84]<2.5){if(f[55]<0.499146){if(f[0]<1.40193e+11){r+=0.733333;}else{r+=-0.5;}}else{if(f[51]<0.579531){r+=-0;}else{r+=-0.976366;}}}else{if(f[64]<0.344816){if(f[82]<1.94452){r+=0.555556;}else{r+=-0.703704;}}else{if(f[0]<1.40004e+11){r+=0.962963;}else{r+=-0;}}}}else{if(f[28]<1.40041e+11){if(f[12]<2.33187e+09){if(f[84]<2.85){r+=-0.979522;}else{r+=-0.314286;}}else{if(f[45]<1.0816){r+=-0.538462;}else{r+=0.714286;}}}else{if(f[28]<1.4071e+11){if(f[55]<1.6066){r+=0.95986;}else{r+=0.0996016;}}else{if(f[28]<1.40715e+11){r+=-0.25;}else{r+=-0.974359;}}}}}}}}else{if(f[13]<1.88752e+09){if(f[40]<7.94263e+07){if(f[84]<2.5){if(f[23]<1.1809e+08){if(f[10]<1.54833e+08){if(f[8]<1.04256e+08){r+=-0.982609;}else{if(f[84]<1.85){r+=-0.818182;}else{r+=0.4;}}}else{if(f[0]<1.42228e+11){r+=-0;}else{r+=0.84;}}}else{if(f[70]<0.979467){r+=-0.948718;}else{if(f[70]<0.99022){if(f[14]<1.41273e+11){r+=0.99549;}else{r+=-0;}}else{r+=-0.918367;}}}}else{if(f[28]<1.41508e+11){if(f[42]<1.00955){r+=0.333333;}else{if(f[15]<6.44621e+06){if(f[12]<6.04441e+07){r+=-0.777778;}else{r+=0.5;}}else{if(f[41]<1.43336e+08){r+=-0.985866;}else{r+=-0.25;}}}}else{r+=0.692308;}}}else{if(f[14]<9.23162e+10){if(f[0]<1.42551e+11){r+=-0.875;}else{r+=0.916667;}}else{if(f[79]<1.40824){if(f[42]<1.01207){if(f[70]<0.989363){if(f[84]<2.5){r+=0.945455;}else{r+=-0.5;}}else{if(f[84]<3.5){r+=-0.88102;}else{r+=0.183673;}}}else{if(f[66]<1.06582){if(f[28]<1.39941e+11){r+=-0.45098;}else{r+=-0.981926;}}else{if(f[27]<1.77046e+08){r+=0.275;}else{r+=-0.853659;}}}}else{if(f[28]<1.3993e+11){if(f[84]<3.35){if(f[10]<3.26331e+08){r+=-0.972414;}else{r+=0.375;}}else{if(f[0]<1.41374e+11){r+=0.833333;}else{r+=-0.25;}}}else{if(f[7]<1.70007e+09){if(f[42]<1.01174){r+=-0.831832;}else{r+=-0.994881;}}else{r+=0.5;}}}}}}else{if(f[70]<0.987477){if(f[14]<1.39811e+11){r+=0.997035;}else{if(f[0]<1.43531e+11){r+=0.5;}else{r+=-0.851852;}}}else{if(f[61]<0.219059){r+=-0;}else{r+=-0.9375;}}}}}else{if(f[28]<1.4459e+11){if(f[13]<2.08135e+10){if(f[14]<1.42607e+11){if(f[37]<1.7824e+08){if(f[23]<4.9397e+07){if(f[82]<2.86692){r+=0.333333;}else{if(f[55]<0.440063){r+=0.111111;}else{r+=-0.938462;}}}else{if(f[84]<1.5){if(f[29]<3.22389e+07){r+=0.938462;}else{r+=-0.111111;}}else{if(f[84]<3.15){if(f[5]<1.91914e+06){r+=-0;}else{r+=-0.826087;}}else{r+=0.454545;}}}}else{if(f[14]<1.37706e+11){if(f[15]<1.14089e+10){if(f[45]<0.0386246){r+=-0;}else{r+=0.897436;}}else{r+=-0.6;}}else{if(f[53]<0.0012737){if(f[58]<0.552428){r+=-0;}else{r+=0.818182;}}else{if(f[0]<1.40083e+11){if(f[69]<0.397717){r+=-0.910714;}else{r+=0.0731707;}}else{if(f[83]<0.931936){r+=-0;}else{r+=-0.99181;}}}}}}else{if(f[84]<1){r+=0.939394;}else{r+=-0.793103;}}}else{if(f[23]<5.39454e+08){if(f[28]<1.40207e+11){if(f[40]<3.20411e+10){r+=0.956989;}else{r+=-0;}}else{r+=-0.5;}}else{if(f[79]<1.78748){if(f[40]<2.7604e+10){if(f[76]<1.67055){r+=0.777778;}else{if(f[14]<1.40186e+11){r+=0.25;}else{r+=-0.555556;}}}else{if(f[56]<1.0026){r+=-0.75;}else{r+=0.25;}}}else{if(f[2]<6.12052e+06){r+=0.333333;}else{if(f[31]<1.61153e+08){if(f[4]<1.06378e+08){r+=-0.6;}else{r+=0.25;}}else{r+=-0.913043;}}}}}}else{if(f[84]<1){if(f[14]<1.39982e+11){r+=0.25;}else{r+=0.991968;}}else{if(f[1]<3.59263e+09){r+=-0.932203;}else{r+=-0;}}}}}}else{if(f[28]<1.78386e+11){if(f[67]<6.78553){if(f[28]<1.40105e+11){if(f[28]<1.40073e+11){if(f[25]<1.39391e+09){if(f[84]<2.85){if(f[0]<1.40297e+11){if(f[0]<1.40075e+11){if(f[84]<2.15){if(f[0]<1.39904e+11){r+=0.25;}else{r+=0.977186;}}else{if(f[14]<1.40236e+11){r+=0.882353;}else{r+=-0.714286;}}}else{if(f[5]<8.72201e+06){if(f[9]<5.73635e+07){r+=-0.636364;}else{r+=-0;}}else{if(f[71]<1.57855){r+=0.818182;}else{r+=-0;}}}}else{r+=-0.870968;}}else{r+=-0.891892;}}else{if(f[17]<3.32968e+08){if(f[16]<2.44193e+07){r+=-0.25;}else{r+=-0.984479;}}else{r+=-0;}}}else{if(f[14]<1.40082e+11){r+=-0.789474;}else{if(f[36]<2.68705e+07){r+=0.983827;}else{r+=-0;}}}}else{if(f[42]<0.998){if(f[70]<1.00407){if(f[14]<1.40526e+11){if(f[14]<1.40475e+11){if(f[15]<5.934e+07){r+=0.666667;}else{r+=-0.866667;}}else{if(f[1]<2.98737e+07){r+=0.985507;}else{r+=0.25;}}}else{r+=-0.931818;}}else{if(f[23]<1.13173e+09){if(f[42]<0.99769){r+=-0.995614;}else{if(f[0]<1.40019e+11){r+=-0.666667;}else{r+=0.25;}}}else{if(f[2]<3.72958e+07){r+=0.666667;}else{r+=-0.25;}}}}else{if(f[19]<2.68799e+08){if(f[66]<1.23347){r+=-0;}else{if(f[40]<5.87134e+08){r+=-0.986672;}else{if(f[9]<1.2864e+10){r+=-0;}else{r+=-0.555556;}}}}else{if(f[0]<1.40764e+11){r+=-0;}else{r+=0.6;}}}}}else{if(f[13]<5.49688e+09){if(f[71]<0.371256){if(f[64]<1.20771){if(f[8]<5.44666e+07){if(f[0]<1.39974e+11){r+=-0;}else{r+=0.692308;}}else{r+=-0.826087;}}else{if(f[41]<2.97153e+09){if(f[68]<8.59271){if(f[28]<1.39925e+11){if(f[84]<2.15){if(f[0]<1.40053e+11){r+=0.909091;}else{r+=-0;}}else{r+=-0.714286;}}else{if(f[13]<4.11379e+09){if(f[8]<2.54519e+09){r+=-0.988304;}else{r+=-0.25;}}else{r+=-0.272727;}}}else{r+=-0.996722;}}else{if(f[1]<2.24866e+08){r+=-0.6;}else{r+=0.5;}}}}else{if(f[28]<1.39939e+11){if(f[84]<2.15){if(f[0]<1.40257e+11){if(f[0]<1.39956e+11){r+=-0;}else{r+=0.960396;}}else{r+=-0.777778;}}else{if(f[0]<1.39985e+11){r+=-0.25;}else{r+=-0.925926;}}}else{if(f[73]<72.7775){if(f[28]<1.40115e+11){if(f[83]<0.419901){if(f[53]<0.380337){r+=0.870968;}else{if(f[76]<0.531827){r+=-0.733333;}else{r+=0.333333;}}}else{if(f[49]<1.06616){if(f[48]<0.885936){r+=-0.985294;}else{r+=-0.428571;}}else{r+=0.0909091;}}}else{if(f[27]<5.49809e+10){if(f[30]<1.71017e+08){if(f[39]<2.08627e+09){r+=-0.983685;}else{r+=-0.25;}}else{r+=-0;}}else{r+=-0;}}}else{r+=0.6;}}}}else{if(f[21]<5.87837e+08){if(f[5]<2.35844e+06){r+=0.333333;}else{if(f[12]<2.90105e+09){r+=-0.25;}else{r+=-0.935484;}}}else{if(f[0]<1.40239e+11){if(f[23]<4.11044e+09){r+=0.111111;}else{r+=0.888889;}}else{if(f[0]<1.40927e+11){r+=-0.555556;}else{r+=0.4;}}}}}}else{if(f[14]<1.4013e+11){r+=0.995429;}else{r+=-0.97861;}}}} return r;}
    static double p19(double[] f){double r=0;if(f[28]<1.39443e+11){if(f[28]<1.39428e+11){if(f[56]<0.994338){if(f[71]<0.866846){r+=0.651819;}else{if(f[84]<1.5){r+=-0.646403;}else{r+=0.263338;}}}else{if(f[21]<5.44959e+10){if(f[10]<3.56819e+10){if(f[55]<0.244931){if(f[53]<0.601866){if(f[57]<26.9103){if(f[52]<0.867886){r+=-0.640876;}else{if(f[31]<1.77453e+07){r+=0.0877494;}else{r+=-0.39722;}}}else{r+=0.0623775;}}else{r+=0.348125;}}else{if(f[9]<2.1034e+08){r+=-0.684131;}else{if(f[41]<2.34988e+08){if(f[42]<1.0095){if(f[28]<1.39421e+11){r+=-0.671108;}else{if(f[14]<1.39832e+11){r+=-0.541916;}else{r+=0.267473;}}}else{if(f[42]<1.01176){if(f[71]<2.03441){r+=1.41989;}else{r+=0.630565;}}else{r+=-0.397804;}}}else{if(f[14]<1.37487e+11){if(f[50]<0.991008){r+=0.312676;}else{r+=-0.588144;}}else{if(f[13]<1.29865e+08){if(f[0]<1.40297e+11){r+=-0.453974;}else{r+=-0.0163518;}}else{r+=-0.678925;}}}}}}else{if(f[23]<2.74496e+10){if(f[13]<3.11632e+10){r+=-0.105331;}else{r+=0.535615;}}else{if(f[13]<2.00914e+10){r+=0.0639184;}else{if(f[84]<2.5){r+=-0.806638;}else{r+=-0.0280891;}}}}}else{r+=0.416318;}}}else{if(f[14]<1.40133e+11){if(f[84]<1){r+=-0.677678;}else{if(f[9]<1.23249e+08){if(f[20]<4.9514e+07){if(f[7]<1.37271e+08){r+=-0.578195;}else{r+=-0.0743272;}}else{r+=0.456263;}}else{if(f[13]<3.40347e+08){r+=0.687175;}else{r+=0.137571;}}}}else{if(f[42]<1.00753){r+=-0.689837;}else{r+=0.190632;}}}}else{if(f[69]<0.548067){if(f[14]<1.40038e+11){if(f[28]<1.39921e+11){if(f[42]<1.0022){if(f[59]<0.00499142){if(f[15]<1.25546e+07){r+=0.616348;}else{r+=0.111809;}}else{r+=-0.684826;}}else{if(f[84]<1.15){if(f[84]<0.5){if(f[14]<1.38966e+11){r+=-0.495525;}else{if(f[0]<1.40144e+11){r+=0.691779;}else{r+=1.24813;}}}else{r+=-0.800602;}}else{if(f[52]<1.01341){if(f[79]<1.13959){if(f[29]<7.0896e+07){r+=0.789724;}else{r+=0.229722;}}else{if(f[84]<2){if(f[3]<3.41308e+07){r+=0.592433;}else{r+=0.0817785;}}else{if(f[0]<1.40778e+11){r+=-0.679885;}else{r+=-0.170162;}}}}else{if(f[29]<9.36755e+07){if(f[53]<3.39183){if(f[30]<6.28437e+07){if(f[31]<6.8782e+07){if(f[28]<1.39921e+11){r+=0.822358;}else{r+=0.289467;}}else{r+=0.215983;}}else{if(f[0]<1.40471e+11){r+=0.639856;}else{r+=-0.337411;}}}else{r+=0.172585;}}else{if(f[84]<2.65){r+=0.657622;}else{r+=-0.320691;}}}}}}else{if(f[84]<0.5){if(f[37]<8.19446e+07){if(f[28]<1.40145e+11){if(f[0]<1.4026e+11){if(f[0]<1.40141e+11){r+=-0.456355;}else{r+=0.80503;}}else{r+=-0.774602;}}else{if(f[56]<0.998736){if(f[14]<1.39522e+11){r+=-0.0580617;}else{if(f[0]<1.40256e+11){r+=0.728811;}else{if(f[38]<1.79038e+08){r+=1.11181;}else{r+=0.480939;}}}}else{if(f[0]<1.40267e+11){if(f[79]<1.25287){if(f[28]<1.40146e+11){r+=0.391629;}else{r+=1.39661;}}else{if(f[41]<2.2253e+09){r+=0.966188;}else{r+=1.42413;}}}else{r+=-0.489399;}}}}else{if(f[28]<1.54003e+11){if(f[64]<7.39057){if(f[42]<0.99784){if(f[12]<1.29027e+08){if(f[39]<2.21076e+08){r+=-0.647905;}else{if(f[79]<1.24606){r+=-1.1826;}else{r+=-0.59867;}}}else{if(f[12]<1.42629e+09){r+=-0.706687;}else{r+=0.0217585;}}}else{if(f[66]<3.47654){if(f[64]<6.46214){if(f[29]<2.13238e+06){r+=-0.120454;}else{r+=-0.702462;}}else{r+=0.0670079;}}else{r+=-0.0285709;}}}else{if(f[0]<1.40266e+11){r+=1.29335;}else{r+=-0.641921;}}}else{if(f[39]<1.02607e+09){r+=0.943076;}else{if(f[67]<0.657832){r+=0.674794;}else{r+=0.127329;}}}}}else{if(f[14]<1.39384e+11){if(f[84]<3.5){if(f[42]<1.00673){if(f[56]<0.993113){r+=-0.68636;}else{if(f[28]<1.40159e+11){if(f[77]<1.24077){if(f[38]<2.549e+08){r+=-1.19203;}else{r+=-0.341533;}}else{if(f[9]<1.73823e+08){r+=0.600399;}else{r+=0.0339302;}}}else{if(f[28]<1.40207e+11){if(f[51]<1.57175){r+=0.793423;}else{r+=-0.243437;}}else{if(f[79]<1.24739){r+=1.44502;}else{r+=0.751234;}}}}}else{if(f[0]<1.40367e+11){if(f[14]<1.38845e+11){if(f[79]<1.25283){if(f[0]<1.40145e+11){r+=-0.611201;}else{r+=-1.22501;}}else{r+=-0.701485;}}else{if(f[42]<1.00981){r+=-0.832973;}else{if(f[28]<1.4005e+11){r+=0.703635;}else{r+=-0.566247;}}}}else{if(f[14]<1.38976e+11){if(f[84]<2.15){if(f[84]<1.5){r+=-1.12303;}else{r+=0.758091;}}else{if(f[79]<1.25343){r+=-1.20353;}else{r+=-0.66653;}}}else{if(f[41]<1.00011e+09){if(f[79]<1.25204){r+=-1.14229;}else{r+=-0.658552;}}else{r+=-0.694194;}}}}}else{if(f[28]<1.40093e+11){if(f[0]<1.7893e+11){r+=0.839167;}else{r+=-0.684524;}}else{r+=-0.72613;}}}else{if(f[25]<1.88466e+07){if(f[0]<1.39991e+11){if(f[37]<5.13647e+07){r+=0.720975;}else{r+=1.41282;}}else{r+=-0.643969;}}else{if(f[10]<1.07138e+08){if(f[42]<1.00047){if(f[23]<8.67407e+07){if(f[84]<2.5){if(f[28]<1.40013e+11){r+=-0.195096;}else{r+=0.862205;}}else{r+=-0.645289;}}else{if(f[28]<1.40131e+11){r+=0.207066;}else{r+=-0.60903;}}}else{if(f[79]<1.25247){if(f[0]<1.40023e+11){if(f[14]<1.39774e+11){r+=0.492904;}else{r+=-1.61159;}}else{if(f[0]<1.40201e+11){r+=-0.685287;}else{r+=-1.21473;}}}else{if(f[51]<0.120701){if(f[29]<2.50463e+07){r+=0.7369;}else{r+=-0.130005;}}else{if(f[84]<1.85){r+=-0.806391;}else{r+=-0.696446;}}}}}else{if(f[28]<1.40149e+11){if(f[0]<1.40221e+11){if(f[78]<1.81786){if(f[0]<1.40145e+11){r+=-0.747001;}else{r+=-1.14392;}}else{if(f[1]<2.1822e+08){r+=-0.686453;}else{r+=0.2054;}}}else{if(f[84]<1.15){if(f[63]<0.313685){r+=-0.10851;}else{r+=-0.769146;}}else{if(f[79]<1.25274){r+=-1.23521;}else{r+=-0.799703;}}}}else{if(f[23]<3.69252e+07){if(f[26]<2.32244e+08){r+=-0.749182;}else{if(f[84]<2.65){r+=0.749517;}else{r+=-0.484732;}}}else{if(f[83]<0.949874){if(f[84]<1.5){r+=0.687141;}else{r+=-0.773909;}}else{if(f[0]<1.39342e+11){r+=0.0447624;}else{r+=-0.696173;}}}}}}}}}}else{if(f[14]<1.40105e+11){if(f[0]<1.40102e+11){if(f[0]<1.39985e+11){if(f[84]<1.5){if(f[0]<1.39983e+11){if(f[79]<1.25286){if(f[28]<1.40178e+11){if(f[28]<1.40114e+11){r+=0.406481;}else{r+=-1.51543;}}else{if(f[12]<1.29049e+08){if(f[39]<2.21823e+08){r+=-0.708974;}else{r+=-1.18448;}}else{if(f[10]<1.79699e+09){r+=-0.733213;}else{r+=0.187315;}}}}else{if(f[29]<4.30706e+06){if(f[39]<3.37347e+08){if(f[10]<5.72828e+07){r+=-0.150503;}else{r+=-0.696957;}}else{r+=0.485253;}}else{if(f[21]<1.49624e+07){r+=-0.11015;}else{if(f[59]<0.0633579){r+=-0.184833;}else{r+=-0.724917;}}}}}else{if(f[28]<1.4018e+11){if(f[54]<0.573845){if(f[38]<1.57225e+08){if(f[15]<6.38132e+06){r+=-0.895524;}else{r+=0.151342;}}else{if(f[26]<3.24331e+08){r+=0.359932;}else{r+=-0.571451;}}}else{if(f[79]<1.24859){r+=-1.40528;}else{r+=-0.674109;}}}else{if(f[79]<1.40426){if(f[43]<2.25825){if(f[71]<2.46182){r+=-0.577925;}else{r+=0.523782;}}else{if(f[30]<4.18551e+06){r+=0.59547;}else{r+=-0.208415;}}}else{if(f[26]<2.24434e+08){r+=-0.498813;}else{if(f[54]<0.464809){r+=0.0450048;}else{r+=0.95192;}}}}}}else{if(f[78]<1.49751){if(f[15]<9.57531e+06){if(f[68]<0.460818){if(f[77]<1.26467){r+=0.741976;}else{if(f[14]<1.40092e+11){r+=-0.108483;}else{r+=0.490489;}}}else{if(f[14]<1.40065e+11){r+=-0.154528;}else{r+=0.406481;}}}else{if(f[64]<2.28375){r+=-0.513042;}else{r+=0.504908;}}}else{if(f[21]<1.93106e+07){if(f[71]<1.65406){r+=0.822084;}else{r+=-0.0415546;}}else{if(f[83]<9.61462){r+=-0.70082;}else{r+=-0.192287;}}}}}else{if(f[28]<1.40184e+11){if(f[0]<1.40068e+11){if(f[28]<1.40179e+11){if(f[41]<9.99027e+07){r+=0.701811;}else{if(f[79]<0.766717){if(f[84]<1.5){r+=-0.822425;}else{r+=0.311718;}}else{if(f[79]<1.25166){r+=-0.984504;}else{r+=-0.682829;}}}}else{if(f[84]<2.5){if(f[54]<0.679339){if(f[33]<2.56409e+06){r+=0.305387;}else{r+=-0.108024;}}else{if(f[28]<1.40182e+11){r+=-1.03292;}else{r+=-0.15108;}}}else{if(f[0]<1.40021e+11){r+=0.711305;}else{r+=-0.59699;}}}}else{if(f[84]<0.85){if(f[38]<2.87907e+08){if(f[0]<1.40089e+11){if(f[29]<1.7005e+07){r+=0.786078;}else{r+=0.990001;}}else{if(f[79]<1.25128){r+=1.64562;}else{r+=0.914276;}}}else{if(f[29]<8.78777e+07){r+=-0.50105;}else{if(f[0]<1.40069e+11){r+=1.26534;}else{r+=0.754369;}}}}else{r+=-1.01066;}}}else{if(f[30]<6.40095e+08){if(f[84]<1){if(f[36]<2.07501e+08){if(f[41]<1.91085e+09){r+=0.977395;}else{r+=0.28074;}}else{r+=0.759702;}}else{if(f[41]<8.4243e+08){r+=-0.807836;}else{if(f[0]<1.39992e+11){if(f[84]<2.5){r+=0.336882;}else{r+=0.712889;}}else{if(f[0]<1.39992e+11){r+=0;}else{r+=0.714299;}}}}}else{if(f[84]<1.40001){r+=0.787462;}else{r+=-0.744919;}}}}}else{if(f[28]<1.84448e+11){if(f[84]<2.85){if(f[79]<1.65623){if(f[38]<6.43952e+07){if(f[42]<1.0006){if(f[84]<1.2){r+=-0.462763;}else{r+=0.630032;}}else{if(f[27]<1.94333e+08){r+=-0.780365;}else{r+=-0.187487;}}}else{if(f[84]<0.85){if(f[75]<0.0711281){r+=-0.131366;}else{if(f[1]<1.52145e+08){r+=-0.766496;}else{r+=-0.149445;}}}else{if(f[32]<1.56475e+07){if(f[0]<1.40145e+11){r+=-0.714871;}else{r+=-1.14765;}}else{if(f[0]<1.4158e+11){r+=-0.650975;}else{r+=0.280479;}}}}}else{if(f[51]<0.17894){r+=0.45531;}else{r+=-0.694399;}}}else{if(f[36]<2.18031e+08){r+=-0.761852;}else{r+=0.707678;}}}else{if(f[79]<1.25213){r+=0.66265;}else{r+=1.60289;}}}}else{if(f[55]<1.27413){if(f[84]<1.15){if(f[84]<0.85){if(f[14]<1.4211e+11){if(f[0]<1.3988e+11){if(f[0]<1.39712e+11){r+=-0.707627;}else{if(f[12]<4.06305e+08){if(f[79]<1.26465){r+=0.847442;}else{r+=1.55442;}}else{r+=-0.502718;}}}else{if(f[0]<1.40261e+11){if(f[0]<1.4003e+11){if(f[56]<1.00363){r+=-0.697847;}else{r+=-0.0470066;}}else{if(f[79]<1.25277){r+=-0.831066;}else{r+=-1.02045;}}}else{if(f[28]<1.42758e+11){if(f[37]<3.03011e+07){r+=-0.885635;}else{r+=-0.70923;}}else{if(f[1]<2.27773e+09){r+=0.521622;}else{r+=-0.420622;}}}}}else{if(f[1]<2.02182e+09){r+=0.686394;}else{r+=-0.358642;}}}else{if(f[29]<1.24592e+07){if(f[56]<1.00155){if(f[56]<1.00026){r+=-0.809563;}else{if(f[14]<1.40151e+11){r+=-0.681714;}else{if(f[12]<3.07512e+07){r+=0.635845;}else{r+=0.897783;}}}}else{if(f[36]<1.46558e+07){r+=0.738813;}else{if(f[50]<3.30186){r+=-1.01384;}else{if(f[42]<0.993503){r+=-0.560752;}else{r+=-0.0255871;}}}}}else{if(f[0]<1.40128e+11){if(f[14]<1.40471e+11){r+=-0.166526;}else{r+=1.3187;}}else{if(f[28]<1.39962e+11){if(f[41]<1.80616e+09){r+=-1.18638;}else{if(f[79]<1.25294){r+=-1.17088;}else{r+=-0.712186;}}}else{if(f[28]<1.4042e+11){r+=0.698568;}else{r+=-0.786043;}}}}}}else{if(f[14]<1.40351e+11){if(f[28]<1.40013e+11){if(f[79]<1.25248){if(f[0]<1.40145e+11){if(f[14]<1.4016e+11){r+=-0.055874;}else{r+=-0.698862;}}else{if(f[83]<1.60374){if(f[6]<2.37328e+07){r+=-0.81762;}else{r+=0.645038;}}else{if(f[77]<0.237523){r+=-0.229955;}else{r+=-1.23792;}}}}else{if(f[84]<2.15){if(f[41]<1.82133e+09){if(f[36]<2.08598e+08){r+=-0.981787;}else{r+=-1.4416;}}else{if(f[28]<1.39967e+11){r+=-0.686713;}else{r+=-1.04368;}}}else{if(f[83]<1.65746){if(f[0]<1.40229e+11){r+=0.725955;}else{r+=-0.204615;}}else{if(f[0]<1.40246e+11){r+=-0.890916;}else{r+=-0.697954;}}}}}else{if(f[56]<1.00146){if(f[66]<4.50885){if(f[28]<1.40801e+11){if(f[14]<1.40107e+11){r+=0.194044;}else{r+=-0.75719;}}else{if(f[0]<1.40622e+11){r+=-0.703991;}else{r+=0.680532;}}}else{if(f[0]<1.39979e+11){r+=-0.324164;}else{if(f[8]<5.42842e+07){r+=0.0247544;}else{r+=0.803174;}}}}else{if(f[28]<1.40067e+11){if(f[0]<1.40162e+11){r+=0.0576534;}else{r+=-0.772835;}}else{if(f[66]<0.215955){r+=-0.487893;}else{if(f[53]<0.969144){r+=0.670854;}else{r+=1.22101;}}}}}}else{if(f[83]<2.31934){if(f[14]<1.40942e+11){if(f[84]<1.5){if(f[28]<1.4057e+11){if(f[56]<1.00127){r+=-0.846814;}else{r+=-1.30486;}}else{if(f[83]<1.77369){r+=0.53483;}else{r+=-0.558264;}}}else{if(f[42]<0.990128){if(f[0]<1.39213e+11){r+=0.234299;}else{r+=-0.66553;}}else{if(f[56]<0.99707){r+=-0.182838;}else{r+=0.554947;}}}}else{if(f[84]<4.85){if(f[56]<0.999337){if(f[28]<1.41215e+11){r+=0.569033;}else{r+=-0.660764;}}else{r+=-0.711068;}}else{if(f[67]<0.717324){if(f[52]<1.13577){r+=-0.276667;}else{r+=-0.713342;}}else{if(f[25]<2.84291e+08){r+=0.634893;}else{r+=-0.577364;}}}}}else{if(f[27]<1.97554e+08){if(f[0]<1.40229e+11){if(f[84]<12){if(f[14]<1.40609e+11){r+=0.381365;}else{r+=-0.638846;}}else{if(f[36]<4.30921e+08){r+=0.966976;}else{r+=-0.640403;}}}else{if(f[42]<1.00263){if(f[84]<4.85){r+=-0.697235;}else{r+=-0.256606;}}else{if(f[25]<9.24296e+07){r+=-0.507109;}else{r+=0.566444;}}}}else{if(f[84]<12){if(f[79]<1.08677){if(f[0]<1.41144e+11){r+=-0.697843;}else{r+=-0.152471;}}else{if(f[0]<1.40997e+11){r+=-0.719652;}else{r+=-0.662169;}}}else{if(f[0]<1.40392e+11){if(f[0]<1.39784e+11){r+=-0.755088;}else{r+=1.11969;}}else{r+=-0.69254;}}}}}}}else{if(f[23]<9.17014e+07){if(f[15]<1.48787e+06){if(f[28]<1.40061e+11){if(f[42]<0.999266){if(f[21]<1.30439e+07){if(f[75]<9.0428){if(f[79]<1.02011){r+=0.762925;}else{r+=1.02866;}}else{r+=-0.0346121;}}else{r+=-0.514953;}}else{if(f[7]<4.70922e+08){r+=-0.930893;}else{r+=-0.00166953;}}}else{if(f[68]<0.745276){if(f[1]<3.89657e+08){r+=-0.720173;}else{r+=0.346431;}}else{r+=0.315301;}}}else{if(f[0]<1.41443e+11){if(f[39]<6.30794e+07){if(f[27]<7.43276e+07){if(f[50]<4.62818){if(f[34]<6.78662e+06){r+=-0.744802;}else{r+=-0.0853659;}}else{if(f[14]<1.40328e+11){r+=0.731379;}else{r+=-0.606618;}}}else{if(f[14]<1.40286e+11){if(f[38]<6.89254e+07){r+=-0.724362;}else{r+=0.504422;}}else{if(f[28]<1.40386e+11){r+=0.696671;}else{r+=-0.318531;}}}}else{if(f[28]<1.39996e+11){if(f[84]<2.85){if(f[28]<1.39968e+11){r+=-1.03419;}else{r+=0.0770229;}}else{if(f[0]<1.40262e+11){r+=0.782314;}else{r+=-0.490125;}}}else{if(f[42]<0.999483){if(f[0]<1.40092e+11){r+=-0.717031;}else{r+=-0.241791;}}else{if(f[84]<2.15){r+=-0.828404;}else{r+=-1.10704;}}}}}else{if(f[84]<0.5){r+=0.686985;}else{if(f[14]<1.40161e+11){r+=0.491734;}else{r+=-0.682944;}}}}}else{if(f[28]<1.39821e+11){if(f[42]<1.00885){if(f[56]<1.00277){r+=0.647456;}else{if(f[14]<1.40496e+11){if(f[42]<1.00116){if(f[41]<1.84043e+09){r+=-1.20764;}else{r+=-0.863518;}}else{r+=-0.682917;}}else{r+=-0.697231;}}}else{if(f[0]<1.42373e+11){r+=0.727034;}else{r+=-0.474241;}}}else{if(f[84]<0.5){if(f[14]<1.40196e+11){if(f[28]<1.39924e+11){r+=0.838757;}else{if(f[77]<0.347207){r+=-0.00418695;}else{r+=-0.813162;}}}else{if(f[0]<1.3984e+11){if(f[14]<1.40416e+11){r+=-0.725823;}else{if(f[28]<1.40476e+11){r+=-0.526514;}else{r+=0.751475;}}}else{if(f[12]<1.30707e+08){if(f[0]<1.40333e+11){r+=0.51425;}else{r+=-0.660731;}}else{if(f[63]<0.363414){r+=-0.382982;}else{r+=0.690184;}}}}}else{if(f[14]<1.40927e+11){if(f[14]<1.40296e+11){if(f[0]<1.4061e+11){if(f[0]<1.40228e+11){r+=-0.511156;}else{r+=-0.9923;}}else{if(f[84]<1.85){r+=-0.674257;}else{r+=0.811421;}}}else{if(f[9]<3.26417e+08){if(f[7]<7.73035e+07){r+=-0.628371;}else{r+=0.651937;}}else{if(f[51]<3.50797){r+=-0.122256;}else{r+=-0.806115;}}}}else{if(f[42]<1.00963){if(f[42]<1.00542){if(f[84]<4.85){r+=-0.695578;}else{r+=-0.292944;}}else{if(f[0]<1.42049e+11){r+=0.547975;}else{r+=-0.685844;}}}else{if(f[56]<1.00804){if(f[32]<5.24857e+07){r+=0.825991;}else{r+=0.117843;}}else{r+=-1.04069;}}}}}}}}}}else{if(f[67]<6.3993){if(f[14]<1.39862e+11){if(f[14]<1.39561e+11){if(f[28]<1.40022e+11){if(f[37]<7.2902e+07){if(f[79]<1.25341){if(f[0]<1.40122e+11){r+=-1.63129;}else{r+=-1.18873;}}else{r+=-0.665534;}}else{if(f[79]<1.15654){if(f[27]<9.88485e+07){if(f[0]<1.40392e+11){r+=0.622806;}else{r+=-1.16209;}}else{if(f[10]<2.06629e+08){if(f[28]<1.39987e+11){if(f[14]<1.38897e+11){r+=-0.691903;}else{r+=0.710406;}}else{if(f[38]<1.71403e+08){r+=0.772;}else{r+=-1.05774;}}}else{if(f[66]<0.352128){r+=-0.710433;}else{if(f[25]<4.30978e+09){r+=0.811042;}else{r+=-0.366863;}}}}}else{if(f[84]<2.15){if(f[27]<1.10034e+08){if(f[0]<1.40595e+11){r+=0.532447;}else{if(f[79]<1.25306){r+=-1.05912;}else{r+=-0.597752;}}}else{if(f[54]<0.225269){if(f[59]<26.4208){r+=0.386018;}else{r+=-0.506179;}}else{if(f[27]<1.18756e+08){r+=0.102499;}else{r+=0.774947;}}}}else{if(f[84]<3.35){if(f[79]<1.25213){r+=-1.20293;}else{r+=-0.688529;}}else{if(f[19]<3.56209e+07){if(f[8]<8.23291e+07){r+=0.277878;}else{r+=0.956825;}}else{r+=0.229722;}}}}}}else{if(f[84]<12){if(f[0]<1.41701e+11){if(f[28]<1.40096e+11){if(f[41]<2.61709e+08){if(f[55]<0.841501){if(f[41]<2.05369e+08){r+=-0.90258;}else{r+=0.631145;}}else{if(f[79]<1.25263){r+=-1.06825;}else{r+=-0.685487;}}}else{if(f[84]<1.5){if(f[14]<1.38906e+11){r+=-1.15122;}else{r+=-0.690236;}}else{if(f[36]<4.94229e+07){r+=-1.14647;}else{r+=0.704513;}}}}else{if(f[42]<0.989142){if(f[84]<2.65){if(f[0]<4.70834e+09){r+=0.10375;}else{r+=0.689709;}}else{r+=-0.643869;}}else{if(f[28]<1.40207e+11){if(f[42]<1.01211){r+=-1.01176;}else{r+=-0.0069618;}}else{if(f[41]<8.5029e+07){r+=-0.848223;}else{r+=-0.709603;}}}}}else{if(f[84]<1.5){if(f[29]<1.20906e+07){r+=0.333855;}else{if(f[12]<1.30459e+08){r+=-0.690591;}else{if(f[41]<9.31871e+07){r+=-0.494064;}else{r+=-1.17489;}}}}else{if(f[84]<2.5){if(f[41]<4.9459e+09){if(f[14]<1.39117e+11){r+=0.827111;}else{r+=0.711971;}}else{r+=-0.0459827;}}else{if(f[84]<3.5){if(f[28]<1.40239e+11){r+=-1.07679;}else{r+=-0.676832;}}else{r+=0.621323;}}}}}else{if(f[0]<1.40208e+11){if(f[56]<0.99067){r+=-0.690564;}else{if(f[28]<1.40152e+11){if(f[38]<2.35764e+08){if(f[28]<1.40147e+11){r+=-1.09143;}else{r+=-0.479983;}}else{if(f[39]<1.92388e+08){r+=0.761714;}else{r+=-0.918418;}}}else{if(f[28]<1.40207e+11){if(f[68]<1.28943){r+=0.795677;}else{r+=1.20047;}}else{if(f[79]<1.25245){r+=1.25835;}else{r+=0.817992;}}}}}else{if(f[23]<1.36901e+08){if(f[28]<1.40207e+11){if(f[0]<1.40225e+11){if(f[28]<1.40156e+11){r+=-1.17361;}else{r+=0.640799;}}else{if(f[79]<1.25745){r+=-1.22208;}else{r+=-0.564968;}}}else{if(f[41]<9.33075e+07){if(f[68]<1.25889){r+=-1.31793;}else{r+=-0.645309;}}else{if(f[0]<1.40577e+11){r+=0.0955932;}else{r+=-0.716159;}}}}else{if(f[0]<1.40226e+11){if(f[28]<1.40158e+11){if(f[28]<1.40148e+11){r+=-0.9996;}else{r+=-0.0889728;}}else{r+=0.857967;}}else{if(f[41]<9.33286e+07){if(f[68]<1.26279){r+=-1.1637;}else{r+=-0.736643;}}else{if(f[28]<1.40175e+11){r+=-1.13893;}else{r+=-0.721502;}}}}}}}}else{if(f[84]<3.15){if(f[65]<0.69919){if(f[28]<1.39792e+11){if(f[84]<1){if(f[14]<1.39669e+11){r+=0.517931;}else{r+=-0.697284;}}else{r+=0.842608;}}else{if(f[28]<1.40464e+11){if(f[36]<2.54619e+07){if(f[12]<2.31021e+08){r+=-0.647197;}else{if(f[0]<1.39485e+11){r+=0.6596;}else{r+=0.978817;}}}else{if(f[28]<1.40042e+11){if(f[84]<1){r+=-0.74333;}else{r+=-1.06683;}}else{if(f[56]<0.997175){r+=-0.741035;}else{r+=0.383661;}}}}else{if(f[42]<1.00652){if(f[84]<0.8){r+=-0.64914;}else{if(f[0]<1.40251e+11){r+=-0.255496;}else{r+=1.10082;}}}else{if(f[84]<6.00001){if(f[0]<1.42566e+11){r+=-0.738217;}else{r+=-0.0210197;}}else{r+=0.541692;}}}}}else{if(f[0]<1.40827e+11){if(f[25]<6.43716e+07){if(f[25]<6.334e+07){r+=-0.843363;}else{if(f[84]<1){r+=-0.5742;}else{if(f[68]<0.89028){r+=-0.167164;}else{r+=0.47448;}}}}else{if(f[14]<1.39656e+11){if(f[0]<1.40294e+11){if(f[34]<7.62255e+07){r+=0.723664;}else{r+=-0.321625;}}else{if(f[84]<1){r+=-0.704034;}else{r+=0.404139;}}}else{if(f[84]<1.85){if(f[84]<1.15){r+=0.621441;}else{r+=0.848664;}}else{if(f[84]<2.15){r+=-0.679268;}else{r+=0.673873;}}}}}else{if(f[0]<1.43429e+11){if(f[84]<0.5){if(f[14]<1.39603e+11){r+=-0.770563;}else{if(f[68]<0.666041){r+=-0.706514;}else{r+=0.692637;}}}else{if(f[23]<5.63671e+07){r+=0.692624;}else{if(f[26]<1.5205e+10){r+=-0.748309;}else{r+=0.438255;}}}}else{if(f[2]<2.34449e+08){if(f[28]<1.40062e+11){r+=-0.583087;}else{if(f[25]<6.96327e+07){r+=-0.44376;}else{r+=0.715821;}}}else{if(f[84]<1){r+=1.35355;}else{r+=0.828996;}}}}}}else{if(f[1]<5.51567e+08){if(f[37]<5.25084e+07){r+=-1.15456;}else{if(f[0]<1.40676e+11){r+=-0.686618;}else{r+=0.0852935;}}}else{r+=1.17418;}}}}else{if(f[28]<1.39818e+11){if(f[9]<2.15658e+08){if(f[14]<1.39882e+11){if(f[0]<1.4006e+11){r+=-0.513162;}else{r+=0.803362;}}else{if(f[84]<2.5){if(f[36]<1.462e+07){if(f[0]<1.40243e+11){if(f[37]<3.56437e+07){if(f[41]<2.29038e+08){r+=-0.806783;}else{r+=0.00185565;}}else{r+=0.533005;}}else{if(f[84]<0.8){r+=-0.741461;}else{if(f[28]<1.39802e+11){r+=-1.21346;}else{r+=-0.956155;}}}}else{if(f[41]<9.32944e+07){if(f[69]<4.41949){if(f[39]<1.85809e+08){r+=-0.713105;}else{r+=-1.2232;}}else{if(f[14]<1.40155e+11){r+=-0.487104;}else{r+=0.523268;}}}else{if(f[25]<8.71571e+07){if(f[0]<1.40191e+11){r+=-0.707996;}else{r+=-1.10392;}}else{if(f[56]<1.00364){r+=-0.0693617;}else{r+=-0.83866;}}}}}else{if(f[56]<1.00367){if(f[0]<1.40145e+11){r+=-0.588534;}else{r+=0.83375;}}else{if(f[36]<4.87844e+07){r+=-1.13156;}else{r+=-0.70167;}}}}}else{if(f[42]<1.00856){if(f[14]<1.39914e+11){if(f[84]<1){r+=-0.427546;}else{r+=0.803309;}}else{if(f[67]<0.21966){if(f[0]<1.40294e+11){r+=0.843567;}else{r+=-0.206736;}}else{if(f[9]<1.6461e+10){if(f[28]<1.39812e+11){if(f[14]<1.40347e+11){r+=-1.01534;}else{r+=-0.713573;}}else{if(f[6]<4.98459e+07){r+=-0.508183;}else{r+=0.406961;}}}else{if(f[51]<1.15065){if(f[75]<0.553941){r+=-0.015006;}else{r+=-0.633243;}}else{if(f[28]<1.39787e+11){r+=0.778063;}else{r+=0.0104711;}}}}}}else{if(f[0]<1.41759e+11){r+=0.825551;}else{if(f[7]<2.37213e+08){r+=0.139554;}else{r+=-0.846375;}}}}}else{if(f[28]<1.40121e+11){if(f[14]<1.39991e+11){if(f[28]<1.40069e+11){if(f[28]<1.40001e+11){if(f[67]<0.425013){if(f[79]<1.23295){if(f[84]<1.2){r+=-0.646093;}else{r+=-1.15959;}}else{r+=-0.676961;}}else{if(f[23]<5.94259e+09){if(f[1]<3.99565e+08){r+=-0.705114;}else{r+=0.0826242;}}else{r+=0.317243;}}}else{if(f[56]<0.999797){if(f[8]<1.52941e+07){r+=0.758772;}else{if(f[27]<5.73738e+09){r+=-0.797882;}else{r+=0.52908;}}}else{if(f[8]<1.20844e+08){r+=0.714709;}else{r+=-0.383099;}}}}else{if(f[41]<9.32439e+07){if(f[10]<7.29181e+07){r+=-0.69317;}else{r+=-1.00017;}}else{if(f[79]<1.25235){if(f[28]<1.40084e+11){if(f[1]<4.32635e+07){r+=-1.59436;}else{r+=-0.558786;}}else{if(f[0]<1.40181e+11){r+=-0.494922;}else{r+=-1.20178;}}}else{if(f[12]<1.06767e+09){r+=-0.721875;}else{r+=0.0787151;}}}}}else{if(f[84]<0.5){if(f[42]<0.997341){if(f[41]<9.14003e+07){if(f[36]<5.44892e+07){r+=-1.1931;}else{r+=-0.391209;}}else{r+=-0.839686;}}else{if(f[79]<1.25266){if(f[41]<2.61677e+07){if(f[28]<1.40026e+11){r+=-0.68301;}else{r+=-1.15085;}}else{if(f[69]<1.8277){r+=-0.757499;}else{r+=-0.69678;}}}else{if(f[38]<7.64808e+07){if(f[0]<1.40281e+11){r+=-1.35105;}else{r+=-0.385154;}}else{r+=-0.773834;}}}}else{if(f[55]<0.799591){if(f[0]<1.40087e+11){if(f[28]<1.40001e+11){if(f[56]<1.00359){r+=-0.68068;}else{r+=0.154146;}}else{if(f[14]<1.40157e+11){r+=0.566844;}else{r+=-0.244998;}}}else{if(f[14]<1.4018e+11){if(f[69]<0.765432){r+=0.184007;}else{r+=-0.668998;}}else{if(f[0]<1.41351e+11){r+=0.345241;}else{r+=-0.677173;}}}}else{if(f[84]<3.5){if(f[0]<1.42887e+11){if(f[52]<1.38238){r+=0.365086;}else{r+=0.151287;}}else{if(f[84]<3.15){r+=-0.733391;}else{r+=0.732232;}}}else{if(f[84]<4.85){r+=-0.689126;}else{if(f[79]<1.25208){r+=-1.57317;}else{r+=-0.616403;}}}}}}}else{if(f[0]<1.40162e+11){if(f[28]<1.40346e+11){if(f[84]<2.85){if(f[28]<1.40124e+11){if(f[0]<1.40095e+11){if(f[84]<1.15){r+=-0.658126;}else{r+=0.708334;}}else{if(f[83]<1.14233){r+=-0.693764;}else{r+=0.598994;}}}else{if(f[31]<3.15508e+08){if(f[66]<5.4287){r+=-0.744463;}else{r+=0.403524;}}else{if(f[84]<1.15){r+=0.90204;}else{r+=-0.683788;}}}}else{if(f[26]<3.6631e+08){if(f[27]<1.15968e+09){if(f[0]<1.40041e+11){r+=-0.74247;}else{r+=-0.583574;}}else{r+=0.613359;}}else{if(f[14]<1.40048e+11){if(f[13]<1.74735e+09){r+=-0.680819;}else{r+=0.507619;}}else{if(f[23]<9.60712e+07){r+=1.01506;}else{r+=0.695845;}}}}}else{if(f[84]<0.5){if(f[28]<1.4069e+11){if(f[28]<1.4052e+11){if(f[37]<7.69921e+08){r+=-0.758606;}else{r+=0.221395;}}else{if(f[14]<1.40566e+11){r+=0.895718;}else{r+=0.689455;}}}else{if(f[14]<1.44666e+11){if(f[78]<3.05861){r+=-0.688329;}else{r+=0.617674;}}else{if(f[14]<1.45808e+11){r+=1.70394;}else{r+=0.683606;}}}}else{if(f[28]<1.40399e+11){if(f[0]<1.40043e+11){if(f[28]<1.40384e+11){r+=-0.762658;}else{r+=0.232187;}}else{if(f[84]<2.5){r+=0.843097;}else{r+=-1.06323;}}}else{if(f[0]<1.39201e+11){if(f[56]<0.996508){r+=0.677967;}else{r+=-0.688162;}}else{if(f[56]<0.989986){r+=-0.325089;}else{r+=-0.725709;}}}}}}else{if(f[9]<1.83243e+08){if(f[56]<1.0062){if(f[0]<1.40278e+11){if(f[84]<1.85){if(f[84]<1.5){r+=0.107136;}else{r+=0.64182;}}else{if(f[28]<1.40136e+11){r+=0.725382;}else{r+=-0.777864;}}}else{if(f[28]<1.4059e+11){if(f[84]<1.85){r+=-1.04089;}else{r+=-0.766016;}}else{if(f[84]<1.15){r+=-0.635067;}else{r+=0.308167;}}}}else{if(f[84]<1){if(f[0]<1.42928e+11){if(f[41]<9.32733e+07){r+=0.777627;}else{r+=0.917724;}}else{if(f[56]<1.00789){r+=0.756881;}else{r+=-1.09421;}}}else{if(f[79]<1.25679){if(f[41]<9.36341e+07){r+=-1.47445;}else{r+=-0.708687;}}else{r+=-0.733884;}}}}else{if(f[28]<1.43136e+11){if(f[84]<4.85){if(f[0]<1.4162e+11){if(f[0]<1.40715e+11){r+=-0.746529;}else{r+=-0.451684;}}else{if(f[84]<1.85){r+=-0.129194;}else{r+=0.69251;}}}else{if(f[14]<1.40556e+11){if(f[38]<4.8118e+08){r+=-1.10111;}else{r+=0.181506;}}else{if(f[28]<1.40724e+11){r+=0.739054;}else{r+=-0.507474;}}}}else{if(f[84]<0.85){if(f[27]<2.03481e+08){if(f[1]<1.29592e+07){r+=0.458168;}else{r+=-0.635878;}}else{if(f[25]<1.81005e+10){r+=0.683189;}else{r+=0.101066;}}}else{r+=-0.702629;}}}}}}}}else{if(f[28]<1.79213e+11){if(f[65]<2.60345){if(f[8]<5.34166e+07){if(f[27]<1.07043e+08){if(f[84]<0.85){r+=0.246154;}else{if(f[67]<9.91702){if(f[37]<2.94483e+06){r+=-0.471086;}else{if(f[84]<6.00001){r+=-1.19032;}else{r+=-0.513702;}}}else{r+=-0.0460243;}}}else{if(f[84]<2.15){if(f[28]<1.40116e+11){if(f[26]<1.13367e+09){if(f[53]<0.37901){r+=0.181118;}else{r+=0.771904;}}else{r+=-0.127918;}}else{r+=-0.574068;}}else{if(f[64]<1.41436){if(f[38]<5.94854e+07){r+=-0.275335;}else{if(f[37]<1.8363e+08){r+=0.60627;}else{r+=0.0917713;}}}else{if(f[20]<3.50836e+06){r+=-0.122995;}else{r+=-0.925228;}}}}}else{if(f[68]<6.48083){if(f[0]<1.40078e+11){if(f[55]<0.528071){if(f[43]<0.416792){r+=-0.0548965;}else{r+=-0.631468;}}else{if(f[44]<4.8532){if(f[63]<2.89982){if(f[53]<0.453107){r+=0.219741;}else{r+=0.7964;}}else{r+=-0.137462;}}else{r+=-0.625324;}}}else{if(f[55]<1.41035){if(f[0]<1.43177e+11){r+=-0.71752;}else{r+=-0.033884;}}else{r+=0.140688;}}}else{if(f[14]<1.39499e+11){r+=0.30784;}else{if(f[12]<3.68646e+09){if(f[59]<0.110796){r+=-0.143608;}else{if(f[13]<3.34908e+09){if(f[9]<7.17481e+07){r+=-0.176081;}else{r+=-0.685584;}}else{r+=-0.163675;}}}else{if(f[12]<4.55094e+09){r+=0.776332;}else{r+=-0.660816;}}}}}}else{if(f[69]<1.55053){if(f[13]<4.12761e+08){if(f[83]<0.729138){r+=0.0284741;}else{r+=-0.727548;}}else{if(f[66]<10.8778){if(f[18]<3.71649e+07){if(f[53]<0.247843){if(f[41]<2.88565e+09){r+=0.433635;}else{r+=-0.481881;}}else{if(f[71]<1.40474){r+=0.713745;}else{r+=0.137242;}}}else{if(f[14]<1.40542e+11){r+=-0.3752;}else{r+=0.0242584;}}}else{r+=-0.53767;}}}else{if(f[10]<5.16249e+09){if(f[37]<6.55475e+07){if(f[0]<1.40005e+11){if(f[28]<1.40118e+11){if(f[84]<0.85){if(f[28]<1.39906e+11){r+=0.785633;}else{r+=0.220745;}}else{if(f[28]<1.40039e+11){r+=-1.14567;}else{r+=0.423755;}}}else{if(f[84]<7.40001){if(f[56]<1.02958){r+=-0.687624;}else{r+=-0.309434;}}else{if(f[0]<1.39939e+11){r+=0.749578;}else{r+=-0.483415;}}}}else{if(f[68]<7.05857){if(f[14]<1.40403e+11){if(f[84]<2.15){r+=-0.627546;}else{r+=-1.16662;}}else{if(f[9]<1.19015e+07){r+=0.386527;}else{r+=-0.72588;}}}else{if(f[72]<27.2089){if(f[41]<5.21115e+08){r+=-0.680895;}else{r+=-0.253583;}}else{r+=0.377271;}}}}else{if(f[28]<1.39844e+11){if(f[56]<1.00217){r+=0.873128;}else{r+=-0.0827501;}}else{if(f[68]<5.10235){if(f[28]<1.40159e+11){if(f[30]<1.1663e+07){r+=-0.996032;}else{r+=-0.239968;}}else{if(f[84]<3.5){r+=0.0558397;}else{r+=0.791034;}}}else{if(f[28]<1.41681e+11){if(f[19]<1.12768e+07){r+=-0.354413;}else{r+=-0.684934;}}else{r+=0.429099;}}}}}else{if(f[39]<4.40472e+08){if(f[1]<3.59157e+07){r+=0.145524;}else{r+=-0.723198;}}else{if(f[12]<3.93292e+09){if(f[52]<0.641821){r+=0.174217;}else{r+=-0.572144;}}else{if(f[69]<5.63702){r+=1.15324;}else{if(f[64]<3.53118){r+=-0.470978;}else{if(f[12]<1.88464e+10){r+=0.70253;}else{r+=-0.0294609;}}}}}}}}}else{if(f[14]<1.40084e+11){r+=0.680307;}else{r+=-0.6719;}}}}} return r;}
    static double p20(double[] f){double r=0;if(f[28]<1.39447e+11){if(f[28]<1.3942e+11){if(f[10]<2.98315e+10){if(f[56]<1.00196){if(f[56]<1.00166){if(f[28]<1.392e+11){r+=-0.570614;}else{if(f[30]<1.68951e+07){r+=0.429321;}else{r+=-0.351519;}}}else{r+=0.410818;}}else{if(f[42]<1.00952){if(f[44]<0.0153638){r+=-0.0574898;}else{r+=-0.59275;}}else{if(f[42]<1.01066){if(f[27]<2.35403e+08){if(f[56]<1.0103){r+=0.761952;}else{r+=0.186957;}}else{r+=-0.49491;}}else{if(f[42]<1.0107){r+=-0.114789;}else{r+=-0.592248;}}}}}else{if(f[14]<1.39625e+11){if(f[34]<2.20858e+09){r+=-0.0729461;}else{r+=-0.578318;}}else{if(f[8]<8.24041e+09){r+=-0.458285;}else{if(f[46]<0.840357){r+=0.862058;}else{if(f[32]<1.10933e+09){r+=0.498752;}else{if(f[34]<5.56765e+09){r+=-0.538287;}else{r+=0.156609;}}}}}}}else{if(f[56]<1.00325){if(f[14]<1.39823e+11){if(f[28]<1.39442e+11){if(f[19]<5.70477e+06){r+=-0.155211;}else{if(f[14]<1.39821e+11){r+=-0.575754;}else{r+=-0.156943;}}}else{if(f[73]<2.09965){r+=-0.89627;}else{r+=-0.185676;}}}else{if(f[14]<1.39837e+11){if(f[54]<0.982508){if(f[43]<0.702044){r+=-0.128459;}else{r+=0.46867;}}else{if(f[81]<2.30414){if(f[50]<1.22816){r+=-0.504793;}else{if(f[50]<1.52512){r+=0.1991;}else{r+=-0.264059;}}}else{r+=0.432141;}}}else{if(f[70]<0.993382){r+=-0.181978;}else{if(f[41]<1.02673e+08){if(f[50]<1.20593){r+=-0.262789;}else{r+=0.498161;}}else{if(f[28]<1.39426e+11){r+=1.37923;}else{if(f[33]<5.70462e+06){r+=0.156744;}else{r+=0.601946;}}}}}}}else{if(f[31]<9.67995e+07){if(f[42]<1.0072){r+=-0.602131;}else{r+=-0.144647;}}else{r+=0.0264686;}}}}else{if(f[68]<7.01228){if(f[81]<1.52361){if(f[0]<1.41609e+11){if(f[35]<7.12006e+07){if(f[38]<5.58246e+07){if(f[14]<1.40128e+11){if(f[14]<1.40107e+11){if(f[0]<1.40093e+11){if(f[38]<4.91606e+07){if(f[38]<4.71513e+07){if(f[14]<1.39991e+11){r+=-0.6002;}else{r+=-0.771655;}}else{if(f[42]<0.999384){r+=0.404091;}else{r+=-0.528697;}}}else{if(f[56]<1.00046){if(f[56]<0.999267){r+=-0.630886;}else{r+=0.52746;}}else{if(f[28]<1.40005e+11){r+=-0.642456;}else{r+=-1.77913;}}}}else{if(f[0]<1.40163e+11){if(f[68]<0.870643){if(f[28]<1.40135e+11){r+=0.527374;}else{r+=-0.69521;}}else{if(f[28]<1.40116e+11){r+=-0.771402;}else{r+=-0.635399;}}}else{if(f[70]<0.998668){if(f[36]<4.11536e+06){r+=0.322258;}else{r+=-0.720467;}}else{if(f[14]<1.40097e+11){r+=0.463326;}else{r+=-1.15096;}}}}}else{if(f[41]<9.40112e+07){if(f[28]<1.40112e+11){if(f[28]<1.4004e+11){if(f[36]<1.0119e+07){r+=0.627877;}else{r+=-0.740548;}}else{r+=0.680368;}}else{if(f[28]<1.40925e+11){if(f[0]<1.40162e+11){r+=-0.61979;}else{r+=-0.972353;}}else{if(f[11]<7.39646e+07){r+=-0.421437;}else{r+=0.682596;}}}}else{if(f[70]<1.00012){if(f[70]<0.999653){if(f[64]<1.76733){r+=-0.604212;}else{r+=0.51359;}}else{if(f[40]<1.18542e+08){r+=-1.87317;}else{r+=-0.388898;}}}else{if(f[28]<1.40982e+11){if(f[28]<1.40081e+11){r+=0.0565129;}else{r+=-0.699674;}}else{r+=0.960208;}}}}}else{if(f[28]<1.39808e+11){if(f[41]<2.25067e+07){if(f[68]<4.32657){r+=-0.062389;}else{r+=0.593446;}}else{if(f[56]<1.00297){if(f[5]<1.04696e+07){r+=-0.265814;}else{r+=0.555936;}}else{if(f[40]<1.11939e+08){if(f[31]<1.21033e+07){r+=-0.710589;}else{r+=-0.281732;}}else{if(f[56]<1.00348){r+=0.223625;}else{r+=-0.651001;}}}}}else{if(f[0]<1.39806e+11){if(f[56]<1.03714){if(f[0]<1.39751e+11){if(f[28]<1.40085e+11){r+=-0.726315;}else{r+=-0.567071;}}else{if(f[28]<1.40074e+11){r+=-0.753034;}else{r+=0.233294;}}}else{if(f[28]<1.41233e+11){r+=-0.222627;}else{r+=0.591178;}}}else{if(f[0]<1.40337e+11){if(f[28]<1.40411e+11){if(f[80]<0.185125){r+=-0.230612;}else{r+=0.269431;}}else{if(f[70]<1.00464){r+=-0.778098;}else{r+=0.541046;}}}else{if(f[53]<0.712189){if(f[28]<1.40594e+11){r+=-0.428494;}else{r+=0.860536;}}else{if(f[56]<1.00097){r+=0.137216;}else{r+=0.649918;}}}}}}}else{if(f[0]<1.40083e+11){if(f[28]<1.39999e+11){if(f[56]<1.00343){if(f[14]<1.39991e+11){r+=-0.60752;}else{if(f[55]<0.799768){if(f[64]<3.09669){r+=-0.587414;}else{r+=-0.0740042;}}else{if(f[28]<1.39998e+11){r+=-0.757505;}else{r+=-0.269459;}}}}else{if(f[53]<1.44361){if(f[81]<1.0126){if(f[28]<1.39865e+11){r+=-0.47272;}else{r+=0.306145;}}else{if(f[51]<1.04577){r+=-1.4259;}else{r+=-0.105469;}}}else{if(f[32]<8.05954e+06){if(f[33]<5.58907e+06){r+=0.657861;}else{r+=0.102097;}}else{r+=-0.287803;}}}}else{if(f[14]<1.40175e+11){if(f[34]<7.94186e+06){if(f[28]<1.40219e+11){if(f[36]<3.00953e+07){r+=0.59568;}else{r+=0.219804;}}else{if(f[27]<2.31574e+09){r+=-0.607061;}else{r+=0.867403;}}}else{if(f[14]<1.40045e+11){if(f[28]<1.40116e+11){r+=-1.55072;}else{r+=-0.46173;}}else{if(f[78]<1.50691){r+=0.591676;}else{r+=-0.440975;}}}}else{if(f[56]<1.03249){if(f[36]<7.82025e+07){if(f[22]<5.2884e+06){r+=0.507429;}else{r+=-0.583962;}}else{if(f[0]<1.39797e+11){r+=-0.568653;}else{r+=0.239568;}}}else{if(f[14]<1.71829e+11){if(f[14]<1.46495e+11){r+=0.133229;}else{r+=0.62585;}}else{r+=-0.590091;}}}}}else{if(f[0]<1.40155e+11){if(f[14]<1.40166e+11){if(f[26]<4.01308e+08){if(f[13]<1.09064e+08){if(f[14]<1.40032e+11){r+=0.639866;}else{r+=-0.571371;}}else{if(f[0]<1.40096e+11){r+=-0.0851711;}else{r+=-0.707397;}}}else{if(f[28]<1.40139e+11){if(f[28]<1.40062e+11){r+=-0.73066;}else{r+=0.625207;}}else{if(f[73]<8.75797){r+=-0.752189;}else{r+=0.087735;}}}}else{if(f[28]<1.40381e+11){if(f[56]<1.00121){if(f[0]<1.40142e+11){r+=-0.633792;}else{r+=0.420361;}}else{if(f[56]<1.00229){r+=0.884294;}else{r+=-0.244914;}}}else{if(f[28]<1.404e+11){if(f[27]<4.23902e+08){r+=-0.402261;}else{r+=0.441557;}}else{if(f[13]<1.81022e+08){r+=-0.827166;}else{r+=-0.618707;}}}}}else{if(f[0]<1.4024e+11){if(f[28]<1.40163e+11){if(f[56]<0.999341){if(f[78]<0.179827){r+=0.218343;}else{r+=-0.712147;}}else{if(f[68]<0.609139){r+=0.031769;}else{r+=0.61758;}}}else{if(f[56]<1.00026){if(f[14]<1.3987e+11){r+=0.641324;}else{r+=-0.769646;}}else{if(f[14]<1.40618e+11){r+=0.315694;}else{r+=-0.565812;}}}}else{if(f[0]<1.40416e+11){if(f[28]<1.39984e+11){if(f[66]<0.884139){r+=-0.691914;}else{r+=0.424595;}}else{if(f[28]<1.41815e+11){r+=-0.855365;}else{r+=0.713959;}}}else{if(f[42]<1.00159){if(f[23]<2.06446e+07){r+=0.797311;}else{r+=-0.593368;}}else{if(f[14]<1.4016e+11){r+=0.0969556;}else{r+=0.672191;}}}}}}}}else{if(f[28]<1.41479e+11){if(f[28]<1.40719e+11){if(f[56]<1.00838){if(f[14]<1.4113e+11){if(f[14]<1.39022e+11){if(f[14]<1.38835e+11){if(f[27]<2.43927e+08){r+=-0.792679;}else{r+=-0.391397;}}else{if(f[70]<0.99891){r+=0.540378;}else{r+=-0.60891;}}}else{if(f[0]<1.40246e+11){if(f[14]<1.40475e+11){r+=-0.0834701;}else{r+=0.403781;}}else{if(f[28]<1.3956e+11){r+=0.471554;}else{r+=-0.364238;}}}}else{if(f[13]<1.19763e+09){if(f[0]<1.41012e+11){r+=-0.544841;}else{if(f[9]<1.83255e+08){r+=0.651117;}else{r+=0.592167;}}}else{if(f[23]<3.06802e+10){if(f[29]<2.77679e+07){r+=0.215266;}else{r+=-0.733913;}}else{r+=0.545064;}}}}else{if(f[42]<0.953964){if(f[10]<1.19751e+10){r+=0.712987;}else{r+=0.108179;}}else{if(f[17]<6.47139e+08){if(f[28]<1.39818e+11){r+=-0.603347;}else{if(f[28]<1.40124e+11){r+=-0.757475;}else{r+=-0.614298;}}}else{if(f[55]<1.25398){r+=-0.55766;}else{r+=0.806815;}}}}}else{if(f[0]<1.40698e+11){if(f[23]<7.96023e+07){if(f[0]<1.40387e+11){if(f[0]<1.40279e+11){if(f[73]<7.89739){r+=-0.58483;}else{r+=-0.0666585;}}else{r+=-0.736303;}}else{if(f[9]<1.39487e+08){if(f[68]<0.454862){r+=-0.611706;}else{r+=-1.03319;}}else{r+=-0.791113;}}}else{if(f[14]<1.45846e+11){if(f[0]<1.4012e+11){if(f[28]<1.41462e+11){r+=-0.59454;}else{r+=0.451054;}}else{if(f[23]<2.7775e+10){r+=-0.675941;}else{r+=0.323162;}}}else{r+=0.604391;}}}else{if(f[14]<1.405e+11){if(f[42]<1.00406){if(f[55]<2.43778){if(f[9]<1.8623e+08){r+=0.832818;}else{r+=1.4203;}}else{r+=0.630682;}}else{if(f[64]<4.23514){r+=-0.642929;}else{r+=-0.0419666;}}}else{if(f[0]<1.4151e+11){if(f[56]<1.00308){if(f[13]<1.86049e+08){r+=-0.786527;}else{r+=-0.646083;}}else{if(f[13]<1.23342e+09){r+=0.615002;}else{r+=-0.58489;}}}else{if(f[56]<0.999208){if(f[4]<1.18806e+08){r+=0.77142;}else{r+=0.0277766;}}else{r+=-0.583933;}}}}}}else{if(f[0]<1.41299e+11){if(f[14]<1.40989e+11){if(f[23]<1.38593e+08){if(f[56]<0.982196){if(f[28]<1.72809e+11){if(f[14]<1.39726e+11){r+=-0.0676226;}else{r+=-0.602103;}}else{r+=0.819893;}}else{if(f[28]<1.42024e+11){if(f[0]<1.39346e+11){r+=0.592635;}else{r+=-0.600501;}}else{if(f[51]<1.67507){r+=0.625544;}else{r+=0.195955;}}}}else{if(f[14]<1.4084e+11){if(f[0]<1.39328e+11){if(f[13]<1.03319e+09){r+=0.590021;}else{r+=-0.45097;}}else{if(f[28]<1.42053e+11){r+=-0.586509;}else{r+=0.482197;}}}else{if(f[41]<2.16567e+08){if(f[0]<1.40226e+11){r+=0.208541;}else{r+=-0.6255;}}else{if(f[0]<1.4037e+11){r+=0.627863;}else{r+=-0.344007;}}}}}else{if(f[56]<0.992847){if(f[70]<1.03488){if(f[23]<1.84419e+08){r+=0.586828;}else{r+=-0.15722;}}else{r+=-0.595928;}}else{if(f[41]<6.13324e+09){if(f[14]<1.41263e+11){r+=-0.756494;}else{r+=-0.601539;}}else{if(f[75]<0.633664){r+=0.232957;}else{r+=-0.281995;}}}}}else{if(f[56]<0.98055){if(f[28]<1.54439e+11){if(f[81]<0.500406){r+=-0.0454257;}else{r+=-0.588668;}}else{if(f[41]<1.74218e+08){r+=-0.733953;}else{r+=-0.597552;}}}else{if(f[28]<1.42108e+11){r+=-0.12411;}else{if(f[43]<0.460214){r+=0.134246;}else{r+=0.533617;}}}}}}}else{if(f[56]<1.00908){if(f[10]<1.70794e+08){if(f[56]<0.990458){if(f[14]<1.40472e+11){r+=-0.648356;}else{if(f[9]<1.42101e+08){r+=-0.996188;}else{r+=-0.763902;}}}else{if(f[28]<1.40134e+11){if(f[14]<1.40156e+11){if(f[28]<1.39987e+11){if(f[68]<0.569755){if(f[28]<1.39974e+11){r+=-0.550407;}else{r+=0.117585;}}else{if(f[63]<2.57381){r+=0.731734;}else{r+=-0.235236;}}}else{if(f[33]<6.18159e+06){if(f[9]<3.47699e+07){r+=-0.272206;}else{r+=-1.15967;}}else{if(f[68]<0.732053){r+=-0.166657;}else{r+=0.451737;}}}}else{if(f[70]<0.988147){if(f[28]<1.40118e+11){if(f[0]<1.42861e+11){r+=-0.805012;}else{r+=-0.610868;}}else{if(f[14]<1.41233e+11){r+=0.797767;}else{r+=-0.857496;}}}else{if(f[14]<1.40259e+11){if(f[8]<1.24037e+08){r+=0.117728;}else{r+=0.896061;}}else{r+=-0.650042;}}}}else{if(f[0]<1.41685e+11){if(f[14]<1.40836e+11){r+=-0.597875;}else{r+=0.363858;}}else{if(f[66]<0.713824){if(f[14]<1.4032e+11){r+=-0.601142;}else{r+=0.64009;}}else{if(f[14]<1.39503e+11){r+=-0.562655;}else{if(f[9]<2.39248e+08){r+=0.613912;}else{r+=-0.409673;}}}}}}}else{if(f[40]<6.25571e+07){if(f[66]<0.949093){if(f[0]<1.42872e+11){r+=0.536799;}else{r+=0.102668;}}else{if(f[14]<1.4019e+11){r+=0.50316;}else{if(f[66]<0.992877){r+=-0.105527;}else{r+=-0.664907;}}}}else{if(f[51]<1.08641){if(f[28]<1.40139e+11){if(f[14]<1.4029e+11){if(f[0]<1.53682e+11){if(f[41]<9.31761e+07){r+=1.27339;}else{r+=0.676443;}}else{r+=-0.590746;}}else{if(f[28]<1.40115e+11){if(f[14]<1.40341e+11){r+=-0.0275431;}else{r+=-0.670639;}}else{if(f[14]<1.41242e+11){r+=0.688396;}else{r+=-0.905768;}}}}else{if(f[14]<1.39524e+11){if(f[1]<2.46097e+08){if(f[14]<1.39513e+11){r+=-0.628848;}else{r+=-0.1591;}}else{if(f[46]<1.24927){r+=0.117526;}else{r+=0.510058;}}}else{if(f[56]<0.988959){if(f[28]<1.43344e+11){r+=-0.81374;}else{r+=0.50078;}}else{if(f[14]<1.4011e+11){r+=0.274608;}else{r+=0.619446;}}}}}else{if(f[27]<1.1288e+08){if(f[56]<0.999953){if(f[67]<1.0213){if(f[33]<3.08525e+06){r+=-0.169032;}else{r+=-0.704603;}}else{r+=-0.0593989;}}else{if(f[70]<0.980567){if(f[30]<3.49746e+06){r+=0.305494;}else{r+=-0.673297;}}else{if(f[56]<1.00804){r+=0.702225;}else{r+=-0.671498;}}}}else{if(f[56]<1.00796){if(f[22]<3.72025e+08){if(f[0]<1.41726e+11){r+=0.723108;}else{r+=0.565875;}}else{if(f[42]<1.01143){r+=0.41357;}else{r+=-0.427234;}}}else{if(f[10]<1.7615e+08){r+=-0.188653;}else{r+=-0.79785;}}}}}}}else{if(f[28]<1.40217e+11){if(f[15]<1.88889e+09){r+=-0.602139;}else{r+=0.717457;}}else{if(f[70]<0.982408){if(f[8]<1.6455e+08){r+=-0.54194;}else{if(f[68]<0.607447){r+=-0.581787;}else{r+=-1.57572;}}}else{if(f[44]<0.296617){r+=-0.103095;}else{r+=-0.831751;}}}}}}else{if(f[42]<1.01046){if(f[36]<8.50888e+08){if(f[53]<1.06268){if(f[41]<1.71099e+09){if(f[29]<6.81175e+07){if(f[0]<1.4074e+11){if(f[0]<1.40262e+11){if(f[39]<3.19522e+08){if(f[26]<6.23706e+07){r+=-0.47726;}else{r+=0.136759;}}else{if(f[55]<1.07035){r+=-0.385502;}else{r+=0.280991;}}}else{if(f[42]<1.00286){if(f[28]<1.39997e+11){r+=0.00658917;}else{r+=-0.685235;}}else{if(f[40]<5.70114e+08){r+=0.566332;}else{r+=-0.340551;}}}}else{if(f[66]<0.219089){if(f[14]<1.4044e+11){if(f[28]<1.40021e+11){r+=-1.12041;}else{r+=0.101981;}}else{if(f[28]<1.39984e+11){r+=-0.296376;}else{r+=0.35898;}}}else{if(f[28]<1.41318e+11){if(f[28]<1.39885e+11){r+=-0.710164;}else{r+=0.436978;}}else{if(f[14]<1.39985e+11){r+=-0.143073;}else{r+=-0.632688;}}}}}else{if(f[0]<1.40067e+11){if(f[42]<0.992726){if(f[56]<1.00661){if(f[36]<5.62802e+08){r+=0.59097;}else{r+=-0.363681;}}else{r+=-0.599027;}}else{if(f[14]<1.39609e+11){if(f[56]<0.993339){r+=-0.466613;}else{r+=0.54612;}}else{if(f[51]<0.178916){r+=0.392585;}else{r+=-0.595532;}}}}else{if(f[14]<1.40414e+11){if(f[38]<1.32059e+09){if(f[11]<5.31012e+07){r+=-0.608343;}else{r+=0.664145;}}else{if(f[14]<1.40377e+11){r+=-0.648031;}else{r+=0.262955;}}}else{if(f[80]<3.45576){if(f[0]<1.41187e+11){r+=-0.566578;}else{r+=-0.136578;}}else{if(f[66]<0.217092){r+=-0.281032;}else{r+=0.494272;}}}}}}else{if(f[14]<1.40082e+11){if(f[31]<5.76498e+07){if(f[28]<1.40147e+11){if(f[70]<0.997913){r+=0.148796;}else{if(f[13]<1.75899e+08){r+=-0.0943034;}else{r+=-0.689063;}}}else{if(f[70]<0.999443){if(f[14]<1.4001e+11){r+=0.668246;}else{r+=-0.593149;}}else{if(f[38]<2.36879e+08){r+=0.0472235;}else{r+=-0.616653;}}}}else{if(f[14]<1.394e+11){r+=-0.448972;}else{r+=0.685366;}}}else{if(f[28]<1.40112e+11){if(f[0]<1.39979e+11){if(f[14]<1.40115e+11){r+=0.803222;}else{if(f[63]<7.37836){r+=-0.822289;}else{r+=-1.78061;}}}else{if(f[14]<1.40187e+11){if(f[52]<1.09299){r+=0.461862;}else{r+=-0.525772;}}else{if(f[28]<1.40022e+11){r+=-0.535459;}else{r+=0.582822;}}}}else{if(f[28]<1.40704e+11){if(f[55]<0.850539){if(f[41]<3.53847e+09){r+=0.562256;}else{r+=0.286985;}}else{if(f[14]<1.40109e+11){r+=0.12421;}else{r+=-0.516963;}}}else{if(f[14]<1.42096e+11){if(f[53]<0.331964){r+=0.220836;}else{r+=-0.588803;}}else{if(f[34]<1.68417e+08){r+=0.513326;}else{r+=-0.403008;}}}}}}}else{if(f[35]<1.2691e+08){if(f[28]<1.39882e+11){if(f[41]<5.80735e+08){if(f[56]<1.00312){if(f[66]<0.859022){if(f[41]<4.26644e+08){r+=0.779948;}else{r+=0.545878;}}else{if(f[76]<0.494203){r+=-0.566386;}else{r+=0.309717;}}}else{r+=-0.680791;}}else{if(f[55]<1.46967){if(f[70]<0.997651){r+=-0.694027;}else{r+=-0.0636478;}}else{if(f[28]<1.39815e+11){r+=-0.590034;}else{if(f[67]<0.122323){r+=-0.413177;}else{r+=0.638065;}}}}}else{if(f[81]<1.38592e+08){if(f[0]<1.41152e+11){if(f[23]<3.24381e+08){if(f[49]<4.75447){r+=-0.556301;}else{r+=0.440407;}}else{if(f[0]<1.40261e+11){r+=0.305947;}else{r+=-0.596137;}}}else{if(f[56]<0.986417){if(f[13]<1.63989e+09){r+=-0.00590027;}else{r+=-0.493264;}}else{if(f[39]<1.65166e+09){r+=0.548125;}else{r+=-0.194641;}}}}else{if(f[14]<1.39831e+11){if(f[14]<1.39656e+11){if(f[17]<5.7064e+06){r+=-0.636709;}else{r+=0.67772;}}else{if(f[28]<1.40181e+11){r+=0.698081;}else{r+=-0.413395;}}}else{if(f[28]<1.4018e+11){r+=-0.868393;}else{r+=-0.630997;}}}}}else{if(f[41]<7.27649e+08){if(f[0]<1.40228e+11){if(f[28]<1.40303e+11){if(f[27]<7.74512e+07){r+=-0.567165;}else{if(f[28]<1.39885e+11){r+=0.671654;}else{r+=0.493064;}}}else{if(f[0]<1.40067e+11){if(f[64]<0.628477){r+=-0.601806;}else{r+=-0.109442;}}else{if(f[52]<1.41894){r+=0.504436;}else{r+=-0.564973;}}}}else{if(f[70]<0.995191){if(f[56]<1.00442){if(f[42]<1.00685){r+=0.634471;}else{r+=-0.635593;}}else{if(f[13]<3.55604e+08){r+=-0.735976;}else{r+=0.359588;}}}else{if(f[28]<1.4051e+11){if(f[28]<1.39882e+11){r+=0.181342;}else{r+=-0.84955;}}else{if(f[70]<1.00279){r+=0.695268;}else{r+=-0.316011;}}}}}else{if(f[0]<1.41194e+11){if(f[0]<1.40113e+11){if(f[70]<1.00539){if(f[28]<1.39696e+11){r+=-0.540299;}else{r+=0.414394;}}else{if(f[28]<1.41526e+11){r+=-0.613491;}else{r+=0.00340238;}}}else{if(f[70]<1.01068){if(f[68]<0.563127){r+=-0.636314;}else{r+=0.0697669;}}else{if(f[27]<2.68399e+08){r+=-0.427861;}else{r+=0.660127;}}}}else{if(f[28]<1.39878e+11){if(f[27]<1.9281e+08){r+=0.0103432;}else{r+=-0.693888;}}else{if(f[34]<9.8943e+07){if(f[42]<1.00671){r+=0.489723;}else{r+=-0.149453;}}else{if(f[39]<1.63401e+09){r+=0.150731;}else{r+=-0.474874;}}}}}}}}else{if(f[66]<1.17533){if(f[28]<1.44054e+11){if(f[52]<0.239849){if(f[14]<1.40987e+11){if(f[43]<0.0189656){r+=0.323424;}else{if(f[21]<3.3886e+08){r+=-0.0666034;}else{r+=-0.610769;}}}else{r+=0.646072;}}else{if(f[31]<2.15182e+09){if(f[11]<2.51112e+10){if(f[14]<1.39766e+11){if(f[70]<1.00117){r+=0.443344;}else{r+=-0.552594;}}else{if(f[31]<1.88191e+09){r+=-0.586315;}else{r+=-0.192467;}}}else{r+=0.553116;}}else{if(f[42]<1.00021){r+=-0.473386;}else{if(f[14]<1.40032e+11){r+=-0.347434;}else{r+=0.595416;}}}}}else{if(f[0]<1.40234e+11){r+=-0.588436;}else{if(f[20]<4.49211e+08){if(f[68]<0.703756){if(f[41]<3.52694e+09){r+=0.154141;}else{r+=0.565049;}}else{r+=-0.101214;}}else{if(f[34]<1.91408e+08){r+=0.323054;}else{r+=-0.385851;}}}}}else{if(f[70]<1.00093){if(f[26]<1.6242e+10){if(f[3]<8.37008e+06){r+=0.174757;}else{if(f[42]<1.00491){if(f[34]<2.46929e+07){r+=-0.164985;}else{r+=-0.59681;}}else{r+=-0.100117;}}}else{if(f[10]<2.7371e+10){r+=0.684237;}else{r+=0.0309017;}}}else{if(f[51]<1.17147){if(f[27]<6.70157e+08){if(f[67]<0.67806){if(f[31]<2.22596e+07){r+=0.510725;}else{r+=0.0448835;}}else{if(f[57]<2.73212){r+=-0.602145;}else{r+=0.130796;}}}else{if(f[53]<0.169258){if(f[14]<1.40516e+11){if(f[67]<1.48466){r+=0.0848913;}else{r+=0.598662;}}else{if(f[27]<1.20987e+10){r+=-0.493292;}else{r+=-0.0119927;}}}else{if(f[34]<3.51014e+08){if(f[55]<0.579385){r+=0.183704;}else{r+=0.594508;}}else{if(f[11]<4.0903e+09){r+=1.08088;}else{r+=0.130097;}}}}}else{r+=-0.486231;}}}}}else{if(f[42]<1.62322){if(f[14]<1.39061e+11){if(f[56]<0.990991){if(f[13]<1.89884e+09){if(f[14]<1.38809e+11){if(f[20]<1.01475e+08){r+=-0.69188;}else{if(f[44]<0.0501886){r+=-0.633473;}else{r+=-1.04349;}}}else{if(f[80]<0.896315){r+=0.146987;}else{if(f[11]<2.17544e+08){r+=-0.60615;}else{if(f[72]<1.676){r+=0.205215;}else{r+=-0.335978;}}}}}else{if(f[14]<1.36827e+11){if(f[78]<5.54342){r+=0.578722;}else{r+=0.0407993;}}else{r+=-0.484395;}}}else{if(f[55]<0.94856){if(f[42]<1.0261){if(f[14]<1.38854e+11){if(f[34]<2.61193e+07){if(f[3]<1.67132e+07){r+=0.0341919;}else{r+=-0.659879;}}else{if(f[54]<0.386006){r+=-0.296355;}else{r+=0.45173;}}}else{if(f[28]<1.40092e+11){if(f[17]<5.31702e+07){r+=0.633845;}else{r+=-0.132568;}}else{if(f[13]<1.9655e+08){r+=-0.626193;}else{r+=0.517039;}}}}else{r+=-0.585356;}}else{if(f[9]<2.6594e+08){if(f[38]<1.95963e+08){r+=-0.106565;}else{r+=-0.62924;}}else{if(f[4]<3.86858e+07){r+=0.529048;}else{r+=-0.135334;}}}}}else{if(f[1]<2.32908e+09){if(f[42]<1.01054){r+=-0.0433736;}else{if(f[51]<1.01237){if(f[42]<1.01158){if(f[0]<1.41999e+11){r+=-0.581863;}else{r+=0.547528;}}else{if(f[0]<1.4078e+11){if(f[9]<1.22628e+08){r+=0.45334;}else{r+=-0.534753;}}else{r+=-0.600013;}}}else{if(f[30]<1.22764e+08){if(f[38]<2.50876e+08){r+=-0.675666;}else{if(f[42]<1.01158){r+=-0.221932;}else{r+=-0.608469;}}}else{if(f[38]<1.05935e+10){r+=0.331522;}else{r+=-0.371041;}}}}}else{r+=0.2543;}}}else{if(f[9]<1.22402e+08){r+=-0.312837;}else{r+=0.591397;}}}}}else{if(f[28]<1.78142e+11){if(f[68]<7.87375){if(f[28]<1.40629e+11){if(f[42]<0.998027){if(f[14]<1.40524e+11){if(f[14]<1.40268e+11){if(f[4]<15339.5){r+=-0.58046;}else{r+=-0.170367;}}else{if(f[28]<1.39842e+11){if(f[41]<2.50014e+07){if(f[39]<1.17e+07){r+=0.018464;}else{r+=0.520451;}}else{if(f[0]<1.40043e+11){r+=-0.730429;}else{r+=0.0265193;}}}else{if(f[70]<0.998987){if(f[9]<5.87725e+07){if(f[44]<0.909534){r+=0.194712;}else{r+=-0.55714;}}else{r+=0.484756;}}else{if(f[14]<1.40466e+11){if(f[0]<1.40027e+11){r+=0.525789;}else{r+=-0.543978;}}else{r+=0.622965;}}}}}else{if(f[29]<3.2184e+07){r+=-0.619962;}else{r+=-0.160242;}}}else{if(f[13]<2.73106e+08){if(f[28]<1.40628e+11){if(f[56]<1.00144){if(f[13]<2.57884e+08){r+=-0.600521;}else{r+=-0.121531;}}else{if(f[56]<1.00149){if(f[9]<3.94276e+07){r+=0.56855;}else{r+=0.0661364;}}else{if(f[70]<0.999249){r+=-0.6038;}else{r+=-0.0461121;}}}}else{r+=1.12917;}}else{if(f[28]<1.40113e+11){if(f[42]<1.00718){if(f[27]<6.00261e+08){r+=-0.40769;}else{if(f[36]<1.74833e+07){r+=-0.00885198;}else{if(f[59]<0.363094){r+=0.165695;}else{r+=0.681457;}}}}else{r+=-0.520105;}}else{if(f[13]<3.11323e+09){r+=-0.54799;}else{r+=0.151225;}}}}}else{if(f[39]<5.14722e+08){if(f[27]<7.76043e+08){r+=-0.699501;}else{if(f[67]<4.25527){if(f[50]<1.21356){r+=0.121916;}else{r+=-0.470384;}}else{r+=-0.600797;}}}else{r+=0.257128;}}}else{if(f[64]<0.371259){r+=0.62337;}else{if(f[72]<0.321937){if(f[41]<4.64109e+09){if(f[63]<0.628572){r+=-0.0245873;}else{if(f[8]<1.63142e+07){if(f[39]<8.84994e+06){r+=-0.596467;}else{if(f[41]<2.42843e+07){if(f[14]<1.40384e+11){r+=0.134328;}else{r+=0.767297;}}else{if(f[4]<3.29095e+06){r+=-0.412281;}else{r+=0.175479;}}}}else{if(f[76]<6.42637){if(f[52]<0.361206){if(f[0]<1.39916e+11){r+=0.473799;}else{r+=-0.553968;}}else{if(f[11]<3.78601e+10){r+=-0.59069;}else{r+=-0.157806;}}}else{r+=-0.083108;}}}}else{r+=0.321487;}}else{if(f[28]<1.39903e+11){if(f[70]<0.997324){if(f[0]<1.40204e+11){r+=-0.0924917;}else{r+=-0.575735;}}else{if(f[43]<0.282416){r+=-0.30696;}else{if(f[11]<6.83005e+07){r+=-0.167114;}else{if(f[66]<2.68202){if(f[63]<3.96761){r+=0.362637;}else{r+=-0.386219;}}else{if(f[52]<1.88608){r+=0.623866;}else{r+=0.136875;}}}}}}else{if(f[28]<1.40117e+11){if(f[53]<0.158014){if(f[19]<1.11062e+06){r+=0.299513;}else{r+=-0.601274;}}else{if(f[70]<0.999475){r+=-0.635845;}else{if(f[38]<1.05492e+07){if(f[29]<3.65111e+06){r+=-0.549522;}else{r+=0.00904894;}}else{if(f[22]<3.73929e+08){r+=0.622792;}else{r+=-0.384597;}}}}}else{if(f[20]<4.71875e+08){if(f[50]<0.00248285){r+=0.0252758;}else{if(f[72]<0.322559){r+=-0.0649788;}else{if(f[41]<3.75398e+09){r+=-0.599734;}else{r+=-0.102438;}}}}else{if(f[13]<2.68484e+09){r+=-0.415305;}else{if(f[32]<5.17014e+07){r+=0.595322;}else{r+=0.00529213;}}}}}}}}}else{if(f[14]<1.40148e+11){r+=0.60647;}else{r+=-0.574987;}}}} return r;}
    static double p21(double[] f){double r=0;if(f[28]<1.39437e+11){if(f[10]<3.28954e+10){if(f[28]<1.39426e+11){if(f[56]<1.00196){if(f[28]<1.39403e+11){if(f[56]<1.0017){r+=-0.95;}else{r+=-0.2;}}else{if(f[1]<6.52105e+07){r+=0.714286;}else{r+=-0;}}}else{if(f[71]<1.79984){r+=-0.996882;}else{if(f[42]<1.00946){if(f[10]<1.67755e+10){r+=-0.99606;}else{r+=-0.111111;}}else{if(f[84]<1.8){r+=-0.964912;}else{if(f[8]<3.99127e+08){r+=0.928571;}else{r+=-0;}}}}}}else{if(f[14]<1.40381e+11){if(f[84]<1.15){r+=-0.974026;}else{if(f[42]<1.0018){r+=-0.809524;}else{if(f[7]<9.87734e+07){r+=0.25;}else{r+=0.954545;}}}}else{r+=-0.989446;}}}else{if(f[28]<1.39093e+11){if(f[45]<0.617414){if(f[6]<3.2322e+09){if(f[7]<1.02774e+10){r+=0.25;}else{r+=-0.5;}}else{r+=0.6;}}else{if(f[53]<1.47781){r+=-0.833333;}else{if(f[6]<4.65476e+09){if(f[0]<1.37148e+11){r+=-0.5;}else{r+=-0;}}else{r+=0.5;}}}}else{if(f[29]<8.35959e+08){r+=-0.5;}else{r+=0.714286;}}}}else{if(f[0]<1.40094e+11){if(f[28]<1.40666e+11){if(f[28]<1.4e+11){if(f[14]<1.39645e+11){if(f[84]<1.5){if(f[29]<1.27159e+08){if(f[0]<1.39566e+11){r+=0.25;}else{r+=0.999251;}}else{r+=-0.666667;}}else{if(f[9]<3.62913e+10){r+=-0.981481;}else{r+=0.25;}}}else{if(f[14]<1.40369e+11){if(f[57]<0.096162){if(f[84]<1.85){if(f[30]<1.76457e+07){r+=-0.6;}else{if(f[27]<6.14056e+09){if(f[0]<1.40094e+11){r+=0.996004;}else{r+=0.25;}}else{r+=-0.25;}}}else{r+=-0.986755;}}else{if(f[27]<1.88687e+09){if(f[14]<1.40313e+11){if(f[42]<1.00174){if(f[28]<1.39999e+11){if(f[27]<1.48842e+09){r+=-0.996409;}else{r+=-0.649123;}}else{if(f[0]<1.39972e+11){r+=-0.888889;}else{r+=0.333333;}}}else{if(f[27]<9.74936e+07){r+=-0.882353;}else{if(f[0]<1.39998e+11){r+=0.955556;}else{r+=-0.25;}}}}else{if(f[28]<1.39878e+11){if(f[84]<2.15){if(f[79]<0.947362){r+=0.954545;}else{r+=-0.8;}}else{if(f[83]<0.514492){r+=-0.974843;}else{r+=0.0909091;}}}else{if(f[84]<1){r+=0.555556;}else{if(f[69]<5.90597){r+=-0.989726;}else{r+=-0.5;}}}}}else{if(f[0]<1.40064e+11){if(f[70]<0.999583){if(f[34]<3.47063e+07){if(f[42]<0.997352){r+=0.555556;}else{r+=-0.862069;}}else{if(f[4]<2.54578e+08){r+=0.833333;}else{r+=-0.25;}}}else{if(f[56]<1.00284){if(f[0]<1.40017e+11){r+=-0.853659;}else{r+=0.0909091;}}else{r+=0.555556;}}}else{if(f[21]<1.48784e+08){r+=-0.5;}else{if(f[14]<1.40017e+11){r+=-0;}else{r+=0.975904;}}}}}}else{if(f[28]<1.39826e+11){if(f[14]<1.40488e+11){if(f[84]<2.15){r+=0.888889;}else{r+=-0.942857;}}else{if(f[25]<3.78259e+10){r+=-0.998307;}else{r+=-0;}}}else{if(f[69]<0.120439){r+=-0.992188;}else{if(f[68]<0.898187){if(f[25]<4.27553e+08){if(f[56]<1.00355){if(f[12]<1.79977e+08){r+=-0.818182;}else{r+=-0;}}else{if(f[37]<5.38708e+06){r+=-0.333333;}else{r+=0.99327;}}}else{if(f[53]<1.76317){if(f[0]<1.3996e+11){r+=-0.961905;}else{r+=-0.25;}}else{if(f[8]<1.83917e+09){r+=-0.4;}else{r+=0.6;}}}}else{if(f[84]<2.15){if(f[28]<1.39907e+11){if(f[0]<1.39705e+11){r+=-0.936508;}else{r+=0.968382;}}else{if(f[84]<1){r+=0.942029;}else{r+=-0.916331;}}}else{if(f[14]<1.40516e+11){r+=-0.997043;}else{if(f[1]<2.15917e+08){r+=0.956044;}else{r+=-0;}}}}}}}}}else{if(f[14]<1.4016e+11){if(f[84]<0.5){if(f[41]<6.20884e+08){if(f[42]<0.99751){if(f[28]<1.40187e+11){r+=0.913043;}else{r+=-0.882353;}}else{r+=-0.997717;}}else{if(f[39]<6.09354e+08){if(f[28]<1.4015e+11){if(f[37]<3.28035e+07){r+=-0;}else{r+=-0.991071;}}else{if(f[29]<1.7215e+07){if(f[27]<2.185e+08){r+=-0.7;}else{if(f[28]<1.4042e+11){r+=0.994416;}else{r+=-0.555556;}}}else{if(f[40]<1.009e+09){if(f[9]<6.13641e+08){r+=-0.976331;}else{r+=-0.25;}}else{if(f[9]<2.05288e+08){r+=-0;}else{r+=0.6;}}}}}else{if(f[14]<1.4009e+11){if(f[13]<1.10247e+10){if(f[79]<0.237749){r+=-0.2;}else{if(f[14]<1.40082e+11){r+=-0.995778;}else{r+=-0.333333;}}}else{if(f[39]<2.46482e+10){r+=0.636364;}else{r+=-0.666667;}}}else{if(f[15]<1.11785e+08){if(f[51]<1.29682){r+=0.981735;}else{r+=-0;}}else{r+=-0.666667;}}}}}else{if(f[41]<7.14748e+07){if(f[83]<2.36345){if(f[74]<10.6072){if(f[32]<3.18178e+07){if(f[71]<0.0427067){r+=-0.25;}else{r+=-0.994695;}}else{r+=-0.25;}}else{if(f[0]<1.3999e+11){r+=-0.826087;}else{r+=0.6;}}}else{r+=0.571429;}}else{if(f[14]<1.3948e+11){if(f[2]<2.2845e+08){if(f[8]<3.40673e+07){r+=0.666667;}else{if(f[15]<3.69066e+08){if(f[0]<4.13857e+10){r+=-0.25;}else{r+=-0.995257;}}else{r+=-0;}}}else{if(f[14]<1.38839e+11){r+=-0.6;}else{if(f[40]<3.67e+08){r+=-0.25;}else{r+=0.789474;}}}}else{if(f[84]<2.85){if(f[36]<5.12283e+08){if(f[82]<23.1843){if(f[14]<1.40133e+11){r+=0.875759;}else{r+=0.541504;}}else{if(f[69]<0.745023){r+=0.647059;}else{r+=-0.973064;}}}else{if(f[40]<5.41932e+08){if(f[0]<1.40041e+11){r+=-0.984032;}else{r+=0.686275;}}else{if(f[36]<1.70259e+09){r+=0.969274;}else{r+=-0.0805687;}}}}else{if(f[28]<1.40176e+11){if(f[9]<2.80688e+09){r+=-0.99809;}else{r+=0.2;}}else{if(f[0]<1.39872e+11){r+=-0.952381;}else{r+=0.999267;}}}}}}}else{if(f[0]<1.39781e+11){if(f[23]<4.2965e+10){if(f[0]<1.39742e+11){if(f[12]<2.97689e+10){if(f[15]<1.76489e+09){if(f[23]<3.77117e+10){if(f[71]<0.00994091){r+=-0.25;}else{r+=-0.996209;}}else{r+=-0.25;}}else{r+=-0.25;}}else{r+=-0;}}else{if(f[41]<4.43095e+08){if(f[34]<9.40692e+06){if(f[27]<1.39329e+08){if(f[56]<1.00178){r+=0.684211;}else{r+=-0.647059;}}else{if(f[14]<1.40308e+11){r+=-0.25;}else{r+=-0.921569;}}}else{if(f[36]<2.03568e+07){if(f[84]<1.85){r+=-0.6;}else{r+=0.666667;}}else{if(f[41]<2.67978e+07){r+=-0.111111;}else{r+=0.96319;}}}}else{if(f[84]<0.65){if(f[4]<3.32287e+07){r+=0.777778;}else{r+=-0;}}else{if(f[28]<1.40127e+11){if(f[8]<7.04786e+09){r+=0.714286;}else{r+=-0.25;}}else{r+=-0.992509;}}}}}else{if(f[28]<1.40139e+11){if(f[7]<5.93582e+09){r+=0.966942;}else{r+=0.25;}}else{if(f[1]<4.09204e+09){r+=-0;}else{r+=-0.8;}}}}else{if(f[14]<1.40213e+11){if(f[63]<0.40285){if(f[0]<1.39998e+11){if(f[21]<1.88445e+08){r+=-0.964602;}else{r+=-0.25;}}else{if(f[84]<1.5){r+=-0.826087;}else{if(f[12]<6.48472e+07){r+=-0.666667;}else{if(f[82]<5.38685){r+=0.996212;}else{r+=-0;}}}}}else{if(f[37]<1.09947e+09){if(f[21]<9.58431e+06){if(f[9]<3.0895e+07){r+=-0.862069;}else{r+=0.870968;}}else{if(f[40]<2.29193e+07){if(f[82]<0.335413){r+=0.764706;}else{r+=-0.692308;}}else{if(f[66]<3.33582){r+=-0.993868;}else{r+=-0.111111;}}}}else{if(f[1]<1.16291e+08){r+=0.934426;}else{r+=-0.25;}}}}else{if(f[69]<0.185216){if(f[28]<1.40376e+11){if(f[28]<1.4011e+11){if(f[56]<1.00193){if(f[51]<1.26085){r+=-0.333333;}else{r+=0.75;}}else{r+=-0.666667;}}else{r+=-0.99738;}}else{if(f[14]<1.40372e+11){r+=-0.666667;}else{if(f[0]<1.40087e+11){if(f[1]<9.20175e+07){r+=0.957895;}else{r+=0.25;}}else{r+=-0;}}}}else{if(f[84]<1.5){if(f[14]<1.40473e+11){if(f[70]<1.00495){if(f[70]<1){r+=0.811321;}else{r+=-0.961165;}}else{if(f[66]<5.01264){r+=0.984615;}else{r+=-0.692308;}}}else{if(f[28]<1.40377e+11){if(f[84]<0.85){r+=-0.983165;}else{r+=0.548387;}}else{if(f[69]<4.01199){r+=0.907683;}else{r+=-0.787234;}}}}else{if(f[83]<0.161663){if(f[28]<1.40159e+11){if(f[84]<4.60001){r+=0.72973;}else{r+=-0.777778;}}else{if(f[37]<1.32143e+08){r+=-0.96139;}else{r+=0.181818;}}}else{if(f[69]<0.529389){if(f[37]<1.25059e+08){r+=-0.52605;}else{r+=0.754277;}}else{if(f[84]<2.5){r+=0.937447;}else{r+=0.652439;}}}}}}}}}}else{if(f[0]<1.39576e+11){if(f[28]<1.41237e+11){if(f[14]<1.46915e+11){if(f[25]<3.09717e+10){if(f[18]<2.84687e+09){r+=-0.999588;}else{r+=-0.25;}}else{if(f[0]<1.38603e+11){r+=-0.555556;}else{r+=-0;}}}else{r+=0.866667;}}else{if(f[0]<1.39026e+11){if(f[84]<2.85){if(f[21]<4.78209e+10){r+=-0.995349;}else{r+=-0.25;}}else{if(f[1]<3.59566e+08){r+=0.5;}else{r+=-0.733333;}}}else{if(f[84]<3.35){if(f[69]<0.683321){if(f[14]<1.45714e+11){if(f[84]<2.85){r+=-0.992933;}else{if(f[0]<1.3934e+11){r+=0.974194;}else{r+=0.25;}}}else{r+=0.992895;}}else{if(f[79]<0.912017){if(f[12]<4.50973e+08){if(f[37]<1.45081e+08){if(f[84]<2.65){r+=0.815865;}else{r+=0.999245;}}else{if(f[51]<1.25553){r+=-0.758621;}else{r+=0.538462;}}}else{if(f[0]<1.39508e+11){r+=-0.940298;}else{r+=-0.25;}}}else{if(f[28]<1.4161e+11){r+=-0.967742;}else{r+=0.944444;}}}}else{r+=-0.98513;}}}}else{if(f[37]<2.12278e+07){if(f[69]<4.62948){if(f[1]<1.81577e+07){r+=0.985765;}else{r+=0.25;}}else{r+=-0.97861;}}else{if(f[79]<5.74268){if(f[29]<2.78304e+06){if(f[64]<1.49652){r+=-0.939394;}else{if(f[28]<1.41225e+11){r+=-0.666667;}else{if(f[53]<1.73334){r+=0.921569;}else{r+=-0;}}}}else{if(f[0]<1.39591e+11){if(f[14]<1.45837e+11){r+=-0.985765;}else{r+=0.938462;}}else{if(f[32]<1.15118e+09){if(f[39]<3.36486e+07){r+=0.2;}else{if(f[71]<0.0373706){if(f[78]<0.0933188){r+=0.5;}else{r+=-0.5;}}else{if(f[29]<3.56238e+06){r+=-0.761194;}else{r+=-0.994793;}}}}else{if(f[56]<0.994614){r+=-0.333333;}else{r+=0.555556;}}}}}else{if(f[84]<6.00001){if(f[28]<1.40757e+11){r+=-0.25;}else{r+=-0.949367;}}else{if(f[0]<1.39903e+11){r+=0.25;}else{r+=0.959596;}}}}}}}else{if(f[83]<2.355){if(f[0]<1.4016e+11){if(f[6]<8.77866e+06){if(f[14]<1.40296e+11){if(f[12]<4.65992e+08){if(f[28]<1.39939e+11){if(f[37]<3.00172e+07){r+=-0.950617;}else{if(f[14]<1.40036e+11){if(f[8]<1.1293e+08){r+=-0.826087;}else{r+=0.6;}}else{if(f[14]<1.40084e+11){r+=0.992995;}else{r+=-0.111111;}}}}else{if(f[83]<1.0549){if(f[41]<2.07643e+08){if(f[56]<1.00118){if(f[0]<1.40157e+11){r+=-0.996511;}else{r+=-0.855953;}}else{if(f[84]<2.3){r+=-0.971143;}else{r+=0.963636;}}}else{if(f[84]<2.85){if(f[56]<1.00125){r+=-0.993072;}else{r+=0.7;}}else{if(f[28]<1.40139e+11){r+=0.623529;}else{r+=-0.956522;}}}}else{if(f[84]<2.85){if(f[56]<1.00126){if(f[14]<1.39846e+11){r+=0.636364;}else{r+=-0.966858;}}else{if(f[39]<2.20586e+08){r+=0.953125;}else{r+=-0.915493;}}}else{if(f[14]<1.40013e+11){r+=-0.994126;}else{if(f[0]<1.40141e+11){r+=-0.794872;}else{r+=0.722403;}}}}}}else{if(f[84]<2.85){if(f[28]<1.39945e+11){if(f[7]<6.81073e+07){r+=-0.7;}else{if(f[0]<1.40133e+11){r+=0.978723;}else{r+=-0.25;}}}else{if(f[56]<1.00056){r+=-0.995311;}else{if(f[14]<1.40082e+11){r+=0.6;}else{r+=-0.793103;}}}}else{if(f[14]<1.40058e+11){r+=-0.84;}else{if(f[41]<2.9134e+08){r+=-0;}else{r+=0.997087;}}}}}else{if(f[56]<1.00134){if(f[10]<2.33982e+07){r+=-0;}else{if(f[84]<6.60001){if(f[55]<0.414691){if(f[84]<1.15){r+=0.5;}else{r+=-0.8;}}else{if(f[4]<126029){r+=-0.25;}else{r+=-0.995798;}}}else{r+=-0.111111;}}}else{if(f[56]<1.00222){if(f[28]<1.40386e+11){if(f[31]<2.48626e+06){if(f[30]<3.77694e+06){if(f[33]<3.7666e+06){r+=0.991939;}else{r+=0.230769;}}else{r+=-0.25;}}else{if(f[6]<4.12373e+06){if(f[8]<2.48632e+07){r+=-0.885714;}else{r+=0.25;}}else{if(f[66]<0.983798){r+=0.111111;}else{r+=0.925926;}}}}else{r+=-0.84;}}else{if(f[14]<1.40323e+11){r+=-0.25;}else{r+=-0.978836;}}}}}else{if(f[28]<1.40396e+11){if(f[14]<1.40225e+11){if(f[28]<1.39956e+11){if(f[84]<1.5){if(f[14]<1.39873e+11){if(f[84]<1){r+=-0.764706;}else{r+=0.953488;}}else{if(f[29]<1.69924e+08){r+=-0.980676;}else{r+=0.25;}}}else{if(f[84]<1.85){if(f[12]<1.52008e+08){r+=-0;}else{if(f[83]<2.01213){r+=0.9991;}else{r+=0.25;}}}else{r+=-0.777778;}}}else{if(f[84]<2.85){if(f[28]<1.40008e+11){if(f[84]<5.40001){if(f[0]<1.40102e+11){r+=0.311828;}else{r+=-0.839456;}}else{if(f[14]<1.40069e+11){r+=0.98374;}else{r+=-0.76;}}}else{if(f[43]<16.3474){if(f[0]<1.40097e+11){r+=-0.371429;}else{r+=-0.989906;}}else{if(f[68]<1.14516){r+=0.692308;}else{r+=-0.6;}}}}else{if(f[27]<4.23163e+08){if(f[14]<1.40103e+11){if(f[25]<2.70443e+08){r+=-0.983051;}else{r+=-0.25;}}else{r+=0.555556;}}else{if(f[68]<2.35012){if(f[21]<2.82987e+07){r+=0.25;}else{r+=0.999048;}}else{r+=-0.333333;}}}}}else{if(f[56]<1.0011){if(f[84]<1.15){if(f[84]<0.85){r+=-0.555556;}else{r+=0.950617;}}else{if(f[28]<1.4018e+11){r+=0.25;}else{if(f[55]<0.635527){r+=-0.25;}else{r+=-0.963636;}}}}else{if(f[56]<1.00237){if(f[84]<0.85){r+=-0.882353;}else{if(f[51]<5.42444){if(f[28]<1.4039e+11){r+=0.995299;}else{r+=-0.0588235;}}else{r+=-0.666667;}}}else{if(f[28]<1.399e+11){if(f[79]<0.300065){if(f[27]<1.89183e+08){r+=-0.764706;}else{r+=0.571429;}}else{if(f[28]<1.39769e+11){r+=-0.5;}else{r+=0.922581;}}}else{if(f[16]<1.3206e+07){if(f[84]<1.85){r+=-0.955056;}else{r+=-0.368421;}}else{if(f[61]<2.15685){r+=0.6;}else{r+=-0.333333;}}}}}}}else{if(f[84]<6.60001){if(f[30]<6.93921e+08){if(f[28]<1.40403e+11){if(f[84]<1.15){if(f[84]<0.65){r+=-0.692308;}else{r+=0.84;}}else{r+=-0.970803;}}else{if(f[28]<1.47353e+11){r+=-0.996837;}else{r+=-0.25;}}}else{r+=0.5;}}else{if(f[36]<1.63022e+08){if(f[0]<1.40102e+11){r+=-0.333333;}else{if(f[9]<1.93184e+08){if(f[5]<3.71681e+06){r+=0.25;}else{r+=0.921569;}}else{r+=-0;}}}else{if(f[28]<1.40434e+11){r+=0.6;}else{r+=-0.945946;}}}}}}else{if(f[0]<1.40259e+11){if(f[84]<2.85){if(f[14]<1.40096e+11){if(f[84]<1.85){if(f[25]<1.68387e+09){if(f[25]<5.50771e+07){if(f[1]<1.80899e+08){if(f[25]<5.34555e+07){r+=-0.986799;}else{r+=-0.375;}}else{r+=-0;}}else{if(f[0]<1.40163e+11){if(f[14]<1.40071e+11){r+=-0.886256;}else{r+=0.934142;}}else{if(f[14]<1.39728e+11){r+=-0.843137;}else{r+=0.871005;}}}}else{if(f[28]<1.50924e+11){if(f[28]<1.39876e+11){r+=0.818182;}else{if(f[1]<1.73416e+08){r+=-0.982609;}else{r+=-0;}}}else{r+=0.924528;}}}else{if(f[10]<1.80366e+10){if(f[43]<12.5693){if(f[0]<1.40258e+11){if(f[0]<1.40249e+11){r+=-0.989301;}else{r+=-0.755102;}}else{r+=-0.0909091;}}else{if(f[7]<1.56798e+08){r+=0.636364;}else{r+=-0.25;}}}else{r+=0.692308;}}}else{if(f[28]<1.40003e+11){if(f[39]<3.04025e+08){if(f[28]<1.39822e+11){if(f[28]<1.39809e+11){if(f[40]<1.89983e+07){r+=-0;}else{r+=-0.985141;}}else{if(f[14]<1.40319e+11){r+=0.78022;}else{r+=-0.834197;}}}else{if(f[14]<1.40155e+11){if(f[9]<3.78616e+08){r+=-0.982684;}else{r+=-0.25;}}else{if(f[27]<1.70188e+08){r+=-0.381107;}else{r+=0.711432;}}}}else{if(f[56]<1.00314){if(f[84]<1.15){if(f[47]<0.608685){r+=0.454545;}else{r+=-0.75;}}else{if(f[15]<8.15166e+07){r+=-0.996055;}else{r+=-0.25;}}}else{if(f[28]<1.39818e+11){r+=-0.931035;}else{if(f[84]<2){r+=-0.75;}else{r+=0.769231;}}}}}else{if(f[28]<1.40387e+11){if(f[41]<7.00252e+07){if(f[84]<0.85){if(f[14]<1.40895e+11){r+=-0.98374;}else{r+=0.25;}}else{if(f[84]<1.5){r+=0.98806;}else{r+=-0.8125;}}}else{if(f[42]<0.998694){if(f[84]<2){r+=-0.586207;}else{r+=0.868421;}}else{if(f[23]<5.20498e+07){r+=-0.968391;}else{r+=-0.561128;}}}}else{if(f[1]<1.845e+09){r+=-0.997496;}else{r+=-0.25;}}}}}else{if(f[14]<1.40014e+11){if(f[23]<1.75939e+09){r+=-0.997028;}else{if(f[39]<2.25088e+10){r+=0.6;}else{r+=-0.5;}}}else{if(f[68]<2.8459){if(f[39]<6.01251e+07){if(f[56]<1.00004){if(f[14]<1.40573e+11){r+=-0.995609;}else{r+=-0.25;}}else{if(f[28]<1.39785e+11){r+=-0.894737;}else{if(f[56]<1.00118){r+=0.601318;}else{r+=0.994911;}}}}else{if(f[84]<3.15){if(f[28]<1.40516e+11){if(f[66]<0.230471){r+=-0.555556;}else{r+=0.996553;}}else{r+=-0.991209;}}else{if(f[56]<1.00312){if(f[37]<2.23034e+09){r+=0.947149;}else{r+=-0.219512;}}else{r+=-0.929825;}}}}else{if(f[28]<1.4048e+11){if(f[78]<0.159244){r+=-0.454545;}else{if(f[1]<6.32723e+07){r+=0.950617;}else{r+=0.25;}}}else{if(f[1]<6.5974e+08){r+=-0.986395;}else{r+=-0.25;}}}}}}else{if(f[0]<1.41618e+11){if(f[36]<6.91074e+06){if(f[14]<1.40394e+11){if(f[0]<1.40414e+11){if(f[14]<1.40007e+11){if(f[9]<2.21262e+07){r+=-0.733333;}else{r+=0.963303;}}else{if(f[25]<8.17936e+07){if(f[29]<1.51331e+07){r+=-0.983584;}else{r+=-0.272727;}}else{if(f[34]<1.58074e+06){r+=-0.882883;}else{r+=-0.0243902;}}}}else{if(f[69]<2.50878){if(f[1]<3.5581e+07){r+=0.98556;}else{r+=0.25;}}else{r+=-0.885714;}}}else{if(f[70]<0.996441){if(f[66]<0.990315){r+=0.636364;}else{r+=-0.967213;}}else{if(f[39]<6.81571e+06){r+=-0.914894;}else{if(f[84]<1.5){if(f[0]<1.40276e+11){r+=-0;}else{r+=0.998798;}}else{r+=-0.809524;}}}}}else{if(f[14]<1.40533e+11){if(f[84]<1.15){if(f[7]<6.59587e+07){if(f[84]<0.85){if(f[56]<0.9976){r+=0.963702;}else{r+=-0.997582;}}else{if(f[28]<1.39953e+11){r+=-0.976608;}else{r+=0.867538;}}}else{if(f[28]<1.47226e+11){if(f[71]<0.0512066){r+=0.28;}else{r+=-0.968505;}}else{r+=0.99439;}}}else{if(f[83]<0.624006){if(f[28]<1.39985e+11){if(f[84]<2.85){r+=-0.526338;}else{r+=0.958384;}}else{if(f[84]<1.5){r+=-0.129218;}else{r+=-0.818658;}}}else{if(f[23]<9.44142e+07){if(f[84]<3.15){r+=-0.679254;}else{r+=0.519926;}}else{if(f[41]<5.52529e+08){r+=0.308541;}else{r+=-0.575055;}}}}}else{if(f[37]<1.99106e+07){if(f[84]<1.15){if(f[28]<1.3993e+11){r+=-0.857143;}else{if(f[28]<1.40491e+11){r+=0.994987;}else{r+=-0.6;}}}else{if(f[67]<1.09638){r+=-0;}else{r+=-0.954545;}}}else{if(f[66]<0.444685){if(f[14]<1.40603e+11){if(f[56]<1.00464){r+=0.951002;}else{r+=-0.818182;}}else{if(f[36]<3.8503e+07){r+=0.870968;}else{r+=-0.975395;}}}else{if(f[0]<1.41558e+11){if(f[84]<7.40001){r+=-0.985369;}else{r+=-0.480412;}}else{if(f[56]<0.99897){r+=0.964912;}else{r+=-0.898305;}}}}}}}else{if(f[84]<2.5){if(f[1]<9.10664e+06){if(f[15]<2.91182e+07){if(f[2]<1.16355e+07){if(f[16]<2.72181e+07){r+=-0.998523;}else{r+=-0;}}else{if(f[8]<2.86782e+08){r+=0.666667;}else{r+=-0.555556;}}}else{if(f[0]<1.46159e+11){if(f[56]<1.00809){if(f[8]<2.17803e+08){r+=0.903614;}else{r+=-0.111111;}}else{r+=-0.5;}}else{r+=-0.818182;}}}else{if(f[14]<1.4127e+11){if(f[28]<1.4167e+11){if(f[23]<2.76371e+08){if(f[40]<6.17168e+07){r+=-0.861386;}else{r+=0.894483;}}else{if(f[28]<1.4025e+11){r+=0.822751;}else{r+=-0.780037;}}}else{if(f[5]<3.37143e+09){if(f[14]<1.36412e+11){r+=-0.25;}else{r+=-0.994878;}}else{r+=0.25;}}}else{if(f[56]<1.00817){if(f[84]<1){if(f[0]<1.41914e+11){r+=-0.307692;}else{r+=0.953279;}}else{if(f[15]<2.80466e+09){r+=-0.9819;}else{r+=-0;}}}else{if(f[63]<15.879){if(f[9]<5.41325e+10){r+=-0.996825;}else{r+=-0.25;}}else{if(f[0]<1.44715e+11){r+=-0;}else{r+=0.555556;}}}}}}else{if(f[14]<1.39136e+11){if(f[56]<0.991561){r+=-0.902439;}else{if(f[84]<3.15){r+=0.992;}else{r+=-0.5;}}}else{if(f[22]<4.23839e+07){if(f[84]<3.15){r+=-0.986622;}else{if(f[28]<1.40002e+11){r+=0.994413;}else{r+=-0.947368;}}}else{if(f[23]<5.62052e+07){if(f[14]<1.40165e+11){r+=0.846154;}else{r+=-0.954023;}}else{r+=-0.997531;}}}}}}}}else{if(f[14]<1.42088e+11){if(f[82]<3.09558e+08){if(f[0]<1.4083e+11){if(f[55]<0.864786){if(f[84]<1.15){if(f[69]<0.818787){if(f[37]<7.97658e+07){if(f[28]<1.40146e+11){if(f[84]<0.85){r+=-0.93959;}else{r+=0.906977;}}else{if(f[0]<1.40182e+11){r+=-0.586957;}else{r+=0.893883;}}}else{if(f[56]<0.999901){if(f[71]<51.698){r+=-0.921875;}else{r+=0.690722;}}else{if(f[84]<0.5){r+=-0.991984;}else{r+=0.185656;}}}}else{if(f[42]<0.999031){if(f[84]<0.65){r+=-0.818182;}else{if(f[28]<1.41168e+11){r+=0.997407;}else{r+=-0.692308;}}}else{if(f[49]<0.0163318){r+=0.714286;}else{if(f[68]<1.54435){r+=-0.895833;}else{r+=-0;}}}}}else{if(f[14]<1.39108e+11){if(f[56]<0.991408){if(f[70]<0.995607){if(f[1]<6.61859e+07){r+=0.733333;}else{r+=-0.5;}}else{if(f[67]<0.482369){r+=0.111111;}else{r+=-0.845161;}}}else{if(f[84]<1.85){r+=-0.666667;}else{if(f[14]<1.38856e+11){r+=-0.5;}else{r+=0.978118;}}}}else{if(f[28]<1.39555e+11){if(f[84]<1.5){if(f[0]<1.40395e+11){r+=0.982379;}else{r+=0.25;}}else{r+=-0.956044;}}else{if(f[25]<6.42259e+07){if(f[42]<0.998007){r+=0.293413;}else{r+=-0.962477;}}else{if(f[0]<1.40119e+11){r+=-0.420561;}else{r+=-0.966115;}}}}}}else{if(f[28]<1.39537e+11){if(f[84]<1.15){r+=-0.971631;}else{if(f[13]<2.53586e+08){r+=0.987421;}else{r+=-0.8125;}}}else{if(f[66]<0.955028){if(f[28]<1.69093e+11){if(f[10]<1.54202e+08){if(f[42]<0.997327){r+=0.0355556;}else{r+=-0.848705;}}else{if(f[53]<0.663427){r+=0.15942;}else{r+=-0.989281;}}}else{if(f[0]<1.40314e+11){r+=0.955056;}else{r+=-0.75;}}}else{if(f[56]<1.00119){if(f[14]<1.39151e+11){if(f[14]<1.38866e+11){r+=-0.854545;}else{r+=0.915493;}}else{if(f[0]<1.4026e+11){r+=-0.472222;}else{r+=-0.973597;}}}else{if(f[39]<2.939e+08){if(f[14]<1.40613e+11){r+=0.654596;}else{r+=-0.666667;}}else{if(f[40]<8.88186e+08){r+=-0.892308;}else{r+=0.0344828;}}}}}}}else{if(f[79]<1.12139){if(f[41]<8.15783e+08){if(f[25]<9.49271e+07){if(f[12]<1.08055e+08){if(f[83]<2.38173){if(f[27]<1.21494e+08){r+=-0.8;}else{r+=0.636364;}}else{if(f[82]<1.18319){r+=-0.166667;}else{r+=-0.967888;}}}else{if(f[84]<1.15){if(f[42]<1.01302){r+=0.927273;}else{r+=-0.714286;}}else{if(f[69]<0.392784){r+=-0.973154;}else{r+=-0.133333;}}}}else{if(f[69]<0.369954){if(f[15]<1.03696e+07){if(f[14]<1.40499e+11){r+=0.647059;}else{r+=-0.5;}}else{if(f[69]<0.317758){r+=-0.972727;}else{r+=-0.463415;}}}else{if(f[56]<1.0014){if(f[15]<6.21916e+07){r+=0.882175;}else{r+=-0.416667;}}else{if(f[37]<1.50948e+08){r+=-0.968;}else{r+=0.0909091;}}}}}else{if(f[69]<0.712797){if(f[14]<1.41858e+11){if(f[15]<1.14097e+07){if(f[84]<1.15){r+=0.166667;}else{r+=-0.956679;}}else{if(f[84]<0.5){r+=-0.65625;}else{r+=-0.986307;}}}else{if(f[28]<1.42845e+11){r+=-0.882353;}else{if(f[84]<1){r+=0.902439;}else{r+=-0.636364;}}}}else{if(f[14]<1.39034e+11){if(f[14]<1.3886e+11){r+=-0.25;}else{if(f[1]<8.70481e+07){r+=0.897436;}else{r+=0.25;}}}else{if(f[1]<2.49997e+08){if(f[67]<0.73342){r+=-0;}else{r+=-0.902439;}}else{r+=-0;}}}}}else{if(f[83]<9.8677e+08){if(f[42]<1.01132){if(f[29]<6.48459e+06){if(f[84]<0.65){r+=-0.909091;}else{if(f[84]<1.15){r+=0.952381;}else{r+=-0.666667;}}}else{if(f[39]<1.77758e+08){if(f[0]<1.41593e+11){r+=-0.826087;}else{r+=0.666667;}}else{if(f[0]<1.42738e+11){r+=-0.969252;}else{r+=0.25;}}}}else{if(f[14]<1.36223e+11){r+=0.25;}else{if(f[14]<1.39025e+11){if(f[42]<1.02058){r+=-0.254237;}else{r+=-0.965909;}}else{if(f[83]<9.09593e+08){r+=-0.999537;}else{r+=-0;}}}}}else{if(f[23]<2.83185e+07){r+=-0.666667;}else{r+=0.833333;}}}}}else{if(f[84]<2.65){if(f[14]<1.39916e+11){if(f[14]<1.39597e+11){if(f[28]<1.40176e+11){if(f[14]<1.38582e+11){r+=-0.5;}else{if(f[14]<1.39582e+11){r+=0.885714;}else{r+=0.25;}}}else{if(f[17]<5.59109e+06){r+=-0.891892;}else{r+=-0;}}}else{r+=0.998197;}}else{r+=-0.945205;}}else{r+=-0.991051;}}}else{if(f[84]<0.8){if(f[10]<1.30451e+10){r+=0.99915;}else{r+=0.25;}}else{r+=-0.994723;}}}}} return r;}
    static double p22(double[] f){double r=0;if(f[28]<1.39435e+11){if(f[56]<1.00186){if(f[36]<1.35272e+08){if(f[28]<1.39336e+11){r+=-0.381517;}else{r+=0.721479;}}else{if(f[53]<0.994581){if(f[30]<4.94498e+09){if(f[14]<1.37954e+11){r+=0.619353;}else{r+=-0.262507;}}else{r+=-0.448335;}}else{r+=-0.680059;}}}else{if(f[10]<3.57921e+10){if(f[28]<1.3942e+11){if(f[27]<5.71165e+10){if(f[42]<1.00377){r+=-0.68419;}else{if(f[42]<1.01073){if(f[84]<2.15){if(f[6]<6.61716e+06){r+=-0.1415;}else{r+=-0.664898;}}else{if(f[42]<1.00875){r+=-0.370739;}else{if(f[71]<1.82299){r+=1.49367;}else{r+=0.622739;}}}}else{r+=-0.68368;}}}else{r+=-0.061157;}}else{if(f[14]<1.39882e+11){if(f[84]<1){r+=-0.666221;}else{if(f[0]<1.40136e+11){r+=-0.422689;}else{if(f[28]<1.39426e+11){r+=1.18576;}else{r+=0.623463;}}}}else{if(f[42]<1.0068){r+=-0.679194;}else{r+=0.232026;}}}}else{if(f[65]<1.15478){if(f[13]<5.73453e+10){if(f[84]<2.15){r+=0.659103;}else{r+=0.0179709;}}else{r+=-0.420076;}}else{if(f[12]<2.71662e+10){r+=-0.0356754;}else{r+=-0.742729;}}}}}else{if(f[67]<6.04592){if(f[0]<1.40093e+11){if(f[28]<1.40665e+11){if(f[28]<1.40001e+11){if(f[14]<1.39647e+11){if(f[28]<1.39886e+11){if(f[84]<1){r+=-0.436439;}else{if(f[12]<2.54974e+10){r+=0.684581;}else{r+=0.0421131;}}}else{r+=-0.672654;}}else{if(f[14]<1.40364e+11){if(f[0]<1.40077e+11){if(f[56]<1.003){if(f[28]<1.39688e+11){if(f[14]<1.3968e+11){r+=0.695531;}else{if(f[12]<1.99611e+10){r+=-0.635498;}else{r+=0.478611;}}}else{if(f[28]<1.39998e+11){if(f[0]<1.40072e+11){r+=-0.680879;}else{r+=-0.365065;}}else{if(f[0]<1.39963e+11){r+=-0.613989;}else{r+=0.151408;}}}}else{if(f[84]<2.15){if(f[66]<1.40877){if(f[84]<0.85){r+=0.447669;}else{r+=-0.535759;}}else{if(f[42]<0.998188){r+=0.670436;}else{r+=-0.0350583;}}}else{if(f[15]<1.335e+08){if(f[43]<4.81176){r+=-0.688309;}else{r+=-0.0964238;}}else{r+=0.215431;}}}}else{if(f[34]<9.12799e+06){if(f[13]<3.97521e+08){if(f[56]<1.00339){r+=-0.671869;}else{r+=0.209315;}}else{r+=0.398207;}}else{if(f[84]<2.15){if(f[42]<0.998099){r+=-0.0102999;}else{if(f[83]<0.421993){r+=0.152986;}else{r+=0.701846;}}}else{if(f[83]<1.00741){r+=-0.630238;}else{r+=0.389901;}}}}}else{if(f[28]<1.39827e+11){if(f[0]<1.40025e+11){if(f[32]<1.3131e+08){if(f[20]<1.76638e+08){if(f[25]<4.46005e+07){r+=-0.0624934;}else{r+=-0.682799;}}else{r+=-0.0450804;}}else{if(f[4]<1.35602e+09){r+=0.516918;}else{r+=-0.296956;}}}else{if(f[37]<1.35404e+07){r+=-0.42745;}else{if(f[28]<1.39789e+11){r+=-0.165067;}else{r+=0.619224;}}}}else{if(f[65]<0.900782){if(f[84]<1.85){if(f[66]<2.23199){if(f[84]<0.85){r+=0.400654;}else{r+=-0.700883;}}else{if(f[28]<1.39907e+11){r+=-1.71176;}else{r+=-0.63971;}}}else{if(f[14]<1.40975e+11){if(f[9]<2.9866e+07){r+=-0.43755;}else{r+=0.688219;}}else{r+=-0.576562;}}}else{if(f[84]<1){if(f[56]<1.00417){r+=0.689676;}else{r+=-0.0451946;}}else{if(f[67]<0.837431){if(f[84]<2.65){r+=-0.391704;}else{r+=0.664817;}}else{if(f[69]<5.53306){r+=-0.575656;}else{r+=-0.00315924;}}}}}}}}else{if(f[14]<1.40109e+11){if(f[34]<5.69135e+06){if(f[38]<4.99013e+07){if(f[55]<0.45624){if(f[84]<1.35){r+=-0.455754;}else{if(f[7]<2.74301e+06){r+=-0.0583149;}else{r+=0.676301;}}}else{if(f[28]<1.40061e+11){if(f[84]<2.5){if(f[42]<1.00045){r+=0.669072;}else{r+=-0.375309;}}else{r+=-0.585117;}}else{if(f[84]<0.85){if(f[41]<9.1376e+07){r+=-0.73468;}else{r+=0.545109;}}else{if(f[41]<1.01991e+08){r+=-0.738992;}else{r+=-1.4309;}}}}}else{if(f[14]<1.39968e+11){if(f[28]<1.40145e+11){if(f[84]<2.65){if(f[7]<9.86553e+06){r+=0.589874;}else{r+=-1.6602;}}else{r+=-0.68213;}}else{if(f[84]<2.85){if(f[28]<1.40425e+11){r+=0.701216;}else{r+=-0.305746;}}else{r+=-0.697577;}}}else{if(f[84]<0.5){r+=-0.6942;}else{if(f[0]<1.40089e+11){if(f[56]<1.00049){r+=0.661364;}else{r+=-0.664791;}}else{if(f[56]<0.99961){r+=0.572796;}else{r+=-1.63859;}}}}}}else{if(f[28]<1.4018e+11){if(f[0]<1.4006e+11){if(f[38]<6.95287e+07){if(f[42]<1){if(f[84]<0.5){r+=-0.526889;}else{r+=0.694406;}}else{if(f[12]<1.69317e+08){r+=-1.1832;}else{r+=0.0851495;}}}else{if(f[84]<1.5){if(f[36]<2.90974e+07){r+=0.399686;}else{r+=-0.678665;}}else{if(f[56]<0.999852){r+=-1.15608;}else{r+=0.723815;}}}}else{if(f[84]<2.5){if(f[10]<6.82702e+08){if(f[41]<9.9218e+07){r+=-0.42786;}else{r+=0.680614;}}else{if(f[28]<1.40042e+11){r+=0.720626;}else{r+=-1.31432;}}}else{if(f[19]<2.5588e+07){r+=-0.72626;}else{r+=0.394627;}}}}else{if(f[27]<7.43487e+07){if(f[84]<1.2){if(f[83]<1.96067){if(f[36]<2.0674e+08){r+=-0.70176;}else{r+=-1.04828;}}else{if(f[41]<7.13774e+07){r+=-0.600687;}else{r+=-1.62819;}}}else{r+=0.650264;}}else{if(f[67]<1.72608){if(f[28]<1.40571e+11){if(f[84]<0.5){r+=-0.00963426;}else{r+=0.667857;}}else{if(f[10]<9.79147e+07){r+=0.323081;}else{r+=-0.762547;}}}else{if(f[41]<7.59241e+08){if(f[42]<0.999996){r+=-0.880782;}else{r+=0.118166;}}else{if(f[14]<1.40027e+11){r+=-0.398544;}else{r+=0.668634;}}}}}}}else{if(f[0]<1.39783e+11){if(f[0]<1.39747e+11){if(f[43]<20.4918){if(f[12]<8.1797e+09){if(f[14]<1.40159e+11){if(f[41]<7.16274e+07){r+=-0.679461;}else{r+=-1.29688;}}else{if(f[56]<0.997537){r+=0.0875163;}else{r+=-0.679431;}}}else{if(f[56]<1.0011){if(f[54]<2.89577){r+=1.01201;}else{r+=-0.290087;}}else{if(f[25]<1.23796e+10){r+=-0.148519;}else{r+=-0.665682;}}}}else{if(f[14]<1.40325e+11){if(f[83]<0.626534){r+=0.757805;}else{r+=0.138164;}}else{r+=-0.501962;}}}else{if(f[28]<1.40522e+11){if(f[69]<2.09206){if(f[28]<1.40127e+11){if(f[84]<1.35){r+=-0.801737;}else{r+=0.54508;}}else{if(f[48]<0.159641){r+=-0.142419;}else{r+=-0.718287;}}}else{if(f[84]<1.5){r+=-0.723333;}else{r+=0.800686;}}}else{if(f[84]<1){r+=0.674277;}else{r+=-0.557254;}}}}else{if(f[84]<1.5){if(f[14]<1.40481e+11){if(f[42]<0.995765){if(f[28]<1.40501e+11){r+=-0.619106;}else{r+=0.686928;}}else{if(f[14]<1.40159e+11){if(f[66]<2.09706){r+=-1.30372;}else{r+=-0.4244;}}else{if(f[14]<1.40473e+11){r+=-0.663888;}else{r+=-1.36626;}}}}else{if(f[28]<1.40379e+11){if(f[84]<0.85){if(f[28]<1.40377e+11){r+=-0.681235;}else{r+=-1.39803;}}else{if(f[42]<0.996104){r+=-0.526791;}else{r+=0.658039;}}}else{if(f[14]<1.40632e+11){if(f[14]<1.40485e+11){r+=0.303662;}else{r+=0.670486;}}else{r+=-1.18407;}}}}else{if(f[68]<0.506422){if(f[0]<1.39992e+11){if(f[37]<1.37453e+08){if(f[27]<8.0991e+07){r+=0.413865;}else{r+=-0.673579;}}else{if(f[14]<1.41009e+11){r+=-0.266162;}else{r+=0.663261;}}}else{if(f[0]<1.40063e+11){if(f[84]<2){r+=0.707801;}else{r+=-0.318862;}}else{if(f[56]<1.00111){r+=-0.563364;}else{r+=0.761266;}}}}else{if(f[28]<1.4017e+11){if(f[84]<3.15){if(f[67]<0.563929){r+=-0.132723;}else{r+=0.683122;}}else{if(f[42]<0.997735){r+=-1.441;}else{r+=-0.621338;}}}else{if(f[84]<3.5){if(f[67]<0.690333){r+=0.507181;}else{r+=-1.11293;}}else{if(f[1]<7.32113e+08){r+=0.759368;}else{r+=-0.711386;}}}}}}}}}else{if(f[0]<1.39581e+11){if(f[28]<1.41235e+11){if(f[14]<1.47074e+11){if(f[78]<3.61733){if(f[30]<7.53795e+08){r+=-0.683798;}else{r+=-0.122607;}}else{if(f[44]<2.06072){r+=0.49351;}else{r+=-0.10917;}}}else{r+=0.461618;}}else{if(f[56]<1.03494){if(f[28]<1.41808e+11){if(f[84]<0.5){if(f[14]<1.43939e+11){r+=-0.675037;}else{r+=0.712483;}}else{if(f[69]<0.683362){r+=-0.678676;}else{if(f[28]<1.41278e+11){r+=-0.709674;}else{if(f[79]<0.912198){r+=-1.5151;}else{r+=-0.656366;}}}}}else{if(f[84]<2.65){if(f[14]<1.46972e+11){r+=-0.684802;}else{r+=-0.141861;}}else{r+=0.685202;}}}else{if(f[84]<1){r+=0.717363;}else{r+=-0.176823;}}}}else{if(f[38]<5.23839e+07){if(f[28]<1.42782e+11){if(f[28]<1.41224e+11){r+=-0.481512;}else{if(f[37]<2.12314e+07){r+=0.677791;}else{if(f[29]<2.80744e+06){r+=0.622516;}else{if(f[4]<1.6879e+07){r+=1.43159;}else{r+=0.508101;}}}}}else{r+=-0.648207;}}else{if(f[53]<0.307215){if(f[84]<6.60001){if(f[13]<9.99457e+07){r+=0.0871021;}else{r+=-0.648418;}}else{if(f[77]<0.82085){r+=0.0882307;}else{r+=0.734151;}}}else{if(f[71]<0.0701623){if(f[14]<1.45617e+11){r+=-0.593706;}else{r+=0.997956;}}else{if(f[28]<1.40672e+11){if(f[0]<1.39941e+11){if(f[3]<5.64495e+06){r+=-0.127229;}else{r+=-0.684408;}}else{if(f[14]<1.40858e+11){r+=-0.48138;}else{if(f[28]<1.40666e+11){r+=0.383046;}else{r+=1.44481;}}}}else{if(f[38]<5.45692e+07){if(f[28]<1.48776e+11){if(f[10]<1.79252e+08){r+=0.851206;}else{r+=-0.381613;}}else{r+=-0.573065;}}else{if(f[2]<1.64426e+09){if(f[37]<2.07966e+07){r+=-0.00441807;}else{r+=-0.682448;}}else{r+=0.183701;}}}}}}}}}else{if(f[83]<2.4587){if(f[0]<1.4016e+11){if(f[6]<8.4718e+06){if(f[14]<1.4029e+11){if(f[26]<4.01952e+08){if(f[28]<1.39941e+11){if(f[14]<1.40035e+11){if(f[30]<1.42783e+07){r+=-0.670482;}else{r+=0.306808;}}else{if(f[0]<1.4012e+11){r+=0.687082;}else{if(f[4]<3.83991e+06){r+=0.0544337;}else{r+=-0.530067;}}}}else{if(f[28]<1.40008e+11){if(f[0]<1.40145e+11){if(f[84]<3.15){r+=-0.674767;}else{r+=0.686782;}}else{if(f[84]<1.15){r+=0.808923;}else{r+=-0.385894;}}}else{if(f[14]<1.39853e+11){if(f[12]<2.21992e+08){r+=-0.738864;}else{r+=0.747431;}}else{if(f[0]<1.40094e+11){r+=-1.05247;}else{r+=-0.67906;}}}}}else{if(f[84]<2.85){if(f[0]<1.40118e+11){if(f[83]<0.896016){if(f[38]<8.98577e+08){r+=-0.621504;}else{r+=0.458453;}}else{if(f[14]<1.40029e+11){r+=-0.430227;}else{r+=0.733716;}}}else{if(f[28]<1.39931e+11){if(f[7]<1.04983e+08){r+=-0.464306;}else{r+=0.579551;}}else{if(f[84]<5.40001){r+=-0.682538;}else{r+=-0.818418;}}}}else{if(f[14]<1.40057e+11){r+=-0.650756;}else{if(f[41]<1.95515e+08){r+=-0.41352;}else{if(f[12]<4.65416e+08){r+=0.78025;}else{r+=0.684618;}}}}}}else{if(f[56]<1.0013){if(f[84]<1.15){if(f[79]<0.693382){if(f[2]<4.60522e+06){r+=-0.127057;}else{r+=-0.592424;}}else{r+=0.695296;}}else{r+=-0.696426;}}else{if(f[56]<1.00222){if(f[79]<0.836262){if(f[84]<0.85){r+=-0.591685;}else{if(f[17]<1.27502e+07){r+=0.688825;}else{r+=-0.0462817;}}}else{if(f[84]<1.8){if(f[10]<2.86971e+07){r+=0.493187;}else{r+=-0.882502;}}else{r+=0.676369;}}}else{if(f[79]<0.429576){r+=0.111718;}else{r+=-0.672285;}}}}}else{if(f[28]<1.40396e+11){if(f[14]<1.40214e+11){if(f[28]<1.39953e+11){if(f[84]<1.5){if(f[14]<1.39879e+11){if(f[84]<1){r+=-0.598848;}else{r+=0.668759;}}else{if(f[6]<9.31618e+06){r+=0.036187;}else{r+=-0.658135;}}}else{if(f[84]<1.85){if(f[67]<0.135577){r+=-0.00636139;}else{r+=0.683858;}}else{r+=-0.599594;}}}else{if(f[84]<2.85){if(f[28]<1.40008e+11){if(f[84]<5.40001){r+=-0.369037;}else{r+=0.575554;}}else{if(f[43]<11.0951){r+=-0.673079;}else{r+=0.0472017;}}}else{if(f[26]<3.73015e+08){if(f[14]<1.40105e+11){r+=-0.686562;}else{r+=0.524138;}}else{if(f[41]<3.8853e+08){r+=-0.019466;}else{r+=0.684069;}}}}}else{if(f[28]<1.40381e+11){if(f[56]<1.00246){if(f[84]<0.85){r+=-0.618523;}else{if(f[56]<1.0007){r+=-0.34705;}else{r+=0.677081;}}}else{if(f[28]<1.39907e+11){if(f[84]<1.85){r+=-0.810639;}else{r+=0.500558;}}else{if(f[37]<5.85981e+07){r+=-0.643534;}else{r+=0.108044;}}}}else{if(f[84]<1.15){if(f[84]<0.85){r+=-0.569386;}else{r+=0.684511;}}else{if(f[14]<1.40563e+11){r+=-0.725146;}else{if(f[84]<1.8){r+=-0.797527;}else{r+=0.511682;}}}}}}else{if(f[84]<6.60001){if(f[12]<2.13781e+10){if(f[28]<1.404e+11){if(f[84]<1.15){if(f[0]<1.40129e+11){r+=0.53456;}else{r+=-0.103546;}}else{r+=-0.697015;}}else{if(f[0]<1.40094e+11){if(f[29]<8.97745e+06){r+=-1.35693;}else{r+=-0.605911;}}else{if(f[3]<556272){r+=-0.141552;}else{r+=-0.678875;}}}}else{r+=0.371373;}}else{if(f[0]<1.40123e+11){if(f[28]<1.41952e+11){if(f[28]<1.40905e+11){r+=-0.049126;}else{r+=-0.667475;}}else{r+=0.418224;}}else{if(f[51]<1.53161){if(f[59]<0.239697){r+=0.190805;}else{r+=0.711167;}}else{r+=-0.217215;}}}}}}else{if(f[0]<1.40245e+11){if(f[84]<2.85){if(f[28]<1.40158e+11){if(f[65]<0.890389){if(f[28]<1.3959e+11){if(f[84]<1){if(f[14]<1.39727e+11){r+=-0.397677;}else{r+=-1.46797;}}else{if(f[14]<1.39872e+11){r+=0.725361;}else{r+=-0.212014;}}}else{if(f[28]<1.40131e+11){if(f[56]<1.00335){r+=-0.690486;}else{r+=0.226567;}}else{if(f[8]<1.19691e+08){r+=0.694044;}else{r+=-0.648864;}}}}else{if(f[84]<0.85){if(f[14]<1.39953e+11){if(f[14]<1.39727e+11){r+=-0.595094;}else{r+=-1.56196;}}else{if(f[28]<1.40102e+11){r+=-0.711702;}else{r+=0.527616;}}}else{if(f[84]<2.5){if(f[28]<1.40126e+11){r+=0.356536;}else{r+=0.709669;}}else{if(f[1]<1.32882e+08){r+=-0.679333;}else{r+=0.498355;}}}}}else{if(f[28]<1.5436e+11){if(f[28]<1.40178e+11){if(f[0]<1.40163e+11){r+=-0.696951;}else{if(f[84]<1.85){r+=-1.57464;}else{r+=-0.6759;}}}else{if(f[28]<1.40388e+11){if(f[83]<1.03969){r+=-0.600423;}else{r+=0.15109;}}else{if(f[83]<2.35618){r+=-0.688836;}else{r+=-0.952251;}}}}else{r+=0.682911;}}}else{if(f[42]<1.00136){if(f[0]<1.40176e+11){if(f[28]<1.40144e+11){if(f[17]<5.78498e+07){if(f[71]<0.0695386){r+=-0.42684;}else{r+=0.674153;}}else{if(f[84]<6.60001){r+=0.577376;}else{r+=-0.829675;}}}else{if(f[14]<1.40561e+11){if(f[39]<6.0131e+07){r+=-0.73544;}else{r+=-1.48519;}}else{if(f[84]<6.60001){r+=-0.827779;}else{r+=0.663877;}}}}else{if(f[83]<0.207652){if(f[77]<0.493235){r+=-1.02389;}else{r+=0.127175;}}else{if(f[28]<1.3977e+11){r+=-0.608732;}else{if(f[67]<1.34347){r+=0.666197;}else{r+=0.535998;}}}}}else{if(f[1]<2.51736e+08){r+=-0.682199;}else{r+=-0.179476;}}}}else{if(f[0]<1.41627e+11){if(f[28]<1.42029e+11){if(f[84]<0.5){if(f[84]<0.60001){if(f[13]<5.75238e+08){if(f[63]<11.1303){r+=-0.691611;}else{r+=0.189777;}}else{if(f[23]<3.47032e+07){r+=0.608797;}else{r+=-0.628038;}}}else{if(f[66]<0.21343){if(f[0]<1.40325e+11){r+=0.688453;}else{r+=-0.6571;}}else{if(f[28]<1.41951e+11){r+=-0.815603;}else{r+=0.741616;}}}}else{if(f[28]<1.40017e+11){if(f[52]<1.4751){if(f[28]<1.3995e+11){r+=-0.0156881;}else{r+=0.461342;}}else{if(f[42]<1.00925){r+=-0.514456;}else{r+=0.632436;}}}else{if(f[84]<2.5){if(f[10]<1.54993e+08){r+=0.160644;}else{r+=-0.388064;}}else{if(f[25]<1.29541e+08){r+=-0.923019;}else{r+=-0.643896;}}}}}else{if(f[84]<0.85){if(f[14]<1.39296e+11){r+=-0.686383;}else{if(f[14]<1.40582e+11){if(f[28]<1.70922e+11){r+=0.911371;}else{r+=0.700918;}}else{if(f[56]<0.982734){r+=-0.670614;}else{r+=1.01345;}}}}else{if(f[84]<1.15){r+=-0.685683;}else{if(f[23]<9.4401e+07){r+=-0.746341;}else{if(f[83]<0.623738){r+=-0.760387;}else{r+=-1.12555;}}}}}}else{if(f[79]<1.45818){if(f[84]<2.5){if(f[56]<1.00805){if(f[0]<1.46414e+11){if(f[0]<1.42614e+11){r+=0.381354;}else{r+=0.690691;}}else{if(f[14]<1.39844e+11){r+=0.686201;}else{r+=-1.07504;}}}else{if(f[14]<1.4127e+11){if(f[17]<1.01466e+07){r+=-0.646469;}else{r+=-1.41638;}}else{if(f[14]<1.43813e+11){r+=-0.680957;}else{r+=0.444859;}}}}else{if(f[14]<1.40166e+11){if(f[28]<1.3999e+11){if(f[66]<0.651064){r+=-0.491025;}else{r+=0.678414;}}else{if(f[14]<1.39138e+11){r+=0.177528;}else{r+=-0.682948;}}}else{r+=-0.684008;}}}else{if(f[56]<0.987746){if(f[8]<4.96503e+07){r+=0.0789061;}else{r+=0.707838;}}else{if(f[84]<0.85){if(f[0]<1.4197e+11){if(f[9]<1.89701e+08){r+=-1.16774;}else{r+=-0.671459;}}else{if(f[0]<1.62954e+11){r+=0.680067;}else{r+=-0.744233;}}}else{if(f[14]<1.39655e+11){if(f[84]<1.85){r+=-1.56908;}else{r+=-0.652694;}}else{if(f[78]<1.09906){r+=-0.379172;}else{r+=-0.700015;}}}}}}}}}else{if(f[14]<1.4201e+11){if(f[69]<0.614743){if(f[84]<1.15){if(f[84]<0.85){if(f[0]<1.40264e+11){if(f[0]<1.40237e+11){if(f[0]<1.40099e+11){if(f[0]<1.40094e+11){r+=0.647637;}else{r+=1.24696;}}else{if(f[83]<9.80759e+08){r+=-0.713206;}else{r+=0.335397;}}}else{if(f[56]<0.999386){if(f[37]<8.04396e+07){r+=0.771726;}else{r+=1.14871;}}else{r+=-0.651284;}}}else{if(f[28]<1.42758e+11){if(f[0]<2.43484e+11){if(f[55]<1.33701){r+=-0.695392;}else{r+=-0.531039;}}else{r+=0.519324;}}else{if(f[0]<1.41878e+11){r+=0.169475;}else{r+=1.15856;}}}}else{if(f[29]<1.05239e+07){if(f[56]<1.00162){if(f[28]<1.40082e+11){if(f[55]<0.864755){r+=0.816858;}else{r+=1.30352;}}else{if(f[13]<1.84886e+08){r+=-0.704333;}else{r+=-0.115683;}}}else{if(f[14]<1.40282e+11){if(f[55]<0.862763){r+=-1.216;}else{r+=-0.632976;}}else{r+=0.384919;}}}else{if(f[42]<1.00138){if(f[0]<1.40128e+11){r+=0.424366;}else{if(f[1]<6.26908e+06){r+=-0.0103457;}else{r+=-0.735019;}}}else{if(f[52]<1.12396){r+=0.961407;}else{r+=0.0886105;}}}}}else{if(f[41]<5.44118e+08){if(f[0]<1.40213e+11){if(f[84]<2.15){if(f[28]<1.3971e+11){r+=0.711913;}else{if(f[39]<5.35801e+08){r+=-0.97608;}else{r+=0.128657;}}}else{if(f[56]<1.00115){if(f[53]<0.736438){r+=-0.190183;}else{r+=-0.776064;}}else{if(f[14]<1.40278e+11){r+=-0.457811;}else{r+=0.880466;}}}}else{if(f[37]<1.7195e+08){if(f[12]<1.1759e+08){if(f[39]<2.81173e+08){r+=-0.66425;}else{r+=0.238059;}}else{if(f[0]<1.4103e+11){r+=-0.659044;}else{r+=0.734078;}}}else{if(f[42]<1.00274){if(f[13]<1.0688e+08){r+=0.203689;}else{r+=-0.663518;}}else{if(f[42]<1.005){r+=0.860469;}else{r+=0.0362312;}}}}}else{if(f[7]<1.01251e+06){if(f[84]<2.65){if(f[0]<1.40214e+11){r+=-0.190183;}else{r+=0.672861;}}else{r+=-0.655267;}}else{if(f[41]<6.81091e+08){if(f[10]<1.78149e+08){if(f[67]<0.212101){r+=0.217491;}else{r+=-0.657994;}}else{if(f[42]<1.00227){r+=-0.604491;}else{r+=0.293092;}}}else{if(f[0]<1.40117e+11){if(f[56]<1.00227){r+=-0.650935;}else{r+=0.593696;}}else{if(f[14]<1.39007e+11){r+=-0.489898;}else{r+=-0.68381;}}}}}}}else{if(f[84]<0.5){if(f[42]<0.998496){if(f[0]<1.40481e+11){if(f[53]<0.507356){if(f[55]<0.393926){r+=0.688273;}else{if(f[69]<0.819425){r+=0.898002;}else{r+=0.632585;}}}else{if(f[84]<0.60001){r+=-0.950757;}else{r+=0.73763;}}}else{r+=-0.62834;}}else{if(f[12]<420378){r+=0.672971;}else{if(f[0]<1.40264e+11){if(f[26]<3.35087e+08){r+=0.579602;}else{if(f[0]<1.40234e+11){r+=-0.583417;}else{r+=0.416051;}}}else{if(f[0]<2.39642e+11){if(f[56]<0.990141){r+=-0.288821;}else{r+=-0.693892;}}else{r+=-0.151777;}}}}}else{if(f[83]<7.05892e+08){if(f[14]<1.39035e+11){if(f[56]<0.991025){if(f[83]<2.5289){r+=0.0863456;}else{if(f[52]<1.34478){r+=-0.72698;}else{r+=-0.15301;}}}else{if(f[84]<1.5){r+=-0.730711;}else{if(f[68]<0.981203){r+=0.715879;}else{r+=-0.41792;}}}}else{if(f[67]<0.395421){if(f[42]<1.00587){if(f[84]<1.5){r+=0.640077;}else{r+=-0.498594;}}else{r+=-0.636441;}}else{if(f[42]<1.96389){if(f[42]<0.996912){r+=-0.00688991;}else{r+=-0.646015;}}else{r+=0.643365;}}}}else{if(f[84]<2.65){if(f[84]<1.2){r+=-0.653609;}else{r+=0.682726;}}else{r+=-0.673921;}}}}}else{if(f[84]<0.8){if(f[28]<1.42503e+11){r+=-0.580768;}else{r+=0.684848;}}else{if(f[12]<5.90326e+09){r+=-0.679865;}else{r+=-0.0958215;}}}}}}else{if(f[28]<1.79213e+11){if(f[68]<7.76073){if(f[28]<1.40629e+11){if(f[42]<0.998356){if(f[84]<1.5){if(f[14]<1.4028e+11){if(f[14]<1.40224e+11){r+=-0.647574;}else{r+=-0.141765;}}else{if(f[14]<1.40623e+11){if(f[52]<0.649311){r+=-0.30959;}else{if(f[42]<0.99737){if(f[28]<1.40398e+11){r+=0.71668;}else{r+=-0.0927558;}}else{r+=0.919421;}}}else{if(f[55]<1.05){r+=-0.789347;}else{r+=0.410815;}}}}else{if(f[37]<9.22207e+06){if(f[53]<0.817843){if(f[84]<2.15){r+=0.547494;}else{if(f[28]<1.40063e+11){r+=-0.578887;}else{r+=0.424944;}}}else{if(f[4]<4.45072e+07){if(f[0]<1.40097e+11){if(f[53]<1.0084){r+=-0.390314;}else{r+=-0.671672;}}else{r+=-0.938752;}}else{r+=0.303625;}}}else{if(f[48]<3.97146){if(f[84]<2.15){if(f[27]<9.36685e+07){r+=-0.512814;}else{if(f[42]<0.993474){r+=-0.0128857;}else{r+=0.666611;}}}else{if(f[14]<1.40339e+11){r+=0.526139;}else{r+=-0.701772;}}}else{if(f[27]<5.07053e+08){r+=0.128073;}else{r+=-0.755464;}}}}}else{if(f[84]<3.5){if(f[27]<1.82385e+09){if(f[28]<1.40628e+11){if(f[28]<1.39517e+11){r+=0.150433;}else{if(f[26]<3.50581e+07){r+=-0.124136;}else{if(f[69]<8.94593){r+=-0.691091;}else{r+=-0.332134;}}}}else{r+=1.16774;}}else{if(f[78]<1.06654){if(f[84]<1.5){if(f[67]<10.49){if(f[10]<5.82355e+09){r+=0.759115;}else{r+=0.0916699;}}else{r+=-0.0868121;}}else{if(f[13]<2.23416e+09){if(f[41]<9.71707e+08){r+=-0.619633;}else{r+=-0.0340011;}}else{r+=0.215242;}}}else{if(f[25]<7.32322e+08){r+=0.256378;}else{if(f[3]<2.05201e+06){r+=-0.110944;}else{r+=-0.704959;}}}}}else{if(f[56]<1.00152){if(f[14]<1.40062e+11){if(f[6]<1.58477e+07){r+=-0.662419;}else{r+=0.0213933;}}else{if(f[21]<6.39911e+07){r+=-0.561494;}else{if(f[14]<1.40311e+11){if(f[3]<1.68974e+07){r+=0.670848;}else{r+=0.927465;}}else{r+=-0.256208;}}}}else{if(f[25]<7.78029e+08){if(f[56]<1.00171){r+=-0.183675;}else{if(f[83]<0.620767){r+=-0.651843;}else{if(f[0]<1.40112e+11){r+=-0.502388;}else{r+=-1.07682;}}}}else{r+=-0.0800518;}}}}}else{if(f[23]<3.93389e+10){if(f[17]<6.15443e+08){if(f[38]<9.55717e+09){if(f[66]<16.0758){if(f[29]<4.45101e+08){if(f[23]<3.31354e+10){r+=-0.690143;}else{r+=-0.164953;}}else{r+=-0.108488;}}else{if(f[14]<1.4034e+11){r+=-0.471992;}else{r+=0.248366;}}}else{r+=0.030389;}}else{r+=0.0748339;}}else{if(f[28]<1.40989e+11){r+=1.03275;}else{r+=-0.407807;}}}}else{if(f[65]<1.11242){if(f[51]<0.938652){r+=-0.624842;}else{if(f[14]<1.40405e+11){r+=0.651791;}else{r+=0.0622276;}}}else{if(f[13]<1.15212e+10){if(f[53]<2.78192){if(f[54]<0.000493024){r+=0.245341;}else{if(f[28]<1.40118e+11){if(f[0]<1.40015e+11){if(f[84]<0.5){r+=-0.665449;}else{if(f[84]<0.85){r+=0.651613;}else{r+=0.00890623;}}}else{if(f[65]<1.91181){if(f[0]<1.40156e+11){r+=0.495808;}else{r+=-0.597844;}}else{if(f[68]<7.87565){r+=-0.11995;}else{r+=-0.684429;}}}}else{if(f[7]<4.38725e+09){if(f[28]<1.40665e+11){if(f[84]<1.15){r+=-0.681233;}else{r+=-0.796779;}}else{if(f[41]<1.11608e+09){r+=-0.689795;}else{r+=-0.228554;}}}else{if(f[53]<0.947111){r+=0.375691;}else{r+=-0.424937;}}}}}else{if(f[10]<1.11846e+09){r+=-0.723395;}else{if(f[84]<2){if(f[0]<1.40162e+11){r+=-0.487473;}else{if(f[14]<1.40619e+11){r+=-1.56419;}else{r+=-0.376733;}}}else{r+=-0.682292;}}}}else{if(f[34]<3.44237e+07){r+=-0.604631;}else{r+=0.713873;}}}}}else{if(f[14]<1.40084e+11){r+=0.702725;}else{r+=-0.674799;}}}} return r;}
    static double p23(double[] f){double r=0;if(f[28]<1.39435e+11){if(f[9]<2.81865e+10){if(f[28]<1.39418e+11){if(f[13]<3.99135e+10){if(f[56]<1.00196){if(f[56]<1.00147){if(f[43]<3.42021){r+=-0.558603;}else{r+=-0.102977;}}else{if(f[46]<0.898581){r+=-0.148892;}else{r+=0.567448;}}}else{if(f[42]<1.00952){r+=-0.592761;}else{if(f[42]<1.01077){if(f[27]<2.34866e+08){if(f[14]<1.40273e+11){r+=0.671106;}else{r+=0.171119;}}else{r+=-0.433393;}}else{if(f[42]<1.01091){r+=-0.177929;}else{r+=-0.592425;}}}}}else{if(f[14]<1.40268e+11){r+=-0.202324;}else{r+=0.349818;}}}else{if(f[56]<1.00321){if(f[14]<1.39837e+11){if(f[0]<1.40006e+11){if(f[14]<1.39655e+11){r+=0.474582;}else{r+=-0.171664;}}else{if(f[14]<1.39823e+11){if(f[19]<5.3994e+06){r+=-0.148259;}else{r+=-0.58735;}}else{if(f[20]<1.84664e+07){if(f[52]<1.16509){r+=-0.0690122;}else{r+=-0.342069;}}else{if(f[44]<1.28268){r+=0.368332;}else{r+=0.00892952;}}}}}else{if(f[0]<1.40349e+11){if(f[9]<1.30489e+08){r+=-0.05786;}else{if(f[41]<1.08786e+08){r+=0.116752;}else{r+=0.649537;}}}else{r+=-0.0683217;}}}else{r+=-0.588639;}}}else{if(f[14]<1.39551e+11){if(f[53]<0.793571){r+=-0.0910539;}else{r+=-0.517203;}}else{if(f[38]<7.23374e+10){if(f[40]<1.49719e+10){r+=-0.202903;}else{if(f[19]<1.95657e+09){r+=0.710303;}else{r+=0.114093;}}}else{r+=-0.331203;}}}}else{if(f[67]<6.02352){if(f[0]<1.40096e+11){if(f[28]<1.40665e+11){if(f[28]<1.4e+11){if(f[14]<1.3964e+11){if(f[70]<1.00212){if(f[4]<2.08614e+08){if(f[28]<1.39877e+11){r+=0.592376;}else{r+=-0.0804289;}}else{r+=-0.378712;}}else{r+=-0.578946;}}else{if(f[14]<1.40363e+11){if(f[0]<1.40077e+11){if(f[28]<1.3977e+11){if(f[14]<1.39668e+11){r+=0.594515;}else{if(f[40]<8.63799e+09){if(f[66]<3.13136){r+=-0.581449;}else{r+=0.419582;}}else{if(f[17]<1.50395e+08){r+=0.0765125;}else{r+=0.671235;}}}}else{if(f[56]<1.003){if(f[26]<2.69714e+09){if(f[57]<0.0392616){r+=-0.195332;}else{r+=-0.588349;}}else{if(f[0]<1.39982e+11){r+=-0.358777;}else{r+=0.731274;}}}else{if(f[38]<1.54439e+07){if(f[20]<6.75608e+06){r+=0.103392;}else{r+=-0.475569;}}else{if(f[28]<1.3987e+11){r+=0.357815;}else{r+=-0.287197;}}}}}else{if(f[63]<0.840293){if(f[34]<4.71666e+06){r+=-0.270121;}else{if(f[54]<1.74093){if(f[29]<9.6916e+07){r+=0.726343;}else{r+=0.603567;}}else{r+=0.077094;}}}else{if(f[56]<1.00332){if(f[31]<4.17741e+07){if(f[28]<1.39994e+11){r+=-0.618782;}else{r+=0.0351386;}}else{r+=0.379463;}}else{if(f[44]<1.92597){r+=0.578482;}else{r+=0.00962286;}}}}}else{if(f[28]<1.39824e+11){if(f[42]<0.997529){if(f[29]<1.2737e+08){if(f[78]<0.191995){r+=-0.172917;}else{r+=-0.591548;}}else{if(f[14]<1.41003e+11){if(f[30]<1.80962e+08){r+=0.557238;}else{r+=-0.0957877;}}else{r+=-0.440092;}}}else{r+=0.356637;}}else{if(f[28]<1.39969e+11){if(f[38]<1.33313e+07){if(f[0]<1.4e+11){if(f[28]<1.39854e+11){r+=-0.181553;}else{r+=0.428737;}}else{if(f[64]<1.84039){r+=-0.0833107;}else{r+=-0.514168;}}}else{if(f[41]<5.26649e+09){if(f[56]<1.00664){r+=0.420082;}else{r+=-0.617763;}}else{if(f[39]<1.03151e+10){r+=-0.593926;}else{r+=0.463056;}}}}else{if(f[68]<0.831614){if(f[26]<2.02565e+08){if(f[0]<1.39497e+11){r+=0.559396;}else{r+=0.141293;}}else{r+=-0.398309;}}else{if(f[42]<0.968939){r+=0.435101;}else{if(f[20]<1.1304e+08){r+=-0.606131;}else{r+=0.0208717;}}}}}}}}else{if(f[14]<1.40154e+11){if(f[36]<3.18446e+07){if(f[38]<5.21355e+07){if(f[38]<4.70749e+07){if(f[55]<0.456472){if(f[26]<6.26444e+07){r+=0.566151;}else{r+=-0.441989;}}else{if(f[0]<1.38349e+11){r+=0.669637;}else{if(f[41]<7.15546e+07){r+=-0.574006;}else{r+=-0.769722;}}}}else{if(f[0]<1.40007e+11){if(f[70]<1.0014){if(f[9]<4.24484e+06){r+=-0.585534;}else{r+=0.603776;}}else{r+=-0.891672;}}else{if(f[14]<1.40069e+11){if(f[14]<1.39978e+11){r+=-0.592806;}else{r+=0.460732;}}else{if(f[13]<8.42089e+07){r+=0.526735;}else{r+=-0.656998;}}}}}else{if(f[28]<1.40013e+11){if(f[0]<1.4003e+11){if(f[3]<7.08379e+06){r+=-0.69423;}else{r+=0.335424;}}else{if(f[14]<1.40141e+11){if(f[13]<1.22827e+08){r+=-0.483188;}else{r+=0.607351;}}else{if(f[0]<1.40069e+11){r+=0.454638;}else{r+=-0.583807;}}}}else{if(f[0]<1.40013e+11){if(f[28]<1.40049e+11){if(f[14]<1.40024e+11){r+=-0.702834;}else{r+=0.603697;}}else{if(f[28]<1.40253e+11){r+=0.552368;}else{r+=-0.876657;}}}else{if(f[14]<1.40023e+11){if(f[56]<0.999487){r+=-0.64148;}else{r+=0.467685;}}else{if(f[14]<1.40105e+11){r+=0.468895;}else{r+=-0.410625;}}}}}}else{if(f[81]<1.3869){if(f[28]<1.40178e+11){if(f[56]<1.00044){if(f[78]<0.516053){if(f[14]<1.40022e+11){r+=-0.473197;}else{r+=0.406543;}}else{if(f[0]<1.40058e+11){r+=-0.718268;}else{r+=-0.0218542;}}}else{if(f[0]<1.40064e+11){if(f[38]<6.23704e+07){r+=-0.117738;}else{r+=0.606248;}}else{if(f[1]<3.36963e+07){r+=-0.598265;}else{r+=0.341404;}}}}else{if(f[64]<0.819952){if(f[22]<1.35027e+08){if(f[27]<7.71668e+07){r+=-0.630468;}else{r+=0.208191;}}else{if(f[36]<5.12146e+08){r+=-0.913844;}else{r+=-0.584673;}}}else{if(f[28]<1.40431e+11){if(f[52]<1.50076){r+=0.585987;}else{r+=-0.398979;}}else{if(f[67]<0.781859){r+=-0.063184;}else{r+=-0.939865;}}}}}else{if(f[28]<1.40112e+11){if(f[0]<1.40069e+11){if(f[67]<0.920729){if(f[14]<1.40146e+11){r+=-0.640568;}else{r+=-0.0872101;}}else{if(f[9]<1.54132e+08){r+=0.714938;}else{r+=-0.199903;}}}else{if(f[14]<1.40036e+11){r+=-0.17052;}else{if(f[52]<1.53528){r+=0.599706;}else{r+=0.0989481;}}}}else{if(f[27]<7.06912e+07){if(f[28]<1.40402e+11){if(f[34]<5.68753e+06){r+=-1.14964;}else{r+=-0.716401;}}else{if(f[49]<0.526422){r+=0.0697202;}else{r+=0.532134;}}}else{if(f[53]<0.157542){if(f[53]<0.0916916){r+=-0.831997;}else{r+=-1.70323;}}else{if(f[42]<1.00173){r+=0.472738;}else{r+=-0.605001;}}}}}}}else{if(f[0]<1.39783e+11){if(f[0]<1.39727e+11){if(f[13]<3.59106e+10){if(f[56]<0.997441){r+=0.165067;}else{if(f[9]<4.68665e+10){if(f[72]<0.00740686){r+=-0.0825941;}else{r+=-0.589649;}}else{r+=0.0105823;}}}else{r+=0.406804;}}else{if(f[28]<1.40561e+11){if(f[28]<1.40148e+11){if(f[28]<1.4008e+11){if(f[56]<1.00176){r+=0.036674;}else{r+=-0.574389;}}else{if(f[78]<0.388054){r+=0.548703;}else{r+=-0.241417;}}}else{if(f[54]<0.563492){r+=-0.0231403;}else{r+=-0.605985;}}}else{if(f[14]<1.40587e+11){if(f[0]<1.39741e+11){r+=1.6336;}else{r+=0.587302;}}else{if(f[80]<2.22806){r+=-0.502461;}else{r+=0.0165796;}}}}}else{if(f[14]<1.40213e+11){if(f[63]<0.402878){if(f[10]<7.06405e+07){if(f[26]<1.86456e+07){r+=-0.162327;}else{r+=-0.625782;}}else{if(f[70]<1.00117){if(f[80]<6.52654){r+=0.607525;}else{r+=-0.514214;}}else{if(f[18]<3.58585e+07){r+=-0.505176;}else{r+=0.552399;}}}}else{if(f[38]<9.45455e+08){if(f[0]<1.40059e+11){if(f[27]<8.34125e+07){r+=-0.110398;}else{r+=-0.612055;}}else{if(f[28]<1.40133e+11){r+=-0.856517;}else{r+=-0.365197;}}}else{if(f[68]<0.516852){r+=-0.401756;}else{if(f[26]<4.04678e+09){r+=0.61156;}else{r+=-0.107877;}}}}}else{if(f[28]<1.40555e+11){if(f[28]<1.40404e+11){if(f[68]<0.416551){if(f[0]<1.4004e+11){r+=-0.559513;}else{r+=0.429583;}}else{if(f[0]<1.39861e+11){r+=0.0796739;}else{r+=0.434342;}}}else{if(f[70]<1.00486){if(f[38]<4.03169e+09){r+=-0.797688;}else{r+=0.186671;}}else{r+=0.519999;}}}else{if(f[42]<0.996661){if(f[1]<3.23634e+08){if(f[78]<0.204106){r+=0.198418;}else{r+=0.620567;}}else{if(f[35]<2.39745e+08){r+=-0.195074;}else{r+=-1.08576;}}}else{if(f[34]<3.15394e+07){r+=-0.783744;}else{r+=0.038286;}}}}}}}}else{if(f[0]<1.39585e+11){if(f[28]<1.41236e+11){if(f[14]<1.47067e+11){if(f[29]<1.63447e+09){if(f[8]<5.04559e+10){r+=-0.592976;}else{r+=-0.131057;}}else{if(f[8]<1.75012e+10){r+=0.631043;}else{r+=-0.409019;}}}else{r+=0.466339;}}else{if(f[56]<1.03494){if(f[28]<1.41826e+11){if(f[42]<0.958872){if(f[13]<1.8684e+08){if(f[0]<1.39531e+11){if(f[40]<7.12055e+07){r+=-0.559201;}else{r+=-0.195588;}}else{if(f[13]<1.46225e+08){r+=-0.0576028;}else{r+=0.405189;}}}else{if(f[41]<1.14157e+08){r+=-0.138118;}else{if(f[66]<2.16568){r+=0.0943815;}else{r+=0.599969;}}}}else{if(f[68]<0.814671){r+=-0.597211;}else{r+=-0.694161;}}}else{if(f[70]<1.02148){if(f[0]<1.39353e+11){r+=0.593557;}else{r+=0.0995406;}}else{r+=-0.595689;}}}else{r+=0.605606;}}}else{if(f[38]<5.17468e+07){if(f[28]<1.42764e+11){if(f[66]<3.86442){r+=0.639052;}else{r+=-0.270688;}}else{r+=-0.541189;}}else{if(f[70]<1.00459){if(f[14]<1.40992e+11){r+=-0.538466;}else{if(f[55]<0.435603){r+=0.539962;}else{r+=1.07473;}}}else{if(f[53]<0.317982){if(f[66]<1.23101){if(f[28]<1.41193e+11){r+=0.686177;}else{r+=-0.351418;}}else{if(f[53]<0.303403){r+=-0.559152;}else{r+=-0.104829;}}}else{if(f[38]<5.59961e+07){if(f[0]<1.39666e+11){if(f[72]<0.0941307){r+=0.665783;}else{r+=0.0831978;}}else{if(f[14]<1.40171e+11){r+=0.517027;}else{r+=-0.55495;}}}else{if(f[1]<3.74897e+09){if(f[32]<199079){r+=0.0400562;}else{if(f[0]<1.39591e+11){r+=-0.352431;}else{r+=-0.58859;}}}else{r+=0.246465;}}}}}}}}else{if(f[68]<0.458776){if(f[28]<1.43873e+11){if(f[28]<1.39879e+11){if(f[40]<5.35968e+08){if(f[56]<1.00321){if(f[14]<1.39739e+11){if(f[8]<2.33682e+08){if(f[81]<3.19187){r+=-0.632762;}else{r+=-0.107453;}}else{if(f[56]<0.994897){r+=0.535797;}else{if(f[11]<1.17844e+08){r+=0.30402;}else{r+=-0.545401;}}}}else{if(f[41]<7.27379e+08){if(f[0]<1.40286e+11){if(f[28]<1.39463e+11){r+=0.0223135;}else{r+=0.591123;}}else{if(f[28]<1.39457e+11){r+=0.421753;}else{r+=0.853363;}}}else{if(f[35]<1.21732e+08){if(f[38]<8.06579e+08){r+=-0.518231;}else{r+=-0.152762;}}else{if(f[56]<1.00273){r+=0.403218;}else{r+=-0.00807141;}}}}}else{if(f[42]<1.00869){if(f[42]<1.00654){if(f[63]<3.40791){r+=-0.634179;}else{r+=-0.179903;}}else{r+=-1.04738;}}else{r+=0.182601;}}}else{if(f[14]<1.39868e+11){if(f[52]<1.11393){if(f[14]<1.3908e+11){r+=-0.510687;}else{r+=-0.00698707;}}else{if(f[36]<2.0385e+08){r+=0.0236016;}else{if(f[26]<2.38042e+08){r+=0.891392;}else{r+=0.445184;}}}}else{if(f[54]<1.83854){if(f[42]<1.00943){if(f[42]<1.00608){if(f[70]<0.997565){r+=-0.632817;}else{r+=-0.204445;}}else{if(f[41]<5.50439e+08){r+=-1.05562;}else{r+=-0.672274;}}}else{r+=0.736796;}}else{if(f[56]<1.00286){if(f[81]<2.87908){if(f[14]<1.40244e+11){r+=0.534714;}else{r+=-0.0676672;}}else{if(f[8]<2.2904e+08){r+=-0.0655755;}else{r+=-0.562015;}}}else{if(f[50]<4.45089){r+=-0.610898;}else{r+=0.17066;}}}}}}else{if(f[14]<1.40158e+11){if(f[11]<7.97203e+07){if(f[0]<1.40261e+11){if(f[27]<1.86955e+08){if(f[0]<1.401e+11){r+=0.601741;}else{if(f[39]<3.29184e+08){r+=-0.630432;}else{r+=-0.0101289;}}}else{if(f[14]<1.40011e+11){if(f[55]<0.863313){r+=0.603157;}else{r+=0.860725;}}else{if(f[39]<3.3528e+08){r+=-0.492436;}else{r+=0.266751;}}}}else{if(f[50]<0.127595){if(f[0]<1.45229e+11){r+=0.161175;}else{r+=-0.304208;}}else{if(f[0]<1.40263e+11){if(f[14]<1.40008e+11){r+=0.37743;}else{r+=-0.539166;}}else{if(f[56]<1.00118){r+=-0.624595;}else{r+=-0.198268;}}}}}else{if(f[78]<0.13791){if(f[0]<1.4026e+11){if(f[14]<1.40011e+11){if(f[15]<1.60306e+07){r+=0.78977;}else{r+=0.16427;}}else{r+=-0.163176;}}else{if(f[0]<1.40264e+11){if(f[28]<1.40147e+11){r+=-0.412906;}else{r+=0.357902;}}else{if(f[57]<4.70051){r+=-0.595729;}else{r+=-0.0684091;}}}}else{if(f[81]<0.982752){if(f[70]<1.00046){if(f[14]<1.40118e+11){r+=-0.576133;}else{r+=0.373386;}}else{if(f[14]<1.39765e+11){r+=0.969119;}else{r+=-0.61836;}}}else{if(f[70]<0.983178){if(f[9]<1.73975e+08){r+=-0.540765;}else{r+=0.576926;}}else{if(f[41]<5.98821e+08){r+=-0.703613;}else{r+=-0.587817;}}}}}}else{if(f[14]<1.4062e+11){if(f[0]<1.40715e+11){if(f[39]<2.63161e+08){if(f[56]<1.0011){if(f[28]<1.40174e+11){r+=-0.180279;}else{r+=-0.610934;}}else{if(f[13]<7.39573e+07){r+=-0.670793;}else{r+=0.364783;}}}else{if(f[40]<8.20827e+08){if(f[14]<1.40597e+11){r+=-0.595902;}else{r+=0.186074;}}else{if(f[14]<1.40186e+11){r+=0.274238;}else{r+=-0.56552;}}}}else{if(f[52]<1.04568){if(f[0]<1.41372e+11){if(f[70]<0.992648){r+=0.451997;}else{r+=-0.246697;}}else{if(f[4]<1.71276e+07){r+=-0.609212;}else{r+=-1.20123;}}}else{if(f[7]<9.06609e+07){if(f[23]<6.38616e+07){r+=-0.566399;}else{r+=-0.137375;}}else{if(f[14]<1.40563e+11){r+=0.833686;}else{r+=-0.383857;}}}}}else{if(f[0]<1.40448e+11){if(f[41]<2.62509e+09){if(f[14]<1.4069e+11){if(f[76]<0.747394){r+=-0.627107;}else{r+=-0.03195;}}else{if(f[54]<1.68741){r+=0.914262;}else{r+=-0.430695;}}}else{if(f[4]<9.5633e+08){r+=-0.647238;}else{r+=-0.063677;}}}else{if(f[81]<1.47352){if(f[42]<1.00462){if(f[28]<1.41894e+11){r+=-0.633245;}else{r+=0.0112466;}}else{if(f[28]<1.40146e+11){r+=-0.469472;}else{r+=0.637638;}}}else{if(f[14]<1.4094e+11){if(f[28]<1.4099e+11){r+=-0.605048;}else{r+=-0.241981;}}else{if(f[14]<1.40941e+11){r+=-0.0609274;}else{r+=-0.601071;}}}}}}}}else{if(f[28]<1.45266e+11){if(f[67]<0.201834){if(f[28]<1.44711e+11){if(f[39]<1.08973e+09){r+=0.199515;}else{if(f[10]<2.73557e+08){r+=-0.0774899;}else{if(f[81]<2.11182){r+=-0.16352;}else{r+=-0.577112;}}}}else{if(f[63]<0.797826){r+=-0.457478;}else{if(f[39]<1.66837e+09){if(f[6]<1.14032e+08){r+=0.529204;}else{r+=0.0484096;}}else{r+=-0.136469;}}}}else{if(f[34]<2.31544e+08){if(f[41]<3.77916e+09){if(f[28]<1.4487e+11){if(f[0]<1.41592e+11){r+=-0.348799;}else{if(f[20]<6.99334e+07){r+=0.445883;}else{r+=0.125281;}}}else{r+=-0.605685;}}else{if(f[7]<3.03672e+07){if(f[56]<0.986301){r+=-0.500827;}else{r+=0.228776;}}else{if(f[20]<4.10976e+08){if(f[63]<0.669709){r+=0.294384;}else{r+=0.570113;}}else{if(f[36]<5.27002e+08){r+=-0.185178;}else{r+=0.481805;}}}}}else{if(f[5]<4.12037e+08){if(f[26]<1.49013e+09){r+=-0.582698;}else{r+=-0.183352;}}else{r+=0.231712;}}}}else{if(f[15]<8.38986e+06){if(f[0]<1.40235e+11){r+=-0.420474;}else{r+=0.639233;}}else{if(f[40]<4.70778e+09){if(f[1]<1.10504e+07){r+=0.250081;}else{if(f[18]<2.3702e+08){if(f[74]<26.5735){if(f[78]<0.247437){r+=-0.249781;}else{r+=-0.598648;}}else{r+=-0.101286;}}else{r+=-0.0777332;}}}else{if(f[34]<1.65796e+08){if(f[52]<1.34906){r+=0.455162;}else{r+=0.0108251;}}else{if(f[10]<2.5531e+09){r+=-0.405626;}else{r+=-0.039642;}}}}}}}else{if(f[0]<1.40159e+11){if(f[6]<7.99913e+06){if(f[14]<1.40296e+11){if(f[26]<3.76868e+08){if(f[28]<1.39941e+11){if(f[14]<1.40035e+11){if(f[9]<1.13682e+08){r+=-0.549572;}else{r+=0.360065;}}else{if(f[14]<1.40064e+11){r+=0.592854;}else{r+=-0.418838;}}}else{if(f[28]<1.40008e+11){if(f[0]<1.40146e+11){if(f[14]<1.40164e+11){r+=-0.573779;}else{r+=0.737148;}}else{if(f[8]<2.81037e+07){r+=0.0738934;}else{r+=0.525232;}}}else{if(f[14]<1.3998e+11){if(f[56]<0.99553){r+=-0.526059;}else{r+=0.661337;}}else{if(f[13]<3.83402e+07){r+=0.60115;}else{r+=-0.589441;}}}}}else{if(f[28]<1.4014e+11){if(f[14]<1.40036e+11){if(f[32]<1.07449e+07){r+=-0.594545;}else{r+=0.383954;}}else{if(f[14]<1.40114e+11){if(f[68]<1.33868){r+=-0.036936;}else{r+=0.453903;}}else{if(f[55]<1.16735){r+=-0.520293;}else{r+=0.00726215;}}}}else{if(f[66]<0.325003){if(f[42]<1.00062){if(f[14]<1.40097e+11){r+=0.277769;}else{r+=-0.368405;}}else{r+=-0.427662;}}else{if(f[38]<7.71118e+09){if(f[63]<2.3376){r+=-0.626812;}else{r+=-0.171158;}}else{r+=-0.073031;}}}}}else{if(f[56]<1.00138){if(f[54]<0.390256){r+=0.0305591;}else{if(f[14]<1.40844e+11){if(f[33]<44917.6){r+=-0.0173124;}else{if(f[0]<1.40098e+11){r+=-0.0881606;}else{r+=-0.582822;}}}else{r+=0.0225523;}}}else{if(f[56]<1.00379){if(f[70]<0.999107){if(f[22]<1.21826e+07){r+=-0.677161;}else{r+=0.241552;}}else{if(f[35]<1.21908e+07){if(f[28]<1.40028e+11){r+=0.188103;}else{r+=0.60167;}}else{if(f[22]<2.32187e+07){r+=-0.262313;}else{r+=0.583606;}}}}else{r+=-0.56924;}}}}else{if(f[28]<1.4009e+11){if(f[0]<1.40151e+11){if(f[14]<1.4002e+11){if(f[14]<1.39898e+11){if(f[73]<0.240616){r+=-0.350201;}else{if(f[13]<1.24251e+08){r+=-0.0889046;}else{r+=0.588131;}}}else{if(f[23]<1.43144e+09){if(f[0]<1.40149e+11){r+=-0.599973;}else{r+=-0.139542;}}else{r+=-0.0300779;}}}else{if(f[28]<1.39962e+11){if(f[14]<1.40083e+11){if(f[28]<1.39956e+11){r+=0.594016;}else{r+=1.3642;}}else{if(f[56]<1.00352){r+=-0.345839;}else{r+=0.445857;}}}else{if(f[14]<1.40165e+11){if(f[11]<6.08002e+09){r+=-0.454995;}else{r+=1.03954;}}else{if(f[56]<1.00247){r+=0.587131;}else{r+=-0.470833;}}}}}else{if(f[10]<2.57738e+08){if(f[68]<0.617606){if(f[34]<6.01892e+06){r+=-0.438961;}else{r+=0.0140655;}}else{if(f[14]<1.39755e+11){r+=-0.126118;}else{if(f[55]<1.92834){r+=0.571351;}else{r+=-0.0823132;}}}}else{if(f[81]<1.02009){if(f[28]<1.40048e+11){if(f[43]<0.89576){r+=0.0797714;}else{r+=-0.463955;}}else{r+=0.520732;}}else{if(f[1]<9.29125e+07){r+=-0.602182;}else{r+=0.187166;}}}}}else{if(f[56]<1.0004){if(f[51]<2.21736){if(f[41]<2.21355e+09){if(f[28]<1.41919e+11){if(f[64]<0.334306){r+=-0.330347;}else{r+=-0.556562;}}else{if(f[14]<1.40587e+11){r+=-0.352377;}else{r+=0.551086;}}}else{if(f[14]<1.40092e+11){if(f[56]<0.998963){r+=-0.57855;}else{r+=0.332785;}}else{if(f[41]<4.27538e+09){r+=-0.480967;}else{r+=-0.212537;}}}}else{if(f[14]<1.40092e+11){if(f[14]<1.4006e+11){if(f[14]<1.39871e+11){r+=0.713467;}else{r+=-0.422411;}}else{if(f[66]<0.297085){r+=0.501396;}else{r+=0.236876;}}}else{if(f[15]<3.92931e+07){if(f[14]<1.40097e+11){r+=-0.274429;}else{r+=-0.5774;}}else{if(f[78]<1.92281){r+=-0.264054;}else{r+=0.175748;}}}}}else{if(f[28]<1.40379e+11){if(f[56]<1.00118){if(f[80]<1.03611){if(f[53]<0.707402){r+=0.127655;}else{r+=-0.589692;}}else{r+=0.312921;}}else{if(f[15]<1.7232e+08){if(f[49]<4.97876){r+=0.610189;}else{r+=-0.091708;}}else{r+=-0.611376;}}}else{if(f[54]<0.499055){if(f[68]<1.52861){if(f[10]<6.81122e+08){r+=0.595275;}else{r+=-0.183269;}}else{if(f[34]<1.67905e+07){r+=-0.435459;}else{r+=0.202505;}}}else{if(f[28]<1.40405e+11){if(f[14]<1.40512e+11){r+=-0.447553;}else{r+=-0.0740672;}}else{if(f[36]<4.55854e+08){r+=-0.585295;}else{r+=0.039594;}}}}}}}}else{if(f[0]<1.40251e+11){if(f[14]<1.39676e+11){if(f[0]<1.40251e+11){if(f[9]<4.04422e+08){if(f[14]<1.39484e+11){if(f[0]<1.4025e+11){r+=-0.618442;}else{r+=-0.188649;}}else{r+=-0.0817385;}}else{if(f[56]<0.991119){r+=-0.599856;}else{if(f[38]<2.51428e+08){r+=1.18118;}else{r+=-0.319179;}}}}else{r+=0.00711927;}}else{if(f[39]<5.48248e+07){if(f[56]<1.00115){if(f[28]<1.40144e+11){if(f[70]<0.998712){r+=-0.924589;}else{if(f[28]<1.40108e+11){r+=0.0205307;}else{r+=0.601489;}}}else{if(f[28]<1.40566e+11){if(f[70]<0.999735){r+=0.389124;}else{r+=-0.636411;}}else{if(f[1]<9.3828e+06){r+=0.434563;}else{r+=-0.612402;}}}}else{if(f[28]<1.39821e+11){if(f[56]<1.00348){if(f[28]<1.39797e+11){r+=-0.514626;}else{r+=0.515635;}}else{if(f[34]<3.04314e+07){r+=-0.702275;}else{r+=0.0858272;}}}else{if(f[55]<1.84791){if(f[14]<1.40617e+11){r+=0.391122;}else{r+=-0.188089;}}else{if(f[28]<1.39853e+11){r+=0.657467;}else{r+=-0.509961;}}}}}else{if(f[10]<3.44381e+08){if(f[52]<1.88821){if(f[9]<1.62898e+07){if(f[14]<1.40292e+11){r+=-0.56244;}else{r+=0.148038;}}else{if(f[55]<1.39729){r+=0.427973;}else{r+=0.214744;}}}else{if(f[28]<1.39872e+11){if(f[56]<1.00363){r+=0.567182;}else{r+=-0.697129;}}else{if(f[42]<0.998901){r+=0.120465;}else{r+=-0.58554;}}}}else{if(f[0]<1.40178e+11){if(f[28]<1.40058e+11){if(f[26]<5.15715e+08){r+=0.251991;}else{r+=-0.65032;}}else{if(f[56]<0.999281){r+=-0.583762;}else{r+=0.0781497;}}}else{if(f[1]<3.94318e+07){if(f[56]<1.00369){r+=0.362195;}else{r+=-0.560089;}}else{if(f[36]<1.45473e+08){r+=-0.538489;}else{r+=0.0393713;}}}}}}}else{if(f[14]<1.39066e+11){if(f[10]<2.29553e+08){if(f[55]<0.905786){if(f[56]<0.99105){if(f[52]<1.23865){r+=-0.749554;}else{r+=-0.162401;}}else{if(f[38]<1.85011e+08){if(f[63]<1.88642){r+=0.760034;}else{r+=-0.597528;}}else{if(f[41]<2.39027e+08){r+=-0.972774;}else{r+=0.294617;}}}}else{if(f[38]<2.12757e+08){if(f[56]<0.993045){if(f[0]<1.41285e+11){r+=-0.858897;}else{r+=-0.612126;}}else{if(f[38]<1.89166e+08){r+=0.630867;}else{r+=-1.03607;}}}else{if(f[28]<1.40017e+11){if(f[28]<1.3995e+11){r+=-1.12156;}else{r+=-1.50754;}}else{if(f[29]<2.03987e+07){r+=-0.132645;}else{r+=-0.747196;}}}}}else{if(f[28]<1.40114e+11){if(f[14]<1.38856e+11){if(f[0]<1.40419e+11){if(f[42]<1.01073){r+=0.404767;}else{r+=-0.927378;}}else{if(f[27]<2.03497e+08){r+=-0.714215;}else{r+=0.608217;}}}else{if(f[9]<2.14473e+08){if(f[43]<1.6832){r+=-0.802495;}else{r+=0.349741;}}else{if(f[0]<1.41626e+11){r+=0.825699;}else{r+=0.519277;}}}}else{if(f[27]<2.43868e+08){if(f[56]<0.984793){r+=-0.671592;}else{r+=-0.897632;}}else{if(f[70]<0.992519){if(f[28]<1.41123e+11){r+=0.495477;}else{r+=-0.401786;}}else{if(f[56]<0.990658){r+=-0.687427;}else{r+=0.270085;}}}}}}else{if(f[14]<1.39541e+11){if(f[41]<2.79174e+08){if(f[0]<1.42664e+11){if(f[28]<1.40066e+11){if(f[80]<0.785467){r+=-0.437287;}else{r+=-1.21485;}}else{if(f[0]<1.41358e+11){r+=-0.745959;}else{r+=-0.923959;}}}else{if(f[1]<1.1009e+08){r+=0.587935;}else{r+=0.125069;}}}else{if(f[26]<1.59309e+08){if(f[56]<0.995716){if(f[9]<2.19122e+08){r+=-0.610574;}else{r+=0.82098;}}else{if(f[14]<1.39479e+11){r+=-1.09216;}else{r+=-0.043188;}}}else{if(f[28]<1.39852e+11){if(f[41]<5.52709e+08){r+=-1.13674;}else{r+=-0.74902;}}else{if(f[51]<1.32797){r+=-0.631765;}else{r+=0.0165038;}}}}}else{if(f[14]<1.39876e+11){if(f[0]<1.40828e+11){if(f[14]<1.39649e+11){if(f[28]<1.41372e+11){r+=-0.395294;}else{r+=0.584919;}}else{if(f[66]<0.418608){r+=-0.55286;}else{r+=0.772351;}}}else{if(f[0]<1.43494e+11){if(f[28]<1.42024e+11){r+=-0.805639;}else{r+=0.429634;}}else{if(f[28]<1.40082e+11){r+=-0.523671;}else{r+=0.605771;}}}}else{if(f[14]<1.40137e+11){if(f[28]<1.72068e+11){if(f[70]<0.98969){r+=-0.189548;}else{r+=-0.745873;}}else{r+=0.59797;}}else{if(f[28]<1.39954e+11){if(f[14]<1.40347e+11){r+=-0.0597089;}else{r+=-0.624806;}}else{if(f[0]<1.40714e+11){r+=-0.202023;}else{r+=0.202925;}}}}}}}}}}}else{if(f[28]<1.80075e+11){if(f[68]<7.99466){if(f[28]<1.40629e+11){if(f[42]<0.998357){if(f[14]<1.4053e+11){if(f[14]<1.40464e+11){if(f[14]<1.4027e+11){if(f[73]<0.314282){r+=-0.583367;}else{r+=0.0756862;}}else{if(f[0]<1.40002e+11){if(f[28]<1.39847e+11){if(f[56]<1.00383){r+=0.281897;}else{r+=-0.482163;}}else{if(f[26]<2.4311e+08){r+=0.508514;}else{r+=-0.0795909;}}}else{if(f[38]<1.42897e+07){if(f[52]<0.905936){r+=0.348017;}else{r+=-0.437513;}}else{if(f[51]<1.41534){r+=-0.0102638;}else{r+=0.550466;}}}}}else{if(f[70]<1.00246){r+=-0.276404;}else{if(f[36]<1.66541e+07){r+=-0.0141722;}else{r+=0.631817;}}}}else{if(f[10]<4.43946e+09){if(f[35]<7.09351e+06){r+=0.167913;}else{if(f[40]<1.54886e+09){r+=-0.667322;}else{r+=-0.10877;}}}else{if(f[6]<4.17232e+08){r+=0.626459;}else{r+=-0.132477;}}}}else{if(f[27]<1.77221e+09){if(f[14]<1.40296e+11){if(f[28]<1.39535e+11){r+=0.570179;}else{if(f[66]<3.44312){if(f[55]<1.16781){r+=-0.54463;}else{r+=0.430492;}}else{if(f[13]<7.84524e+08){r+=-0.604866;}else{if(f[8]<1.12363e+08){r+=0.275986;}else{r+=-0.496783;}}}}}else{if(f[14]<1.40297e+11){r+=0.708512;}else{if(f[28]<1.40628e+11){if(f[0]<1.40149e+11){r+=0.162668;}else{if(f[72]<10.4024){r+=-0.620295;}else{r+=-0.00995425;}}}else{r+=0.54384;}}}}else{if(f[78]<1.19768){if(f[0]<1.404e+11){if(f[51]<0.148363){r+=-0.407948;}else{if(f[50]<1.58015){if(f[20]<2.62036e+06){r+=0.148533;}else{r+=0.678175;}}else{r+=-0.279168;}}}else{if(f[68]<6.77959){r+=-0.594835;}else{r+=0.137733;}}}else{if(f[68]<2.92934){r+=0.180106;}else{if(f[78]<1.39165){r+=-0.164452;}else{r+=-0.632672;}}}}}}else{if(f[21]<1.81711e+09){if(f[66]<23.0463){if(f[32]<1.62286e+08){r+=-0.596007;}else{r+=-0.106806;}}else{r+=-0.0875873;}}else{if(f[39]<4.70217e+08){r+=-0.675104;}else{if(f[0]<1.40247e+11){r+=-0.546564;}else{if(f[78]<1.19749){if(f[56]<0.995487){r+=0.809801;}else{r+=0.0549127;}}else{r+=-0.207225;}}}}}}else{if(f[13]<5.93235e+09){if(f[66]<3.54034){if(f[23]<6.90459e+07){if(f[11]<1.25359e+08){r+=-0.17244;}else{r+=0.552202;}}else{if(f[0]<1.41838e+11){r+=-0.585995;}else{r+=0.121576;}}}else{if(f[68]<10.4474){if(f[28]<1.39928e+11){if(f[41]<3.03179e+07){if(f[11]<7.97362e+07){r+=-0.320057;}else{if(f[49]<1.89239){r+=0.602044;}else{r+=0.0984892;}}}else{if(f[0]<1.39999e+11){r+=0.168435;}else{if(f[66]<4.54364){r+=-0.14788;}else{r+=-0.585414;}}}}else{if(f[14]<1.42259e+11){if(f[13]<3.72321e+09){if(f[64]<0.953033){if(f[18]<1.54602e+06){r+=0.283955;}else{r+=-0.484471;}}else{if(f[68]<10.4354){r+=-0.590873;}else{r+=-0.151199;}}}else{if(f[60]<1.96472){r+=0.509786;}else{r+=-0.422993;}}}else{r+=0.187069;}}}else{if(f[72]<21.8359){if(f[41]<5.37708e+09){if(f[14]<1.4009e+11){if(f[56]<0.999923){if(f[28]<1.53448e+11){r+=-0.65533;}else{r+=-0.174067;}}else{if(f[13]<8.0651e+08){r+=0.31684;}else{r+=-0.592222;}}}else{if(f[39]<1.61922e+09){if(f[14]<1.40091e+11){r+=-0.162794;}else{r+=-0.594073;}}else{r+=-0.0994676;}}}else{if(f[1]<5.54839e+07){r+=-0.372495;}else{r+=0.117524;}}}else{r+=-0.0331116;}}}}else{if(f[36]<5.96863e+08){if(f[4]<6.8654e+06){r+=-0.163995;}else{r+=-0.575983;}}else{if(f[9]<1.83736e+10){r+=0.593663;}else{r+=-0.273266;}}}}}else{if(f[14]<1.40148e+11){r+=0.596675;}else{r+=-0.577311;}}}} return r;}
}

/*
 * Free FFT and convolution (Java)
 *
 * Copyright (c) 2014 Project Nayuki
 * http://www.nayuki.io/page/free-small-fft-in-multiple-languages
 *
 * (MIT License)
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 * - The above copyright notice and this permission notice shall be included in
 *   all copies or substantial portions of the Software.
 * - The Software is provided "as is", without warranty of any kind, express or
 *   implied, including but not limited to the warranties of merchantability,
 *   fitness for a particular purpose and noninfringement. In no event shall the
 *   authors or copyright holders be liable for any claim, damages or other
 *   liability, whether in an action of contract, tort or otherwise, arising from,
 *   out of or in connection with the Software or the use or other dealings in the
 *   Software.
 */


class Fft {

    /*
     * Computes the discrete Fourier transform (DFT) of the given complex vector, storing the result back into the vector.
     * The vector can have any length. This is a wrapper function.
     */
    public static void transform(double[] real, double[] imag) {
        if (real.length != imag.length)
            throw new IllegalArgumentException("Mismatched lengths");

        int n = real.length;
        if (n == 0)
            return;
        else if ((n & (n - 1)) == 0)  // Is power of 2
            transformRadix2(real, imag);
        else  // More complicated algorithm for arbitrary sizes
            transformBluestein(real, imag);
    }


    /*
     * Computes the inverse discrete Fourier transform (IDFT) of the given complex vector, storing the result back into the vector.
     * The vector can have any length. This is a wrapper function. This transform does not perform scaling, so the inverse is not a true inverse.
     */
    public static void inverseTransform(double[] real, double[] imag) {
        transform(imag, real);
    }


    /*
     * Computes the discrete Fourier transform (DFT) of the given complex vector, storing the result back into the vector.
     * The vector's length must be a power of 2. Uses the Cooley-Tukey decimation-in-time radix-2 algorithm.
     */
    public static void transformRadix2(double[] real, double[] imag) {
        // Initialization
        if (real.length != imag.length)
            throw new IllegalArgumentException("Mismatched lengths");
        int n = real.length;
        int levels = 31 - Integer.numberOfLeadingZeros(n);  // Equal to floor(log2(n))
        if (1 << levels != n)
            throw new IllegalArgumentException("Length is not a power of 2");
        double[] cosTable = new double[n / 2];
        double[] sinTable = new double[n / 2];
        for (int i = 0; i < n / 2; i++) {
            cosTable[i] = Math.cos(2 * Math.PI * i / n);
            sinTable[i] = Math.sin(2 * Math.PI * i / n);
        }

        // Bit-reversed addressing permutation
        for (int i = 0; i < n; i++) {
            int j = Integer.reverse(i) >>> (32 - levels);
            if (j > i) {
                double temp = real[i];
                real[i] = real[j];
                real[j] = temp;
                temp = imag[i];
                imag[i] = imag[j];
                imag[j] = temp;
            }
        }

        // Cooley-Tukey decimation-in-time radix-2 FFT
        for (int size = 2; size <= n; size *= 2) {
            int halfsize = size / 2;
            int tablestep = n / size;
            for (int i = 0; i < n; i += size) {
                for (int j = i, k = 0; j < i + halfsize; j++, k += tablestep) {
                    double tpre =  real[j+halfsize] * cosTable[k] + imag[j+halfsize] * sinTable[k];
                    double tpim = -real[j+halfsize] * sinTable[k] + imag[j+halfsize] * cosTable[k];
                    real[j + halfsize] = real[j] - tpre;
                    imag[j + halfsize] = imag[j] - tpim;
                    real[j] += tpre;
                    imag[j] += tpim;
                }
            }
            if (size == n)  // Prevent overflow in 'size *= 2'
                break;
        }
    }


    /*
     * Computes the discrete Fourier transform (DFT) of the given complex vector, storing the result back into the vector.
     * The vector can have any length. This requires the convolution function, which in turn requires the radix-2 FFT function.
     * Uses Bluestein's chirp z-transform algorithm.
     */
    public static void transformBluestein(double[] real, double[] imag) {
        // Find a power-of-2 convolution length m such that m >= n * 2 + 1
        if (real.length != imag.length)
            throw new IllegalArgumentException("Mismatched lengths");
        int n = real.length;
        if (n >= 0x20000000)
            throw new IllegalArgumentException("Array too large");
        int m = Integer.highestOneBit(n * 2 + 1) << 1;

        // Trignometric tables
        double[] cosTable = new double[n];
        double[] sinTable = new double[n];
        for (int i = 0; i < n; i++) {
            int j = (int)((long)i * i % (n * 2));  // This is more accurate than j = i * i
            cosTable[i] = Math.cos(Math.PI * j / n);
            sinTable[i] = Math.sin(Math.PI * j / n);
        }

        // Temporary vectors and preprocessing
        double[] areal = new double[m];
        double[] aimag = new double[m];
        for (int i = 0; i < n; i++) {
            areal[i] =  real[i] * cosTable[i] + imag[i] * sinTable[i];
            aimag[i] = -real[i] * sinTable[i] + imag[i] * cosTable[i];
        }
        double[] breal = new double[m];
        double[] bimag = new double[m];
        breal[0] = cosTable[0];
        bimag[0] = sinTable[0];
        for (int i = 1; i < n; i++) {
            breal[i] = breal[m - i] = cosTable[i];
            bimag[i] = bimag[m - i] = sinTable[i];
        }

        // Convolution
        double[] creal = new double[m];
        double[] cimag = new double[m];
        convolve(areal, aimag, breal, bimag, creal, cimag);

        // Postprocessing
        for (int i = 0; i < n; i++) {
            real[i] =  creal[i] * cosTable[i] + cimag[i] * sinTable[i];
            imag[i] = -creal[i] * sinTable[i] + cimag[i] * cosTable[i];
        }
    }


    /*
     * Computes the circular convolution of the given real vectors. Each vector's length must be the same.
     */
    public static void convolve(double[] x, double[] y, double[] out) {
        if (x.length != y.length || x.length != out.length)
            throw new IllegalArgumentException("Mismatched lengths");
        int n = x.length;
        convolve(x, new double[n], y, new double[n], out, new double[n]);
    }


    /*
     * Computes the circular convolution of the given complex vectors. Each vector's length must be the same.
     */
    public static void convolve(double[] xreal, double[] ximag, double[] yreal, double[] yimag, double[] outreal, double[] outimag) {
        if (xreal.length != ximag.length || xreal.length != yreal.length || yreal.length != yimag.length || xreal.length != outreal.length || outreal.length != outimag.length)
            throw new IllegalArgumentException("Mismatched lengths");

        int n = xreal.length;
        xreal = xreal.clone();
        ximag = ximag.clone();
        yreal = yreal.clone();
        yimag = yimag.clone();

        transform(xreal, ximag);
        transform(yreal, yimag);
        for (int i = 0; i < n; i++) {
            double temp = xreal[i] * yreal[i] - ximag[i] * yimag[i];
            ximag[i] = ximag[i] * yreal[i] + xreal[i] * yimag[i];
            xreal[i] = temp;
        }
        inverseTransform(xreal, ximag);
        for (int i = 0; i < n; i++) {  // Scaling (because this FFT implementation omits it)
            outreal[i] = xreal[i] / n;
            outimag[i] = ximag[i] / n;
        }
    }

}